/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.auth.provider.ldap;

import com.yammer.metrics.core.MetricName;
import io.confluent.security.auth.provider.ldap.LdapConfig;
import io.confluent.security.auth.provider.ldap.LdapContextCreator;
import io.confluent.security.auth.provider.ldap.LdapException;
import io.confluent.security.auth.store.data.UserKey;
import io.confluent.security.auth.store.data.UserValue;
import io.confluent.security.auth.store.external.ExternalStoreListener;
import io.confluent.security.auth.utils.MetricsUtils;
import io.confluent.security.auth.utils.RetryBackoff;
import io.confluent.security.authorizer.provider.ProviderFailedException;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.naming.InterruptedNamingException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.PartialResultException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.Control;
import javax.naming.ldap.HasControls;
import javax.naming.ldap.LdapContext;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.PagedResultsControl;
import javax.naming.ldap.PagedResultsResponseControl;
import javax.naming.ldap.Rdn;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LdapGroupManager {
    private static final Logger log = LoggerFactory.getLogger(LdapGroupManager.class);
    private static final String METRIC_GROUP = "confluent.metadata";
    private static final String METRIC_TYPE = LdapGroupManager.class.getSimpleName();
    private static final int CLOSE_TIMEOUT_MS = 30000;
    private final LdapConfig config;
    private final Time time;
    private final LdapContextCreator contextCreator;
    private final Map<String, Set<String>> userGroupCache;
    private final ScheduledExecutorService executorService;
    private final ResultEntryConfig resultEntryConfig;
    private final SearchControls searchControls;
    private final RetryBackoff retryBackoff;
    private final AtomicLong failureStartMs;
    private final AtomicInteger retryCount;
    private final AtomicBoolean alive;
    private final PersistentSearch persistentSearch;
    private final ExternalStoreListener<UserKey, UserValue> listener;
    private final Set<MetricName> metricNames;
    private volatile LdapContext context;
    private volatile Future<?> searchFuture;

    public LdapGroupManager(LdapConfig config, Time time) {
        this(config, time, null);
    }

    public LdapGroupManager(LdapConfig config, Time time, ExternalStoreListener<UserKey, UserValue> listener) {
        this.config = config;
        this.time = time;
        this.listener = listener;
        this.contextCreator = new LdapContextCreator(config);
        this.userGroupCache = new ConcurrentHashMap<String, Set<String>>();
        this.persistentSearch = config.persistentSearch ? new PersistentSearch() : null;
        this.searchControls = new SearchControls();
        switch (config.searchMode) {
            case GROUPS: {
                this.searchControls.setReturningAttributes(new String[]{config.groupNameAttribute, config.groupMemberAttribute});
                this.resultEntryConfig = new ResultEntryConfig(config.groupNameAttribute, config.groupNameAttributePattern, config.groupMemberAttribute, config.groupMemberAttributePattern);
                this.searchControls.setSearchScope(config.groupSearchScope);
                break;
            }
            case USERS: {
                this.searchControls.setReturningAttributes(new String[]{config.userNameAttribute, config.userMemberOfAttribute});
                this.resultEntryConfig = new ResultEntryConfig(config.userNameAttribute, config.userNameAttributePattern, config.userMemberOfAttribute, config.userMemberOfAttributePattern);
                this.searchControls.setSearchScope(config.userSearchScope);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported search mode " + (Object)((Object)config.searchMode));
            }
        }
        this.alive = new AtomicBoolean(true);
        this.failureStartMs = new AtomicLong(0L);
        this.retryCount = new AtomicInteger(0);
        this.retryBackoff = new RetryBackoff(config.retryBackoffMs, config.retryMaxBackoffMs);
        this.metricNames = new HashSet<MetricName>();
        this.metricNames.add(MetricsUtils.newGauge(METRIC_GROUP, METRIC_TYPE, "failure-start-seconds-ago", Collections.emptyMap(), () -> MetricsUtils.elapsedSeconds(time, this.failureStartMs.get())));
        this.executorService = Executors.newSingleThreadScheduledExecutor(runnable -> {
            Thread thread = new Thread(runnable, "ldap-group-manager");
            thread.setDaemon(true);
            return thread;
        });
        log.info("LDAP group manager created with config: {}", (Object)config);
    }

    public void start() {
        log.info("Starting LDAP group manager");
        boolean done = false;
        do {
            try {
                if (this.listener != null) {
                    Set<String> currentSearchEntries = this.searchAndProcessResults(false);
                    this.listener.start();
                    this.removeDeletedEntries(currentSearchEntries);
                } else {
                    this.searchAndProcessResults(false);
                }
                done = true;
            }
            catch (InterruptedNamingException e) {
                throw new RuntimeException("LDAP group manager has been Interrupted", e);
            }
            catch (Throwable e) {
                try {
                    if (this.failed()) {
                        throw e;
                    }
                    int backoffMs = this.processFailureAndGetBackoff(e);
                    Thread.sleep(backoffMs);
                }
                catch (Throwable t) {
                    throw new LdapException("Ldap group manager initialization failed", t);
                }
            }
            if (this.alive.get()) continue;
            throw new RuntimeException("LDAP group manager has been shutdown");
        } while (!done);
        this.onStartup();
        if (this.config.persistentSearch) {
            this.schedulePersistentSearch(0L, false);
        } else {
            this.schedulePeriodicSearch(this.config.refreshIntervalMs, this.config.refreshIntervalMs);
        }
        log.info("Startup of LDAP group manager completed");
    }

    public void close() {
        this.alive.set(false);
        if (this.searchFuture != null) {
            this.searchFuture.cancel(true);
        }
        this.executorService.shutdownNow();
        try {
            this.executorService.awaitTermination(30000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            log.debug("LdapGroupManager.close() was interrupted", (Throwable)e);
        }
        try {
            if (this.context != null) {
                this.context.close();
            }
        }
        catch (NamingException e) {
            log.debug("Could not close LDAP context", (Throwable)e);
        }
        MetricsUtils.removeMetrics(this.metricNames);
    }

    public Set<String> groups(String userPrincipal) {
        if (!this.alive.get()) {
            throw new IllegalStateException("LDAP Group manager is not active");
        }
        if (this.failed()) {
            throw new ProviderFailedException("LDAP Group provider has failed");
        }
        return this.userGroupCache.getOrDefault(userPrincipal, Collections.emptySet());
    }

    public boolean failed() {
        long failedMs = this.failureStartMs.get();
        return failedMs != 0L && this.time.milliseconds() > failedMs + this.config.retryTimeoutMs;
    }

    private void resetFailure() {
        if (this.retryCount.getAndSet(0) != 0) {
            log.info("LDAP search succeeded, resetting failed status");
        }
        this.failureStartMs.set(0L);
        if (this.listener != null) {
            this.listener.resetFailure();
        }
    }

    private int processFailureAndGetBackoff(Throwable exception) {
        if (!this.alive.get()) {
            return 0;
        }
        this.failureStartMs.compareAndSet(0L, this.time.milliseconds());
        if (this.failed()) {
            log.error("LDAP search failed. Configured retry timeout of " + this.config.retryTimeoutMs + " has expired without a successful search. All requests will fail authorization until the next successful search.", exception);
            if (this.listener != null) {
                this.listener.fail("LDAP search failed with exception: " + exception);
            }
        } else {
            log.error("LDAP search failed, search will be retried. Groups from the last successful search will continue to be applied until the configured retry timeout or the next successful search.", exception);
        }
        try {
            if (this.searchFuture != null) {
                this.searchFuture.cancel(false);
            }
            if (this.context != null) {
                this.context.close();
            }
        }
        catch (Exception e) {
            log.error("Context could not be closed", (Throwable)e);
        }
        this.context = null;
        return this.retryBackoff.backoffMs(this.retryCount.getAndIncrement());
    }

    private void persistentSearch() throws NamingException, IOException {
        if (this.context == null) {
            this.context = this.contextCreator.createLdapContext();
        }
        try {
            this.context.setRequestControls(new Control[]{this.persistentSearch.control});
            this.searchControls.setTimeLimit(0);
        }
        catch (Exception e) {
            throw new LdapException("Request controls could not be created");
        }
        log.trace("Starting persistent search");
        NamingEnumeration<SearchResult> enumeration = this.search(this.searchControls);
        this.resetFailure();
        while (this.alive.get()) {
            this.processPersistentSearchResults(enumeration);
        }
    }

    private void schedulePersistentSearch(long initialDelayMs, boolean initializeCache) {
        log.trace("Scheduling persistent search, initialDelayMs={}, initializeCache={}", (Object)initialDelayMs, (Object)initializeCache);
        this.searchFuture = this.executorService.schedule(() -> {
            try {
                if (initializeCache) {
                    this.searchAndProcessResults();
                }
                this.persistentSearch();
            }
            catch (Throwable e) {
                int backoffMs = this.processFailureAndGetBackoff(e);
                this.schedulePersistentSearch(backoffMs, true);
            }
        }, initialDelayMs, TimeUnit.MILLISECONDS);
    }

    private void schedulePeriodicSearch(long initialDelayMs, long refreshIntervalMs) {
        log.trace("Scheduling periodic search with initialDelayMs={}, refreshIntervalMs {}", (Object)initialDelayMs, (Object)refreshIntervalMs);
        this.searchFuture = this.executorService.scheduleWithFixedDelay(() -> {
            try {
                this.searchAndProcessResults();
            }
            catch (Throwable e) {
                int backoffMs = this.processFailureAndGetBackoff(e);
                this.schedulePeriodicSearch(backoffMs, this.config.refreshIntervalMs);
            }
        }, initialDelayMs, refreshIntervalMs, TimeUnit.MILLISECONDS);
    }

    protected Set<String> searchAndProcessResults() throws NamingException, IOException {
        return this.searchAndProcessResults(true);
    }

    protected Set<String> searchAndProcessResults(boolean removeDeletedEntries) throws NamingException, IOException {
        if (this.context == null) {
            this.context = this.contextCreator.createLdapContext();
            this.maybeSetPagingControl(null);
        }
        HashSet<String> currentSearchEntries = new HashSet<String>();
        byte[] cookie = null;
        do {
            NamingEnumeration<SearchResult> enumeration = this.search(this.searchControls);
            currentSearchEntries.addAll(this.processFullSearchResults(enumeration));
            this.resetFailure();
            Control[] responseControls = this.context.getResponseControls();
            if (this.config.searchPageSize > 0 && responseControls != null) {
                for (Control responseControl : responseControls) {
                    if (responseControl instanceof PagedResultsResponseControl) {
                        PagedResultsResponseControl pc = (PagedResultsResponseControl)responseControl;
                        cookie = pc.getCookie();
                        log.debug("Search returned page, totalSize {}", (Object)pc.getResultSize());
                        break;
                    }
                    log.debug("Ignoring response control {}", (Object)responseControl);
                }
            }
            this.maybeSetPagingControl(cookie);
        } while (cookie != null);
        if (removeDeletedEntries) {
            this.removeDeletedEntries(currentSearchEntries);
        }
        log.debug("Search completed, group cache is {}", this.userGroupCache);
        return currentSearchEntries;
    }

    private void removeDeletedEntries(Set<String> currentSearchEntries) {
        HashSet<String> prevSearchEntries = new HashSet<String>();
        if (this.config.searchMode == LdapConfig.SearchMode.USERS) {
            prevSearchEntries.addAll(this.userGroupCache.keySet());
        } else {
            this.userGroupCache.values().forEach(prevSearchEntries::addAll);
        }
        prevSearchEntries.stream().filter(name -> !currentSearchEntries.contains(name)).forEach(this::processSearchResultDelete);
    }

    private void onUpdate(Set<String> users) {
        if (this.listener != null) {
            users.forEach(user -> {
                Set<String> groups = this.userGroupCache.get(user);
                if (groups != null) {
                    this.listener.update(this.userKey((String)user), this.userValue(groups));
                } else {
                    this.listener.delete(this.userKey((String)user));
                }
            });
        }
    }

    private void onStartup() {
        if (this.listener != null) {
            this.listener.initialize(this.userGroupCache.entrySet().stream().collect(Collectors.toMap(e -> this.userKey((String)e.getKey()), e -> this.userValue((Set)e.getValue()))));
        }
    }

    private UserKey userKey(String user) {
        return new UserKey(new KafkaPrincipal("User", user));
    }

    private UserValue userValue(Set<String> groups) {
        return new UserValue((Collection)groups.stream().map(group -> new KafkaPrincipal("Group", group)).collect(Collectors.toSet()));
    }

    private NamingEnumeration<SearchResult> search(SearchControls searchControls) throws NamingException {
        if (this.config.searchMode == LdapConfig.SearchMode.GROUPS) {
            log.trace("Searching groups with base {} filter {}: ", (Object)this.config.groupSearchBase, (Object)this.config.groupSearchFilter);
            return this.context.search(this.config.groupSearchBase, this.config.groupSearchFilter, searchControls);
        }
        log.trace("Searching users with base {} filter {}: ", (Object)this.config.userSearchBase, (Object)this.config.userSearchFilter);
        return this.context.search(this.config.userSearchBase, this.config.userSearchFilter, searchControls);
    }

    private Set<String> processFullSearchResults(NamingEnumeration<SearchResult> enumeration) throws NamingException {
        HashSet<String> currentSearchEntries;
        block3: {
            currentSearchEntries = new HashSet<String>();
            try {
                while (enumeration.hasMore()) {
                    SearchResult searchResult = enumeration.next();
                    log.trace("Processing full search result {}", (Object)searchResult);
                    ResultEntry resultEntry = this.searchResultEntry(searchResult);
                    if (resultEntry == null) continue;
                    currentSearchEntries.add(resultEntry.name);
                    this.processSearchResultModify(resultEntry);
                }
            }
            catch (PartialResultException e) {
                log.debug(Arrays.toString(e.getStackTrace()));
                if (this.ignorePartialResultException()) break block3;
                throw e;
            }
        }
        return currentSearchEntries;
    }

    private void processPersistentSearchResults(NamingEnumeration<SearchResult> enumeration) throws NamingException {
        block15: {
            try {
                while (enumeration.hasMore()) {
                    SearchResult searchResult = enumeration.next();
                    log.trace("Processing search result {}", (Object)searchResult);
                    ResultEntry resultEntry = this.searchResultEntry(searchResult);
                    if (resultEntry == null) continue;
                    Control changeResponseControl = null;
                    if (searchResult instanceof HasControls) {
                        Control[] controls;
                        for (Control control : controls = ((HasControls)((Object)searchResult)).getControls()) {
                            if (this.persistentSearch.isEntryChangeResponseControl(control)) {
                                changeResponseControl = control;
                                log.debug("Entry change search response control {}", (Object)control);
                                continue;
                            }
                            log.debug("Ignoring search response control {}", (Object)control);
                        }
                    }
                    ChangeType changeType = changeResponseControl != null ? this.persistentSearch.changeType(changeResponseControl) : ChangeType.MODIFY;
                    switch (changeType) {
                        case ADD: 
                        case MODIFY: {
                            this.processSearchResultModify(resultEntry);
                            break;
                        }
                        case DELETE: {
                            this.processSearchResultDelete(resultEntry.name);
                            break;
                        }
                        case RENAME: {
                            String previousDn = this.persistentSearch.previousDn(changeResponseControl);
                            Pattern pattern = this.config.searchMode == LdapConfig.SearchMode.GROUPS ? this.config.groupDnNamePattern : this.config.userDnNamePattern;
                            String previousName = null;
                            if (pattern == null) {
                                List<Rdn> rdns = new LdapName(previousDn).getRdns();
                                for (Rdn rdn : rdns) {
                                    if (!this.resultEntryConfig.nameAttribute.equals(rdn.getType())) continue;
                                    previousName = (String)rdn.getValue();
                                }
                            } else {
                                previousName = LdapGroupManager.attributeValue(previousDn, pattern, "", "rename entry", this.config.searchMode);
                            }
                            if (previousName != null) {
                                this.processSearchResultDelete(previousName);
                            }
                            this.processSearchResultModify(resultEntry);
                            break;
                        }
                        default: {
                            throw new IllegalArgumentException("Unsupported response control type " + (Object)((Object)changeType));
                        }
                    }
                    log.debug("Group cache after change notification is {}", this.userGroupCache);
                }
            }
            catch (PartialResultException e) {
                log.debug(Arrays.toString(e.getStackTrace()));
                if (this.ignorePartialResultException()) break block15;
                throw e;
            }
        }
    }

    private void processSearchResultModify(ResultEntry resultEntry) {
        HashSet<String> updatedUsers = new HashSet<String>();
        if (resultEntry != null) {
            if (this.config.searchMode == LdapConfig.SearchMode.GROUPS) {
                String group = resultEntry.name;
                Set<String> members = resultEntry.members;
                for (String string : members) {
                    Set groups = this.userGroupCache.computeIfAbsent(string, u -> new HashSet());
                    if (groups.contains(group)) continue;
                    groups.add(group);
                    updatedUsers.add(string);
                }
                for (Map.Entry entry : this.userGroupCache.entrySet()) {
                    String user = (String)entry.getKey();
                    Set userGroups = (Set)entry.getValue();
                    if (!userGroups.contains(group) || members.contains(user)) continue;
                    userGroups.remove(group);
                    if (userGroups.isEmpty()) {
                        this.userGroupCache.remove(user);
                    }
                    updatedUsers.add(user);
                }
            } else {
                String user = resultEntry.name;
                Set<String> groups = resultEntry.members;
                Set<String> oldValue = this.userGroupCache.put(user, groups);
                if (oldValue == null || !oldValue.equals(groups)) {
                    updatedUsers.add(user);
                }
            }
        }
        this.onUpdate(updatedUsers);
    }

    private void processSearchResultDelete(String name) {
        if (this.config.searchMode == LdapConfig.SearchMode.GROUPS) {
            this.processSearchResultModify(new ResultEntry(name, Collections.emptySet()));
        } else {
            this.userGroupCache.remove(name);
            this.onUpdate(Collections.singleton(name));
        }
    }

    private ResultEntry searchResultEntry(SearchResult searchResult) throws NamingException {
        Attributes attributes = searchResult.getAttributes();
        Attribute nameAttr = attributes.get(this.resultEntryConfig.nameAttribute);
        if (nameAttr != null) {
            String name = LdapGroupManager.attributeValue(nameAttr.get(), this.resultEntryConfig.nameAttributePattern, "", "search result", this.config.searchMode);
            if (name == null) {
                return null;
            }
            HashSet<String> members = new HashSet<String>();
            Attribute memberAttr = attributes.get(this.resultEntryConfig.memberAttribute);
            if (memberAttr != null) {
                NamingEnumeration<?> attrs = memberAttr.getAll();
                while (attrs.hasMore()) {
                    Object member = attrs.next();
                    String memberName = LdapGroupManager.attributeValue(member, this.resultEntryConfig.memberAttributePattern, name, "member", this.config.searchMode);
                    if (memberName == null) continue;
                    members.add(memberName);
                }
            }
            return new ResultEntry(name, members);
        }
        return null;
    }

    private void maybeSetPagingControl(byte[] cookie) {
        try {
            if (this.config.searchPageSize > 0) {
                PagedResultsControl control = new PagedResultsControl(this.config.searchPageSize, cookie, cookie != null);
                this.context.setRequestControls(new Control[]{control});
            }
        }
        catch (IOException | NamingException e) {
            log.warn("Paging control could not be set", (Throwable)e);
        }
    }

    private boolean ignorePartialResultException() {
        return this.config.ignorePartialResultException;
    }

    static String attributeValue(Object value, Pattern pattern, String parent, String attrDesc, LdapConfig.SearchMode searchMode) {
        if (value == null) {
            log.error("Ignoring null {} in LDAP {} {}", new Object[]{attrDesc, searchMode, parent});
            return null;
        }
        if (pattern == null) {
            return String.valueOf(value);
        }
        Matcher matcher = pattern.matcher(value.toString());
        if (!matcher.matches()) {
            log.debug("Ignoring {} in LDAP {} {} that doesn't match pattern: {}", new Object[]{attrDesc, searchMode, parent, value});
            return null;
        }
        return matcher.group(1);
    }

    private static class PersistentSearch {
        final Control control;
        private Class<? extends Control> entryChangeResponseControlClass;
        private final Method changeTypeMethod;
        private final Method previousDnMethod;

        PersistentSearch() {
            try {
                Class controlClass = Utils.loadClass((String)"com.sun.jndi.ldap.PersistentSearchControl", Control.class);
                Constructor constructor = controlClass.getConstructor(Integer.TYPE, Boolean.TYPE, Boolean.TYPE, Boolean.TYPE);
                this.control = (Control)constructor.newInstance(15, false, true, true);
                this.entryChangeResponseControlClass = Utils.loadClass((String)"com.sun.jndi.ldap.EntryChangeResponseControl", Control.class);
                this.changeTypeMethod = this.entryChangeResponseControlClass.getMethod("getChangeType", new Class[0]);
                this.previousDnMethod = this.entryChangeResponseControlClass.getMethod("getPreviousDN", new Class[0]);
            }
            catch (Exception e) {
                throw new ConfigException("Persistent search could not be enabled", (Object)e);
            }
        }

        ChangeType changeType(Control changeResponseControl) {
            try {
                Integer changeTypeValue = (Integer)this.changeTypeMethod.invoke((Object)changeResponseControl, new Object[0]);
                for (ChangeType type : ChangeType.values()) {
                    if (type.value != changeTypeValue) continue;
                    return type;
                }
                return ChangeType.UNKNOWN;
            }
            catch (IllegalAccessException | InvocationTargetException e) {
                throw new LdapException("Could not get change type", e);
            }
        }

        String previousDn(Control changeResponseControl) {
            try {
                return (String)this.previousDnMethod.invoke((Object)changeResponseControl, new Object[0]);
            }
            catch (IllegalAccessException | InvocationTargetException e) {
                throw new LdapException("Could not get change type", e);
            }
        }

        boolean isEntryChangeResponseControl(Control control) {
            return this.entryChangeResponseControlClass.isInstance(control);
        }
    }

    private static class ResultEntry {
        final String name;
        final Set<String> members;

        ResultEntry(String name, Set<String> members) {
            this.name = name;
            this.members = members;
        }
    }

    private static class ResultEntryConfig {
        final String nameAttribute;
        final Pattern nameAttributePattern;
        final String memberAttribute;
        final Pattern memberAttributePattern;

        ResultEntryConfig(String nameAttribute, Pattern nameAttributePattern, String memberAttribute, Pattern memberAttributePattern) {
            this.nameAttribute = nameAttribute;
            this.nameAttributePattern = nameAttributePattern;
            this.memberAttribute = memberAttribute;
            this.memberAttributePattern = memberAttributePattern;
        }
    }

    private static enum ChangeType {
        ADD(1),
        DELETE(2),
        MODIFY(4),
        RENAME(8),
        UNKNOWN(-1);

        final int value;

        private ChangeType(int value) {
            this.value = value;
        }
    }
}

