/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.storage.checksum;

import com.google.flatbuffers.FlatBufferBuilder;
import io.confluent.kafka.storage.checksum.ChecksumAlgorithm;
import io.confluent.kafka.storage.checksum.ChecksumHeader;
import io.confluent.kafka.storage.checksum.ChecksumInfo;
import io.confluent.kafka.storage.checksum.ChecksumStoreReaderWriter;
import io.confluent.kafka.storage.checksum.E2EChecksumMetrics;
import io.confluent.kafka.storage.checksum.serdes.Entry;
import io.confluent.kafka.storage.checksum.serdes.Header;
import io.confluent.kafka.storage.checksum.serdes.Store;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.zip.Checksum;
import org.apache.kafka.common.utils.Checksums;
import org.apache.kafka.common.utils.Crc32C;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ChecksumStore {
    private static final Logger LOG = LoggerFactory.getLogger(ChecksumStore.class);
    protected final long entryTTL;
    private final Time time;
    private final ChecksumStoreReaderWriter storeReaderWriter;
    private final E2EChecksumMetrics metrics;
    protected ChecksumHeader header;
    private int storeBufferInitLength = 0x100000;
    private Map<String, ChecksumInfo> entries = new ConcurrentHashMap<String, ChecksumInfo>();

    protected ChecksumStore(ChecksumHeader header, long entryTTL, Time time, ChecksumStoreReaderWriter storeReaderWriter, E2EChecksumMetrics metrics) {
        this.entryTTL = entryTTL;
        this.header = header;
        this.time = time;
        this.storeReaderWriter = storeReaderWriter;
        this.metrics = metrics;
    }

    public synchronized void setByteBufferStoreInitLength(int length) {
        this.storeBufferInitLength = length;
    }

    public synchronized void clear() {
        this.entries.clear();
    }

    protected abstract ChecksumAlgorithm algorithm();

    public E2EChecksumMetrics metrics() {
        return this.metrics;
    }

    private ChecksumInfo emptyEntry(boolean shouldPersist) {
        return new ChecksumInfo(this.emptyChecksum(), 0L, this.time.milliseconds(), shouldPersist);
    }

    public abstract Checksum emptyChecksum();

    protected abstract Checksum convertLongToChecksum(long var1);

    public boolean update(String key, ByteBuffer buffer) {
        return this.update(key, buffer, this.time.milliseconds());
    }

    public boolean update(String key, ByteBuffer buffer, long updatedAtMs) {
        Optional<ChecksumInfo> checksumInfoOpt = this.get(key);
        if (checksumInfoOpt.isPresent()) {
            ChecksumInfo checksumInfo = checksumInfoOpt.get();
            int bufferLength = buffer.remaining();
            Checksums.update((Checksum)checksumInfo.checksum(), (ByteBuffer)buffer, (int)bufferLength);
            this.put(key, new ChecksumInfo(checksumInfo.checksum(), checksumInfo.sizeInBytes() + (long)bufferLength, updatedAtMs, checksumInfo.shouldPersist()));
        }
        return checksumInfoOpt.isPresent();
    }

    public boolean update(String key, int value) {
        return this.update(key, value, this.time.milliseconds());
    }

    public boolean update(String key, int value, long updatedAtMs) {
        Optional<ChecksumInfo> checksumInfoOpt = this.get(key);
        if (checksumInfoOpt.isPresent()) {
            ChecksumInfo checksumInfo = checksumInfoOpt.get();
            Checksums.updateInt((Checksum)checksumInfo.checksum(), (int)value);
            this.put(key, new ChecksumInfo(checksumInfo.checksum(), checksumInfo.sizeInBytes() + 4L, updatedAtMs, checksumInfo.shouldPersist()));
        }
        return checksumInfoOpt.isPresent();
    }

    public boolean update(String key, long value) {
        return this.update(key, value, this.time.milliseconds());
    }

    public boolean update(String key, long value, long updatedAtMs) {
        Optional<ChecksumInfo> checksumInfoOpt = this.get(key);
        if (checksumInfoOpt.isPresent()) {
            ChecksumInfo checksumInfo = checksumInfoOpt.get();
            Checksums.updateLong((Checksum)checksumInfo.checksum(), (long)value);
            this.put(key, new ChecksumInfo(checksumInfo.checksum(), checksumInfo.sizeInBytes() + 8L, updatedAtMs, checksumInfo.shouldPersist()));
        }
        return checksumInfoOpt.isPresent();
    }

    public boolean update(String key, int value1, int value2, long updatedAtMs) {
        Optional<ChecksumInfo> checksumInfoOpt = this.get(key);
        if (checksumInfoOpt.isPresent()) {
            ChecksumInfo checksumInfo = checksumInfoOpt.get();
            Checksums.updateInt((Checksum)checksumInfo.checksum(), (int)value1);
            Checksums.updateInt((Checksum)checksumInfo.checksum(), (int)value2);
            this.put(key, new ChecksumInfo(checksumInfo.checksum(), checksumInfo.sizeInBytes() + 4L + 4L, updatedAtMs, checksumInfo.shouldPersist()));
        }
        return checksumInfoOpt.isPresent();
    }

    public boolean update(String key, long value1, int value2, long updatedAtMs) {
        Optional<ChecksumInfo> checksumInfoOpt = this.get(key);
        if (checksumInfoOpt.isPresent()) {
            ChecksumInfo checksumInfo = checksumInfoOpt.get();
            Checksums.updateLong((Checksum)checksumInfo.checksum(), (long)value1);
            Checksums.updateInt((Checksum)checksumInfo.checksum(), (int)value2);
            this.put(key, new ChecksumInfo(checksumInfo.checksum(), checksumInfo.sizeInBytes() + 8L + 4L, updatedAtMs, checksumInfo.shouldPersist()));
        }
        return checksumInfoOpt.isPresent();
    }

    protected abstract Checksum truncate(Checksum var1, ByteBuffer var2);

    public boolean truncate(String key, long oldSizeBytes, ByteBuffer truncatedBytes) {
        Optional<ChecksumInfo> checksumInfoOpt = this.get(key);
        if (!checksumInfoOpt.isPresent()) {
            LOG.debug("key : {} is not present in checksum store, skipping truncation", (Object)key);
            return false;
        }
        ChecksumInfo checksumInfo = checksumInfoOpt.get();
        if (checksumInfo.sizeInBytes() != oldSizeBytes) {
            LOG.error("Actual size provided : {} for {} is not equal to the size registered in the checksum store : {} , removing the entry to avoid the usage of incorrect checksum ", new Object[]{oldSizeBytes, key, checksumInfo});
            this.remove(key);
            return false;
        }
        if (oldSizeBytes - (long)truncatedBytes.remaining() == 0L) {
            this.initializeEntry(key);
            return true;
        }
        return this.truncate(key, checksumInfo, truncatedBytes);
    }

    private boolean truncate(String key, ChecksumInfo checksumInfo, ByteBuffer truncatedBytes) {
        long newSize = checksumInfo.sizeInBytes() - (long)truncatedBytes.remaining();
        Checksum truncatedChecksum = this.truncate(checksumInfo.checksum(), truncatedBytes);
        this.put(key, new ChecksumInfo(truncatedChecksum, newSize, this.time.milliseconds(), checksumInfo.shouldPersist()));
        return true;
    }

    public void initializeEntry(String key) {
        this.initializeEntry(key, true);
    }

    public void initializeEntry(String key, boolean shouldPersist) {
        LOG.debug("Initializing entry for {} in checksum store", (Object)key);
        this.put(key, this.emptyEntry(shouldPersist));
    }

    public boolean contains(String key) {
        return this.get(key).isPresent();
    }

    public Optional<ChecksumInfo> get(String key) {
        if (key == null) {
            return Optional.empty();
        }
        return Optional.ofNullable(this.entries.get(key));
    }

    private void put(String key, ChecksumInfo checksumInfo) {
        if (key == null) {
            LOG.error("Adding null key to checksum store is not permitted");
            return;
        }
        this.entries.put(key, checksumInfo);
    }

    public Optional<ChecksumInfo> remove(String key) {
        if (key == null || !this.contains(key)) {
            return Optional.empty();
        }
        LOG.debug("Removing entry for {} from checksum store", (Object)key);
        return Optional.ofNullable(this.entries.remove(key));
    }

    public void replace(String oldKey, String newKey) {
        Optional<ChecksumInfo> checksumInfo = this.remove(oldKey);
        checksumInfo.ifPresent(info -> this.put(newKey, (ChecksumInfo)info));
    }

    public int size() {
        return this.entries.size();
    }

    public void recover() throws IOException {
        ByteBuffer buffer = this.storeReaderWriter.read();
        this.deserialize(buffer);
    }

    public void checkpoint() throws IOException {
        ByteBuffer buffer = this.serialize();
        this.metrics().recordStoreFileSize(buffer.remaining());
        this.storeReaderWriter.write(buffer);
    }

    public ByteBuffer serialize() {
        FlatBufferBuilder builder = new FlatBufferBuilder(this.storeBufferInitLength).forceDefaults(true);
        int headerOffset = Header.createHeader(builder, this.header.version(), this.header.algorithm().value());
        ArrayList<Integer> entryOffsets = new ArrayList<Integer>();
        for (Map.Entry<String, ChecksumInfo> entry : this.entries.entrySet()) {
            String key = entry.getKey();
            ChecksumInfo value = entry.getValue();
            if (!value.shouldPersist()) continue;
            long entryChecksum = this.calculateEntryChecksum(key, value);
            int keyOffset = builder.createString((CharSequence)key);
            int entryOffset = Entry.createEntry(builder, keyOffset, value.checksum().getValue(), value.sizeInBytes(), value.lastModifiedMs(), entryChecksum);
            entryOffsets.add(entryOffset);
        }
        int entryVectorOffset = Store.createEntriesVector(builder, entryOffsets.stream().mapToInt(i -> i).toArray());
        int storeOffset = Store.createStore(builder, headerOffset, entryVectorOffset);
        builder.finish(storeOffset);
        return builder.dataBuffer().slice();
    }

    public synchronized void deserialize(ByteBuffer buffer) {
        if (!buffer.hasRemaining()) {
            LOG.warn("provided buffer is empty , nothing to deserialize. number of entries in the store : {}", (Object)this.entries.size());
            return;
        }
        Store store = Store.getRootAsStore(buffer);
        Header hdr = store.header();
        ConcurrentHashMap<String, ChecksumInfo> newEntries = new ConcurrentHashMap<String, ChecksumInfo>();
        for (int i = 0; i < store.entriesLength(); ++i) {
            Entry entry = store.entries(i);
            ChecksumInfo info = new ChecksumInfo(this.convertLongToChecksum(entry.checksum()), entry.sizeInBytes(), entry.lastModifiedMs(), true);
            if (this.validEntry(entry.key(), info, entry.entryChecksum())) {
                newEntries.put(entry.key(), info);
                continue;
            }
            LOG.warn("Ignoring entry as validation failed while loading , entry : {}", (Object)info);
        }
        this.header = new ChecksumHeader(hdr.version(), ChecksumAlgorithm.fromValue(hdr.algorithm()));
        this.entries = newEntries;
        LOG.info("Deserialization completed and checksum store state restored with {} entries.", (Object)this.entries.size());
    }

    private boolean validEntry(String key, ChecksumInfo checksumInfo, long expectedCrc) {
        return this.isEntryChecksumValid(key, checksumInfo, expectedCrc) && this.isRecentlyUpdated(checksumInfo);
    }

    private boolean isRecentlyUpdated(ChecksumInfo checksumInfo) {
        return this.time.milliseconds() - checksumInfo.lastModifiedMs() < this.entryTTL;
    }

    private boolean isEntryChecksumValid(String key, ChecksumInfo checksumInfo, long expectedCrc) {
        long actualCrc = this.calculateEntryChecksum(key, checksumInfo);
        return actualCrc == expectedCrc;
    }

    protected long calculateEntryChecksum(String key, ChecksumInfo checksumInfo) {
        Checksum crc = Crc32C.create();
        ByteBuffer keyBuffer = ByteBuffer.wrap(key.getBytes());
        Checksums.update((Checksum)crc, (ByteBuffer)keyBuffer, (int)keyBuffer.remaining());
        Checksums.updateLong((Checksum)crc, (long)checksumInfo.checksum().getValue());
        Checksums.updateLong((Checksum)crc, (long)checksumInfo.sizeInBytes());
        Checksums.updateLong((Checksum)crc, (long)checksumInfo.lastModifiedMs());
        return crc.getValue();
    }
}

