/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import io.confluent.kafka.availability.FilesWrapper;
import io.confluent.kafka.availability.ThreadCountersManager;
import io.confluent.kafka.storage.checksum.ChecksumParams;
import io.confluent.kafka.storage.checksum.E2EChecksumProtectedObjectType;
import io.confluent.kafka.storage.checksum.E2EChecksumStore;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.protocol.types.ArrayOf;
import org.apache.kafka.common.protocol.types.Field;
import org.apache.kafka.common.protocol.types.Schema;
import org.apache.kafka.common.protocol.types.SchemaException;
import org.apache.kafka.common.protocol.types.Struct;
import org.apache.kafka.common.protocol.types.Type;
import org.apache.kafka.common.utils.ByteUtils;
import org.apache.kafka.common.utils.Crc32C;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.storage.internals.log.AppendOrigin;
import org.apache.kafka.storage.internals.log.BatchMetadata;
import org.apache.kafka.storage.internals.log.CompletedTxn;
import org.apache.kafka.storage.internals.log.CorruptSnapshotException;
import org.apache.kafka.storage.internals.log.LogFileUtils;
import org.apache.kafka.storage.internals.log.LogOffsetMetadata;
import org.apache.kafka.storage.internals.log.ProducerAppendInfo;
import org.apache.kafka.storage.internals.log.ProducerIdQuotaRecorder;
import org.apache.kafka.storage.internals.log.ProducerStateEntry;
import org.apache.kafka.storage.internals.log.ProducerStateManagerConfig;
import org.apache.kafka.storage.internals.log.SnapshotFile;
import org.apache.kafka.storage.internals.log.TxnMetadata;
import org.apache.kafka.storage.internals.log.VerificationStateEntry;
import org.slf4j.Logger;

public class ProducerStateManager {
    public static final long LATE_TRANSACTION_BUFFER_MS = 300000L;
    private static final short PRODUCER_SNAPSHOT_VERSION = 1;
    private static final String VERSION_FIELD = "version";
    private static final String CRC_FIELD = "crc";
    private static final String PRODUCER_ID_FIELD = "producer_id";
    private static final String LAST_SEQUENCE_FIELD = "last_sequence";
    private static final String PRODUCER_EPOCH_FIELD = "epoch";
    private static final String LAST_OFFSET_FIELD = "last_offset";
    private static final String OFFSET_DELTA_FIELD = "offset_delta";
    private static final String TIMESTAMP_FIELD = "timestamp";
    private static final String PRODUCER_ENTRIES_FIELD = "producer_entries";
    private static final String COORDINATOR_EPOCH_FIELD = "coordinator_epoch";
    private static final String CURRENT_TXN_FIRST_OFFSET_FIELD = "current_txn_first_offset";
    private static final int VERSION_OFFSET = 0;
    private static final int CRC_OFFSET = 2;
    private static final int PRODUCER_ENTRIES_OFFSET = 6;
    private static final Schema PRODUCER_SNAPSHOT_ENTRY_SCHEMA = new Schema(new Field[]{new Field("producer_id", (Type)Type.INT64, "The producer ID"), new Field("epoch", (Type)Type.INT16, "Current epoch of the producer"), new Field("last_sequence", (Type)Type.INT32, "Last written sequence of the producer"), new Field("last_offset", (Type)Type.INT64, "Last written offset of the producer"), new Field("offset_delta", (Type)Type.INT32, "The difference of the last sequence and first sequence in the last written batch"), new Field("timestamp", (Type)Type.INT64, "Max timestamp from the last written entry"), new Field("coordinator_epoch", (Type)Type.INT32, "The epoch of the last transaction coordinator to send an end transaction marker"), new Field("current_txn_first_offset", (Type)Type.INT64, "The first offset of the on-going transaction (-1 if there is none)")});
    private static final Schema PID_SNAPSHOT_MAP_SCHEMA = new Schema(new Field[]{new Field("version", (Type)Type.INT16, "Version of the snapshot file"), new Field("crc", (Type)Type.UNSIGNED_INT32, "CRC of the snapshot data"), new Field("producer_entries", (Type)new ArrayOf((Type)PRODUCER_SNAPSHOT_ENTRY_SCHEMA), "The entries in the producer table")});
    private final Logger log;
    private final TopicPartition topicPartition;
    private final Optional<String> tenantIdOpt;
    private final int maxTransactionTimeoutMs;
    private final ProducerStateManagerConfig producerStateManagerConfig;
    private final Time time;
    private Optional<ProducerIdQuotaRecorder> producerIdQuotaRecorder;
    private final Map<Long, ProducerStateEntry> producers = new HashMap<Long, ProducerStateEntry>();
    private final Map<Long, VerificationStateEntry> verificationStates = new HashMap<Long, VerificationStateEntry>();
    private final TreeMap<Long, TxnMetadata> ongoingTxns = new TreeMap();
    private final TreeMap<Long, TxnMetadata> unreplicatedTxns = new TreeMap();
    private volatile File logDir;
    private volatile int producerIdCount = 0;
    private volatile long oldestTxnStartTimestamp = -1L;
    private ConcurrentSkipListMap<Long, SnapshotFile> snapshots;
    private long lastMapOffset = 0L;
    private long lastSnapOffset = 0L;
    private final Optional<E2EChecksumStore> checksumStoreOpt;
    private final boolean e2eChecksumEnabledForTopic;

    public ProducerStateManager(TopicPartition topicPartition, File logDir, int maxTransactionTimeoutMs, ProducerStateManagerConfig producerStateManagerConfig, Time time, Optional<ProducerIdQuotaRecorder> producerIdQuotaRecorder, ChecksumParams checksumParams) throws IOException {
        this.topicPartition = topicPartition;
        this.tenantIdOpt = Optional.ofNullable(ProducerStateManager.extractTenantPrefix(topicPartition.topic()));
        this.logDir = logDir;
        this.maxTransactionTimeoutMs = maxTransactionTimeoutMs;
        this.producerStateManagerConfig = producerStateManagerConfig;
        this.time = time;
        this.log = new LogContext("[ProducerStateManager partition=" + topicPartition + "] ").logger(ProducerStateManager.class);
        this.snapshots = this.loadSnapshots();
        this.producerIdQuotaRecorder = producerIdQuotaRecorder;
        this.checksumStoreOpt = checksumParams.checksumStoreOpt();
        this.e2eChecksumEnabledForTopic = checksumParams.e2eChecksumEnabledForTopic();
    }

    public int maxTransactionTimeoutMs() {
        return this.maxTransactionTimeoutMs;
    }

    public ProducerStateManagerConfig producerStateManagerConfig() {
        return this.producerStateManagerConfig;
    }

    public Optional<ProducerIdQuotaRecorder> producerIdQuotaRecorder() {
        return this.producerIdQuotaRecorder;
    }

    public boolean hasLateTransaction(long currentTimeMs) {
        long lastTimestamp = this.oldestTxnStartTimestamp;
        return lastTimestamp > 0L && currentTimeMs - lastTimestamp > (long)this.maxTransactionTimeoutMs + 300000L;
    }

    public void truncateFullyAndReloadSnapshots() throws IOException {
        this.log.info("Reloading the producer state snapshots");
        this.truncateFullyAndStartAt(0L);
        this.snapshots = this.loadSnapshots();
    }

    public int producerIdCount() {
        return this.producerIdCount;
    }

    private void addProducerId(long producerId, ProducerStateEntry entry) {
        this.producers.put(producerId, entry);
        this.producerIdCount = this.producers.size();
    }

    private void removeProducerIds(List<Long> keys) {
        keys.forEach(this.producers::remove);
        this.producerIdCount = this.producers.size();
    }

    private void clearProducerIds() {
        this.producers.clear();
        this.producerIdCount = 0;
    }

    public void clearProducerIdQuotaRecorder() {
        this.maybeRecordProducerIdQuota(this.topicPartition.topic(), -1 * this.producers.size(), this.time.milliseconds());
        this.producerIdQuotaRecorder = Optional.empty();
    }

    public VerificationStateEntry maybeCreateVerificationStateEntry(long producerId, int sequence, short epoch) {
        VerificationStateEntry entry = this.verificationStates.computeIfAbsent(producerId, pid -> new VerificationStateEntry(this.time.milliseconds(), sequence, epoch));
        entry.maybeUpdateLowestSequenceAndEpoch(sequence, epoch);
        return entry;
    }

    public VerificationStateEntry verificationStateEntry(long producerId) {
        return this.verificationStates.get(producerId);
    }

    public void clearVerificationStateEntry(long producerId) {
        this.verificationStates.remove(producerId);
    }

    private ConcurrentSkipListMap<Long, SnapshotFile> loadSnapshots() throws IOException {
        ConcurrentSkipListMap<Long, SnapshotFile> offsetToSnapshots = new ConcurrentSkipListMap<Long, SnapshotFile>();
        List snapshotFiles = (List)ThreadCountersManager.wrapIOChecked(() -> ProducerStateManager.listSnapshotFiles(this.logDir, this.e2eChecksumEnabledForTopic));
        for (SnapshotFile snapshotFile : snapshotFiles) {
            offsetToSnapshots.put(snapshotFile.offset, snapshotFile);
        }
        return offsetToSnapshots;
    }

    public void removeStraySnapshots(Collection<Long> segmentBaseOffsets) throws IOException {
        SnapshotFile removedSnapshot;
        long maxOffset;
        long strayOffset;
        OptionalLong maxSegmentBaseOffset = segmentBaseOffsets.isEmpty() ? OptionalLong.empty() : OptionalLong.of(segmentBaseOffsets.stream().max(Long::compare).get());
        HashSet<Long> baseOffsets = new HashSet<Long>(segmentBaseOffsets);
        Optional<Object> latestStraySnapshot = Optional.empty();
        ConcurrentSkipListMap<Long, SnapshotFile> snapshots = this.loadSnapshots();
        for (SnapshotFile snapshot : snapshots.values()) {
            long key = snapshot.offset;
            if (latestStraySnapshot.isPresent()) {
                SnapshotFile prev = (SnapshotFile)latestStraySnapshot.get();
                if (baseOffsets.contains(key)) continue;
                prev.deleteIfExists(this.checksumStoreOpt);
                snapshots.remove(prev.offset);
                latestStraySnapshot = Optional.of(snapshot);
                continue;
            }
            if (baseOffsets.contains(key)) continue;
            latestStraySnapshot = Optional.of(snapshot);
        }
        if (latestStraySnapshot.isPresent() && maxSegmentBaseOffset.isPresent() && (strayOffset = ((SnapshotFile)latestStraySnapshot.get()).offset) < (maxOffset = maxSegmentBaseOffset.getAsLong()) && (removedSnapshot = snapshots.remove(strayOffset)) != null) {
            removedSnapshot.deleteIfExists(this.checksumStoreOpt);
        }
        this.snapshots = snapshots;
    }

    public void flushSnapshots(long fromOffset, long toOffset) throws IOException {
        if (toOffset < fromOffset) {
            throw new IllegalArgumentException("Invalid producer snapshot range: requested snapshots in " + this.topicPartition + " from offset " + fromOffset + " which is greater than limit offset " + toOffset);
        }
        for (SnapshotFile snapshotFile : this.snapshots.subMap((Object)fromOffset, true, (Object)toOffset, true).values()) {
            snapshotFile.flush();
            snapshotFile.close();
        }
    }

    public void flushSnapshots(long fromOffset) throws IOException {
        for (SnapshotFile snapshotFile : this.snapshots.tailMap((Object)fromOffset).values()) {
            snapshotFile.flush();
            snapshotFile.close();
        }
    }

    public Optional<LogOffsetMetadata> firstUnstableOffset() {
        Optional<LogOffsetMetadata> unreplicatedFirstOffset = Optional.ofNullable(this.unreplicatedTxns.firstEntry()).map(e -> ((TxnMetadata)e.getValue()).firstOffset);
        Optional<LogOffsetMetadata> undecidedFirstOffset = Optional.ofNullable(this.ongoingTxns.firstEntry()).map(e -> ((TxnMetadata)e.getValue()).firstOffset);
        if (!unreplicatedFirstOffset.isPresent()) {
            return undecidedFirstOffset;
        }
        if (!undecidedFirstOffset.isPresent()) {
            return unreplicatedFirstOffset;
        }
        if (undecidedFirstOffset.get().messageOffset < unreplicatedFirstOffset.get().messageOffset) {
            return undecidedFirstOffset;
        }
        return unreplicatedFirstOffset;
    }

    public void onHighWatermarkUpdated(long highWatermark) {
        this.removeUnreplicatedTransactions(highWatermark);
    }

    public OptionalLong firstUndecidedOffset() {
        Map.Entry<Long, TxnMetadata> firstEntry = this.ongoingTxns.firstEntry();
        return firstEntry != null ? OptionalLong.of(firstEntry.getValue().firstOffset.messageOffset) : OptionalLong.empty();
    }

    public long mapEndOffset() {
        return this.lastMapOffset;
    }

    public Map<Long, ProducerStateEntry> activeProducers() {
        return Collections.unmodifiableMap(this.producers);
    }

    public Map<Long, TxnMetadata> ongoingTxns() {
        return Collections.unmodifiableMap(this.ongoingTxns);
    }

    public boolean isEmpty() {
        return this.producers.isEmpty() && this.unreplicatedTxns.isEmpty();
    }

    private void loadFromSnapshot(long logStartOffset, long currentTimeMs) throws IOException {
        Optional<SnapshotFile> latestSnapshotFileOptional;
        int producerIdExpirationMs = this.producerStateManagerConfig.producerIdExpirationMs(this.tenantIdOpt);
        while ((latestSnapshotFileOptional = this.latestSnapshotFile()).isPresent()) {
            SnapshotFile snapshot = latestSnapshotFileOptional.get();
            try {
                this.log.info("Loading producer state from snapshot file '{}'", (Object)snapshot);
                Stream<ProducerStateEntry> loadedProducers = ProducerStateManager.readSnapshot(snapshot.file()).stream().filter(producerEntry -> !this.isProducerExpired(currentTimeMs, producerIdExpirationMs, (ProducerStateEntry)producerEntry));
                loadedProducers.forEach(producer -> this.loadProducerEntry((ProducerStateEntry)producer, currentTimeMs));
                this.lastMapOffset = this.lastSnapOffset = snapshot.offset;
                this.updateOldestTxnTimestamp();
                return;
            }
            catch (CorruptSnapshotException e) {
                this.log.warn("Failed to load producer snapshot from '{}': {}", (Object)snapshot.file(), (Object)e.getMessage());
                this.removeAndDeleteSnapshot(snapshot.offset);
            }
        }
        this.lastSnapOffset = logStartOffset;
        this.lastMapOffset = logStartOffset;
    }

    public void loadProducerEntry(ProducerStateEntry entry, long loadTimeMs) {
        long producerId = entry.producerId();
        if (!this.producers.containsKey(producerId)) {
            this.maybeRecordProducerIdQuota(this.topicPartition.topic(), 1.0, loadTimeMs);
        }
        this.addProducerId(producerId, entry);
        entry.currentTxnFirstOffset().ifPresent(offset -> this.ongoingTxns.put(offset, new TxnMetadata(producerId, offset, loadTimeMs)));
    }

    private boolean isProducerExpired(long currentTimeMs, int producerIdExpirationMs, ProducerStateEntry producerState) {
        return !producerState.currentTxnFirstOffset().isPresent() && currentTimeMs - producerState.lastTimestamp() >= (long)producerIdExpirationMs;
    }

    private void maybeRecordProducerIdQuota(String topicName, double value, long timeMs) {
        this.producerIdQuotaRecorder.ifPresent(quotaRecorder -> quotaRecorder.maybeRecord(topicName, value, timeMs));
    }

    public void removeExpiredProducers(long currentTimeMs) {
        int producerIdExpirationMs = this.producerStateManagerConfig.producerIdExpirationMs(this.tenantIdOpt);
        List<Long> expiredProducers = this.producers.entrySet().stream().filter(entry -> this.isProducerExpired(currentTimeMs, producerIdExpirationMs, (ProducerStateEntry)entry.getValue())).map(Map.Entry::getKey).collect(Collectors.toList());
        this.maybeRecordProducerIdQuota(this.topicPartition.topic(), -1 * expiredProducers.size(), currentTimeMs);
        this.removeProducerIds(expiredProducers);
        List<Long> verificationKeys = this.verificationStates.entrySet().stream().filter(entry -> currentTimeMs - ((VerificationStateEntry)entry.getValue()).timestamp() >= (long)producerIdExpirationMs).map(Map.Entry::getKey).collect(Collectors.toList());
        verificationKeys.forEach(this.verificationStates::remove);
    }

    public void truncateAndReload(long logStartOffset, long logEndOffset, long loadTimeMs) throws IOException {
        for (SnapshotFile snapshot : this.snapshots.values()) {
            if (snapshot.offset <= logEndOffset && snapshot.offset > logStartOffset) continue;
            this.removeAndDeleteSnapshot(snapshot.offset);
        }
        if (logEndOffset != this.mapEndOffset()) {
            this.maybeRecordProducerIdQuota(this.topicPartition.topic(), -1 * this.producers.size(), loadTimeMs);
            this.clearProducerIds();
            this.ongoingTxns.clear();
            this.updateOldestTxnTimestamp();
            this.unreplicatedTxns.clear();
            this.loadFromSnapshot(logStartOffset, loadTimeMs);
        } else {
            this.onLogStartOffsetIncremented(logStartOffset);
        }
    }

    public void reloadFromTieredSnapshot(long currentTimeMs, ByteBuffer snapshotBuffer, long snapshotOffset) throws IOException {
        if (!this.activeProducers().isEmpty()) {
            throw new IllegalStateException("expected producer state to be fully truncated before reloading tiered snapshot");
        }
        int producerIdExpirationMs = this.producerStateManagerConfig.producerIdExpirationMs(this.tenantIdOpt);
        try {
            List<ProducerStateEntry> loadedProducers = ProducerStateManager.readSnapshot(snapshotBuffer).stream().filter(producerEntry -> !this.isProducerExpired(currentTimeMs, producerIdExpirationMs, (ProducerStateEntry)producerEntry)).collect(Collectors.toList());
            this.log.info("Restored state for {} producers from tiered storage", (Object)loadedProducers.size());
            loadedProducers.forEach(producer -> this.loadProducerEntry((ProducerStateEntry)producer, currentTimeMs));
            this.lastMapOffset = snapshotOffset;
        }
        catch (CorruptSnapshotException e) {
            this.log.warn("Failed to load producer snapshot from buffer {}", (Object)e.getMessage());
            throw e;
        }
    }

    public ProducerAppendInfo prepareUpdate(long producerId, AppendOrigin origin, long currentTimeMs) {
        ProducerStateEntry currentEntry = this.lastEntry(producerId).orElse(ProducerStateEntry.empty(producerId));
        return new ProducerAppendInfo(this.topicPartition, producerId, currentEntry, origin, currentTimeMs, this.verificationStateEntry(producerId));
    }

    public void update(ProducerAppendInfo appendInfo) {
        if (appendInfo.producerId() == -1L) {
            throw new IllegalArgumentException("Invalid producer id " + appendInfo.producerId() + " passed to update for partition" + this.topicPartition);
        }
        this.log.trace("Updated producer {} state to {}", (Object)appendInfo.producerId(), (Object)appendInfo);
        ProducerStateEntry updatedEntry = appendInfo.toEntry();
        ProducerStateEntry currentEntry = this.producers.get(appendInfo.producerId());
        if (currentEntry != null) {
            currentEntry.update(updatedEntry);
        } else {
            this.maybeRecordProducerIdQuota(this.topicPartition.topic(), 1.0, this.time.milliseconds());
            this.addProducerId(appendInfo.producerId(), updatedEntry);
        }
        appendInfo.startedTransactions().forEach(txn -> this.ongoingTxns.put(txn.firstOffset.messageOffset, (TxnMetadata)txn));
        this.updateOldestTxnTimestamp();
    }

    private void updateOldestTxnTimestamp() {
        Map.Entry<Long, TxnMetadata> firstEntry = this.ongoingTxns.firstEntry();
        this.oldestTxnStartTimestamp = firstEntry == null ? -1L : firstEntry.getValue().startTimeUpperBoundMs;
    }

    public void updateMapEndOffset(long lastOffset) {
        this.lastMapOffset = lastOffset;
    }

    public Optional<ProducerStateEntry> lastEntry(long producerId) {
        return Optional.ofNullable(this.producers.get(producerId));
    }

    public void takeSnapshot() throws IOException {
        if (this.lastMapOffset > this.lastSnapOffset) {
            SnapshotFile snapshotFile = (SnapshotFile)ThreadCountersManager.wrapIO(() -> new SnapshotFile(LogFileUtils.producerSnapshotFile(this.logDir, this.lastMapOffset), this.e2eChecksumEnabledForTopic));
            long start = this.time.hiResClockMs();
            ThreadCountersManager.wrapIOChecked(() -> this.writeSnapshot(snapshotFile, this.producers));
            this.log.info("Wrote producer snapshot at offset {} with {} producer ids in {} ms.", new Object[]{this.lastMapOffset, this.producers.size(), this.time.hiResClockMs() - start});
            this.snapshots.put(snapshotFile.offset, snapshotFile);
            this.lastSnapOffset = this.lastMapOffset;
        }
    }

    public void updateParentDir(File parentDir) {
        this.logDir = parentDir;
        this.snapshots.forEach((k, v) -> v.updateParentDir(parentDir, this.checksumStoreOpt));
    }

    public OptionalLong latestSnapshotOffset() {
        Optional<SnapshotFile> snapshotFileOptional = this.latestSnapshotFile();
        return snapshotFileOptional.map(snapshotFile -> OptionalLong.of(snapshotFile.offset)).orElseGet(OptionalLong::empty);
    }

    public OptionalLong oldestSnapshotOffset() {
        Optional<SnapshotFile> snapshotFileOptional = this.oldestSnapshotFile();
        return snapshotFileOptional.map(snapshotFile -> OptionalLong.of(snapshotFile.offset)).orElseGet(OptionalLong::empty);
    }

    public void onLogStartOffsetIncremented(long logStartOffset) {
        this.removeUnreplicatedTransactions(logStartOffset);
        if (this.lastMapOffset < logStartOffset) {
            this.lastMapOffset = logStartOffset;
        }
        this.lastSnapOffset = this.latestSnapshotOffset().orElse(logStartOffset);
    }

    private void removeUnreplicatedTransactions(long offset) {
        Iterator<Map.Entry<Long, TxnMetadata>> iterator = this.unreplicatedTxns.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<Long, TxnMetadata> txnEntry = iterator.next();
            OptionalLong lastOffset = txnEntry.getValue().lastOffset;
            if (!lastOffset.isPresent() || lastOffset.getAsLong() >= offset) continue;
            iterator.remove();
        }
    }

    public void truncateFullyAndStartAt(long offset) throws IOException {
        this.maybeRecordProducerIdQuota(this.topicPartition.topic(), -1 * this.producers.size(), this.time.milliseconds());
        this.clearProducerIds();
        this.ongoingTxns.clear();
        this.unreplicatedTxns.clear();
        for (SnapshotFile snapshotFile : this.snapshots.values()) {
            this.removeAndDeleteSnapshot(snapshotFile.offset);
        }
        this.lastSnapOffset = 0L;
        this.lastMapOffset = offset;
        this.updateOldestTxnTimestamp();
    }

    public long proposedLastStableOffset(CompletedTxn completedTxn) {
        for (Map.Entry<Long, TxnMetadata> ongoingTxn : this.ongoingTxns.entrySet()) {
            if (ongoingTxn.getKey() == completedTxn.firstOffset) continue;
            return ongoingTxn.getValue().firstOffset.messageOffset;
        }
        return completedTxn.lastOffset + 1L;
    }

    public void completeTxn(CompletedTxn completedTxn) {
        TxnMetadata txnMetadata = this.ongoingTxns.remove(completedTxn.firstOffset);
        if (txnMetadata == null) {
            throw new IllegalArgumentException("Attempted to complete transaction " + completedTxn + " on partition " + this.topicPartition + " which was not started");
        }
        txnMetadata.lastOffset = OptionalLong.of(completedTxn.lastOffset);
        this.unreplicatedTxns.put(completedTxn.firstOffset, txnMetadata);
        this.updateOldestTxnTimestamp();
    }

    public void deleteSnapshotsBefore(long offset) throws IOException {
        for (SnapshotFile snapshot : this.snapshots.subMap((Object)0L, (Object)offset).values()) {
            this.removeAndDeleteSnapshot(snapshot.offset);
        }
    }

    public Optional<File> fetchSnapshot(long offset) {
        return Optional.ofNullable(this.snapshots.get(offset)).map(x -> x.file());
    }

    private Optional<SnapshotFile> oldestSnapshotFile() {
        return Optional.ofNullable(this.snapshots.firstEntry()).map(x -> (SnapshotFile)x.getValue());
    }

    private Optional<SnapshotFile> latestSnapshotFile() {
        return Optional.ofNullable(this.snapshots.lastEntry()).map(e -> (SnapshotFile)e.getValue());
    }

    public Optional<SnapshotFile> snapshotFileForOffset(long offset) {
        return Optional.ofNullable(this.snapshots.get(offset));
    }

    private void removeAndDeleteSnapshot(long snapshotOffset) throws IOException {
        SnapshotFile snapshotFile = this.snapshots.remove(snapshotOffset);
        if (snapshotFile != null) {
            snapshotFile.deleteIfExists(this.checksumStoreOpt);
        }
    }

    public Optional<SnapshotFile> removeAndMarkSnapshotForDeletion(long snapshotOffset) throws IOException {
        SnapshotFile snapshotFile = this.snapshots.remove(snapshotOffset);
        if (snapshotFile != null) {
            try {
                snapshotFile.close();
                snapshotFile.renameToDelete(this.checksumStoreOpt);
                return Optional.of(snapshotFile);
            }
            catch (NoSuchFileException ex) {
                this.log.info("Failed to rename producer state snapshot {} with deletion suffix because it was already deleted", (Object)snapshotFile.file().getAbsoluteFile());
            }
        }
        return Optional.empty();
    }

    public static List<ProducerStateEntry> readSnapshot(File file) throws IOException {
        byte[] buffer = FilesWrapper.readAllBytes((Path)file.toPath());
        return ProducerStateManager.readSnapshot(ByteBuffer.wrap(buffer));
    }

    public static List<ProducerStateEntry> readSnapshot(ByteBuffer buffer) throws IOException {
        try {
            long computedCrc;
            ByteBuffer crcBuffer = buffer.duplicate();
            int crcBufferSize = buffer.remaining();
            Struct struct = PID_SNAPSHOT_MAP_SCHEMA.read(buffer);
            Short version = struct.getShort(VERSION_FIELD);
            if (version != 1) {
                throw new CorruptSnapshotException("Snapshot contained an unknown file version " + version);
            }
            long crc = struct.getUnsignedInt(CRC_FIELD);
            if (crc != (computedCrc = Crc32C.compute((ByteBuffer)crcBuffer, (int)6, (int)(crcBufferSize - 6)))) {
                throw new CorruptSnapshotException("Snapshot is corrupt (CRC is no longer valid). Stored crc: " + crc + ". Computed crc: " + computedCrc);
            }
            Object[] producerEntryFields = struct.getArray(PRODUCER_ENTRIES_FIELD);
            ArrayList<ProducerStateEntry> entries = new ArrayList<ProducerStateEntry>(producerEntryFields.length);
            for (Object producerEntryObj : producerEntryFields) {
                Struct producerEntryStruct = (Struct)producerEntryObj;
                long producerId = producerEntryStruct.getLong(PRODUCER_ID_FIELD);
                short producerEpoch = producerEntryStruct.getShort(PRODUCER_EPOCH_FIELD);
                int seq = producerEntryStruct.getInt(LAST_SEQUENCE_FIELD);
                long offset = producerEntryStruct.getLong(LAST_OFFSET_FIELD);
                long timestamp = producerEntryStruct.getLong(TIMESTAMP_FIELD);
                int offsetDelta = producerEntryStruct.getInt(OFFSET_DELTA_FIELD);
                int coordinatorEpoch = producerEntryStruct.getInt(COORDINATOR_EPOCH_FIELD);
                long currentTxnFirstOffset = producerEntryStruct.getLong(CURRENT_TXN_FIRST_OFFSET_FIELD);
                OptionalLong currentTxnFirstOffsetVal = currentTxnFirstOffset >= 0L ? OptionalLong.of(currentTxnFirstOffset) : OptionalLong.empty();
                Optional<BatchMetadata> batchMetadata = offset >= 0L ? Optional.of(new BatchMetadata(seq, offset, offsetDelta, timestamp)) : Optional.empty();
                entries.add(new ProducerStateEntry(producerId, producerEpoch, coordinatorEpoch, timestamp, currentTxnFirstOffsetVal, batchMetadata));
            }
            return entries;
        }
        catch (SchemaException e) {
            throw new CorruptSnapshotException("Snapshot failed schema validation: " + e.getMessage());
        }
    }

    private void writeSnapshot(SnapshotFile snapshotFile, Map<Long, ProducerStateEntry> entries) throws IOException {
        Struct struct = new Struct(PID_SNAPSHOT_MAP_SCHEMA);
        struct.set(VERSION_FIELD, (Object)1);
        struct.set(CRC_FIELD, (Object)0L);
        Struct[] structEntries = new Struct[entries.size()];
        int i = 0;
        for (Map.Entry<Long, ProducerStateEntry> producerIdEntry : entries.entrySet()) {
            Long producerId = producerIdEntry.getKey();
            ProducerStateEntry entry = producerIdEntry.getValue();
            Struct producerEntryStruct = struct.instance(PRODUCER_ENTRIES_FIELD);
            producerEntryStruct.set(PRODUCER_ID_FIELD, (Object)producerId).set(PRODUCER_EPOCH_FIELD, (Object)entry.producerEpoch()).set(LAST_SEQUENCE_FIELD, (Object)entry.lastSeq()).set(LAST_OFFSET_FIELD, (Object)entry.lastDataOffset()).set(OFFSET_DELTA_FIELD, (Object)entry.lastOffsetDelta()).set(TIMESTAMP_FIELD, (Object)entry.lastTimestamp()).set(COORDINATOR_EPOCH_FIELD, (Object)entry.coordinatorEpoch()).set(CURRENT_TXN_FIRST_OFFSET_FIELD, (Object)entry.currentTxnFirstOffset().orElse(-1L));
            structEntries[i++] = producerEntryStruct;
        }
        struct.set(PRODUCER_ENTRIES_FIELD, (Object)structEntries);
        ByteBuffer buffer = ByteBuffer.allocate(struct.sizeOf());
        struct.writeTo(buffer);
        buffer.flip();
        long crc = Crc32C.compute((ByteBuffer)buffer, (int)6, (int)(buffer.limit() - 6));
        ByteUtils.writeUnsignedInt((ByteBuffer)buffer, (int)2, (long)crc);
        snapshotFile.maybeOpenForWrite();
        this.checksumStoreOpt.ifPresent(checksumStore -> this.mayUpdateChecksumStoreEntry((E2EChecksumStore)checksumStore, buffer.duplicate(), snapshotFile.file().getAbsolutePath()));
        snapshotFile.write(buffer);
    }

    private void mayUpdateChecksumStoreEntry(E2EChecksumStore checksumStore, ByteBuffer buffer, String key) {
        if (this.e2eChecksumEnabledForTopic && checksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.PRODUCER_STATE)) {
            checksumStore.store().initializeEntry(key);
            checksumStore.store().update(key, buffer);
        }
    }

    private static boolean isSnapshotFile(Path path) {
        return Files.isRegularFile(path, new LinkOption[0]) && path.getFileName().toString().endsWith(".snapshot");
    }

    public static List<SnapshotFile> listSnapshotFiles(File dir) throws IOException {
        return ProducerStateManager.listSnapshotFiles(dir, true);
    }

    public static List<SnapshotFile> listSnapshotFiles(File dir, boolean e2eChecksumEnabledForTopic) throws IOException {
        if (dir.exists() && dir.isDirectory()) {
            try (Stream<Path> paths = Files.list(dir.toPath());){
                List<SnapshotFile> list = paths.filter(ProducerStateManager::isSnapshotFile).map(path -> new SnapshotFile(path.toFile(), e2eChecksumEnabledForTopic)).collect(Collectors.toList());
                return list;
            }
        }
        return Collections.emptyList();
    }

    private static String extractTenantPrefix(String name) {
        if (!name.startsWith("lkc-")) {
            return null;
        }
        int delimIndex = name.indexOf(95);
        if (delimIndex == -1) {
            return null;
        }
        return name.substring(0, delimIndex);
    }
}

