/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.raft;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.ShortNode;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.OptionalInt;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.raft.ElectionState;
import org.apache.kafka.raft.QuorumStateStore;
import org.apache.kafka.raft.generated.QuorumStateData;
import org.apache.kafka.raft.generated.QuorumStateDataJsonConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileBasedStateStore
implements QuorumStateStore {
    private static final Logger log = LoggerFactory.getLogger(FileBasedStateStore.class);
    private final File stateFile;
    static final String DATA_VERSION = "data_version";
    static final short HIGHEST_SUPPORTED_VERSION = 0;

    public FileBasedStateStore(File stateFile) {
        this.stateFile = stateFile;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private QuorumStateData readStateFromFile(File file) {
        try (BufferedReader reader = Files.newBufferedReader(file.toPath(), StandardCharsets.UTF_8);){
            String line = reader.readLine();
            if (line == null) {
                throw new EOFException("File ended prematurely.");
            }
            ObjectMapper objectMapper = new ObjectMapper();
            JsonNode readNode = objectMapper.readTree(line);
            if (!(readNode instanceof ObjectNode)) {
                throw new IOException("Deserialized node " + readNode + " is not an object node");
            }
            ObjectNode dataObject = (ObjectNode)readNode;
            JsonNode dataVersionNode = dataObject.get(DATA_VERSION);
            if (dataVersionNode == null) {
                throw new IOException("Deserialized node " + readNode + " does not have " + DATA_VERSION + " field");
            }
            short dataVersion = dataVersionNode.shortValue();
            QuorumStateData quorumStateData = QuorumStateDataJsonConverter.read((JsonNode)dataObject, dataVersion);
            return quorumStateData;
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format("Error while reading the Quorum status from the file %s", file), e);
        }
    }

    @Override
    public ElectionState readElectionState() {
        if (!this.stateFile.exists()) {
            return null;
        }
        QuorumStateData data = this.readStateFromFile(this.stateFile);
        return new ElectionState(data.leaderEpoch(), data.leaderId() == -1 ? OptionalInt.empty() : OptionalInt.of(data.leaderId()), data.votedId() == -1 ? OptionalInt.empty() : OptionalInt.of(data.votedId()), data.currentVoters().stream().map(QuorumStateData.Voter::voterId).collect(Collectors.toSet()));
    }

    @Override
    public void writeElectionState(ElectionState latest) {
        QuorumStateData data = new QuorumStateData().setLeaderEpoch(latest.epoch).setVotedId(latest.hasVoted() ? latest.votedId() : -1).setLeaderId(latest.hasLeader() ? latest.leaderId() : -1).setCurrentVoters(this.voters(latest.voters()));
        this.writeElectionStateToFile(this.stateFile, data);
    }

    private List<QuorumStateData.Voter> voters(Set<Integer> votersId) {
        return votersId.stream().map(voterId -> new QuorumStateData.Voter().setVoterId((int)voterId)).collect(Collectors.toList());
    }

    private void writeElectionStateToFile(File stateFile, QuorumStateData state) {
        File temp = new File(stateFile.getAbsolutePath() + ".tmp");
        this.deleteFileIfExists(temp);
        log.trace("Writing tmp quorum state {}", (Object)temp.getAbsolutePath());
        try {
            try (FileOutputStream fileOutputStream = new FileOutputStream(temp);
                 BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)fileOutputStream, StandardCharsets.UTF_8));){
                ObjectNode jsonState = (ObjectNode)QuorumStateDataJsonConverter.write(state, (short)0);
                jsonState.set(DATA_VERSION, (JsonNode)new ShortNode(0));
                writer.write(jsonState.toString());
                writer.flush();
                fileOutputStream.getFD().sync();
            }
            Utils.atomicMoveWithFallback((Path)temp.toPath(), (Path)stateFile.toPath());
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format("Error while writing the Quorum status from the file %s", stateFile.getAbsolutePath()), e);
        }
        finally {
            this.deleteFileIfExists(temp);
        }
    }

    @Override
    public void clear() {
        this.deleteFileIfExists(this.stateFile);
        this.deleteFileIfExists(new File(this.stateFile.getAbsolutePath() + ".tmp"));
    }

    public String toString() {
        return "Quorum state filepath: " + this.stateFile.getAbsolutePath();
    }

    private void deleteFileIfExists(File file) {
        try {
            Files.deleteIfExists(file.toPath());
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format("Error while deleting file %s", file.getAbsoluteFile()), e);
        }
    }
}

