/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.record;

import io.confluent.kafka.availability.FileChannelWrapper;
import io.confluent.kafka.availability.FilesWrapper;
import io.confluent.kafka.availability.ThreadCountersManager;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.network.TransferableChannel;
import org.apache.kafka.common.record.AbstractRecords;
import org.apache.kafka.common.record.ConvertedRecords;
import org.apache.kafka.common.record.FileLogInputStream;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.RecordBatchIterator;
import org.apache.kafka.common.record.RecordValidationStats;
import org.apache.kafka.common.record.Records;
import org.apache.kafka.common.record.RecordsUtil;
import org.apache.kafka.common.record.UnalignedFileRecords;
import org.apache.kafka.common.utils.AbstractIterator;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;

public class FileRecords
extends AbstractRecords
implements Closeable {
    private static final Path DEVNULL_PATH = Paths.get("/dev/null", new String[0]);
    private final boolean isSlice;
    private final int start;
    private final int end;
    private final Iterable<FileLogInputStream.FileChannelRecordBatch> batches;
    private final AtomicInteger size;
    private final FileChannel channel;
    private volatile File file;
    private Time time;
    private volatile long transferTotalTimeNanos;

    FileRecords(File file, FileChannel channel, int start, int end, boolean isSlice) throws IOException {
        this.file = file;
        this.channel = channel;
        this.start = start;
        this.end = end;
        this.isSlice = isSlice;
        this.size = new AtomicInteger();
        this.transferTotalTimeNanos = 0L;
        this.time = Time.SYSTEM;
        if (isSlice) {
            this.size.set(end - start);
        } else {
            if (channel.size() > Integer.MAX_VALUE) {
                throw new KafkaException("The size of segment " + file + " (" + channel.size() + ") is larger than the maximum allowed segment size of " + Integer.MAX_VALUE);
            }
            int limit = Math.min((int)channel.size(), end);
            this.size.set(limit - start);
            channel.position(limit);
        }
        this.batches = this.batchesFrom(start);
    }

    @Override
    public int sizeInBytes() {
        return this.size.get();
    }

    public File file() {
        return this.file;
    }

    public FileChannel channel() {
        return this.channel;
    }

    public void readInto(ByteBuffer buffer, int position) throws IOException {
        Utils.readFully(this.channel, buffer, position + this.start);
        buffer.flip();
    }

    public FileRecords slice(int position, int size) throws IOException {
        int availableBytes = this.availableBytes(position, size);
        int startPosition = this.start + position;
        return new FileRecords(this.file, this.channel, startPosition, startPosition + availableBytes, true);
    }

    public UnalignedFileRecords sliceUnaligned(int position, int size) {
        int availableBytes = this.availableBytes(position, size);
        return new UnalignedFileRecords(this.channel, this.start + position, availableBytes);
    }

    private int availableBytes(int position, int size) {
        int currentSizeInBytes = this.sizeInBytes();
        if (position < 0) {
            throw new IllegalArgumentException("Invalid position: " + position + " in read from " + this);
        }
        if (position > currentSizeInBytes - this.start) {
            throw new IllegalArgumentException("Slice from position " + position + " exceeds end position of " + this);
        }
        if (size < 0) {
            throw new IllegalArgumentException("Invalid size: " + size + " in read from " + this);
        }
        int end = this.start + position + size;
        if (end < 0 || end > this.start + currentSizeInBytes) {
            end = this.start + currentSizeInBytes;
        }
        return end - (this.start + position);
    }

    public int append(MemoryRecords records) throws IOException {
        if (records.sizeInBytes() > Integer.MAX_VALUE - this.size.get()) {
            throw new IllegalArgumentException("Append of size " + records.sizeInBytes() + " bytes is too large for segment with current file position at " + this.size.get());
        }
        int written = ThreadCountersManager.wrapIOChecked(() -> records.writeFullyTo(this.channel));
        this.size.getAndAdd(written);
        return written;
    }

    public void flush() throws IOException {
        FileChannelWrapper.force(this.channel, true);
    }

    @Override
    public void close() throws IOException {
        this.flush();
        this.trim();
        FileChannelWrapper.close(this.channel);
    }

    @Override
    public void release() {
    }

    public void closeHandlers() throws IOException {
        FileChannelWrapper.close(this.channel);
    }

    public boolean deleteIfExists() throws IOException {
        Utils.closeQuietly(this.channel, "FileChannel");
        return FilesWrapper.deleteIfExists(this.file.toPath());
    }

    public void trim() throws IOException {
        this.truncateTo(this.sizeInBytes());
    }

    public void updateParentDir(File parentDir) {
        this.file = new File(parentDir, this.file.getName());
    }

    public void renameTo(File f) throws IOException {
        try {
            Utils.atomicMoveWithFallback(this.file.toPath(), f.toPath(), false);
        }
        finally {
            this.file = f;
        }
    }

    public int truncateTo(int targetSize) throws IOException {
        int originalSize = this.sizeInBytes();
        if (targetSize > originalSize || targetSize < 0) {
            throw new KafkaException("Attempt to truncate log segment " + this.file + " to " + targetSize + " bytes failed,  size of this log segment is " + originalSize + " bytes.");
        }
        if (targetSize < (int)this.channel.size()) {
            FileChannelWrapper.truncate(this.channel, targetSize);
            this.size.set(targetSize);
        }
        return originalSize - targetSize;
    }

    @Override
    public ConvertedRecords<? extends Records> downConvert(byte toMagic, long firstOffset, Time time) {
        ConvertedRecords convertedRecords = ThreadCountersManager.wrapIO(() -> RecordsUtil.downConvert(this.batches, toMagic, firstOffset, time));
        if (convertedRecords.recordConversionStats().numRecordsConverted() == 0) {
            return new ConvertedRecords<FileRecords>(this, RecordValidationStats.EMPTY);
        }
        return convertedRecords;
    }

    @Override
    public int writeTo(TransferableChannel destChannel, int offset, int length) throws IOException {
        int oldSize;
        long newSize = Math.min(this.channel.size(), (long)this.end) - (long)this.start;
        if (newSize < (long)(oldSize = this.sizeInBytes())) {
            throw new KafkaException(String.format("Size of FileRecords %s has been truncated during write: old size %d, new size %d", this.file.getAbsolutePath(), oldSize, newSize));
        }
        long position = this.start + offset;
        long count = Math.min(length, oldSize - offset);
        long transferStartTime = this.time.nanoseconds();
        long bytesTransferred = ThreadCountersManager.wrapIOChecked(() -> destChannel.transferFrom(this.channel, position, count));
        this.transferTotalTimeNanos += this.time.nanoseconds() - transferStartTime;
        return (int)bytesTransferred;
    }

    public LogOffsetPosition searchForOffsetWithSize(long targetOffset, int startingPosition) {
        for (FileLogInputStream.FileChannelRecordBatch batch : ThreadCountersManager.wrapIO(() -> this.batchesFrom(startingPosition))) {
            long offset = batch.lastOffset();
            if (offset < targetOffset) continue;
            return new LogOffsetPosition(offset, batch.position(), batch.sizeInBytes());
        }
        return null;
    }

    public FileTimestampAndOffset searchForTimestamp(long targetTimestamp, int startingPosition, long startingOffset) {
        for (RecordBatch batch : ThreadCountersManager.wrapIO(() -> this.batchesFrom(startingPosition))) {
            if (batch.maxTimestamp() < targetTimestamp) continue;
            for (Record record : batch) {
                long timestamp = record.timestamp();
                if (timestamp < targetTimestamp || record.offset() < startingOffset) continue;
                return new FileTimestampAndOffset(timestamp, record.offset(), this.maybeLeaderEpoch(batch.partitionLeaderEpoch()));
            }
        }
        return null;
    }

    public FileTimestampAndOffset largestTimestampAfter(int startingPosition) {
        long maxTimestamp = -1L;
        long offsetOfMaxTimestamp = -1L;
        int leaderEpochOfMaxTimestamp = -1;
        for (RecordBatch batch : ThreadCountersManager.wrapIO(() -> this.batchesFrom(startingPosition))) {
            long timestamp = batch.maxTimestamp();
            if (timestamp <= maxTimestamp) continue;
            maxTimestamp = timestamp;
            offsetOfMaxTimestamp = batch.lastOffset();
            leaderEpochOfMaxTimestamp = batch.partitionLeaderEpoch();
        }
        return new FileTimestampAndOffset(maxTimestamp, offsetOfMaxTimestamp, this.maybeLeaderEpoch(leaderEpochOfMaxTimestamp));
    }

    private Optional<Integer> maybeLeaderEpoch(int leaderEpoch) {
        return leaderEpoch == -1 ? Optional.empty() : Optional.of(leaderEpoch);
    }

    public Iterable<FileLogInputStream.FileChannelRecordBatch> batches() {
        return this.batches;
    }

    public void setTime(Time time) {
        this.time = time;
    }

    public long transferTotalTimeNanos() {
        return this.transferTotalTimeNanos;
    }

    public String toString() {
        return "FileRecords(size=" + this.sizeInBytes() + ", file=" + this.file + ", start=" + this.start + ", end=" + this.end + ")";
    }

    public Iterable<FileLogInputStream.FileChannelRecordBatch> batchesFrom(int start) {
        return () -> this.batchIterator(start);
    }

    public AbstractIterator<FileLogInputStream.FileChannelRecordBatch> batchIterator() {
        return this.batchIterator(this.start);
    }

    private AbstractIterator<FileLogInputStream.FileChannelRecordBatch> batchIterator(int start) {
        int end = this.isSlice ? this.end : this.sizeInBytes();
        FileLogInputStream inputStream = new FileLogInputStream(this, start, end);
        return new RecordBatchIterator<FileLogInputStream.FileChannelRecordBatch>(inputStream);
    }

    public void loadIntoPageCache() throws IOException {
        long size = Math.min(FileChannelWrapper.size(this.channel), (long)this.end) - (long)this.start;
        try (FileChannel devnullChannel = FileChannelWrapper.open(DEVNULL_PATH, StandardOpenOption.WRITE);){
            FileChannelWrapper.transferTo(this.channel, this.start, size, devnullChannel);
        }
    }

    public static FileRecords open(File file, boolean mutable, boolean fileAlreadyExists, int initFileSize, boolean preallocate) throws IOException {
        LogContext logContext = new LogContext();
        Logger logger = logContext.logger(FileRecords.class);
        FileChannel channel = ThreadCountersManager.wrapIOChecked(() -> FileRecords.openChannel(file, mutable, fileAlreadyExists, initFileSize, preallocate, logger));
        int end = !fileAlreadyExists && preallocate ? 0 : Integer.MAX_VALUE;
        return new FileRecords(file, channel, 0, end, false);
    }

    public static FileRecords open(File file, boolean fileAlreadyExists, int initFileSize, boolean preallocate) throws IOException {
        return FileRecords.open(file, true, fileAlreadyExists, initFileSize, preallocate);
    }

    public static FileRecords open(File file, boolean mutable) throws IOException {
        return FileRecords.open(file, mutable, false, 0, false);
    }

    public static FileRecords open(File file) throws IOException {
        return FileRecords.open(file, true);
    }

    private static FileChannel openChannel(File file, boolean mutable, boolean fileAlreadyExists, int initFileSize, boolean preallocate, Logger logger) throws IOException {
        if (mutable) {
            return FileRecords.openMutableChannel(file, fileAlreadyExists, initFileSize, preallocate, logger);
        }
        return FileChannel.open(file.toPath(), new OpenOption[0]);
    }

    private static void failOnNonZeroChannelSize(File file, FileChannel channel) throws IOException {
        if (channel.size() > 0L) {
            channel.close();
            throw new IllegalStateException(String.format("Non-empty file %s already exists while it shouldn't.", file));
        }
    }

    private static FileChannel openMutableChannel(File file, boolean fileAlreadyExists, int initFileSize, boolean preallocate, Logger logger) throws IOException {
        if (fileAlreadyExists || !preallocate) {
            FileChannel channel;
            if (fileAlreadyExists) {
                return FileChannel.open(file.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
            }
            try {
                channel = FileChannel.open(file.toPath(), StandardOpenOption.CREATE_NEW, StandardOpenOption.READ, StandardOpenOption.WRITE);
                FileRecords.failOnNonZeroChannelSize(file, channel);
            }
            catch (FileAlreadyExistsException e) {
                channel = FileChannel.open(file.toPath(), StandardOpenOption.READ, StandardOpenOption.WRITE);
                if (channel.size() == 0L) {
                    logger.warn("Empty file {} already exists while it shouldn't.", (Object)file);
                }
                FileRecords.failOnNonZeroChannelSize(file, channel);
            }
            return channel;
        }
        RandomAccessFile randomAccessFile = new RandomAccessFile(file, "rw");
        long fileLength = randomAccessFile.length();
        if (fileLength == 0L) {
            logger.warn("Empty file {} already exists while it shouldn't.", (Object)file);
        } else if (fileLength > 0L) {
            randomAccessFile.close();
            throw new IllegalStateException(String.format("Non-empty file %s already exists while it shouldn't.", file));
        }
        randomAccessFile.setLength(initFileSize);
        return randomAccessFile.getChannel();
    }

    public static class FileTimestampAndOffset {
        public final long timestamp;
        public final Optional<Integer> leaderEpoch;
        public long offset;
        public Exception exception = null;

        public FileTimestampAndOffset(long timestamp, long offset, Optional<Integer> leaderEpoch) {
            this.timestamp = timestamp;
            this.offset = offset;
            this.leaderEpoch = leaderEpoch;
        }

        public FileTimestampAndOffset(long timestamp, Optional<Integer> leaderEpoch, Exception exception) {
            this.timestamp = timestamp;
            this.leaderEpoch = leaderEpoch;
            this.exception = exception;
        }

        public long timestamp() {
            return this.timestamp;
        }

        public long offset() {
            return this.offset;
        }

        public Exception exception() {
            return this.exception;
        }

        public Optional<Integer> leaderEpoch() {
            return this.leaderEpoch;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            FileTimestampAndOffset that = (FileTimestampAndOffset)o;
            return this.timestamp == that.timestamp && this.offset == that.offset && Objects.equals(this.leaderEpoch, that.leaderEpoch);
        }

        public int hashCode() {
            return Objects.hash(this.timestamp, this.offset, this.leaderEpoch);
        }

        public String toString() {
            return "TimestampAndOffset(timestamp=" + this.timestamp + ", offset=" + this.offset + ", leaderEpoch=" + this.leaderEpoch + ')';
        }
    }

    public static class LogOffsetPosition {
        public final long offset;
        public final int position;
        public final int size;

        public LogOffsetPosition(long offset, int position, int size) {
            this.offset = offset;
            this.position = position;
            this.size = size;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            LogOffsetPosition that = (LogOffsetPosition)o;
            return this.offset == that.offset && this.position == that.position && this.size == that.size;
        }

        public int hashCode() {
            int result = Long.hashCode(this.offset);
            result = 31 * result + this.position;
            result = 31 * result + this.size;
            return result;
        }

        public String toString() {
            return "LogOffsetPosition(offset=" + this.offset + ", position=" + this.position + ", size=" + this.size + ')';
        }
    }
}

