/*
 * Decompiled with CFR 0.152.
 */
package io.spiffe.internal;

import io.spiffe.internal.AsymmetricKeyAlgorithm;
import io.spiffe.internal.KeyFileFormat;
import io.spiffe.internal.KeyUsage;
import io.spiffe.spiffeid.SpiffeId;
import io.spiffe.spiffeid.TrustDomain;
import java.io.ByteArrayInputStream;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateParsingException;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.security.spec.EncodedKeySpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;

public class CertificateUtils {
    private static final String SPIFFE_PREFIX = "spiffe://";
    private static final int SAN_VALUE_INDEX = 1;
    private static final String PUBLIC_KEY_INFRASTRUCTURE_ALGORITHM = "PKIX";
    private static final String X509_CERTIFICATE_TYPE = "X.509";

    private CertificateUtils() {
    }

    public static List<X509Certificate> generateCertificates(byte[] input) throws CertificateParsingException {
        Collection<? extends Certificate> certificates;
        if (input.length == 0) {
            throw new CertificateParsingException("No certificates found");
        }
        CertificateFactory certificateFactory = CertificateUtils.getCertificateFactory();
        try {
            certificates = certificateFactory.generateCertificates(new ByteArrayInputStream(input));
        }
        catch (CertificateException e) {
            throw new CertificateParsingException("Certificate could not be parsed from cert bytes", e);
        }
        return certificates.stream().map(X509Certificate.class::cast).collect(Collectors.toList());
    }

    public static PrivateKey generatePrivateKey(byte[] privateKeyBytes, AsymmetricKeyAlgorithm algorithm, KeyFileFormat keyFileFormat) throws InvalidKeySpecException, NoSuchAlgorithmException, InvalidKeyException {
        EncodedKeySpec kspec = CertificateUtils.getEncodedKeySpec(privateKeyBytes, keyFileFormat);
        return CertificateUtils.generatePrivateKeyWithSpec(kspec, algorithm);
    }

    public static void validate(List<X509Certificate> chain, Collection<X509Certificate> trustedCerts) throws CertificateException, CertPathValidatorException {
        if (chain == null || chain.size() == 0) {
            throw new IllegalArgumentException("Chain of certificates is empty");
        }
        CertificateFactory certificateFactory = CertificateUtils.getCertificateFactory();
        try {
            PKIXParameters pkixParameters = CertificateUtils.toPkixParameters(trustedCerts);
            CertPath certPath = certificateFactory.generateCertPath(chain);
            CertificateUtils.getCertPathValidator().validate(certPath, pkixParameters);
        }
        catch (InvalidAlgorithmParameterException | NoSuchAlgorithmException e) {
            throw new CertificateException(e);
        }
    }

    public static SpiffeId getSpiffeId(X509Certificate certificate) throws CertificateException {
        List<String> spiffeIds = CertificateUtils.getSpiffeIds(certificate);
        if (spiffeIds.size() > 1) {
            throw new CertificateException("Certificate contains multiple SPIFFE IDs");
        }
        if (spiffeIds.size() < 1) {
            throw new CertificateException("Certificate does not contain SPIFFE ID in the URI SAN");
        }
        return SpiffeId.parse(spiffeIds.get(0));
    }

    public static TrustDomain getTrustDomain(List<X509Certificate> chain) throws CertificateException {
        SpiffeId spiffeId = CertificateUtils.getSpiffeId(chain.get(0));
        return spiffeId.getTrustDomain();
    }

    public static boolean isCA(X509Certificate cert) {
        return cert.getBasicConstraints() != -1;
    }

    public static boolean hasKeyUsageCertSign(X509Certificate cert) {
        boolean[] keyUsage = cert.getKeyUsage();
        return keyUsage[KeyUsage.KEY_CERT_SIGN.index()];
    }

    public static boolean hasKeyUsageDigitalSignature(X509Certificate cert) {
        boolean[] keyUsage = cert.getKeyUsage();
        return keyUsage[KeyUsage.DIGITAL_SIGNATURE.index()];
    }

    public static boolean hasKeyUsageCRLSign(X509Certificate cert) {
        boolean[] keyUsage = cert.getKeyUsage();
        return keyUsage[KeyUsage.CRL_SIGN.index()];
    }

    private static EncodedKeySpec getEncodedKeySpec(byte[] privateKeyBytes, KeyFileFormat keyFileFormat) throws InvalidKeyException {
        PKCS8EncodedKeySpec keySpec;
        if (keyFileFormat == KeyFileFormat.PEM) {
            byte[] keyDer = CertificateUtils.toDerFormat(privateKeyBytes);
            keySpec = new PKCS8EncodedKeySpec(keyDer);
        } else {
            keySpec = new PKCS8EncodedKeySpec(privateKeyBytes);
        }
        return keySpec;
    }

    private static List<String> getSpiffeIds(X509Certificate certificate) throws CertificateParsingException {
        if (certificate.getSubjectAlternativeNames() == null) {
            return Collections.emptyList();
        }
        return certificate.getSubjectAlternativeNames().stream().map(san -> (String)san.get(1)).filter(uri -> StringUtils.startsWith((CharSequence)uri, (CharSequence)SPIFFE_PREFIX)).collect(Collectors.toList());
    }

    private static PrivateKey generatePrivateKeyWithSpec(EncodedKeySpec keySpec, AsymmetricKeyAlgorithm algorithm) throws NoSuchAlgorithmException, InvalidKeySpecException {
        PrivateKey privateKey = null;
        switch (algorithm) {
            case EC: {
                privateKey = KeyFactory.getInstance(AsymmetricKeyAlgorithm.EC.value()).generatePrivate(keySpec);
                break;
            }
            case RSA: {
                privateKey = KeyFactory.getInstance(AsymmetricKeyAlgorithm.RSA.value()).generatePrivate(keySpec);
            }
        }
        return privateKey;
    }

    private static PKIXParameters toPkixParameters(Collection<X509Certificate> trustedCerts) throws CertificateException, InvalidAlgorithmParameterException {
        if (trustedCerts == null || trustedCerts.isEmpty()) {
            throw new CertificateException("No trusted Certs");
        }
        PKIXParameters pkixParameters = new PKIXParameters(trustedCerts.stream().map(c -> new TrustAnchor((X509Certificate)c, null)).collect(Collectors.toSet()));
        pkixParameters.setRevocationEnabled(false);
        return pkixParameters;
    }

    private static CertPathValidator getCertPathValidator() throws NoSuchAlgorithmException {
        return CertPathValidator.getInstance(PUBLIC_KEY_INFRASTRUCTURE_ALGORITHM);
    }

    private static CertificateFactory getCertificateFactory() {
        try {
            return CertificateFactory.getInstance(X509_CERTIFICATE_TYPE);
        }
        catch (CertificateException e) {
            throw new IllegalStateException("Could not create Certificate Factory", e);
        }
    }

    private static byte[] toDerFormat(byte[] privateKeyPem) throws InvalidKeyException {
        String privateKeyAsString = new String(privateKeyPem);
        privateKeyAsString = privateKeyAsString.replaceAll("(-+BEGIN PRIVATE KEY-+\\r?\\n|-+END PRIVATE KEY-+\\r?\\n?)", "");
        privateKeyAsString = privateKeyAsString.replaceAll("\n", "");
        Base64.Decoder decoder = Base64.getDecoder();
        try {
            return decoder.decode(privateKeyAsString);
        }
        catch (Exception e) {
            throw new InvalidKeyException(e);
        }
    }
}

