/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.license;

import io.confluent.license.InvalidLicenseException;
import io.confluent.license.util.StringUtils;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.Key;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.apache.kafka.common.utils.Time;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jws.JsonWebSignature;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.MalformedClaimException;
import org.jose4j.jwt.NumericDate;
import org.jose4j.jwt.consumer.ErrorCodeValidator;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.IssValidator;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.jose4j.jwt.consumer.SubValidator;
import org.jose4j.lang.JoseException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class License {
    private static final Logger log = LoggerFactory.getLogger(License.class);
    private static final String DATE_FORMAT_STR = "yyyy-MM-dd";
    static final String MONITORING_CLAIM = "monitoring";
    static final String TYPE_CLAIM_NAME = "licenseType";
    private static final Collection<String> RELEVANT_CLAIM_NAMES = Collections.unmodifiableList(Arrays.asList("aud", "iss", "sub", "exp", "monitoring", "licenseType"));
    private final JwtClaims jwtClaims;
    private final Time clock;
    private final String serialized;
    private final String platformId;

    public static JwtClaims baseClaims(String audience, long expiration, boolean monitoring) {
        JwtClaims claims = new JwtClaims();
        claims.setIssuer("Confluent");
        claims.setAudience(audience);
        claims.setExpirationTime(NumericDate.fromMilliseconds((long)expiration));
        claims.setGeneratedJwtId();
        claims.setIssuedAtToNow();
        claims.setNotBeforeMinutesInThePast(2.0f);
        claims.setSubject("Confluent Enterprise");
        claims.setClaim(MONITORING_CLAIM, (Object)monitoring);
        return claims;
    }

    public static String sign(PrivateKey key, String audience, long expiration, boolean monitoring) throws JoseException {
        JwtClaims claims = License.baseClaims(audience, expiration, monitoring);
        JsonWebSignature jws = new JsonWebSignature();
        jws.setPayload(claims.toJson());
        jws.setKey((Key)key);
        jws.setAlgorithmHeaderValue("RS256");
        return jws.getCompactSerialization();
    }

    public static JwtClaims verify(PublicKey key, String license) throws InvalidJwtException {
        JwtConsumer jwtConsumer = License.commonVerifyOptions().setRequireSubject().setExpectedIssuer("Confluent").setVerificationKey((Key)key).build();
        return jwtConsumer.processToClaims(license);
    }

    public static JwtClaims verifyStored(PublicKey key, String license) throws InvalidJwtException {
        JwtConsumer jwtConsumer = License.commonVerifyOptions().setDisableRequireSignature().setSkipVerificationKeyResolutionOnNone().setJwsAlgorithmConstraints(AlgorithmConstraints.NO_CONSTRAINTS).setVerificationKey((Key)key).build();
        return jwtConsumer.processToClaims(license);
    }

    private static JwtConsumerBuilder commonVerifyOptions() throws InvalidJwtException {
        return new JwtConsumerBuilder().setSkipAllDefaultValidators().registerValidator((ErrorCodeValidator)new IssValidator("Confluent", true)).registerValidator((ErrorCodeValidator)new SubValidator(true));
    }

    public static long getExpiration(JwtClaims claims) throws Throwable {
        try {
            return License.expiration(claims);
        }
        catch (MalformedClaimException e) {
            log.warn("Unable to extract expiration due to malformed claim: ", (Throwable)e);
            throw e;
        }
        catch (Throwable t) {
            log.warn("Unable to extract expiration: ", t);
            throw t;
        }
    }

    protected static long expiration(JwtClaims claims) throws Throwable {
        NumericDate expirationDate = claims.getExpirationTime();
        return expirationDate != null ? expirationDate.getValueInMillis() : Long.MIN_VALUE;
    }

    public static List<String> getAudience(JwtClaims claims) {
        try {
            return claims.getAudience();
        }
        catch (Throwable t) {
            log.warn("unable to getAudience", t);
            return new ArrayList<String>();
        }
    }

    public static boolean verifyMonitoring(JwtClaims claims) {
        try {
            return (Boolean)claims.getClaimValue(MONITORING_CLAIM, Boolean.class);
        }
        catch (MalformedClaimException e) {
            return false;
        }
        catch (Throwable t) {
            log.warn("unable to verifyMonitoring", t);
            return false;
        }
    }

    public static PrivateKey loadPrivateKey(InputStream is) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        byte[] keyBytes = License.readFully(is);
        PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(keyBytes);
        return KeyFactory.getInstance("RSA").generatePrivate(spec);
    }

    public static PublicKey loadPublicKey() throws NoSuchAlgorithmException, IOException, InvalidKeySpecException {
        try (InputStream keyData = License.class.getClassLoader().getResourceAsStream("secrets/public_key.der");){
            PublicKey publicKey = License.loadPublicKey(keyData);
            return publicKey;
        }
    }

    public static PublicKey loadPublicKey(InputStream is) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        X509EncodedKeySpec xspec = new X509EncodedKeySpec(License.readFully(is));
        return KeyFactory.getInstance("RSA").generatePublic(xspec);
    }

    private static byte[] readFully(InputStream in) throws IOException {
        int n;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        byte[] buffer = new byte[1024];
        while ((n = in.read(buffer)) != -1) {
            baos.write(buffer, 0, n);
        }
        return baos.toByteArray();
    }

    public static String generateTrialLicense(JwtClaims trialClaims) throws InvalidLicenseException {
        JsonWebSignature jws = new JsonWebSignature();
        jws.setAlgorithmConstraints(AlgorithmConstraints.NO_CONSTRAINTS);
        jws.setAlgorithmHeaderValue("none");
        jws.setPayload(trialClaims.toJson());
        try {
            return jws.getCompactSerialization();
        }
        catch (JoseException e) {
            log.error("Error while attempting to start trial period: ", (Throwable)e);
            throw new InvalidLicenseException("Error creating license for trial version: ", e);
        }
    }

    public String platformId() {
        return this.platformId;
    }

    public String licenseId() {
        try {
            return this.jwtClaims.getJwtId();
        }
        catch (Throwable t) {
            log.warn("unable to get licenseId", t);
            return "";
        }
    }

    public License(JwtClaims claims, Time clock, String serialized) {
        this(claims, clock, serialized, null);
    }

    public License(JwtClaims claims, Time clock, String serialized, String platformId) {
        this.jwtClaims = claims;
        this.clock = clock;
        this.serialized = serialized;
        this.platformId = platformId;
    }

    public String subject() {
        try {
            return this.jwtClaims.getSubject();
        }
        catch (MalformedClaimException e) {
            return "";
        }
    }

    public boolean isTrial() {
        return this.hasLicenseType("trial") || "trial".equalsIgnoreCase(this.audienceString());
    }

    public boolean isFreeTier() {
        return this.hasLicenseType("free");
    }

    public Type type() {
        if (this.isFreeTier()) {
            return Type.FREE_TIER;
        }
        if (this.isTrial()) {
            return Type.TRIAL;
        }
        return Type.ENTERPRISE;
    }

    public Date expirationDate() {
        return new Date(this.expirationMillis());
    }

    public String expirationDateString() {
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT_STR);
        return dateFormat.format(this.expirationDate());
    }

    public boolean isExpired() {
        return !this.isFreeTier() && this.timeRemaining(TimeUnit.MILLISECONDS) <= 0L;
    }

    public boolean isValid() {
        return !this.isExpired();
    }

    public boolean isEquivalentTo(License other) {
        return this.hasMatchingClaims(other, TYPE_CLAIM_NAME);
    }

    public boolean isRenewalOf(License other) {
        return this.hasMatchingClaims(other, "exp", TYPE_CLAIM_NAME) && other != null && other.expiresBefore(this);
    }

    public long timeRemaining(TimeUnit unit) {
        long timeRemainingMillis = this.expirationMillis() - this.clock.milliseconds();
        return unit.convert(timeRemainingMillis, TimeUnit.MILLISECONDS);
    }

    public List<String> audience() {
        return License.getAudience(this.jwtClaims);
    }

    public String audienceString() {
        return String.join((CharSequence)",", this.audience());
    }

    public boolean expiresBefore(License other) {
        if (other == null) {
            return false;
        }
        return this.expirationMillis() < other.expirationMillis();
    }

    public int hashCode() {
        return Objects.hash(this.serialized, this.platformId);
    }

    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (obj instanceof License) {
            License that = (License)obj;
            return this.jwtClaims().equals(that.jwtClaims()) && Objects.equals(this.platformId, that.platformId);
        }
        return false;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        if (this.isFreeTier()) {
            sb.append("Free Tier license ");
        } else if (this.isTrial()) {
            sb.append("Trial license ");
        } else {
            sb.append("License ");
        }
        if (!StringUtils.isBlank(this.subject())) {
            sb.append("for ").append(this.subject());
        }
        try {
            if (this.isExpired()) {
                sb.append(" expired on ").append(this.expirationDateString());
            } else if (!this.isFreeTier()) {
                long remainingMillis = this.timeRemaining(TimeUnit.MILLISECONDS);
                long daysRemaining = TimeUnit.MILLISECONDS.toDays(remainingMillis);
                sb.append(" expires in ").append(daysRemaining).append(" days on ").append(this.expirationDateString());
            }
        }
        catch (Throwable t) {
            sb.append(" with invalid expiration");
        }
        sb.append(".");
        return sb.toString();
    }

    JwtClaims jwtClaims() {
        return this.jwtClaims;
    }

    public String serializedForm() {
        return this.serialized;
    }

    protected long expirationMillis() {
        try {
            return License.expiration(this.jwtClaims);
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Throwable t) {
            throw new RuntimeException(t);
        }
    }

    protected boolean hasLicenseType(String expected) {
        try {
            return expected.equalsIgnoreCase(this.jwtClaims().getStringClaimValue(TYPE_CLAIM_NAME));
        }
        catch (MalformedClaimException e) {
            return false;
        }
    }

    protected boolean hasMatchingClaims(License other, String ... excludeClaims) {
        if (other == this) {
            return true;
        }
        if (other == null) {
            return false;
        }
        HashSet<String> exclusions = new HashSet<String>(Arrays.asList(excludeClaims));
        JwtClaims thisClaims = this.jwtClaims();
        JwtClaims thatClaims = other.jwtClaims();
        for (String claimName : RELEVANT_CLAIM_NAMES) {
            Object laterClaim;
            Object prevClaim;
            if (exclusions.contains(claimName) || Objects.equals(prevClaim = thisClaims.getClaimValue(claimName), laterClaim = thatClaims.getClaimValue(claimName))) continue;
            return false;
        }
        return true;
    }

    public static enum Type {
        ENTERPRISE,
        TRIAL,
        FREE_TIER;

    }
}

