/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import com.yammer.metrics.core.MetricName;
import com.yammer.metrics.core.Timer;
import com.yammer.metrics.core.TimerContext;
import io.confluent.kafka.availability.FilesWrapper;
import io.confluent.kafka.availability.ThreadCountersManager;
import io.confluent.kafka.storage.checksum.ChecksumParams;
import io.confluent.kafka.storage.checksum.ChecksumStore;
import io.confluent.kafka.storage.checksum.E2EChecksumProtectedObjectType;
import io.confluent.kafka.storage.checksum.E2EChecksumStore;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.kafka.common.InvalidRecordException;
import org.apache.kafka.common.errors.CorruptRecordException;
import org.apache.kafka.common.record.FileLogInputStream;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.Records;
import org.apache.kafka.common.utils.BufferSupplier;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.server.metrics.KafkaMetricsGroup;
import org.apache.kafka.storage.internals.epoch.LeaderEpochFileCache;
import org.apache.kafka.storage.internals.log.AbortedTxn;
import org.apache.kafka.storage.internals.log.AppendOrigin;
import org.apache.kafka.storage.internals.log.CompletedTxn;
import org.apache.kafka.storage.internals.log.FetchDataInfo;
import org.apache.kafka.storage.internals.log.FetchedTimestampAndOffset;
import org.apache.kafka.storage.internals.log.LazyIndex;
import org.apache.kafka.storage.internals.log.LogConfig;
import org.apache.kafka.storage.internals.log.LogFileUtils;
import org.apache.kafka.storage.internals.log.LogOffsetMetadata;
import org.apache.kafka.storage.internals.log.LogSegmentOffsetOverflowException;
import org.apache.kafka.storage.internals.log.OffsetIndex;
import org.apache.kafka.storage.internals.log.OffsetPosition;
import org.apache.kafka.storage.internals.log.ProducerAppendInfo;
import org.apache.kafka.storage.internals.log.ProducerStateManager;
import org.apache.kafka.storage.internals.log.RollParams;
import org.apache.kafka.storage.internals.log.StorageAction;
import org.apache.kafka.storage.internals.log.TimeIndex;
import org.apache.kafka.storage.internals.log.TimestampOffset;
import org.apache.kafka.storage.internals.log.TransactionIndex;
import org.apache.kafka.storage.internals.log.TxnIndexSearchResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public class LogSegment
implements Closeable {
    private static final long UNKNOWN_LAST_OFFSET = -1L;
    private static final Logger LOGGER = LoggerFactory.getLogger(LogSegment.class);
    private static final Timer LOG_FLUSH_TIMER;
    private static final Timer SEGMENT_APPEND_TIME_MS;
    private static final Timer OFFSET_INDEX_APPEND_TIME_MS;
    private static final Timer TIMESTAMP_INDEX_APPEND_TIME_MS;
    private static final String FUTURE_DIR_SUFFIX = "-future";
    private static final Pattern FUTURE_DIR_PATTERN;
    private final FileRecords log;
    private final LazyIndex<OffsetIndex> lazyOffsetIndex;
    private final LazyIndex<TimeIndex> lazyTimeIndex;
    private final TransactionIndex txnIndex;
    private final long baseOffset;
    private final int indexIntervalBytes;
    private final long rollJitterMs;
    private final Time time;
    private final Optional<E2EChecksumStore> checksumStoreOpt;
    private final boolean e2eChecksumEnabledForTopic;
    private final boolean shouldPersistChecksum;
    private final AtomicLong lastSegmentOffset = new AtomicLong(-1L);
    private volatile OptionalLong rollingBasedTimestamp = OptionalLong.empty();
    private volatile OptionalLong lastFlushedTimeMs = OptionalLong.empty();
    private volatile TimestampOffset maxTimestampAndOffsetSoFar = TimestampOffset.UNKNOWN;
    private long created;
    private long segmentCreateTime;
    private int bytesSinceLastIndexEntry;

    public LogSegment(FileRecords log, LazyIndex<OffsetIndex> lazyOffsetIndex, LazyIndex<TimeIndex> lazyTimeIndex, TransactionIndex txnIndex, long baseOffset, int indexIntervalBytes, long rollJitterMs, Time time, ChecksumParams checksumParams) throws IOException {
        this.log = log;
        this.lazyOffsetIndex = lazyOffsetIndex;
        this.lazyTimeIndex = lazyTimeIndex;
        this.txnIndex = txnIndex;
        this.baseOffset = baseOffset;
        this.indexIntervalBytes = indexIntervalBytes;
        this.rollJitterMs = rollJitterMs;
        this.time = time;
        this.checksumStoreOpt = checksumParams.checksumStoreOpt();
        this.e2eChecksumEnabledForTopic = checksumParams.e2eChecksumEnabledForTopic();
        this.shouldPersistChecksum = checksumParams.shouldPersistChecksum();
        this.created = time.milliseconds();
        this.segmentCreateTime = FilesWrapper.getFileCreationTime((Path)log.file().toPath()).orElse(this.created);
    }

    public OffsetIndex offsetIndex() throws IOException {
        return this.lazyOffsetIndex.get();
    }

    public File offsetIndexFile() {
        return this.lazyOffsetIndex.file();
    }

    public TimeIndex timeIndex() throws IOException {
        return this.lazyTimeIndex.get();
    }

    public File timeIndexFile() {
        return this.lazyTimeIndex.file();
    }

    public long baseOffset() {
        return this.baseOffset;
    }

    public FileRecords log() {
        return this.log;
    }

    public long rollJitterMs() {
        return this.rollJitterMs;
    }

    public TransactionIndex txnIndex() {
        return this.txnIndex;
    }

    public OptionalLong lastFlushedTimeMs() {
        return this.lastFlushedTimeMs;
    }

    public void setSegmentCreateTime(long ms) {
        this.segmentCreateTime = ms;
    }

    public ChecksumParams checksumParams() {
        return new ChecksumParams(this.checksumStoreOpt, this.e2eChecksumEnabledForTopic, this.shouldPersistChecksum);
    }

    public boolean shouldRoll(RollParams rollParams) throws IOException {
        boolean reachedRollMs = this.timeWaitedForRoll(rollParams.now, rollParams.maxTimestampInMessages, rollParams.rollBasedOnSystemTime) > rollParams.maxSegmentMs - this.rollJitterMs;
        int size = this.size();
        return size > rollParams.maxSegmentBytes - rollParams.messagesSize || size > 0 && reachedRollMs || this.offsetIndex().isFull() || this.timeIndex().isFull() || !this.canConvertToRelativeOffset(rollParams.maxOffsetInMessages);
    }

    public void resizeIndexes(int size) throws IOException {
        ThreadCountersManager.wrapIOChecked(() -> {
            this.offsetIndex().resize(size);
            this.timeIndex().resize(size);
        });
    }

    public void sanityCheck(boolean timeIndexFileNewlyCreated) throws IOException {
        ThreadCountersManager.wrapIOChecked(() -> {
            if (this.offsetIndexFile().exists()) {
                if (timeIndexFileNewlyCreated) {
                    this.timeIndex().resize(0);
                }
            } else {
                throw new NoSuchFileException("Offset index file " + this.offsetIndexFile().getAbsolutePath() + " does not exist");
            }
            this.txnIndex.sanityCheck();
        });
    }

    public TimestampOffset readMaxTimestampAndOffsetSoFar() throws IOException {
        if (this.maxTimestampAndOffsetSoFar == TimestampOffset.UNKNOWN) {
            this.maxTimestampAndOffsetSoFar = this.timeIndex().lastEntry();
        }
        return this.maxTimestampAndOffsetSoFar;
    }

    public long maxTimestampSoFar() throws IOException {
        return this.readMaxTimestampAndOffsetSoFar().timestamp;
    }

    public long shallowOffsetOfMaxTimestampSoFar() throws IOException {
        return this.readMaxTimestampAndOffsetSoFar().offset;
    }

    public int size() {
        return this.log.sizeInBytes();
    }

    public boolean canConvertToRelativeOffset(long offset) throws IOException {
        return this.offsetIndex().canAppendOffset(offset);
    }

    public void append(long lastOffset, MemoryRecords records) throws IOException {
        this.append(lastOffset, records, this.time.milliseconds());
    }

    public void append(long lastOffset, MemoryRecords records, long appendTimeMs) throws IOException {
        ThreadCountersManager.wrapIOChecked(() -> {
            if (records.sizeInBytes() > 0) {
                int position = this.log.sizeInBytes();
                boolean traceEnabled = LOGGER.isTraceEnabled();
                if (traceEnabled) {
                    LOGGER.trace("Inserting {} bytes at end offset {} at position {}", new Object[]{records.sizeInBytes(), lastOffset, this.log.sizeInBytes()});
                }
                this.ensureOffsetInRange(lastOffset);
                int appendedBytes = LogSegment.time(SEGMENT_APPEND_TIME_MS, () -> this.log.append(records));
                this.checksumStoreOpt.ifPresent(store -> this.mayUpdateChecksum((E2EChecksumStore)store, records.buffer().duplicate(), appendTimeMs));
                if (traceEnabled) {
                    LOGGER.trace("Appended {} to {} at end offset {}", new Object[]{appendedBytes, this.log.file(), lastOffset});
                }
                for (RecordBatch batch : records.batches()) {
                    this.updateSegmentMetadata(position, batch, appendTimeMs);
                    position += batch.sizeInBytes();
                }
            }
        });
    }

    private void mayUpdateChecksum(E2EChecksumStore checksumStore, ByteBuffer buffer, long appendTimeMs) {
        if (this.e2eChecksumEnabledForTopic && checksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.SEGMENT)) {
            checksumStore.store().update(this.log.file().getAbsolutePath(), buffer, appendTimeMs);
        }
    }

    private void ensureOffsetInRange(long offset) throws IOException {
        if (!this.canConvertToRelativeOffset(offset)) {
            throw new LogSegmentOffsetOverflowException(this, offset);
        }
    }

    private void updateSegmentMetadata(int position, RecordBatch batch, long appendTimeMs) throws IOException {
        long lastOffset = batch.lastOffset();
        long batchMaxTimestamp = batch.maxTimestamp();
        this.lastSegmentOffset.set(lastOffset);
        this.ensureOffsetInRange(lastOffset);
        if (position == 0) {
            this.rollingBasedTimestamp = OptionalLong.of(batchMaxTimestamp);
        }
        if (batchMaxTimestamp > this.maxTimestampSoFar()) {
            this.maxTimestampAndOffsetSoFar = new TimestampOffset(batchMaxTimestamp, lastOffset);
        }
        if (this.bytesSinceLastIndexEntry > this.indexIntervalBytes) {
            if (!this.offsetIndex().isFull()) {
                LogSegment.time(OFFSET_INDEX_APPEND_TIME_MS, () -> {
                    this.offsetIndex().append(lastOffset, position, appendTimeMs);
                    return null;
                });
            }
            if (!this.timeIndex().isFull()) {
                LogSegment.time(TIMESTAMP_INDEX_APPEND_TIME_MS, () -> {
                    this.timeIndex().maybeAppend(this.maxTimestampSoFar(), this.shallowOffsetOfMaxTimestampSoFar(), appendTimeMs);
                    return null;
                });
            }
            this.bytesSinceLastIndexEntry = 0;
        }
        this.bytesSinceLastIndexEntry += batch.sizeInBytes();
    }

    private int appendChunkFromFile(FileRecords records, int position, BufferSupplier bufferSupplier) throws IOException {
        FileLogInputStream.FileChannelRecordBatch batch;
        int bytesToAppend = 0;
        long maxOffset = Long.MIN_VALUE;
        ByteBuffer readBuffer = bufferSupplier.get(0x100000);
        Iterator<FileLogInputStream.FileChannelRecordBatch> nextBatches = records.batchesFrom(position).iterator();
        while ((batch = this.nextAppendableBatch(nextBatches, readBuffer, bytesToAppend)) != null) {
            maxOffset = batch.lastOffset();
            bytesToAppend += batch.sizeInBytes();
        }
        if (bytesToAppend > 0) {
            if (readBuffer.capacity() < bytesToAppend) {
                readBuffer = bufferSupplier.get(bytesToAppend);
            }
            readBuffer.limit(bytesToAppend);
            records.readInto(readBuffer, position);
            this.append(maxOffset, MemoryRecords.readableRecords((ByteBuffer)readBuffer));
        }
        bufferSupplier.release(readBuffer);
        return bytesToAppend;
    }

    private FileLogInputStream.FileChannelRecordBatch nextAppendableBatch(Iterator<FileLogInputStream.FileChannelRecordBatch> recordBatches, ByteBuffer readBuffer, int bytesToAppend) throws IOException {
        FileLogInputStream.FileChannelRecordBatch batch;
        if (recordBatches.hasNext() && this.canConvertToRelativeOffset((batch = recordBatches.next()).lastOffset()) && (bytesToAppend == 0 || bytesToAppend + batch.sizeInBytes() < readBuffer.capacity())) {
            return batch;
        }
        return null;
    }

    public int appendFromFile(FileRecords records, int start) throws IOException {
        return (Integer)ThreadCountersManager.wrapIOChecked(() -> {
            int position;
            int bytesAppended;
            BufferSupplier.GrowableBufferSupplier bufferSupplier = new BufferSupplier.GrowableBufferSupplier();
            for (position = start; position < start + records.sizeInBytes(); position += bytesAppended) {
                bytesAppended = this.appendChunkFromFile(records, position, (BufferSupplier)bufferSupplier);
                if (bytesAppended != 0) continue;
                return position - start;
            }
            return position - start;
        });
    }

    public void updateTxnIndex(CompletedTxn completedTxn, long lastStableOffset, long appendTimeMs) throws IOException {
        ThreadCountersManager.wrapIOChecked(() -> {
            if (completedTxn.isAborted) {
                LOGGER.trace("Writing aborted transaction {} to transaction index, last stable offset is {}", (Object)completedTxn, (Object)lastStableOffset);
                this.txnIndex.append(new AbortedTxn(completedTxn, lastStableOffset), appendTimeMs);
            }
        });
    }

    private void updateProducerState(ProducerStateManager producerStateManager, RecordBatch batch, long currentTimeMs) throws IOException {
        if (batch.hasProducerId()) {
            long producerId = batch.producerId();
            ProducerAppendInfo appendInfo = producerStateManager.prepareUpdate(producerId, AppendOrigin.REPLICATION, currentTimeMs);
            Optional<CompletedTxn> maybeCompletedTxn = appendInfo.append(batch, Optional.empty());
            producerStateManager.update(appendInfo);
            if (maybeCompletedTxn.isPresent()) {
                CompletedTxn completedTxn = maybeCompletedTxn.get();
                long lastStableOffset = producerStateManager.proposedLastStableOffset(completedTxn);
                this.updateTxnIndex(completedTxn, lastStableOffset, currentTimeMs);
                producerStateManager.completeTxn(completedTxn);
            }
        }
        producerStateManager.updateMapEndOffset(batch.lastOffset() + 1L);
    }

    public FileRecords.LogOffsetPosition translateOffset(long offset) throws IOException {
        return this.translateOffset(offset, 0);
    }

    public FileRecords.LogOffsetPosition translateOffset(long offset, int startingFilePosition) throws IOException {
        OffsetPosition mapping = this.offsetIndex().lookup(offset);
        return this.log.searchForOffsetWithSize(offset, Math.max(mapping.position, startingFilePosition));
    }

    public FetchDataInfo read(long startOffset, int maxSize) throws IOException {
        return this.read(startOffset, maxSize, this.size());
    }

    public FetchDataInfo read(long startOffset, int maxSize, long maxPosition) throws IOException {
        return this.read(startOffset, maxSize, Optional.of(maxPosition), false);
    }

    public FetchDataInfo read(long startOffset, int maxSize, Optional<Long> maxPositionOpt, boolean minOneMessage) throws IOException {
        return (FetchDataInfo)ThreadCountersManager.wrapIOChecked(() -> {
            if (maxSize < 0) {
                throw new IllegalArgumentException("Invalid max size " + maxSize + " for log read from segment " + this.log);
            }
            FileRecords.LogOffsetPosition startOffsetAndSize = this.translateOffset(startOffset);
            if (startOffsetAndSize == null) {
                return null;
            }
            int startPosition = startOffsetAndSize.position;
            LogOffsetMetadata offsetMetadata = new LogOffsetMetadata(startOffset, this.baseOffset, startPosition);
            int adjustedMaxSize = maxSize;
            if (minOneMessage) {
                adjustedMaxSize = Math.max(maxSize, startOffsetAndSize.size);
            }
            if (adjustedMaxSize == 0 || !maxPositionOpt.isPresent()) {
                return new FetchDataInfo(offsetMetadata, (Records)MemoryRecords.EMPTY);
            }
            int fetchSize = Math.min((int)((Long)maxPositionOpt.get() - (long)startPosition), adjustedMaxSize);
            return new FetchDataInfo(offsetMetadata, (Records)this.log.slice(startPosition, fetchSize), adjustedMaxSize < startOffsetAndSize.size, Optional.empty());
        });
    }

    public OptionalLong fetchUpperBoundOffset(OffsetPosition startOffsetPosition, int fetchSize) throws IOException {
        return (OptionalLong)ThreadCountersManager.wrapIOChecked(() -> {
            Optional<OffsetPosition> position = this.offsetIndex().fetchUpperBoundOffset(startOffsetPosition, fetchSize);
            if (position.isPresent()) {
                return OptionalLong.of(position.get().offset);
            }
            return OptionalLong.empty();
        });
    }

    public int recover(ProducerStateManager producerStateManager, Optional<LeaderEpochFileCache> leaderEpochCache) throws IOException {
        return (Integer)ThreadCountersManager.wrapIOChecked(() -> {
            this.checksumStoreOpt.ifPresent(this::mayInitializeChecksumEntries);
            this.offsetIndex().reset();
            this.timeIndex().reset();
            this.txnIndex.reset();
            int validBytes = 0;
            this.maxTimestampAndOffsetSoFar = TimestampOffset.UNKNOWN;
            long currentTimeMs = this.time.milliseconds();
            try {
                for (RecordBatch batch : this.log.batches()) {
                    batch.ensureValid();
                    this.checksumStoreOpt.ifPresent(checksumStore -> this.mayUpdateChecksum((E2EChecksumStore)checksumStore, batch.buffer(), currentTimeMs));
                    this.updateSegmentMetadata(validBytes, batch, currentTimeMs);
                    validBytes += batch.sizeInBytes();
                    if (batch.magic() < 2) continue;
                    leaderEpochCache.ifPresent(cache -> {
                        if (!(batch.partitionLeaderEpoch() < 0 || cache.latestEpoch().isPresent() && batch.partitionLeaderEpoch() <= cache.latestEpoch().getAsInt())) {
                            cache.assign(batch.partitionLeaderEpoch(), batch.baseOffset());
                        }
                    });
                    this.updateProducerState(producerStateManager, batch, currentTimeMs);
                }
            }
            catch (InvalidRecordException | CorruptRecordException e) {
                LOGGER.warn("Found invalid messages in log segment {} at byte offset {}: {}. {}", new Object[]{this.log.file().getAbsolutePath(), validBytes, e.getMessage(), e.getCause()});
            }
            int truncated = this.log.sizeInBytes() - validBytes;
            if (truncated > 0) {
                LOGGER.debug("Truncated {} invalid bytes at the end of segment {} during recovery", (Object)truncated, (Object)this.log.file().getAbsolutePath());
            }
            this.log.truncateTo(validBytes);
            this.offsetIndex().trimToValidSize();
            this.timeIndex().maybeAppend(this.maxTimestampSoFar(), this.shallowOffsetOfMaxTimestampSoFar(), true);
            this.timeIndex().trimToValidSize();
            return truncated;
        });
    }

    public void mayInitializeChecksumEntries(E2EChecksumStore e2eChecksumStore) {
        if (!this.e2eChecksumEnabledForTopic) {
            return;
        }
        ChecksumStore store = e2eChecksumStore.store();
        if (e2eChecksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.SEGMENT)) {
            store.initializeEntry(this.log.file().getAbsolutePath(), this.shouldPersistChecksum);
        }
        if (e2eChecksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.OFFSET_INDEX)) {
            store.initializeEntry(this.lazyOffsetIndex.file().getAbsolutePath(), this.shouldPersistChecksum);
        }
        if (e2eChecksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.TIMESTAMP_INDEX)) {
            store.initializeEntry(this.lazyTimeIndex.file().getAbsolutePath(), this.shouldPersistChecksum);
        }
    }

    public boolean hasOverflow() throws IOException {
        long nextOffset = this.readNextOffset();
        return nextOffset > this.baseOffset && !this.canConvertToRelativeOffset(nextOffset - 1L);
    }

    public TxnIndexSearchResult collectAbortedTxns(long fetchOffset, long upperBoundOffset, boolean shouldValidateChecksum) {
        return (TxnIndexSearchResult)ThreadCountersManager.wrapIO(() -> this.txnIndex.collectAbortedTxns(fetchOffset, upperBoundOffset, shouldValidateChecksum));
    }

    public String toString() {
        return "LogSegment(baseOffset=" + this.baseOffset + ", size=" + this.size() + ", lastModifiedTime=" + this.lastModified() + ", largestRecordTimestamp=" + this.maxTimestampAndOffsetSoFar.timestamp + ")";
    }

    public int truncateTo(long offset) throws IOException {
        return (Integer)ThreadCountersManager.wrapIOChecked(() -> {
            int bytesTruncated;
            FileRecords.LogOffsetPosition mapping = this.translateOffset(offset);
            OffsetIndex offsetIndex = this.offsetIndex();
            TimeIndex timeIndex = this.timeIndex();
            offsetIndex.truncateTo(offset);
            timeIndex.truncateTo(offset);
            this.txnIndex.truncateTo(offset);
            offsetIndex.resize(offsetIndex.maxIndexSize());
            timeIndex.resize(timeIndex.maxIndexSize());
            if (mapping == null) {
                bytesTruncated = 0;
            } else {
                if (this.checksumStoreOpt.isPresent()) {
                    this.mayTruncateChecksum(this.checksumStoreOpt.get(), mapping.position);
                }
                bytesTruncated = this.log.truncateTo(mapping.position);
            }
            if (this.log.sizeInBytes() == 0) {
                this.created = this.time.milliseconds();
                this.rollingBasedTimestamp = OptionalLong.empty();
            }
            this.bytesSinceLastIndexEntry = 0;
            if (this.maxTimestampSoFar() >= 0L) {
                this.maxTimestampAndOffsetSoFar = this.readLargestTimestamp();
            }
            this.lastSegmentOffset.set(-1L);
            return bytesTruncated;
        });
    }

    private TimestampOffset readLargestTimestamp() throws IOException {
        TimestampOffset lastTimeIndexEntry = this.timeIndex().lastEntry();
        OffsetPosition offsetPosition = this.offsetIndex().lookup(lastTimeIndexEntry.offset);
        FileRecords.FileTimestampAndOffset maxTimestampOffsetAfterLastEntry = this.log.largestTimestampAfter(offsetPosition.position);
        if (maxTimestampOffsetAfterLastEntry.timestamp > lastTimeIndexEntry.timestamp) {
            return new TimestampOffset(maxTimestampOffsetAfterLastEntry.timestamp, maxTimestampOffsetAfterLastEntry.offset);
        }
        return lastTimeIndexEntry;
    }

    private void mayTruncateChecksum(E2EChecksumStore e2eChecksumStore, int targetSize) throws IOException {
        if (!this.e2eChecksumEnabledForTopic || !e2eChecksumStore.checksumProtectionEnabled(E2EChecksumProtectedObjectType.SEGMENT)) {
            return;
        }
        if (!e2eChecksumStore.store().contains(this.log.file().getAbsolutePath())) {
            return;
        }
        if (this.shouldTruncateMoreThanHalf(targetSize)) {
            this.recalculateFromBeginning(e2eChecksumStore, targetSize);
        } else {
            this.truncateFromEnd(e2eChecksumStore, targetSize);
        }
    }

    private boolean shouldTruncateMoreThanHalf(int targetSize) {
        return targetSize < this.log.sizeInBytes() / 2;
    }

    private void truncateFromEnd(E2EChecksumStore e2eChecksumStore, int targetSize) throws IOException {
        int currentSize = this.log.sizeInBytes();
        if (targetSize < currentSize) {
            int bytesToRead = currentSize - targetSize;
            ByteBuffer buffer = ByteBuffer.allocate(bytesToRead);
            this.log.readInto(buffer, targetSize);
            e2eChecksumStore.store().truncate(this.log.file().getAbsolutePath(), currentSize, buffer);
        }
    }

    private void recalculateFromBeginning(E2EChecksumStore e2eChecksumStore, int targetSize) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(targetSize);
        this.log.readInto(buffer, 0);
        ChecksumStore store = e2eChecksumStore.store();
        store.initializeEntry(this.log.file().getAbsolutePath());
        store.update(this.log.file().getAbsolutePath(), buffer);
    }

    public long readNextOffset() throws IOException {
        return (Long)ThreadCountersManager.wrapIOChecked(() -> {
            long scannedNextOffset;
            do {
                long cachedLastSegmentOffset;
                if ((cachedLastSegmentOffset = this.lastSegmentOffset.get()) != -1L) {
                    return cachedLastSegmentOffset + 1L;
                }
                scannedNextOffset = this.scanNextOffset();
            } while (!this.lastSegmentOffset.compareAndSet(-1L, scannedNextOffset - 1L));
            return scannedNextOffset;
        });
    }

    public long scanNextOffset() throws IOException {
        FetchDataInfo fetchData = this.read(this.offsetIndex().lastOffset(), this.log.sizeInBytes());
        if (fetchData == null) {
            return this.baseOffset;
        }
        return fetchData.records.lastBatch().map(batch -> batch.nextOffset()).orElse(this.baseOffset);
    }

    public void flush() throws IOException {
        ThreadCountersManager.wrapIOChecked(() -> {
            try {
                LOG_FLUSH_TIMER.time((Callable)new Callable<Void>(){

                    @Override
                    public Void call() throws IOException {
                        LogSegment.this.log.flush();
                        LogSegment.this.offsetIndex().flush();
                        LogSegment.this.timeIndex().flush();
                        LogSegment.this.txnIndex.flush();
                        LogSegment.this.lastFlushedTimeMs = OptionalLong.of(LogSegment.this.time.milliseconds());
                        return null;
                    }
                });
            }
            catch (Exception e) {
                if (e instanceof IOException) {
                    throw (IOException)e;
                }
                if (e instanceof RuntimeException) {
                    throw (RuntimeException)e;
                }
                throw new IllegalStateException("Unexpected exception thrown: " + e, e);
            }
        });
    }

    void updateParentDir(File dir) {
        ThreadCountersManager.wrapIOVoid(() -> {
            String oldSegmentPath = this.log.file().getAbsolutePath();
            String oldOffsetIndexPath = this.offsetIndexFile().getAbsolutePath();
            String oldTimeIndexPath = this.timeIndexFile().getAbsolutePath();
            String oldTxnIndexPath = this.txnIndex.file().getAbsolutePath();
            this.log.updateParentDir(dir);
            this.lazyOffsetIndex.updateParentDir(dir);
            this.lazyTimeIndex.updateParentDir(dir);
            this.txnIndex.updateParentDir(dir);
            this.checksumStoreOpt.ifPresent(checksumStore -> this.mayReplaceChecksumStoreEntries((E2EChecksumStore)checksumStore, oldSegmentPath, oldOffsetIndexPath, oldTimeIndexPath, oldTxnIndexPath));
        });
    }

    public void changeFileSuffixes(String oldSuffix, String newSuffix) throws IOException {
        ThreadCountersManager.wrapIOChecked(() -> {
            String oldSegmentPath = this.log.file().getAbsolutePath();
            String oldOffsetIndexPath = this.offsetIndexFile().getAbsolutePath();
            String oldTimeIndexPath = this.timeIndexFile().getAbsolutePath();
            String oldTxnIndexPath = this.txnIndex.file().getAbsolutePath();
            this.log.renameTo(new File(Utils.replaceSuffix((String)this.log.file().getPath(), (String)oldSuffix, (String)newSuffix)));
            this.lazyOffsetIndex.renameTo(new File(Utils.replaceSuffix((String)this.offsetIndexFile().getPath(), (String)oldSuffix, (String)newSuffix)));
            this.lazyTimeIndex.renameTo(new File(Utils.replaceSuffix((String)this.timeIndexFile().getPath(), (String)oldSuffix, (String)newSuffix)));
            this.txnIndex.renameTo(new File(Utils.replaceSuffix((String)this.txnIndex.file().getPath(), (String)oldSuffix, (String)newSuffix)));
            this.checksumStoreOpt.ifPresent(checksumStore -> this.mayReplaceChecksumStoreEntries((E2EChecksumStore)checksumStore, oldSegmentPath, oldOffsetIndexPath, oldTimeIndexPath, oldTxnIndexPath));
        });
    }

    private void mayReplaceChecksumStoreEntries(E2EChecksumStore e2eChecksumStore, String oldSegmentPath, String oldOffsetIndexPath, String oldTimeIndexPath, String oldTxnIndexPath) {
        if (!this.e2eChecksumEnabledForTopic || !e2eChecksumStore.checksumProtectionEnabled()) {
            return;
        }
        ChecksumStore store = e2eChecksumStore.store();
        store.replace(oldSegmentPath, this.log.file().getAbsolutePath());
        store.replace(oldOffsetIndexPath, this.offsetIndexFile().getAbsolutePath());
        store.replace(oldTimeIndexPath, this.timeIndexFile().getAbsolutePath());
        store.replace(oldTxnIndexPath, this.txnIndex.file().getAbsolutePath());
    }

    public boolean hasSuffix(String suffix) {
        return this.log.file().getName().endsWith(suffix) && this.offsetIndexFile().getName().endsWith(suffix) && this.timeIndexFile().getName().endsWith(suffix) && this.txnIndex.file().getName().endsWith(suffix);
    }

    public void onBecomeInactiveSegment() throws IOException {
        ThreadCountersManager.wrapIOChecked(() -> {
            this.timeIndex().maybeAppend(this.maxTimestampSoFar(), this.shallowOffsetOfMaxTimestampSoFar(), true);
            this.offsetIndex().trimToValidSize();
            this.timeIndex().trimToValidSize();
            this.log.trim();
        });
    }

    private void loadFirstBatchTimestamp() {
        Iterator iter;
        if (!this.rollingBasedTimestamp.isPresent() && (iter = this.log.batches().iterator()).hasNext()) {
            this.rollingBasedTimestamp = OptionalLong.of(((FileLogInputStream.FileChannelRecordBatch)iter.next()).maxTimestamp());
        }
    }

    public long timeWaitedForRoll(long now, long messageTimestamp, boolean rollBasedOnSystemTime) {
        return (Long)ThreadCountersManager.wrapIO(() -> {
            long allowedTolerance = now + 3600000L;
            this.loadFirstBatchTimestamp();
            OptionalLong ts = this.rollingBasedTimestamp;
            if (rollBasedOnSystemTime && ts.isPresent() && (ts.getAsLong() > allowedTolerance || messageTimestamp > allowedTolerance)) {
                return now - this.segmentCreateTime;
            }
            if (ts.isPresent() && ts.getAsLong() >= 0L) {
                return messageTimestamp - ts.getAsLong();
            }
            return now - this.created;
        });
    }

    public long timeWaitedForForcedRoll(long now) {
        return (Long)ThreadCountersManager.wrapIO(() -> now - this.segmentCreateTime);
    }

    public long getFirstBatchTimestamp() {
        return (Long)ThreadCountersManager.wrapIO(() -> {
            this.loadFirstBatchTimestamp();
            OptionalLong timestamp = this.rollingBasedTimestamp;
            if (timestamp.isPresent() && timestamp.getAsLong() >= 0L) {
                return timestamp.getAsLong();
            }
            return Long.MAX_VALUE;
        });
    }

    public Optional<FetchedTimestampAndOffset> findOffsetByTimestamp(long timestampMs, long startingOffset) throws IOException {
        return (Optional)ThreadCountersManager.wrapIOChecked(() -> {
            TimestampOffset timestampOffset = this.timeIndex().lookup(timestampMs);
            int position = this.offsetIndex().lookup((long)Math.max((long)timestampOffset.offset, (long)startingOffset)).position;
            return Optional.ofNullable(this.log.searchForTimestamp(timestampMs, position, startingOffset)).map(found -> {
                if (found.exception != null) {
                    return new FetchedTimestampAndOffset(found.exception);
                }
                return new FetchedTimestampAndOffset(found.timestamp, found.offset, found.leaderEpoch);
            });
        });
    }

    @Override
    public void close() throws IOException {
        ThreadCountersManager.wrapIOChecked(() -> {
            if (this.maxTimestampAndOffsetSoFar != TimestampOffset.UNKNOWN) {
                Utils.swallow((Logger)LOGGER, (Level)Level.WARN, (String)"maybeAppend", () -> this.timeIndex().maybeAppend(this.maxTimestampSoFar(), this.shallowOffsetOfMaxTimestampSoFar(), true));
            }
            Utils.closeQuietly(this.lazyOffsetIndex, (String)"offsetIndex", (Logger)LOGGER);
            Utils.closeQuietly(this.lazyTimeIndex, (String)"timeIndex", (Logger)LOGGER);
            Utils.closeQuietly((AutoCloseable)this.log, (String)"log", (Logger)LOGGER);
            Utils.closeQuietly((AutoCloseable)this.txnIndex, (String)"txnIndex", (Logger)LOGGER);
            this.lastSegmentOffset.set(-1L);
        });
    }

    void closeHandlers() {
        ThreadCountersManager.wrapIOVoid(() -> {
            Utils.swallow((Logger)LOGGER, (Level)Level.WARN, (String)"offsetIndex", () -> this.lazyOffsetIndex.closeHandler());
            Utils.swallow((Logger)LOGGER, (Level)Level.WARN, (String)"timeIndex", () -> this.lazyTimeIndex.closeHandler());
            Utils.swallow((Logger)LOGGER, (Level)Level.WARN, (String)"log", () -> this.log.closeHandlers());
            Utils.closeQuietly((AutoCloseable)this.txnIndex, (String)"txnIndex", (Logger)LOGGER);
        });
    }

    public void deleteIfExists() throws IOException {
        this.checksumStoreOpt.ifPresent(store -> this.mayRemoveChecksumEntries((E2EChecksumStore)store));
        ThreadCountersManager.wrapIOChecked(() -> {
            try {
                Utils.tryAll(Arrays.asList(() -> this.deleteTypeIfExists(() -> this.log.deleteIfExists(), "log", this.log.file(), true), () -> this.deleteTypeIfExists(() -> this.lazyOffsetIndex.deleteIfExists(), "offset index", this.offsetIndexFile(), true), () -> this.deleteTypeIfExists(() -> this.lazyTimeIndex.deleteIfExists(), "time index", this.timeIndexFile(), true), () -> this.deleteTypeIfExists(() -> this.txnIndex.deleteIfExists(), "transaction index", this.txnIndex.file(), false)));
            }
            catch (Throwable t) {
                if (t instanceof IOException) {
                    throw (IOException)t;
                }
                if (t instanceof Error) {
                    throw (Error)t;
                }
                if (t instanceof RuntimeException) {
                    throw (RuntimeException)t;
                }
                throw new IllegalStateException("Unexpected exception: " + t.getMessage(), t);
            }
        });
        this.lastSegmentOffset.set(-1L);
    }

    private Void deleteTypeIfExists(StorageAction<Boolean, IOException> delete, String fileType, File file, boolean logIfMissing) throws IOException {
        try {
            if (delete.execute().booleanValue()) {
                LOGGER.info("Deleted {} {}.", (Object)fileType, (Object)file.getAbsolutePath());
            } else {
                String topicPartitionAbsolutePath;
                File fallbackFile;
                Matcher dirMatcher;
                if (logIfMissing) {
                    LOGGER.info("Failed to delete {} {} because it does not exist.", (Object)fileType, (Object)file.getAbsolutePath());
                }
                if ((dirMatcher = FUTURE_DIR_PATTERN.matcher(file.getParent())).matches() && (fallbackFile = new File(topicPartitionAbsolutePath = dirMatcher.group(1) + "-" + dirMatcher.group(2), file.getName())).exists() && file.getName().endsWith(".deleted") && fallbackFile.delete()) {
                    LOGGER.info("Fallback to delete {} {}.", (Object)fileType, (Object)fallbackFile.getAbsolutePath());
                }
            }
            return null;
        }
        catch (IOException e) {
            throw new IOException("Delete of " + fileType + " " + file.getAbsolutePath() + " failed.", e);
        }
    }

    private void mayRemoveChecksumEntries(E2EChecksumStore e2eChecksumStore) {
        if (!this.e2eChecksumEnabledForTopic || !e2eChecksumStore.checksumProtectionEnabled()) {
            return;
        }
        ChecksumStore store = e2eChecksumStore.store();
        store.remove(this.log.file().getAbsolutePath());
        store.remove(this.timeIndexFile().getAbsolutePath());
        store.remove(this.offsetIndexFile().getAbsolutePath());
        store.remove(this.txnIndex.file().getAbsolutePath());
    }

    public boolean deleted() {
        return (Boolean)ThreadCountersManager.wrapIO(() -> !this.log.file().exists() && !this.offsetIndexFile().exists() && !this.timeIndexFile().exists() && !this.txnIndex.file().exists());
    }

    public long lastModified() {
        return (Long)ThreadCountersManager.wrapIO(() -> this.log.file().lastModified());
    }

    public OptionalLong largestRecordTimestamp() throws IOException {
        long maxTimestampSoFar = this.maxTimestampSoFar();
        if (maxTimestampSoFar >= 0L) {
            return OptionalLong.of(maxTimestampSoFar);
        }
        return OptionalLong.empty();
    }

    public long largestTimestamp() throws IOException {
        long maxTimestampSoFar = this.maxTimestampSoFar();
        if (maxTimestampSoFar >= 0L) {
            return maxTimestampSoFar;
        }
        return this.lastModified();
    }

    public void setLastModified(long ms) throws IOException {
        FileTime fileTime = FileTime.fromMillis(ms);
        FilesWrapper.setLastModifiedTime((Path)this.log.file().toPath(), (FileTime)fileTime);
        FilesWrapper.setLastModifiedTime((Path)this.offsetIndexFile().toPath(), (FileTime)fileTime);
        FilesWrapper.setLastModifiedTime((Path)this.timeIndexFile().toPath(), (FileTime)fileTime);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static <T> T time(Timer timer, StorageAction<T, IOException> action) throws IOException {
        TimerContext context = timer.time();
        try {
            T t = action.execute();
            return t;
        }
        finally {
            context.stop();
        }
    }

    public static LogSegment open(File dir, long baseOffset, LogConfig config, Time time, int initFileSize, boolean preallocate, ChecksumParams checksumParams) throws IOException {
        return LogSegment.open(dir, baseOffset, config, time, false, initFileSize, preallocate, "", checksumParams);
    }

    public static LogSegment open(File dir, long baseOffset, LogConfig config, Time time, boolean fileAlreadyExists, int initFileSize, boolean preallocate, String fileSuffix, ChecksumParams checksumParams) throws IOException {
        int maxIndexSize = config.maxIndexSize;
        Optional<E2EChecksumStore> checksumStoreOpt = checksumParams.checksumStoreOpt();
        LogSegment segment = (LogSegment)ThreadCountersManager.wrapIOChecked(() -> new LogSegment(FileRecords.open((File)LogFileUtils.logFile(dir, baseOffset, fileSuffix), (boolean)fileAlreadyExists, (int)initFileSize, (boolean)preallocate), LazyIndex.forOffset(LogFileUtils.offsetIndexFile(dir, baseOffset, fileSuffix), fileAlreadyExists, baseOffset, maxIndexSize, checksumParams), LazyIndex.forTime(LogFileUtils.timeIndexFile(dir, baseOffset, fileSuffix), fileAlreadyExists, baseOffset, maxIndexSize, checksumParams), new TransactionIndex(baseOffset, LogFileUtils.transactionIndexFile(dir, baseOffset, fileSuffix), fileAlreadyExists, checksumParams), baseOffset, config.indexInterval, config.randomSegmentJitter(), time, checksumParams));
        if (!fileAlreadyExists) {
            checksumStoreOpt.ifPresent(segment::mayInitializeChecksumEntries);
        }
        return segment;
    }

    public static void deleteIfExists(File dir, long baseOffset, String fileSuffix) throws IOException {
        LogSegment.deleteFileIfExists(LogFileUtils.offsetIndexFile(dir, baseOffset, fileSuffix));
        LogSegment.deleteFileIfExists(LogFileUtils.timeIndexFile(dir, baseOffset, fileSuffix));
        LogSegment.deleteFileIfExists(LogFileUtils.transactionIndexFile(dir, baseOffset, fileSuffix));
        LogSegment.deleteFileIfExists(LogFileUtils.logFile(dir, baseOffset, fileSuffix));
    }

    private static boolean deleteFileIfExists(File file) throws IOException {
        return FilesWrapper.deleteIfExists((Path)file.toPath());
    }

    static {
        FUTURE_DIR_PATTERN = Pattern.compile("^(\\S+)-(\\S+)\\.(\\S+)-future");
        KafkaMetricsGroup logFlushStatsMetricsGroup = new KafkaMetricsGroup(LogSegment.class){

            public MetricName metricName(String name, Map<String, String> tags) {
                return KafkaMetricsGroup.explicitMetricName((String)"kafka.log", (String)"LogFlushStats", (String)name, tags);
            }
        };
        LOG_FLUSH_TIMER = logFlushStatsMetricsGroup.newTimer("LogFlushRateAndTimeMs", TimeUnit.MILLISECONDS, TimeUnit.SECONDS);
        KafkaMetricsGroup segmentStatsGroup = new KafkaMetricsGroup(LogSegment.class){

            public MetricName metricName(String name, Map<String, String> tags) {
                return KafkaMetricsGroup.explicitMetricName((String)"kafka.log", (String)"SegmentStats", (String)name, tags);
            }
        };
        SEGMENT_APPEND_TIME_MS = segmentStatsGroup.newTimer("SegmentAppendTimeMs", TimeUnit.MILLISECONDS, TimeUnit.SECONDS);
        OFFSET_INDEX_APPEND_TIME_MS = segmentStatsGroup.newTimer("OffsetIndexAppendTimeMs", TimeUnit.MILLISECONDS, TimeUnit.SECONDS);
        TIMESTAMP_INDEX_APPEND_TIME_MS = segmentStatsGroup.newTimer("TimestampIndexAppendTimeMs", TimeUnit.MILLISECONDS, TimeUnit.SECONDS);
    }
}

