/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.detector;

import com.linkedin.kafka.cruisecontrol.KafkaCruiseControlOperationMetricsTracker;
import com.linkedin.kafka.cruisecontrol.analyzer.UpdatableBalancingConstraint;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.thresholds.DistributionThresholdUtils;
import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.detector.ResourceUtilizationDetector;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.ClusterModel;
import com.yammer.metrics.core.Histogram;
import io.confluent.databalancer.metrics.GeneralSBCMetricsRegistry;
import io.confluent.databalancer.operation.BrokerAdditionV2StateManager;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.SortedSet;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BrokerAdditionDetector
implements ResourceUtilizationDetector {
    private static final Logger LOG = LoggerFactory.getLogger(BrokerAdditionDetector.class);
    public static final String BROKER_ADDITION_COMPLETION_TIMER_METRIC_NAME = "broker-addition-completion-timer";
    public static final String BROKER_ADDITION_DESIRED_NEW_BROKER_CPU_GAUGE_METRIC_NAME = "broker-addition-desired-new-broker-cpu";
    public static final String BROKER_ADDITION_COMPUTED_MEAN_CPU_GAUGE_METRIC_NAME = "broker-addition-computed-mean-cluster-cpu";
    public static final String IN_PROGRESS_ADDITIONS_METRIC_NAME = "in-progress-additions";
    private final double cpuPercentCompletionThreshold;
    private final Time time;
    private final int brokerAdditionCompletionDurationThresholdMs;
    private final Histogram brokerAdditionCompletionTimer;
    private final BrokerAdditionV2StateManager brokerAdditionV2StateManager;
    private final KafkaCruiseControlOperationMetricsTracker operationMetricsTracker;
    private final UpdatableBalancingConstraint updatableBalancingConstraint;
    private volatile double clusterMeanCpuUsage;
    private volatile double desiredBrokerCpuUsage;

    public BrokerAdditionDetector(KafkaCruiseControlConfig kccConfig, Time time, GeneralSBCMetricsRegistry metricsRegistry, BrokerAdditionV2StateManager brokerAdditionV2StateManager, KafkaCruiseControlOperationMetricsTracker operationMetricsTracker, UpdatableBalancingConstraint updatableBalancingConstraint) {
        this.brokerAdditionCompletionTimer = metricsRegistry.newHistogram(BrokerAdditionDetector.class, BROKER_ADDITION_COMPLETION_TIMER_METRIC_NAME);
        metricsRegistry.newGauge(BrokerAdditionDetector.class, BROKER_ADDITION_DESIRED_NEW_BROKER_CPU_GAUGE_METRIC_NAME, this::desiredBrokerCpuUsage);
        metricsRegistry.newGauge(BrokerAdditionDetector.class, BROKER_ADDITION_COMPUTED_MEAN_CPU_GAUGE_METRIC_NAME, this::clusterMeanCpuUsage);
        metricsRegistry.newGauge(BrokerAdditionDetector.class, IN_PROGRESS_ADDITIONS_METRIC_NAME, this::numInProgressAdditions);
        this.cpuPercentCompletionThreshold = kccConfig.getDouble("broker.addition.mean.cpu.percent.completion.threshold");
        this.brokerAdditionCompletionDurationThresholdMs = kccConfig.getInt("broker.addition.elapsed.time.ms.completion.threshold");
        this.brokerAdditionV2StateManager = brokerAdditionV2StateManager;
        this.operationMetricsTracker = operationMetricsTracker;
        this.updatableBalancingConstraint = updatableBalancingConstraint;
        this.time = time;
    }

    private double clusterMeanCpuUsage() {
        return this.clusterMeanCpuUsage;
    }

    private double desiredBrokerCpuUsage() {
        return this.desiredBrokerCpuUsage;
    }

    private int numInProgressAdditions() {
        return this.brokerAdditionV2StateManager.pendingBrokerAdditions().size();
    }

    @Override
    public void detectResourceUtilization(ClusterModel clusterModel) {
        try {
            if (this.numInProgressAdditions() == 0) {
                this.operationMetricsTracker.completeOperation(KafkaCruiseControlOperationMetricsTracker.Operation.BROKER_ADDITION);
                this.changeCpuLowUtilizationThreshold(false, "No broker additions in progress");
                return;
            }
            LOG.info("Broker addition detector detected broker additions in progress: {}", this.brokerAdditionV2StateManager.pendingBrokerAdditions());
            if (clusterModel.isCellEnabled()) {
                this.handleCellEnabledClusters();
                return;
            }
            this.changeCpuLowUtilizationThreshold(true, "Broker additions in progress");
            this.operationMetricsTracker.beginOperation(KafkaCruiseControlOperationMetricsTracker.Operation.BROKER_ADDITION);
            SortedSet<Broker> brokerSet = clusterModel.allBrokers();
            double lowCpuUtilizationThreshold = this.cpuLowUtilizationThreshold();
            boolean brokerExistsAboveLowUtil = brokerSet.stream().anyMatch(broker -> broker.load().expectedUtilizationFor(Resource.CPU) / 100.0 >= lowCpuUtilizationThreshold);
            List<BrokerAdditionV2StateManager.PendingAddition> pendingBrokers = this.brokerAdditionV2StateManager.pendingBrokerAdditions();
            if (!brokerExistsAboveLowUtil) {
                LOG.info("Marking all pending broker additions as complete, as all brokers have CPU Utilization less than minimum CPU Utilization threshold: {}. Broker Ids are: {}.", (Object)lowCpuUtilizationThreshold, (Object)Arrays.toString(pendingBrokers.stream().map(b -> b.brokerId).toArray()));
                for (BrokerAdditionV2StateManager.PendingAddition brokerToAdd : pendingBrokers) {
                    this.brokerAdditionV2StateManager.completeAddition(brokerToAdd.brokerId);
                }
                return;
            }
            this.clusterMeanCpuUsage = DistributionThresholdUtils.clusterUtilizationAverage(clusterModel, Resource.CPU);
            this.desiredBrokerCpuUsage = this.clusterMeanCpuUsage * this.cpuPercentCompletionThreshold;
            LOG.debug("Expected CPU Utilization for broker addition operation to complete is : {} ({}% of mean Utilization {}), since one or more brokers have CPU usage more than {}% ", new Object[]{this.desiredBrokerCpuUsage, this.cpuPercentCompletionThreshold, this.clusterMeanCpuUsage, lowCpuUtilizationThreshold});
            HashMap<Integer, Double> notCompletedBrokers = new HashMap<Integer, Double>();
            for (BrokerAdditionV2StateManager.PendingAddition brokerToAdd : pendingBrokers) {
                try {
                    Broker broker2 = clusterModel.broker(brokerToAdd.brokerId);
                    if (broker2 == null) {
                        LOG.warn("Unable to find appropriate broker from cluster model for broker id: {}", (Object)brokerToAdd.brokerId);
                        continue;
                    }
                    double brokerCpuUsage = broker2.load().expectedUtilizationFor(Resource.CPU) / 100.0;
                    long brokerCreationTime = brokerToAdd.createdTimeMs;
                    long currentTime = this.time.milliseconds();
                    long timeSinceCreation = currentTime - brokerCreationTime;
                    if (brokerCpuUsage >= this.desiredBrokerCpuUsage) {
                        this.markBrokerAdditionCompleted(brokerToAdd.brokerId, timeSinceCreation);
                        LOG.info("Marking the broker addition operation for broker: {} as complete. Time taken for completion is: {}ms. Current CPU of Broker:{}, Mean CPU of Cluster:{}, Desired CPU of Broker:{}.", new Object[]{brokerToAdd, timeSinceCreation, brokerCpuUsage, this.clusterMeanCpuUsage, this.desiredBrokerCpuUsage});
                        continue;
                    }
                    if (timeSinceCreation >= (long)this.brokerAdditionCompletionDurationThresholdMs) {
                        this.timeoutV2BrokerAddition(brokerToAdd.brokerId, brokerCpuUsage, timeSinceCreation);
                        continue;
                    }
                    notCompletedBrokers.put(brokerToAdd.brokerId, brokerCpuUsage);
                }
                catch (Exception e2) {
                    LOG.error("Broker Addition Detector encountered an unexpected exception, while evaluating the completion of the broker addition operations for broker Id:{}.", (Object)brokerToAdd.brokerId, (Object)e2);
                }
            }
            if (!notCompletedBrokers.isEmpty()) {
                String notCompletedBrokersStr = String.join((CharSequence)", ", (CharSequence[])notCompletedBrokers.entrySet().stream().map(e -> String.format("kafka-%d:%.02f", e.getKey(), e.getValue())).toArray(String[]::new));
                LOG.info("Finished the broker addition completion check and was left with {} brokers that have not completed their addition yet because they didn't reach the desired {}% CPU threshold ({}% of mean Utilization {}). The brokers and their current CPU usage: {}", new Object[]{notCompletedBrokers.size(), this.desiredBrokerCpuUsage, this.cpuPercentCompletionThreshold, this.clusterMeanCpuUsage, notCompletedBrokersStr});
            }
        }
        catch (Exception e3) {
            LOG.error("Broker Addition Detector encountered an unexpected exception, while evaluating the completion of the broker addition operations for brokers.", (Throwable)e3);
        }
    }

    private void handleCellEnabledClusters() throws InterruptedException {
        LOG.info("Marking all pending broker additions as complete as it is cell enabled cluster. The current expansion threshold expects all brokers to have an average CPU but SBC doesn\u2019t balance across cells. Broker Ids are: {}.", this.brokerAdditionV2StateManager.pendingBrokerAdditions());
        List<BrokerAdditionV2StateManager.PendingAddition> pendingBrokers = this.brokerAdditionV2StateManager.pendingBrokerAdditions();
        for (BrokerAdditionV2StateManager.PendingAddition brokerToAdd : pendingBrokers) {
            this.brokerAdditionV2StateManager.completeAddition(brokerToAdd.brokerId);
        }
    }

    private void markBrokerAdditionCompleted(Integer brokerToAdd, long timeSinceCreation) throws InterruptedException {
        this.brokerAdditionV2StateManager.completeAddition(brokerToAdd);
        this.brokerAdditionCompletionTimer.update(timeSinceCreation);
    }

    private synchronized void timeoutV2BrokerAddition(int brokerId, double brokerCPUUsage, long timeSinceCreation) throws InterruptedException {
        long timeoutInSeconds = Duration.ofMillis(this.brokerAdditionCompletionDurationThresholdMs).getSeconds();
        String cause = "Broker Addition Timeout Occurred (" + timeoutInSeconds + " seconds)";
        BrokerAdditionV2StateManager.CancellationResult cancellationResult = this.brokerAdditionV2StateManager.maybeCancelAddition(brokerId, cause);
        if (cancellationResult.cancelled) {
            LOG.warn("Broker Addition Detector detected an addition timeout for broker {}. The broker has been created {} seconds before and it took more than allocated time of {} seconds to reach the expected cpu utilization threshold of {} -- the broker is still at {}", new Object[]{brokerId, Duration.ofMillis(timeSinceCreation).getSeconds(), timeoutInSeconds, this.desiredBrokerCpuUsage, brokerCPUUsage});
        } else {
            LOG.warn("Broker Addition Detector detected a timeout for broker {} as its already been {} seconds since its creation but it could not cancel it because of {}. (the reason for cancellation was: {}). It's possible that the addition was in a terminal state (already cancelled or completed).", new Object[]{brokerId, Duration.ofMillis(timeSinceCreation).getSeconds(), cancellationResult.failureReason, cause});
        }
    }

    private Double cpuLowUtilizationThreshold() {
        return this.updatableBalancingConstraint.balancingConstraint().lowUtilizationRatio(Resource.CPU);
    }

    void changeCpuLowUtilizationThreshold(boolean useCpuLowUtilizationForBrokerAdditions, String reason) {
        boolean willChange;
        boolean bl = willChange = this.updatableBalancingConstraint.useCpuLowUtilizationThresholdForBrokerAddition() != useCpuLowUtilizationForBrokerAdditions;
        if (willChange) {
            LOG.info("Changing the CPU low utilization threshold to use the one for broker addition(s): '{}' (if false, the default CPU low utilization value is set), this is due to '{}'. Broker additions: {}", new Object[]{useCpuLowUtilizationForBrokerAdditions, reason, this.brokerAdditionV2StateManager.pendingBrokerAdditions()});
        }
        if (useCpuLowUtilizationForBrokerAdditions) {
            this.updatableBalancingConstraint.setCpuLowUtilizationThresholdForBrokerAddition();
            return;
        }
        this.updatableBalancingConstraint.setDefaultCpuLowUtilizationThreshold();
    }
}

