/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.detector;

import com.linkedin.kafka.cruisecontrol.KafkaCruiseControlOperationMetricsTracker;
import com.linkedin.kafka.cruisecontrol.analyzer.UpdatableBalancingConstraint;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.thresholds.DistributionThresholdUtils;
import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.detector.ResourceUtilizationDetector;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.ClusterModel;
import com.yammer.metrics.core.Histogram;
import io.confluent.databalancer.metrics.GeneralSBCMetricsRegistry;
import io.confluent.databalancer.operation.BrokerAdditionV2StateManager;
import java.time.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BrokerAdditionDetectorWithTopicDistribution
implements ResourceUtilizationDetector {
    private static final Logger LOG = LoggerFactory.getLogger(BrokerAdditionDetectorWithTopicDistribution.class);
    public static final String BROKER_ADDITION_COMPLETION_TIMER_METRIC_NAME = "broker-addition-completion-timer";
    public static final String BROKER_ADDITION_DESIRED_NEW_BROKER_CPU_GAUGE_METRIC_NAME = "broker-addition-desired-new-broker-cpu";
    public static final String BROKER_ADDITION_COMPUTED_MEAN_CPU_GAUGE_METRIC_NAME = "broker-addition-computed-mean-cluster-cpu";
    public static final String IN_PROGRESS_ADDITIONS_METRIC_NAME = "in-progress-additions";
    public static final String TOPIC_DISTRIBUTION_NUM_BALANCED_TOPICS = "topic-distribution-num-balanced-topics";
    public static final String TOPIC_DISTRIBUTION_NUM_TOPICS_REQUIRED_FOR_COMPLETION = "topic-distribution-num-topics-required-for-completion";
    private static final double NO_IMBALANCE_SCORE = Double.MAX_VALUE;
    private final double cpuPercentCompletionThreshold;
    private final double topicDistributionPercentCompletionThreshold;
    private final double balancedTopicScoreThreshold;
    private final Time time;
    private final int brokerAdditionCompletionDurationThresholdMs;
    private final Histogram brokerAdditionCompletionTimer;
    private final BrokerAdditionV2StateManager brokerAdditionV2StateManager;
    private final KafkaCruiseControlOperationMetricsTracker operationMetricsTracker;
    private final UpdatableBalancingConstraint updatableBalancingConstraint;
    private long numTotalTopics;
    private volatile double clusterMeanCpuUsage;
    private volatile double desiredBrokerCpuUsage;
    private volatile long numBalancedTopics;
    private volatile long numBalancedTopicsRequiredForCompletion;

    public BrokerAdditionDetectorWithTopicDistribution(KafkaCruiseControlConfig kccConfig, Time time, GeneralSBCMetricsRegistry metricsRegistry, BrokerAdditionV2StateManager brokerAdditionV2StateManager, KafkaCruiseControlOperationMetricsTracker operationMetricsTracker, UpdatableBalancingConstraint updatableBalancingConstraint) {
        this.brokerAdditionCompletionTimer = metricsRegistry.newHistogram(BrokerAdditionDetectorWithTopicDistribution.class, BROKER_ADDITION_COMPLETION_TIMER_METRIC_NAME);
        metricsRegistry.newGauge(BrokerAdditionDetectorWithTopicDistribution.class, BROKER_ADDITION_DESIRED_NEW_BROKER_CPU_GAUGE_METRIC_NAME, () -> this.desiredBrokerCpuUsage);
        metricsRegistry.newGauge(BrokerAdditionDetectorWithTopicDistribution.class, BROKER_ADDITION_COMPUTED_MEAN_CPU_GAUGE_METRIC_NAME, () -> this.clusterMeanCpuUsage);
        metricsRegistry.newGauge(BrokerAdditionDetectorWithTopicDistribution.class, IN_PROGRESS_ADDITIONS_METRIC_NAME, this::numInProgressAdditions);
        metricsRegistry.newGauge(BrokerAdditionDetectorWithTopicDistribution.class, TOPIC_DISTRIBUTION_NUM_BALANCED_TOPICS, () -> this.numBalancedTopics);
        metricsRegistry.newGauge(BrokerAdditionDetectorWithTopicDistribution.class, TOPIC_DISTRIBUTION_NUM_TOPICS_REQUIRED_FOR_COMPLETION, () -> this.numBalancedTopicsRequiredForCompletion);
        this.cpuPercentCompletionThreshold = kccConfig.getDouble("broker.addition.mean.cpu.percent.completion.threshold");
        this.topicDistributionPercentCompletionThreshold = kccConfig.getDouble("topic.balancing.broker.addition.completion.percentage");
        this.balancedTopicScoreThreshold = kccConfig.getDouble("topic.balancing.imbalanced.score.threshold");
        this.brokerAdditionCompletionDurationThresholdMs = kccConfig.getInt("broker.addition.elapsed.time.ms.completion.threshold");
        this.brokerAdditionV2StateManager = brokerAdditionV2StateManager;
        this.operationMetricsTracker = operationMetricsTracker;
        this.updatableBalancingConstraint = updatableBalancingConstraint;
        this.numTotalTopics = 0L;
        this.time = time;
    }

    private int numInProgressAdditions() {
        return this.brokerAdditionV2StateManager.pendingBrokerAdditions().size();
    }

    @Override
    public void detectResourceUtilization(ClusterModel clusterModel) {
        try {
            if (this.numInProgressAdditions() == 0) {
                this.resetMetrics();
                return;
            }
            if (clusterModel.isCellEnabled()) {
                this.handleCellEnabledClusters();
                return;
            }
            long currentTime = this.time.milliseconds();
            List<BrokerAdditionV2StateManager.PendingAddition> pendingBrokerAdditions = this.filterOutTimedOutAdditions(clusterModel, this.brokerAdditionV2StateManager.pendingBrokerAdditions(), currentTime);
            Double lowCpuUtilizationThreshold = this.lowCpuUtilizationThreshold();
            boolean clusterHasHighCpuUsage = clusterModel.allBrokers().stream().anyMatch(broker -> broker.load().expectedUtilizationFor(Resource.CPU) / 100.0 > lowCpuUtilizationThreshold);
            if (clusterHasHighCpuUsage) {
                this.operationMetricsTracker.beginOperation(KafkaCruiseControlOperationMetricsTracker.Operation.BROKER_ADDITION_WITH_CPU);
                this.clusterMeanCpuUsage = DistributionThresholdUtils.clusterUtilizationAverage(clusterModel, Resource.CPU);
                this.desiredBrokerCpuUsage = this.clusterMeanCpuUsage * this.cpuPercentCompletionThreshold;
                LOG.info("Checking the CPU usage of brokers pending addition where pending brokers should have at least {}% CPU usage ({}% of mean Utilization {}%), since one or more brokers have CPU usage more than {}%, pending broker CPU usages: {}", new Object[]{this.desiredBrokerCpuUsage * 100.0, this.cpuPercentCompletionThreshold * 100.0, this.clusterMeanCpuUsage * 100.0, lowCpuUtilizationThreshold * 100.0, this.pendingBrokersCpuUsageString(pendingBrokerAdditions, clusterModel)});
                this.tryCompletingBasedOnCpu(clusterModel, pendingBrokerAdditions, currentTime);
            } else {
                this.operationMetricsTracker.beginOperation(KafkaCruiseControlOperationMetricsTracker.Operation.BROKER_ADDITION_WITH_TRDG);
                this.numTotalTopics = clusterModel.topics().size();
                this.numBalancedTopicsRequiredForCompletion = Math.round((double)this.numTotalTopics * this.topicDistributionPercentCompletionThreshold);
                this.numBalancedTopics = clusterModel.topics().stream().filter(topic -> this.isTopicBalanced((String)topic, clusterModel)).count();
                LOG.info("Checking cluster's topic distribution, the aim is to have {} topics balanced out of all {} ({}% of all topics), since it's underutilized on CPU, meaning all brokers have CPU usage less than {}%, a balanced topic should have an imbalance score of {} or below, pending broker CPU usages: {}", new Object[]{this.numBalancedTopicsRequiredForCompletion, this.numTotalTopics, this.topicDistributionPercentCompletionThreshold * 100.0, lowCpuUtilizationThreshold * 100.0, this.balancedTopicScoreThreshold, this.pendingBrokersCpuUsageString(pendingBrokerAdditions, clusterModel)});
                this.tryCompletingBasedOnTopicDistribution(clusterModel, pendingBrokerAdditions, currentTime);
            }
        }
        catch (Exception e) {
            LOG.error("Broker Addition Detector encountered an unexpected exception, while evaluating the completion of the broker addition operations for brokers: {}", this.brokerAdditionV2StateManager.pendingBrokerAdditions(), (Object)e);
        }
    }

    private void handleCellEnabledClusters() throws InterruptedException {
        LOG.info("Marking all pending broker additions as complete as it is cell enabled cluster. The current expansion threshold expects all brokers to have an average CPU but SBC doesn\u2019t balance across cells. Broker Ids are: {}.", this.brokerAdditionV2StateManager.pendingBrokerAdditions());
        List<BrokerAdditionV2StateManager.PendingAddition> pendingBrokers = this.brokerAdditionV2StateManager.pendingBrokerAdditions();
        for (BrokerAdditionV2StateManager.PendingAddition brokerToAdd : pendingBrokers) {
            this.brokerAdditionV2StateManager.completeAddition(brokerToAdd.brokerId);
        }
    }

    private void resetMetrics() {
        this.operationMetricsTracker.completeOperation(KafkaCruiseControlOperationMetricsTracker.Operation.BROKER_ADDITION);
        this.clusterMeanCpuUsage = 0.0;
        this.desiredBrokerCpuUsage = 0.0;
        this.numBalancedTopicsRequiredForCompletion = 0L;
        this.numBalancedTopics = 0L;
    }

    private void tryCompletingBasedOnTopicDistribution(ClusterModel clusterModel, List<BrokerAdditionV2StateManager.PendingAddition> pendingAdditions, long currentTime) {
        boolean enoughTopicsAreBalanced;
        boolean clusterHasNoTopics;
        Set missingBrokersFromClusterModel = pendingAdditions.stream().map(pendingAddition -> pendingAddition.brokerId).filter(brokerId -> Objects.isNull(clusterModel.broker((int)brokerId))).collect(Collectors.toSet());
        if (!missingBrokersFromClusterModel.isEmpty()) {
            LOG.warn("Cluster model is stale, skipping the check for balanced topic distribution because we couldn't find some brokers in the cluster model: {}", missingBrokersFromClusterModel);
            return;
        }
        boolean bl = clusterHasNoTopics = this.numTotalTopics == 0L;
        if (clusterHasNoTopics) {
            LOG.warn("No topics are present in the cluster, will mark all pending broker additions as completed");
        }
        boolean bl2 = enoughTopicsAreBalanced = clusterHasNoTopics || (double)this.numBalancedTopics / (double)this.numTotalTopics > this.topicDistributionPercentCompletionThreshold;
        if (!enoughTopicsAreBalanced) {
            LOG.info("Not enough topics are balanced, i.e. {}, to mark broker additions as complete (we need at least {} out of all {}, i.e. {}% of all topics), waiting for more topics to be balanced", new Object[]{this.numBalancedTopics, this.numBalancedTopicsRequiredForCompletion, this.numTotalTopics, this.topicDistributionPercentCompletionThreshold * 100.0});
            return;
        }
        LOG.info("Enough topics are balanced - {} (for threshold {} out of {}, i.e. at least {}% of all topics), marking all pending broker additions as completed - {}", new Object[]{this.numBalancedTopics, this.numBalancedTopicsRequiredForCompletion, this.numTotalTopics, this.topicDistributionPercentCompletionThreshold * 100.0, pendingAdditions.stream().map(pendingAddition -> pendingAddition.brokerId).collect(Collectors.toList())});
        pendingAdditions.forEach(pendingAddition -> {
            try {
                this.markAdditionCompleted(pendingAddition.brokerId, currentTime - pendingAddition.createdTimeMs);
            }
            catch (Exception e) {
                LOG.error("Something went wrong while trying to mark broker {} as successfully added based on well enough balanced topic distribution", (Object)pendingAddition.brokerId, (Object)e);
            }
        });
    }

    boolean isTopicBalanced(String topic, ClusterModel clusterModel) {
        double topicImbalanceScore = clusterModel.topicImbalanceScore(topic);
        if (topicImbalanceScore == Double.MAX_VALUE) {
            LOG.warn("Topic {} doesn't have a recorded imbalance score, considering it imbalanced", (Object)topic);
            return false;
        }
        return topicImbalanceScore < this.balancedTopicScoreThreshold;
    }

    private void tryCompletingBasedOnCpu(ClusterModel clusterModel, List<BrokerAdditionV2StateManager.PendingAddition> pendingBrokers, long currentTime) {
        HashSet<Integer> notCompletedBrokerIds = new HashSet<Integer>();
        for (BrokerAdditionV2StateManager.PendingAddition brokerToAdd : pendingBrokers) {
            try {
                Broker broker = clusterModel.broker(brokerToAdd.brokerId);
                if (broker == null) {
                    LOG.warn("Unable to find appropriate broker from cluster model for broker id: {}", (Object)brokerToAdd.brokerId);
                    continue;
                }
                double brokerCpuUsage = broker.load().expectedUtilizationFor(Resource.CPU) / 100.0;
                long brokerCreationTime = brokerToAdd.createdTimeMs;
                long timeSinceCreation = currentTime - brokerCreationTime;
                if (brokerCpuUsage >= this.desiredBrokerCpuUsage) {
                    this.markAdditionCompleted(brokerToAdd.brokerId, timeSinceCreation);
                    LOG.info("Marked the broker addition operation for broker: {} as complete. Time taken for completion is: {}ms. Current CPU of Broker:{}%, Mean CPU of Cluster:{}%, Desired CPU of Broker:{}%.", new Object[]{brokerToAdd, timeSinceCreation, brokerCpuUsage * 100.0, this.clusterMeanCpuUsage * 100.0, this.desiredBrokerCpuUsage * 100.0});
                    continue;
                }
                notCompletedBrokerIds.add(brokerToAdd.brokerId);
            }
            catch (Exception e) {
                LOG.error("The broker addition detector encountered an unexpected exception, while evaluating the cpu-based completion criteria of the broker addition operation for broker: {}.", (Object)brokerToAdd.brokerId, (Object)e);
            }
        }
        if (!notCompletedBrokerIds.isEmpty()) {
            pendingBrokers.removeIf(pendingAddition -> !notCompletedBrokerIds.contains(pendingAddition.brokerId));
            LOG.info("Finished the broker addition completion check and was left with {} brokers that have not completed their addition yet because they didn't reach the desired {}% CPU threshold ({}% of mean Utilization {}%). The brokers and their current CPU usage: {}", new Object[]{notCompletedBrokerIds.size(), this.desiredBrokerCpuUsage * 100.0, this.cpuPercentCompletionThreshold * 100.0, this.clusterMeanCpuUsage * 100.0, this.pendingBrokersCpuUsageString(pendingBrokers, clusterModel)});
        }
    }

    List<BrokerAdditionV2StateManager.PendingAddition> filterOutTimedOutAdditions(ClusterModel clusterModel, List<BrokerAdditionV2StateManager.PendingAddition> pendingAdditions, long currentTime) {
        return pendingAdditions.stream().filter(pendingAddition -> {
            boolean additionShouldTimeout;
            boolean bl = additionShouldTimeout = currentTime - pendingAddition.createdTimeMs >= (long)this.brokerAdditionCompletionDurationThresholdMs;
            if (additionShouldTimeout) {
                try {
                    double brokerCpuUsage = clusterModel.broker(pendingAddition.brokerId).load().expectedUtilizationFor(Resource.CPU);
                    this.timeoutAddition(pendingAddition.brokerId, brokerCpuUsage, currentTime - pendingAddition.createdTimeMs);
                }
                catch (Exception e) {
                    LOG.error("Timed out failing the broker addition for broker {} failed", (Object)pendingAddition.brokerId, (Object)e);
                }
            }
            return !additionShouldTimeout;
        }).collect(Collectors.toList());
    }

    private void markAdditionCompleted(Integer brokerToAdd, long timeSinceCreation) throws InterruptedException {
        this.brokerAdditionV2StateManager.completeAddition(brokerToAdd);
        this.brokerAdditionCompletionTimer.update(timeSinceCreation);
    }

    private synchronized void timeoutAddition(int brokerId, double brokerCPUUsage, long timeSinceCreation) throws InterruptedException {
        long timeoutInSeconds = Duration.ofMillis(this.brokerAdditionCompletionDurationThresholdMs).getSeconds();
        String cause = "Broker Addition Timeout Occurred (" + timeoutInSeconds + " seconds)";
        BrokerAdditionV2StateManager.CancellationResult cancellationResult = this.brokerAdditionV2StateManager.maybeCancelAddition(brokerId, cause);
        if (cancellationResult.cancelled) {
            LOG.warn("Broker Addition Detector detected an addition timeout for broker {}. The broker has been created {} seconds before and it took more than allocated time of {} seconds to reach the expected cpu utilization threshold of {}% -- the broker is still at {}% and the topic distribution is as follows: {} balanced topics with a requirement to have at least {} topics balanced", new Object[]{brokerId, timeSinceCreation, timeoutInSeconds, this.desiredBrokerCpuUsage * 100.0, brokerCPUUsage * 100.0, this.numBalancedTopics, this.numBalancedTopicsRequiredForCompletion});
        } else {
            LOG.warn("Broker Addition Detector detected a timeout for broker {} as its already been {} seconds since its creation but it could not cancel it because of {}. (the reason for cancellation was: {}). It's possible that the addition was in a terminal state (already cancelled or completed).", new Object[]{brokerId, timeSinceCreation, cancellationResult.failureReason, cause});
        }
    }

    private String pendingBrokersCpuUsageString(List<BrokerAdditionV2StateManager.PendingAddition> pendingAdditions, ClusterModel clusterModel) {
        return pendingAdditions.stream().map(pendingAddition -> String.format("(kafka-%s: %.0f%%)", pendingAddition.brokerId, clusterModel.broker(pendingAddition.brokerId).load().expectedUtilizationFor(Resource.CPU))).collect(Collectors.joining(", "));
    }

    private Double lowCpuUtilizationThreshold() {
        return this.updatableBalancingConstraint.balancingConstraint().lowUtilizationRatio(Resource.CPU);
    }
}

