/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.executor;

import com.linkedin.cruisecontrol.common.utils.Utils;
import com.linkedin.kafka.cruisecontrol.ConfigFetchErrorHandler;
import com.linkedin.kafka.cruisecontrol.KafkaCruiseControlUtils;
import com.linkedin.kafka.cruisecontrol.common.BatchedConfigsFetcher;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.executor.ExecutionTask;
import com.linkedin.kafka.cruisecontrol.executor.PartitionProposal;
import com.linkedin.kafka.cruisecontrol.model.ReplicaId;
import com.linkedin.kafka.cruisecontrol.monitor.SamplerStateHandle;
import io.confluent.databalancer.utils.RetryableResult;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.AlterConfigOp;
import org.apache.kafka.clients.admin.Config;
import org.apache.kafka.clients.admin.ConfigEntry;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.config.internals.ConfluentConfigs;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.common.AdminOperationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ReplicationThrottleHelper {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicationThrottleHelper.class);
    static final String LEADER_THROTTLED_RATE = "leader.replication.throttled.rate";
    static final String FOLLOWER_THROTTLED_RATE = "follower.replication.throttled.rate";
    static final String LEADER_THROTTLED_REPLICAS = "leader.replication.throttled.replicas";
    static final String FOLLOWER_THROTTLED_REPLICAS = "follower.replication.throttled.replicas";
    private final Admin adminClient;
    Long throttleRate;
    private final boolean overrideStaticThrottleRate;
    private final boolean dynamicThrottlingEnabled;
    boolean autoThrottleEnabled;
    private final KafkaCruiseControlConfig config;
    private final Time time;
    private final long alterConfigTimeoutMs;

    ReplicationThrottleHelper(Admin adminClient, KafkaCruiseControlConfig config, Time time) {
        this.adminClient = adminClient;
        this.config = config;
        this.time = time;
        this.alterConfigTimeoutMs = config.getLong("alter.configs.response.timeout.ms");
        this.throttleRate = this.config.getLong("throttle.bytes.per.second");
        this.dynamicThrottlingEnabled = this.config.getBoolean("dynamic.throttling.enabled");
        this.autoThrottleEnabled = this.throttleRate == KafkaCruiseControlConfig.AUTO_THROTTLE;
        this.overrideStaticThrottleRate = this.config.getBoolean("static.throttle.rate.override.enabled");
        if (this.dynamicThrottlingEnabled) {
            LOG.info("Set throttle rate {}. Will " + (this.overrideStaticThrottleRate ? "" : "not") + " override static throttles when setting the rate.", (Object)this.throttleRate);
        } else {
            LOG.info("Dynamic throttling is disabled - SBC will not attempt setting a throttle rate / throttled replicas during reassignment execution.");
        }
    }

    Long getThrottleRate() {
        if (this.autoThrottleEnabled) {
            return KafkaCruiseControlConfig.AUTO_THROTTLE;
        }
        return this.throttleRate;
    }

    public void setThrottleRate(Long throttleRate) {
        if (throttleRate != null) {
            this.autoThrottleEnabled = throttleRate == KafkaCruiseControlConfig.AUTO_THROTTLE;
        }
        this.throttleRate = throttleRate;
    }

    public void setThrottles(List<PartitionProposal> replicaMovementProposals, SamplerStateHandle samplerStateHandle, Set<Integer> removedBrokers) {
        if (!this.dynamicThrottlingEnabled()) {
            LOG.debug("Skipped setting a rebalance throttle rate because dynamic throttling is not enabled");
            return;
        }
        if (!this.throttlingRateEnabled()) {
            LOG.debug("Skipped setting a rebalance throttle rate because it is not enabled");
            return;
        }
        try {
            this.doSetThrottles(replicaMovementProposals, samplerStateHandle, removedBrokers);
        }
        catch (Exception e) {
            LOG.error("Unexpected exception while setting rebalance throttle.", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    public void removeAllThrottles() {
        if (!this.dynamicThrottlingEnabled()) {
            LOG.debug("Skipped removing all rebalance throttles because dynamic throttling is not enabled");
            return;
        }
        Collection<String> topics = KafkaCruiseControlUtils.getAllTopicsInCluster(this.adminClient);
        Map<ConfigResource, Config> topicConfigs = KafkaCruiseControlUtils.getEntityConfigs(this.adminClient, this.config, ConfigResource.Type.TOPIC, this.time, topics, true);
        LinkedList<String> throttledTopics = new LinkedList<String>();
        topicConfigs.forEach((topic, config) -> {
            Optional<ConfigEntry> leaderThrottle = KafkaCruiseControlUtils.getConfigEntry(config, LEADER_THROTTLED_REPLICAS);
            Optional<ConfigEntry> followerThrottle = KafkaCruiseControlUtils.getConfigEntry(config, FOLLOWER_THROTTLED_REPLICAS);
            if (leaderThrottle.isPresent() && !leaderThrottle.get().isDefault() || followerThrottle.isPresent() && !followerThrottle.get().isDefault()) {
                throttledTopics.add(topic.name());
            }
        });
        if (!throttledTopics.isEmpty()) {
            HashMap<String, String> throttleConfigs = new HashMap<String, String>(2);
            throttleConfigs.put(LEADER_THROTTLED_REPLICAS, null);
            throttleConfigs.put(FOLLOWER_THROTTLED_REPLICAS, null);
            KafkaCruiseControlUtils.setEntityConfigs(this.adminClient, ConfigResource.Type.TOPIC, throttledTopics, AlterConfigOp.OpType.DELETE, throttleConfigs, this.alterConfigTimeoutMs);
        }
        LOG.info("Removed throttled replicas config for topics: {}", topics);
        int throttledBrokers = this.updateOrRemoveBrokerThrottleRate(null);
        LOG.info("Removed throttle rate config from {} brokers", (Object)throttledBrokers);
    }

    public void clearThrottles(List<ExecutionTask> completedTasks, List<ExecutionTask> inProgressTasks, Set<Integer> removedBrokers) {
        if (!this.dynamicThrottlingEnabled()) {
            LOG.debug("Skipped clearing all rebalance throttles because dynamic throttling is not enabled");
            return;
        }
        if (!this.throttlingRateEnabled()) {
            LOG.debug("Skipped clearing all rebalance throttles because it is not enabled");
            return;
        }
        List<PartitionProposal> completedProposals = completedTasks.stream().filter(this::shouldRemoveThrottleForTask).map(ExecutionTask::partitionProposal).collect(Collectors.toList());
        Set<Integer> participatingBrokers = this.getParticipatingBrokers(completedProposals);
        List<PartitionProposal> inProgressProposals = inProgressTasks.stream().filter(this::taskIsInProgress).map(ExecutionTask::partitionProposal).collect(Collectors.toList());
        Set<Integer> brokersWithInProgressTasks = this.getParticipatingBrokers(inProgressProposals);
        participatingBrokers.removeAll(brokersWithInProgressTasks);
        participatingBrokers.removeAll(removedBrokers);
        LOG.info("Removing replica movement throttles from brokers in the cluster: {}", participatingBrokers);
        participatingBrokers.forEach(this::removeThrottledRateFromBroker);
        Map<String, Set<String>> throttledReplicas = this.getThrottledReplicasByTopic(completedProposals, Collections.emptySet());
        throttledReplicas.forEach(this::removeThrottledReplicasFromTopic);
    }

    public int updateOrRemoveBrokerThrottleRate(Long newThrottle) {
        if (!this.dynamicThrottlingEnabled()) {
            LOG.debug("Skipped updating or removing all broker throttle rates because dynamic throttling is not enabled");
            return 0;
        }
        String newThrottleStr = newThrottle == null ? null : Long.toString(newThrottle);
        LinkedList<String> updatedBrokers = new LinkedList<String>();
        Collection brokers = KafkaCruiseControlUtils.getAllBrokersInCluster(this.adminClient).stream().map(String::valueOf).collect(Collectors.toList());
        Map<ConfigResource, Config> brokerConfigs = KafkaCruiseControlUtils.getEntityConfigs(this.adminClient, this.config, ConfigResource.Type.BROKER, this.time, brokers, false);
        brokerConfigs.forEach((broker, brokerConfig) -> {
            boolean updateFollowerRate;
            boolean updateLeaderRate;
            Optional<ConfigEntry> leaderThrottleRate = KafkaCruiseControlUtils.getConfigEntry(brokerConfig, LEADER_THROTTLED_RATE);
            Optional<ConfigEntry> followerThrottleRate = KafkaCruiseControlUtils.getConfigEntry(brokerConfig, FOLLOWER_THROTTLED_RATE);
            if (newThrottle == null) {
                updateLeaderRate = leaderThrottleRate.isPresent() && !leaderThrottleRate.get().isDefault() && leaderThrottleRate.get().value() != null && !leaderThrottleRate.get().value().trim().isEmpty();
                updateFollowerRate = followerThrottleRate.isPresent() && !followerThrottleRate.get().isDefault() && followerThrottleRate.get().value() != null && !followerThrottleRate.get().value().trim().isEmpty();
            } else {
                Optional<Boolean> leaderRateUpdated = leaderThrottleRate.map(ConfigEntry::value).map(rate -> !rate.equals(newThrottleStr));
                updateLeaderRate = !leaderRateUpdated.isPresent() || leaderRateUpdated.get() != false;
                Optional<Boolean> followerRateUpdated = followerThrottleRate.map(ConfigEntry::value).map(rate -> !rate.equals(newThrottleStr));
                boolean bl = updateFollowerRate = !followerRateUpdated.isPresent() || followerRateUpdated.get() != false;
            }
            if (updateLeaderRate || updateFollowerRate) {
                LOG.debug("Setting throttle to {} bytes/second for broker {}", (Object)newThrottle, broker);
                updatedBrokers.add(broker.name());
            } else {
                LOG.debug("Not setting throttle {} for broker {} because pre-existing throttle is same. Leader rate: {}, follower rate:{}", new Object[]{newThrottle, broker, leaderThrottleRate, followerThrottleRate});
            }
        });
        if (!updatedBrokers.isEmpty()) {
            HashMap<String, String> throttleConfigs = new HashMap<String, String>(2);
            throttleConfigs.put(LEADER_THROTTLED_RATE, newThrottleStr);
            throttleConfigs.put(FOLLOWER_THROTTLED_RATE, newThrottleStr);
            KafkaCruiseControlUtils.setEntityConfigs(this.adminClient, ConfigResource.Type.BROKER, updatedBrokers, newThrottle == null ? AlterConfigOp.OpType.DELETE : AlterConfigOp.OpType.SET, throttleConfigs, this.alterConfigTimeoutMs);
            LOG.debug("Setting throttle to {} bytes/second for brokers {}", throttleConfigs, updatedBrokers);
        }
        return updatedBrokers.size();
    }

    private void doSetThrottles(List<PartitionProposal> replicaMovementProposals, SamplerStateHandle samplerStateHandle, Set<Integer> removedBrokers) {
        if (this.autoThrottleEnabled) {
            this.throttleRate = samplerStateHandle.computeThrottle();
        }
        Set<Integer> participatingBrokers = this.getParticipatingBrokers(replicaMovementProposals);
        TreeSet<Integer> participatingOnlineBrokers = new TreeSet<Integer>(participatingBrokers);
        participatingOnlineBrokers.removeAll(removedBrokers);
        if (!removedBrokers.isEmpty()) {
            LOG.info("Skipping fetching configs for brokers currently being removed: {}", removedBrokers);
        }
        Map<ConfigResource, Config> brokerConfigs = this.fetchBrokerConfigs(participatingOnlineBrokers);
        Set<Integer> brokersWithStaticThrottles = this.filterBrokersWithStaticThrottles(brokerConfigs);
        Map<String, Set<String>> throttledReplicas = this.getThrottledReplicasByTopic(replicaMovementProposals, brokersWithStaticThrottles);
        LOG.info("Setting a rebalance throttle of {} bytes/sec to {} brokers and {} topics. Following Brokers already have static throttles set: {}", new Object[]{this.throttleRate, Utils.join(participatingOnlineBrokers, ", "), Utils.join(throttledReplicas.keySet(), ", "), Utils.join(brokersWithStaticThrottles, ", ")});
        for (Integer brokerId : participatingOnlineBrokers) {
            Config brokerConfig = brokerConfigs.get(new ConfigResource(ConfigResource.Type.BROKER, String.valueOf(brokerId)));
            if (brokerConfig == null) {
                LOG.warn("Skipping setting throttle rates on broker {} as it seems to be offline.", (Object)brokerId);
                continue;
            }
            this.setLeaderThrottledRateIfUnset(brokerId, brokerConfig);
            this.setFollowerThrottledRateIfUnset(brokerId, brokerConfig);
        }
        throttledReplicas.forEach(this::setLeaderThrottledReplicas);
        throttledReplicas.forEach(this::setFollowerThrottledReplicas);
    }

    private Map<ConfigResource, Config> fetchBrokerConfigs(Set<Integer> participatingBrokers) {
        LOG.info("Fetching configs for brokers {}", participatingBrokers);
        BatchedConfigsFetcher batchedConfigsFetcher = BatchedConfigsFetcher.of(this.adminClient, this.config, ConfigResource.Type.BROKER, this.time).entities(participatingBrokers.stream().map(String::valueOf).collect(Collectors.toList())).errorHandler((configResource, e) -> {
            if (e.getCause() instanceof TimeoutException) {
                LOG.warn("Could not fetch broker configs for broker {} when setting replication throttles. This could be because the broker is offline. Ignoring it.", (Object)configResource.name());
                return RetryableResult.Success.of(new HashMap());
            }
            throw new RuntimeException(e);
        }).build();
        return batchedConfigsFetcher.getConfigs();
    }

    private Set<Integer> filterBrokersWithStaticThrottles(Map<ConfigResource, Config> configs) {
        Predicate<Map.Entry> staticThrottleFilter = configResourceConfigEntry -> {
            ConfigEntry leaderConfig = ((Config)configResourceConfigEntry.getValue()).get(LEADER_THROTTLED_REPLICAS);
            ConfigEntry followerConfig = ((Config)configResourceConfigEntry.getValue()).get(FOLLOWER_THROTTLED_REPLICAS);
            return leaderConfig != null && leaderConfig.value() != null && leaderConfig.value().equals("*") && followerConfig != null && followerConfig.value() != null && followerConfig.value().equals("*");
        };
        return configs.entrySet().stream().filter(staticThrottleFilter).map(entry -> Integer.parseInt(((ConfigResource)entry.getKey()).name())).collect(Collectors.toSet());
    }

    boolean shouldRemoveThrottleForTask(ExecutionTask task) {
        return task.state() != ExecutionTask.State.IN_PROGRESS && task.state() != ExecutionTask.State.PENDING && task.type() == ExecutionTask.TaskType.INTER_BROKER_REPLICA_ACTION;
    }

    boolean taskIsInProgress(ExecutionTask task) {
        return task.state() == ExecutionTask.State.IN_PROGRESS && task.type() == ExecutionTask.TaskType.INTER_BROKER_REPLICA_ACTION;
    }

    private boolean dynamicThrottlingEnabled() {
        return this.dynamicThrottlingEnabled;
    }

    private boolean throttlingRateEnabled() {
        return this.throttleRate != null && !ConfluentConfigs.BALANCER_THROTTLE_NO_THROTTLE.equals(this.throttleRate);
    }

    private Set<Integer> getParticipatingBrokers(List<PartitionProposal> replicaMovementProposals) {
        TreeSet<Integer> participatingBrokers = new TreeSet<Integer>();
        for (PartitionProposal proposal : replicaMovementProposals) {
            participatingBrokers.addAll(proposal.oldReplicas().stream().map(ReplicaId::brokerId).collect(Collectors.toSet()));
            participatingBrokers.addAll(proposal.newReplicas().stream().map(ReplicaId::brokerId).collect(Collectors.toSet()));
        }
        return participatingBrokers;
    }

    private Map<String, Set<String>> getThrottledReplicasByTopic(List<PartitionProposal> replicaMovementProposals, Set<Integer> brokerWithStaticThrottles) {
        HashMap<String, Set<String>> throttledReplicasByTopic = new HashMap<String, Set<String>>();
        for (PartitionProposal proposal : replicaMovementProposals) {
            String topic = proposal.topic();
            int partitionId = proposal.partitionId();
            List<Integer> brokers = Stream.concat(proposal.oldReplicas().stream().map(ReplicaId::brokerId), proposal.replicasToAdd().stream().map(ReplicaId::brokerId)).collect(Collectors.toList());
            if (brokerWithStaticThrottles.containsAll(brokers)) continue;
            Set throttledReplicas = throttledReplicasByTopic.computeIfAbsent(topic, x -> new TreeSet());
            brokers.forEach(brokerId -> throttledReplicas.add(partitionId + ":" + brokerId));
        }
        return throttledReplicasByTopic;
    }

    private void setLeaderThrottledRateIfUnset(int brokerId, Config brokerConfig) {
        this.setThrottledRateIfUnset(brokerId, brokerConfig, LEADER_THROTTLED_RATE);
    }

    private void setFollowerThrottledRateIfUnset(int brokerId, Config brokerConfig) {
        this.setThrottledRateIfUnset(brokerId, brokerConfig, FOLLOWER_THROTTLED_RATE);
    }

    private void setThrottledRateIfUnset(int brokerId, Config brokerConfig, String configKey) {
        ConfigEntry entry;
        assert (this.throttleRate != null);
        assert (configKey.equals(LEADER_THROTTLED_RATE) || configKey.equals(FOLLOWER_THROTTLED_RATE));
        if (!this.overrideStaticThrottleRate && (entry = brokerConfig.get(configKey)) != null && entry.source() == ConfigEntry.ConfigSource.STATIC_BROKER_CONFIG && entry.value() != null && !entry.value().isEmpty()) {
            LOG.debug("Not setting {} for broker {} because pre-existing throttle of {} was already set statically.", new Object[]{configKey, brokerId, entry.value()});
            return;
        }
        Optional<ConfigEntry> oldThrottleRate = KafkaCruiseControlUtils.getConfigEntry(brokerConfig, configKey);
        if (!oldThrottleRate.isPresent() || oldThrottleRate.get().isDefault()) {
            LOG.debug("Setting {} to {} bytes/second for broker {}", new Object[]{configKey, this.throttleRate, brokerId});
            KafkaCruiseControlUtils.setEntityConfigs(this.adminClient, ConfigResource.Type.BROKER, Collections.singleton(String.valueOf(brokerId)), AlterConfigOp.OpType.SET, Collections.singletonMap(configKey, String.valueOf(this.throttleRate)), this.alterConfigTimeoutMs);
        } else {
            LOG.debug("Not setting {} value {} for broker {} because pre-existing throttle of {} was already set", new Object[]{configKey, this.throttleRate, brokerId, oldThrottleRate.get()});
        }
    }

    private void setLeaderThrottledReplicas(String topic, Set<String> replicas) {
        this.setThrottledReplicas(topic, replicas, LEADER_THROTTLED_REPLICAS);
    }

    private void setFollowerThrottledReplicas(String topic, Set<String> replicas) {
        this.setThrottledReplicas(topic, replicas, FOLLOWER_THROTTLED_REPLICAS);
    }

    private void setThrottledReplicas(String topic, Set<String> replicas, String configKey) {
        assert (configKey.equals(LEADER_THROTTLED_REPLICAS) || configKey.equals(FOLLOWER_THROTTLED_REPLICAS));
        String throttledReplicasStr = String.join((CharSequence)",", replicas);
        try {
            KafkaCruiseControlUtils.setEntityConfigs(this.adminClient, ConfigResource.Type.TOPIC, Collections.singleton(topic), AlterConfigOp.OpType.APPEND, Collections.singletonMap(configKey, throttledReplicasStr), this.alterConfigTimeoutMs);
        }
        catch (UnknownTopicOrPartitionException | AdminOperationException e) {
            LOG.info("Not setting throttled replicas {} for topic {} because it does not exist", (Object)throttledReplicasStr, (Object)topic);
        }
    }

    static String removeReplicasFromConfig(String throttleConfig, Set<String> replicas) {
        ArrayList<String> throttles = new ArrayList<String>(Arrays.asList(throttleConfig.split(",")));
        throttles.removeIf(replicas::contains);
        return String.join((CharSequence)",", throttles);
    }

    private Optional<String> removeLeaderThrottledReplicasFromTopic(Config config, String topic, Set<String> replicas) {
        Optional<ConfigEntry> oldLeaderThrottledReplicas = KafkaCruiseControlUtils.getConfigEntry(config, LEADER_THROTTLED_REPLICAS);
        if (oldLeaderThrottledReplicas.isPresent() && !oldLeaderThrottledReplicas.get().isDefault()) {
            LOG.debug("Removing leader throttles for topic {} for replicas {}", (Object)topic, replicas);
            return Optional.of(ReplicationThrottleHelper.removeReplicasFromConfig(oldLeaderThrottledReplicas.get().value(), replicas));
        }
        return Optional.empty();
    }

    private Optional<String> removeFollowerThrottledReplicasFromTopic(Config config, String topic, Set<String> replicas) {
        Optional<ConfigEntry> oldFollowerThrottledReplicas = KafkaCruiseControlUtils.getConfigEntry(config, FOLLOWER_THROTTLED_REPLICAS);
        if (oldFollowerThrottledReplicas.isPresent() && !oldFollowerThrottledReplicas.get().isDefault()) {
            LOG.debug("Removing follower throttles for topic {} and replicas {}", (Object)topic, replicas);
            return Optional.of(ReplicationThrottleHelper.removeReplicasFromConfig(oldFollowerThrottledReplicas.get().value(), replicas));
        }
        return Optional.empty();
    }

    void removeThrottledReplicasFromTopic(String topic, Set<String> replicas) {
        try {
            ConfigFetchErrorHandler errorHandler = (configResource, e) -> {
                if (e.getCause() instanceof UnknownTopicOrPartitionException) {
                    throw (RuntimeException)e.getCause();
                }
                return RetryableResult.Incomplete.instance();
            };
            Config config = KafkaCruiseControlUtils.getEntityConfigs(this.adminClient, this.config, ConfigResource.Type.TOPIC, this.time, topic, errorHandler);
            ArrayList<AlterConfigOp> alterConfigOps = new ArrayList<AlterConfigOp>(2);
            Optional<String> newLeaderThrottledReplicas = this.removeLeaderThrottledReplicasFromTopic(config, topic, replicas);
            newLeaderThrottledReplicas.ifPresent(s -> alterConfigOps.add(new AlterConfigOp(new ConfigEntry(LEADER_THROTTLED_REPLICAS, s), s.isEmpty() ? AlterConfigOp.OpType.DELETE : AlterConfigOp.OpType.SET)));
            Optional<String> newFollowerThrottledReplicas = this.removeFollowerThrottledReplicasFromTopic(config, topic, replicas);
            newFollowerThrottledReplicas.ifPresent(s -> alterConfigOps.add(new AlterConfigOp(new ConfigEntry(FOLLOWER_THROTTLED_REPLICAS, s), s.isEmpty() ? AlterConfigOp.OpType.DELETE : AlterConfigOp.OpType.SET)));
            KafkaCruiseControlUtils.setEntityConfigs(this.adminClient, ConfigResource.Type.TOPIC, Collections.singleton(topic), alterConfigOps, this.alterConfigTimeoutMs);
        }
        catch (UnknownTopicOrPartitionException | AdminOperationException e2) {
            LOG.warn("Skip removing throttled replicas {} for topic {} due to error {}", new Object[]{replicas, topic, e2});
        }
    }

    private void removeThrottledRateFromBroker(Integer brokerId) {
        if (brokerId == null) {
            throw new RuntimeException("Cannot remove throttle for null broker id.");
        }
        LOG.debug("Removing leader and follower throttle on broker {}", (Object)brokerId);
        HashMap<String, String> throttleConfigs = new HashMap<String, String>(2);
        throttleConfigs.put(LEADER_THROTTLED_RATE, null);
        throttleConfigs.put(FOLLOWER_THROTTLED_RATE, null);
        KafkaCruiseControlUtils.setEntityConfigs(this.adminClient, ConfigResource.Type.BROKER, Collections.singleton(String.valueOf(brokerId)), AlterConfigOp.OpType.DELETE, throttleConfigs, this.alterConfigTimeoutMs);
    }
}

