/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.monitor.sampling;

import com.linkedin.kafka.cruisecontrol.KafkaCruiseControlUtils;
import com.linkedin.kafka.cruisecontrol.config.BrokerCapacityConfigResolver;
import com.linkedin.kafka.cruisecontrol.config.BrokerCapacityInfo;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.metricsreporter.metric.CruiseControlMetric;
import com.linkedin.kafka.cruisecontrol.metricsreporter.metric.RawMetricType;
import com.linkedin.kafka.cruisecontrol.monitor.MonitorUtils;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.MetricSampler;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.SamplingUtils;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.BrokerLoad;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.PartitionMetricSample;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.ReplicaMetricSample;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.PartitionInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CruiseControlMetricsProcessor {
    private static final Logger LOG = LoggerFactory.getLogger(CruiseControlMetricsProcessor.class);
    private static final long INIT_METRIC_MAX_TIMESTAMP = -1L;
    private static final long INIT_METRIC_MIN_TIMESTAMP = Long.MAX_VALUE;
    private static final int MAX_PARTITION_ERROR_LOGS = 10;
    private static final int MAX_REPLICA_ERROR_LOGS = 10;
    private final Map<Integer, BrokerLoad> brokerLoad = new HashMap<Integer, BrokerLoad>();
    private final Map<Integer, Short> cachedNumCoresByBroker = new HashMap<Integer, Short>();
    private final BrokerCapacityConfigResolver brokerCapacityConfigResolver;
    private final boolean allowCpuCapacityEstimation;
    private long minMetricTimestamp;
    private long maxMetricTimestamp;
    private final double requestContributionWeight;
    private final double bytesContributionWeight;

    public CruiseControlMetricsProcessor(BrokerCapacityConfigResolver brokerCapacityConfigResolver, KafkaCruiseControlConfig config) {
        this.brokerCapacityConfigResolver = brokerCapacityConfigResolver;
        this.allowCpuCapacityEstimation = config.getBoolean("sampling.allow.cpu.capacity.estimation");
        this.requestContributionWeight = config.getDouble("request.cpu.contribution.weight");
        this.bytesContributionWeight = config.getDouble("bytes.cpu.contribution.weight");
        this.minMetricTimestamp = Long.MAX_VALUE;
        this.maxMetricTimestamp = -1L;
    }

    public void addMetric(CruiseControlMetric metric) {
        int brokerId = metric.brokerId();
        LOG.trace("Adding cruise control metric {}", (Object)metric);
        this.minMetricTimestamp = Math.min(metric.time(), this.minMetricTimestamp);
        this.maxMetricTimestamp = Math.max(metric.time(), this.maxMetricTimestamp);
        this.brokerLoad.compute(brokerId, (bid, load) -> {
            BrokerLoad brokerLoad = load == null ? new BrokerLoad() : load;
            brokerLoad.recordMetric(metric);
            return brokerLoad;
        });
    }

    BrokerLoad brokerLoad(int brokerId) {
        return this.brokerLoad.get(brokerId);
    }

    private void updateCachedNumCoresByBroker(Cluster cluster) {
        for (int brokerId : this.brokerLoad.keySet()) {
            Node node = cluster.nodeById(brokerId);
            if (node == null) continue;
            this.cachedNumCoresByBroker.computeIfAbsent(brokerId, bid -> {
                BrokerCapacityInfo capacity = this.brokerCapacityConfigResolver.capacityForBroker(MonitorUtils.getRackHandleNull(node), node.host(), (int)bid);
                return !this.allowCpuCapacityEstimation && capacity.isEstimated() ? null : Short.valueOf(capacity.numCpuCores());
            });
        }
    }

    private void updateDiskCapacityByBroker(Cluster cluster) {
        for (Map.Entry<Integer, BrokerLoad> entry : this.brokerLoad.entrySet()) {
            Integer brokerId = entry.getKey();
            BrokerLoad brokerLoad = entry.getValue();
            Node node = cluster.nodeById(brokerId.intValue());
            if (node == null || !brokerLoad.brokerMetricAvailable(RawMetricType.BROKER_DISK_CAPACITY)) continue;
            double brokerDiskCapacityInMiB = brokerLoad.brokerMetric(RawMetricType.BROKER_DISK_CAPACITY) / 1048576.0;
            this.brokerCapacityConfigResolver.updateDiskCapacityForBroker(MonitorUtils.getRackHandleNull(node), node.host(), brokerId, brokerDiskCapacityInMiB);
        }
    }

    Map<Integer, Short> cachedNumCoresByBroker() {
        return this.cachedNumCoresByBroker;
    }

    private void prepareReplicaMetrics(Cluster cluster, long time) {
        this.setLeaderReplicationBytesOut(cluster, time);
    }

    void setLeaderReplicationBytesOut(Cluster cluster, long time) {
        for (Node node : cluster.nodes()) {
            BrokerLoad brokerLoad = this.brokerLoad.get(node.id());
            if (brokerLoad == null) {
                LOG.debug("Skipping broker {} as broker load is not present.", (Object)node);
                continue;
            }
            HashMap<String, Integer> leaderTopicsWithRf = new HashMap<String, Integer>();
            for (PartitionInfo partitionInfo : cluster.partitionsForNode(node.id())) {
                if (leaderTopicsWithRf.containsKey(partitionInfo.topic())) continue;
                leaderTopicsWithRf.put(partitionInfo.topic(), partitionInfo.replicas().length);
            }
            leaderTopicsWithRf.forEach((topic, rf) -> {
                if (brokerLoad.topicMetricsAvailable((String)topic)) {
                    double topicBytesIn = brokerLoad.topicMetrics((String)topic, RawMetricType.TOPIC_BYTES_IN, false);
                    double replicationBytesOut = topicBytesIn * (double)(rf - 1);
                    brokerLoad.setTopicMetrics((String)topic, RawMetricType.TOPIC_REPLICATION_BYTES_OUT, replicationBytesOut, time);
                } else {
                    LOG.debug("Skipping setting replication bytes out for topic {} as topic metrics are not available.", topic);
                }
            });
        }
    }

    Map<Integer, List<PartitionInfo>> followerPartitionsForNodes(Cluster cluster) {
        HashMap<Integer, List<PartitionInfo>> followerPartitionsByNode = new HashMap<Integer, List<PartitionInfo>>();
        cluster.topics().stream().flatMap(topic -> cluster.partitionsForTopic(topic).stream()).filter(partitionInfo -> partitionInfo.leader() != null && !partitionInfo.leader().isEmpty()).forEach(partitionInfo -> Arrays.stream(partitionInfo.replicas()).filter(replica -> replica != null && !replica.isEmpty()).filter(replica -> !replica.equals((Object)partitionInfo.leader())).forEach(replica -> followerPartitionsByNode.computeIfAbsent(replica.id(), ArrayList::new).add(partitionInfo)));
        cluster.nodes().forEach(node -> followerPartitionsByNode.computeIfAbsent(node.id(), ignored -> Collections.emptyList()));
        return followerPartitionsByNode;
    }

    public MetricSampler.Samples process(Cluster cluster, Set<PartitionInfo> partitions) {
        this.updateCachedNumCoresByBroker(cluster);
        this.updateDiskCapacityByBroker(cluster);
        this.prepareReplicaMetrics(cluster, this.maxMetricTimestamp);
        this.brokerLoad.forEach((broker, load) -> load.prepareBrokerMetrics(cluster, (int)broker, this.maxMetricTimestamp));
        HashSet<PartitionInfo> validPartitions = new HashSet<PartitionInfo>();
        int skippedReplicas = 0;
        int skippedPartitions = 0;
        for (PartitionInfo topicPartition : partitions) {
            Node leader;
            if (SamplingUtils.skipBuildingMetricSample(topicPartition, leader = topicPartition.leader(), this.brokerLoad, this.cachedNumCoresByBroker)) {
                ++skippedPartitions;
                continue;
            }
            validPartitions.add(topicPartition);
        }
        int numBrokers = cluster.nodes().size();
        HashMap<Integer, Map<String, Integer>> leaderDistribution = new HashMap<Integer, Map<String, Integer>>(numBrokers);
        HashMap<Integer, Map<String, Integer>> followerDistribution = new HashMap<Integer, Map<String, Integer>>(numBrokers);
        SamplingUtils.populateReplicaDistribution(cluster, leaderDistribution, followerDistribution);
        HashSet<ReplicaMetricSample> replicaMetricSamples = new HashSet<ReplicaMetricSample>();
        HashSet<PartitionMetricSample> partitionMetricSamples = new HashSet<PartitionMetricSample>();
        LOG.info("Generated {}{} replica metrics samples, {}{} partition metric samples from time {} to {} ({} to {}).", new Object[]{replicaMetricSamples.size(), (skippedReplicas += this.addReplicaMetricSamples(validPartitions, replicaMetricSamples, leaderDistribution, followerDistribution, this.minMetricTimestamp, this.maxMetricTimestamp)) > 0 ? "(" + skippedReplicas + " skipped)" : "", partitionMetricSamples.size(), skippedPartitions > 0 ? "(" + (skippedPartitions += this.addPartitionMetricSamples(validPartitions, partitionMetricSamples, leaderDistribution, this.minMetricTimestamp, this.maxMetricTimestamp)) + " skipped)" : "", KafkaCruiseControlUtils.toTimeString(this.minMetricTimestamp), KafkaCruiseControlUtils.toTimeString(this.maxMetricTimestamp), this.minMetricTimestamp, this.maxMetricTimestamp});
        return new MetricSampler.Samples(replicaMetricSamples, partitionMetricSamples);
    }

    public void clear() {
        this.brokerLoad.clear();
        this.maxMetricTimestamp = -1L;
        this.minMetricTimestamp = Long.MAX_VALUE;
    }

    private int addReplicaMetricSamples(Set<PartitionInfo> topicPartitions, Set<ReplicaMetricSample> replicaMetricSamples, Map<Integer, Map<String, Integer>> leaderDistribution, Map<Integer, Map<String, Integer>> followerDistribution, long minMetricTimestamp, long maxMetricTimestamp) {
        int skippedReplicas = 0;
        int loggedReplicaErrors = 0;
        for (PartitionInfo topicPartition : topicPartitions) {
            for (Node replica : topicPartition.replicas()) {
                try {
                    if (SamplingUtils.skipBuildingMetricSample(topicPartition, replica, this.brokerLoad, this.cachedNumCoresByBroker)) {
                        ++skippedReplicas;
                        continue;
                    }
                    boolean isLeader = replica.id() == topicPartition.leader().id();
                    Integer leaderCount = leaderDistribution.getOrDefault(replica.id(), Collections.emptyMap()).getOrDefault(topicPartition.topic(), 0);
                    Integer followerCount = followerDistribution.getOrDefault(replica.id(), Collections.emptyMap()).getOrDefault(topicPartition.topic(), 0);
                    if (isLeader && leaderCount == 0 || !isLeader && followerCount == 0) continue;
                    ReplicaMetricSample sample = SamplingUtils.buildReplicaMetricSample(topicPartition, replica, leaderCount, followerCount, this.brokerLoad, minMetricTimestamp, maxMetricTimestamp, this.cachedNumCoresByBroker, this.requestContributionWeight, this.bytesContributionWeight);
                    LOG.trace("Added replica metrics sample for {}-{}, replica: {}.", new Object[]{topicPartition.topic(), topicPartition.partition(), replica});
                    replicaMetricSamples.add(sample);
                }
                catch (Exception e) {
                    if (loggedReplicaErrors < 10) {
                        LOG.error("Error building replica metric sample for {}-{}, replica: {}. Error: {}", new Object[]{topicPartition.topic(), topicPartition.partition(), replica, e.getMessage(), e});
                        if (++loggedReplicaErrors == 10) {
                            LOG.info("Already logged {} errors. Switching to trace level logging now.", (Object)10);
                        }
                    } else {
                        LOG.trace("Error building replica metric sample for {}-{}, replica: {}. Error: {}", new Object[]{topicPartition.topic(), topicPartition.partition(), replica, e.getMessage(), e});
                    }
                    ++skippedReplicas;
                }
            }
        }
        return skippedReplicas;
    }

    private int addPartitionMetricSamples(Set<PartitionInfo> topicPartitions, Set<PartitionMetricSample> partitionMetricSamples, Map<Integer, Map<String, Integer>> leaderDistribution, long minMetricTimestamp, long maxMetricTimestamp) {
        int skippedPartition = 0;
        int loggedPartitionErrors = 0;
        for (PartitionInfo topicPartition : topicPartitions) {
            try {
                PartitionMetricSample sample = SamplingUtils.buildPartitionMetricSample(leaderDistribution, topicPartition, this.brokerLoad, minMetricTimestamp, maxMetricTimestamp);
                LOG.trace("Added partition metrics sample for {}.", (Object)topicPartition);
                partitionMetricSamples.add(sample);
            }
            catch (Exception e) {
                if (loggedPartitionErrors < 10) {
                    LOG.warn("Error building partition metric sample for {}.", (Object)topicPartition, (Object)e);
                    ++loggedPartitionErrors;
                } else {
                    LOG.trace("Error building partition metric sample for {}.", (Object)topicPartition, (Object)e);
                }
                ++skippedPartition;
            }
        }
        return skippedPartition;
    }
}

