/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.server.common;

import io.confluent.kafka.availability.FilesWrapper;
import io.confluent.kafka.availability.ThreadCountersManager;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.server.common.CheckpointFileChecksumUtils;
import org.apache.kafka.server.common.CheckpointFileConfig;

public class CheckpointFile<T> {
    private final int version;
    private final CheckpointFileConfig config;
    private final EntryFormatter<T> formatter;
    private final Object lock = new Object();
    private final Path fileWithoutCrc32cPath;
    private final Path fileWithCrc32cPath;

    public CheckpointFile(File file, int version, EntryFormatter<T> formatter) throws IOException {
        this(file, version, new CheckpointFileConfig(false), formatter);
    }

    public CheckpointFile(File file, int version, CheckpointFileConfig config, EntryFormatter<T> formatter) throws IOException {
        this.version = version;
        this.config = config;
        this.formatter = formatter;
        String filePathPrefix = file.toPath().toAbsolutePath().toString();
        this.fileWithCrc32cPath = Paths.get(filePathPrefix + ".crc32c", new String[0]);
        this.fileWithoutCrc32cPath = Paths.get(filePathPrefix, new String[0]);
        try {
            if (config.checksumProtected()) {
                FilesWrapper.createFile((Path)this.fileWithCrc32cPath, (FileAttribute[])new FileAttribute[0]);
            } else {
                FilesWrapper.createFile((Path)this.fileWithoutCrc32cPath, (FileAttribute[])new FileAttribute[0]);
            }
        }
        catch (FileAlreadyExistsException fileAlreadyExistsException) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateChecksumProtection(boolean enable) {
        Object object = this.lock;
        synchronized (object) {
            this.config.setChecksumProtection(enable);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void write(Collection<T> entries) throws IOException {
        Object object = this.lock;
        synchronized (object) {
            Path tempFilePath = this.config.checksumProtected() ? Paths.get(this.fileWithCrc32cPath.toString() + ".tmp", new String[0]) : Paths.get(this.fileWithoutCrc32cPath.toString() + ".tmp", new String[0]);
            ThreadCountersManager.wrapIOChecked(() -> {
                try (FileOutputStream fileOutputStream = new FileOutputStream(tempFilePath.toFile());
                     BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)fileOutputStream, StandardCharsets.UTF_8));){
                    CheckpointWriteBuffer<T> checkpointWriteBuffer = new CheckpointWriteBuffer<T>(writer, this.version, this.config, this.formatter);
                    checkpointWriteBuffer.write(entries);
                    writer.flush();
                    fileOutputStream.getFD().sync();
                }
            });
            if (this.config.checksumProtected()) {
                Utils.atomicMoveWithFallback((Path)tempFilePath, (Path)this.fileWithCrc32cPath, (boolean)true);
                Files.deleteIfExists(this.fileWithoutCrc32cPath);
            } else {
                Utils.atomicMoveWithFallback((Path)tempFilePath, (Path)this.fileWithoutCrc32cPath, (boolean)true);
                Files.deleteIfExists(this.fileWithCrc32cPath);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public byte[] writeToByteArray(Collection<T> entries) throws IOException {
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        try (BufferedWriter bufferedWriter = new BufferedWriter(new OutputStreamWriter((OutputStream)byteArrayOutputStream, StandardCharsets.UTF_8));){
            CheckpointWriteBuffer<T> checkpointWriteBuffer = new CheckpointWriteBuffer<T>(bufferedWriter, this.version, this.config, this.formatter);
            checkpointWriteBuffer.write(entries);
            bufferedWriter.flush();
            byte[] byArray = byteArrayOutputStream.toByteArray();
            return byArray;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<T> read() throws IOException {
        Object object = this.lock;
        synchronized (object) {
            Path file = this.getFileToRead();
            return (List)ThreadCountersManager.wrapIOChecked(() -> {
                try (BufferedReader reader = Files.newBufferedReader(file, StandardCharsets.UTF_8);){
                    CheckpointReadBuffer<T> checkpointBuffer = new CheckpointReadBuffer<T>(file.toString(), reader, this.version, this.formatter);
                    List<T> list = checkpointBuffer.read();
                    return list;
                }
            });
        }
    }

    private Path getFileToRead() throws IOException {
        if (this.fileWithoutCrc32cPath.toFile().exists() && this.fileWithCrc32cPath.toFile().exists()) {
            if (this.config.checksumProtected()) {
                Files.deleteIfExists(this.fileWithoutCrc32cPath);
                return this.fileWithCrc32cPath;
            }
            Files.deleteIfExists(this.fileWithCrc32cPath);
            return this.fileWithoutCrc32cPath;
        }
        if (this.fileWithoutCrc32cPath.toFile().exists()) {
            return this.fileWithoutCrc32cPath;
        }
        return this.fileWithCrc32cPath;
    }

    public static interface EntryFormatter<T> {
        public String toString(T var1);

        public Optional<T> fromString(String var1);
    }

    public static class CheckpointReadBuffer<T> {
        private final String location;
        private final BufferedReader reader;
        private final int version;
        private final EntryFormatter<T> formatter;
        private final CheckpointFileChecksumUtils checksumUtils;

        public CheckpointReadBuffer(String location, BufferedReader reader, int version, EntryFormatter<T> formatter) {
            this.location = location;
            this.reader = reader;
            this.version = version;
            this.formatter = formatter;
            this.checksumUtils = new CheckpointFileChecksumUtils(location.endsWith(".crc32c"));
        }

        public List<T> read() throws IOException {
            String line = this.reader.readLine();
            if (line == null) {
                return Collections.emptyList();
            }
            int readVersion = this.toInt(line);
            if (readVersion != this.version) {
                throw new IOException("Unrecognised version:" + readVersion + ", expected version: " + this.version + " in checkpoint file at: " + this.location);
            }
            line = this.reader.readLine();
            if (line == null) {
                return Collections.emptyList();
            }
            int expectedSize = this.toInt(line);
            this.checksumUtils.mayUpdateChecksum(readVersion, expectedSize);
            ArrayList<T> entries = new ArrayList<T>(expectedSize);
            line = this.reader.readLine();
            while (line != null && entries.size() < expectedSize) {
                Optional<T> maybeEntry = this.formatter.fromString(line);
                if (!maybeEntry.isPresent()) {
                    throw this.buildMalformedLineException(line);
                }
                entries.add(maybeEntry.get());
                this.checksumUtils.mayUpdateChecksum(line);
                line = this.reader.readLine();
            }
            if (entries.size() < expectedSize) {
                throw new IOException("Expected [" + expectedSize + "] entries in checkpoint file [" + this.location + "], but found only [" + entries.size() + "]");
            }
            this.checksumUtils.mayValidateChecksum(line);
            if (this.reader.readLine() != null) {
                throw new IOException("Checkpoint file " + this.location + " has more lines than expected");
            }
            return entries;
        }

        private int toInt(String line) throws IOException {
            try {
                return Integer.parseInt(line);
            }
            catch (NumberFormatException e) {
                throw this.buildMalformedLineException(line);
            }
        }

        private IOException buildMalformedLineException(String line) {
            return new IOException(String.format("Malformed line in checkpoint file [%s]: %s", this.location, line));
        }
    }

    public static class CheckpointWriteBuffer<T> {
        private final BufferedWriter writer;
        private final int version;
        private final EntryFormatter<T> formatter;
        private final CheckpointFileChecksumUtils checksumUtils;

        public CheckpointWriteBuffer(BufferedWriter writer, int version, CheckpointFileConfig config, EntryFormatter<T> formatter) {
            this.writer = writer;
            this.version = version;
            this.formatter = formatter;
            this.checksumUtils = new CheckpointFileChecksumUtils(config.checksumProtected());
        }

        public CheckpointWriteBuffer(BufferedWriter writer, int version, EntryFormatter<T> formatter) {
            this.writer = writer;
            this.version = version;
            this.formatter = formatter;
            this.checksumUtils = new CheckpointFileChecksumUtils(false);
        }

        public void write(Collection<T> entries) throws IOException {
            this.writer.write(Integer.toString(this.version));
            this.writer.newLine();
            this.writer.write(Integer.toString(entries.size()));
            this.writer.newLine();
            this.checksumUtils.mayUpdateChecksum(this.version, entries.size());
            for (T entry : entries) {
                String entryString = this.formatter.toString(entry);
                this.writer.write(entryString);
                this.writer.newLine();
                this.checksumUtils.mayUpdateChecksum(entryString);
            }
            this.checksumUtils.mayWriteChecksumLine(this.writer);
        }
    }
}

