/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.auth.mtls;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.confluent.security.auth.mtls.CaCertificatesUtils;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.AlterConfigOp;
import org.apache.kafka.clients.admin.AlterConfigsOptions;
import org.apache.kafka.clients.admin.AlterConfigsResult;
import org.apache.kafka.clients.admin.ConfigEntry;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.config.internals.ConfluentConfigs;
import org.apache.kafka.common.network.ListenerName;
import org.apache.kafka.common.security.mtls.MTlsTruststoreManager;
import org.apache.kafka.common.utils.FileWatchService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultMTlsTruststoreManager
implements MTlsTruststoreManager,
FileWatchService.Listener {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultMTlsTruststoreManager.class);
    private static final Long ADMIN_CLIENT_CLOSE_TIMEOUT_MS = TimeUnit.SECONDS.toMillis(30L);
    private static final Long FILE_WATCH_START_THREAD_JOIN_TIMEOUT_MS = TimeUnit.SECONDS.toMillis(30L);
    private static final AlterConfigsOptions ALTER_OPTIONS = new AlterConfigsOptions().timeoutMs(Integer.valueOf(30000));
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final String SSL_PEM_FULLCHAIN_FILENAME_FIELD = "ssl_pem_fullchain_filename";
    static final String LEAF_CERT_ALIAS = "confluent-leaf-certificate".toLowerCase(Locale.ENGLISH);
    static final String EXPIRED_CERTIFICATE_REMOVAL_INTERVAL_MILLIS_CONFIG = "confluent.mtls.ca.certificates.manager.expired.certificate.removal.interval.ms";
    static final long EXPIRED_CERTIFICATE_REMOVAL_INTERVAL_MILLIS_DEFAULT = Duration.ofHours(1L).toMillis();
    public static final String ALIAS_SEPARATOR = String.valueOf('\u001e');
    private KeyStore truststore;
    private String nodeId;
    private String listenerName;
    private String truststoreLocation;
    private String truststoreType;
    private String truststorePassword;
    private ScheduledExecutorService executorService;
    private FileWatchService fileWatchService;
    private Path fullChainPemPath;
    private AdminClient adminClient;
    private Supplier<AdminClient> adminClientSupplier;
    private Thread fileWatchStartThread;

    public void configure(Map<String, ?> configs) {
        String fullChainPemFilename;
        Path sslCertsDir;
        LOG.debug("Configuring {}", (Object)this.getClass().getName());
        this.nodeId = this.getNodeId(configs.get("broker.id"), configs.get("node.id"));
        this.listenerName = ConfluentConfigs.getMTlsListenerName(configs);
        String configPrefix = ListenerName.normalised((String)this.listenerName).configPrefix();
        this.truststoreLocation = (String)configs.get(configPrefix + "ssl.truststore.location");
        if (this.truststoreLocation == null) {
            throw new ConfigException("Missing required configuration: " + configPrefix + "ssl.truststore.location");
        }
        this.truststoreType = (String)configs.get(configPrefix + "ssl.truststore.type");
        if (this.truststoreType == null) {
            throw new ConfigException("Missing required configuration: " + configPrefix + "ssl.truststore.type");
        }
        this.truststorePassword = (String)configs.get(configPrefix + "ssl.truststore.password");
        if (this.truststorePassword == null) {
            throw new ConfigException("Missing required configuration: " + configPrefix + "ssl.truststore.password");
        }
        try {
            this.initTruststore();
        }
        catch (Exception e) {
            throw new ConfigException("Failed to initialize truststore from configuration", (Object)e);
        }
        Object specFile = configs.get("multitenant.metadata.ssl.certs.path");
        if (specFile != null) {
            Path specFilePath = Paths.get(specFile.toString(), new String[0]);
            sslCertsDir = specFilePath.getParent();
            fullChainPemFilename = DefaultMTlsTruststoreManager.readJsonFieldAsString(specFilePath.toString(), SSL_PEM_FULLCHAIN_FILENAME_FIELD);
            if (fullChainPemFilename == null) {
                throw new ConfigException("Missing required field: ssl_pem_fullchain_filename in " + specFile);
            }
        } else {
            throw new ConfigException("Missing required configuration: multitenant.metadata.ssl.certs.path");
        }
        this.fullChainPemPath = sslCertsDir.resolve(fullChainPemFilename);
        this.fileWatchService = new FileWatchService();
        this.fileWatchStartThread = new Thread(() -> this.fileWatchService.add((FileWatchService.Listener)this), "mTlsTruststoreManager-Start-FileWatchService-Thread");
        this.fileWatchStartThread.start();
        long cleanerIntervalMs = DefaultMTlsTruststoreManager.getLong(configs, EXPIRED_CERTIFICATE_REMOVAL_INTERVAL_MILLIS_CONFIG, EXPIRED_CERTIFICATE_REMOVAL_INTERVAL_MILLIS_DEFAULT);
        if (cleanerIntervalMs > 0L) {
            this.executorService = Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, "mTlsTruststoreManager-Expiration-Cleaner-Thread"));
            this.executorService.scheduleAtFixedRate(this::removeExpiredCertificates, cleanerIntervalMs, cleanerIntervalMs, TimeUnit.MILLISECONDS);
        }
    }

    public synchronized void addCertificates(String orgId, String providerId, X509Certificate[] certificates) {
        this.ensureTruststore();
        try {
            for (X509Certificate certificate : certificates) {
                certificate.checkValidity();
            }
        }
        catch (CertificateExpiredException e) {
            throw new RuntimeException(e);
        }
        catch (CertificateNotYetValidException e) {
            // empty catch block
        }
        boolean truststoreChanged = false;
        for (X509Certificate certificate : certificates) {
            String alias = DefaultMTlsTruststoreManager.generateUniqueAlias(certificate, providerId, orgId);
            try {
                if (this.truststore.isCertificateEntry(alias)) {
                    LOG.info("CA certificate with alias {} already exists in truststore", (Object)alias);
                    continue;
                }
                LOG.info("Adding CA certificate with alias {}", (Object)alias);
                this.truststore.setCertificateEntry(alias, certificate);
                truststoreChanged = true;
            }
            catch (KeyStoreException e) {
                LOG.error("Failed to add CA certificate with alias {}", (Object)alias, (Object)e);
                throw new RuntimeException(e);
            }
        }
        if (truststoreChanged) {
            this.reloadTruststore();
        }
    }

    public synchronized void removeCertificates(String orgId, String providerId, X509Certificate[] certificates) {
        this.ensureTruststore();
        boolean truststoreChanged = false;
        for (X509Certificate certificate : certificates) {
            String alias = DefaultMTlsTruststoreManager.generateUniqueAlias(certificate, providerId, orgId);
            try {
                if (this.truststore.isCertificateEntry(alias)) {
                    LOG.info("Removing CA certificate with alias {}", (Object)alias);
                    this.truststore.deleteEntry(alias);
                    truststoreChanged = true;
                    continue;
                }
                LOG.info("CA certificate with alias {} not found in truststore", (Object)alias);
            }
            catch (KeyStoreException e) {
                LOG.error("Failed to remove CA certificate with alias {}", (Object)alias, (Object)e);
                throw new RuntimeException(e);
            }
        }
        if (truststoreChanged) {
            this.reloadTruststore();
        }
    }

    public synchronized void removeCertificates(String organizationId, String providerId) {
        this.ensureTruststore();
        ArrayList<String> aliasesToRemove = new ArrayList<String>();
        try {
            Enumeration<String> aliases = this.truststore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                if (!alias.startsWith(DefaultMTlsTruststoreManager.generateAliasPrefix(providerId, organizationId))) continue;
                aliasesToRemove.add(alias);
            }
            for (String alias : aliasesToRemove) {
                LOG.info("Removing CA certificate with alias {}", (Object)alias);
                this.truststore.deleteEntry(alias);
            }
        }
        catch (KeyStoreException e) {
            LOG.error("Failed to remove all CA certificates for provider {} and organization {}", new Object[]{providerId, organizationId, e});
            throw new RuntimeException(e);
        }
        if (!aliasesToRemove.isEmpty()) {
            this.reloadTruststore();
        }
    }

    public void close() throws IOException {
        LOG.info("Closing {}", (Object)this.getClass().getName());
        try {
            this.awaitTerminationAfterShutdown(this.executorService);
        }
        catch (Exception e) {
            LOG.error("Failed to shutdown executor service", (Throwable)e);
        }
        try {
            if (this.fileWatchService != null) {
                this.fileWatchService.close();
            }
            if (this.fileWatchStartThread != null) {
                this.fileWatchStartThread.join(FILE_WATCH_START_THREAD_JOIN_TIMEOUT_MS);
            }
        }
        catch (Exception e) {
            LOG.error("Failed to close file watch service", (Throwable)e);
        }
        if (this.adminClient != null) {
            this.adminClient.close(Duration.ofMillis(ADMIN_CLIENT_CLOSE_TIMEOUT_MS));
        }
    }

    public void setAdminSupplierAndCreateClient(Supplier<AdminClient> adminClientSupplier) {
        this.adminClientSupplier = adminClientSupplier;
        LOG.info("AdminClientSupplier is set, attempting to create AdminClient now");
        this.createAdminClient();
    }

    public File file() {
        return this.fullChainPemPath != null ? this.fullChainPemPath.toFile() : null;
    }

    public void onInit() {
        this.addConfluentLeafCertToTruststore();
    }

    public void onUpdate() {
        this.addConfluentLeafCertToTruststore();
    }

    @VisibleForTesting
    KeyStore getTruststore() {
        return this.truststore;
    }

    @VisibleForTesting
    AdminClient getAdminClient() {
        return this.adminClient;
    }

    @VisibleForTesting
    static long getLong(Map<String, ?> configs, String key, long defaultValue) {
        Object value = configs.get(key);
        try {
            if (value instanceof Number) {
                return ((Number)value).longValue();
            }
            if (value instanceof String) {
                return Long.parseLong((String)value);
            }
            return defaultValue;
        }
        catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    private static String generateAliasPrefix(String providerId, String orgId) {
        return (orgId + ALIAS_SEPARATOR + providerId).toLowerCase(Locale.ENGLISH);
    }

    public static String generateUniqueAlias(X509Certificate certificate, String providerId, String orgId) {
        return (orgId + ALIAS_SEPARATOR + providerId + ALIAS_SEPARATOR + certificate.getIssuerX500Principal().getName("CANONICAL") + ALIAS_SEPARATOR + certificate.getSerialNumber().toString(16)).toLowerCase(Locale.ENGLISH);
    }

    static String readJsonFieldAsString(String jsonFile, String field) {
        try {
            return OBJECT_MAPPER.readTree(new File(jsonFile)).get(field).asText();
        }
        catch (Exception e) {
            return null;
        }
    }

    private String getNodeId(Object brokerIdValue, Object controllerIdValue) {
        if (controllerIdValue != null) {
            return controllerIdValue.toString();
        }
        if (brokerIdValue != null) {
            return brokerIdValue.toString();
        }
        throw new ConfigException("Config has no broker.id or node.id config");
    }

    private char[] getTruststorePassword() {
        Preconditions.checkState((this.truststorePassword != null ? 1 : 0) != 0, (Object)"Truststore password is not set");
        return this.truststorePassword.toCharArray();
    }

    private void initTruststore() throws KeyStoreException, CertificateException, NoSuchAlgorithmException, IOException {
        LOG.debug("Loading truststore from {}", (Object)this.truststoreLocation);
        char[] truststorePassword = this.getTruststorePassword();
        try (FileInputStream fis = new FileInputStream(this.truststoreLocation);){
            this.truststore = KeyStore.getInstance(this.truststoreType);
            this.truststore.load(fis, truststorePassword);
        }
        LOG.debug("Truststore loaded successfully with #{} entries", (Object)this.truststore.size());
    }

    private void reloadTruststore() {
        try {
            try (FileOutputStream fos = new FileOutputStream(this.truststoreLocation);){
                this.truststore.store(fos, this.getTruststorePassword());
            }
            this.setDynamicBrokerSslTruststoreConfigs();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private List<AlterConfigOp> truststoreSslAlterConfigOps() {
        ArrayList<AlterConfigOp> sslConfigs = new ArrayList<AlterConfigOp>();
        String configPrefix = ListenerName.normalised((String)this.listenerName).configPrefix();
        ConfigEntry typeConfig = new ConfigEntry(configPrefix + "ssl.truststore.type", this.truststoreType);
        sslConfigs.add(new AlterConfigOp(typeConfig, AlterConfigOp.OpType.SET));
        return sslConfigs;
    }

    private void setDynamicBrokerSslTruststoreConfigs() throws Exception {
        ConfigResource configResource = new ConfigResource(ConfigResource.Type.BROKER, this.nodeId);
        HashMap<ConfigResource, Collection<AlterConfigOp>> alterConfigs = new HashMap<ConfigResource, Collection<AlterConfigOp>>();
        alterConfigs.put(configResource, this.truststoreSslAlterConfigOps());
        this.invokeAdminClient(alterConfigs);
    }

    private synchronized void invokeAdminClient(Map<ConfigResource, Collection<AlterConfigOp>> alterConfigs) throws Exception {
        if (this.adminClient == null) {
            LOG.info("Admin client has not been created yet, trying again now");
            this.createAdminClient();
            if (this.adminClient == null) {
                LOG.info("Admin client instance not created, would not proceed to CA certificate update");
                return;
            }
        }
        try {
            AlterConfigsResult alterConfigsResult = this.adminClient.incrementalAlterConfigs(alterConfigs, ALTER_OPTIONS);
            alterConfigsResult.all().get();
            LOG.info("Altered truststore SSL configs for broker {}, listener {}, with {}", new Object[]{this.nodeId, this.listenerName, alterConfigsResult});
        }
        catch (Exception e) {
            LOG.warn("IncrementalAlterConfigs threw an exception, marking AdminClient for recreation", (Throwable)e);
            this.adminClient.close(Duration.ofMillis(ADMIN_CLIENT_CLOSE_TIMEOUT_MS));
            this.adminClient = null;
            throw e;
        }
    }

    private synchronized void createAdminClient() {
        if (this.adminClient == null) {
            if (this.adminClientSupplier != null) {
                try {
                    this.adminClient = this.adminClientSupplier.get();
                    LOG.info("Successfully created admin client");
                }
                catch (Exception e) {
                    LOG.warn("Failed to create admin client, will retry next time the client is invoked", (Throwable)e);
                }
            } else {
                LOG.debug("AdminClientSupplier is not set yet, this is expected when a broker first starts up");
            }
        }
    }

    private synchronized void removeExpiredCertificates() {
        if (this.truststore == null) {
            return;
        }
        try {
            LOG.info("Removing expired certificates from truststore");
            Enumeration<String> aliases = this.truststore.aliases();
            ArrayList<String> aliasesToRemove = new ArrayList<String>();
            while (aliases.hasMoreElements()) {
                Certificate certificate;
                String alias = aliases.nextElement();
                if (alias.equals(LEAF_CERT_ALIAS) || !((certificate = this.truststore.getCertificate(alias)) instanceof X509Certificate)) continue;
                X509Certificate x509Certificate = (X509Certificate)certificate;
                try {
                    x509Certificate.checkValidity();
                }
                catch (CertificateExpiredException e) {
                    aliasesToRemove.add(alias);
                }
                catch (CertificateNotYetValidException e) {
                    LOG.warn("Certificate with alias {} is not yet valid", (Object)alias);
                }
            }
            for (String alias : aliasesToRemove) {
                LOG.info("Removing expired CA certificate with alias {}", (Object)alias);
                this.truststore.deleteEntry(alias);
            }
            if (!aliasesToRemove.isEmpty()) {
                this.reloadTruststore();
            }
        }
        catch (Exception e) {
            LOG.warn("Failed to remove expired certificates from truststore", (Throwable)e);
        }
    }

    private synchronized void addConfluentLeafCertToTruststore() {
        if (this.truststore == null || this.fullChainPemPath == null) {
            return;
        }
        try {
            LOG.info("Adding Confluent leaf certificate to truststore");
            X509Certificate[] certificates = CaCertificatesUtils.x509CertificatesFromPemFile(this.fullChainPemPath);
            boolean truststoreChanged = false;
            for (X509Certificate certificate : certificates) {
                if (CaCertificatesUtils.isCertificateAuthority(certificate)) continue;
                this.truststore.setCertificateEntry(LEAF_CERT_ALIAS, certificate);
                truststoreChanged = true;
                break;
            }
            if (truststoreChanged) {
                this.reloadTruststore();
            }
        }
        catch (Exception e) {
            LOG.error("Failed to add Confluent leaf certificate to truststore", (Throwable)e);
        }
    }

    private void awaitTerminationAfterShutdown(ExecutorService threadPool) {
        if (threadPool == null) {
            return;
        }
        threadPool.shutdown();
        try {
            if (!threadPool.awaitTermination(60L, TimeUnit.SECONDS)) {
                threadPool.shutdownNow();
            }
        }
        catch (InterruptedException ex) {
            threadPool.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private void ensureTruststore() {
        if (this.truststore == null) {
            throw new IllegalStateException("Missing configure called, truststore is not initialized");
        }
    }
}

