/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.store.kafka.clients;

import io.confluent.security.authorizer.utils.ThreadUtils;
import io.confluent.security.store.KeyValueStore;
import io.confluent.security.store.MetadataStoreStatus;
import io.confluent.security.store.kafka.KafkaStoreConfig;
import io.confluent.security.store.kafka.clients.ConsumerListener;
import io.confluent.security.store.kafka.clients.KafkaUtils;
import io.confluent.security.store.kafka.clients.StatusListener;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.KafkaAdminClient;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.AuthorizationException;
import org.apache.kafka.common.errors.InterruptException;
import org.apache.kafka.common.errors.TopicExistsException;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaReader<K, V>
implements Runnable {
    private static final Logger log = LoggerFactory.getLogger(KafkaReader.class);
    private final String topic;
    private final int numPartitions;
    private final Future<Consumer<K, V>> consumerFuture;
    private Consumer<K, V> consumer;
    protected final KeyValueStore<K, V> cache;
    private final Time time;
    private final ExecutorService executor;
    private final AtomicBoolean alive;
    protected final Map<TopicPartition, PartitionState> partitionStates;
    protected final ConsumerListener<K, V> consumerListener;
    private final StatusListener statusListener;
    private final KafkaStoreConfig storeConfig;
    protected boolean prepareReaderStartupState = true;

    public KafkaReader(String topic, int numPartitions, Future<Consumer<K, V>> consumerFuture, KeyValueStore<K, V> cache, ConsumerListener<K, V> consumerListener, StatusListener statusListener, KafkaStoreConfig storeConfig, Time time) {
        this.topic = Objects.requireNonNull(topic, "topic");
        this.numPartitions = numPartitions;
        this.consumerFuture = Objects.requireNonNull(consumerFuture, "consumerFuture");
        this.cache = Objects.requireNonNull(cache, "cache");
        this.time = Objects.requireNonNull(time, "time");
        this.consumerListener = consumerListener;
        this.statusListener = statusListener;
        this.executor = Executors.newSingleThreadExecutor(ThreadUtils.createThreadFactory((String)"auth-reader-%d", (boolean)true));
        this.alive = new AtomicBoolean(true);
        this.partitionStates = new HashMap<TopicPartition, PartitionState>();
        this.storeConfig = storeConfig;
    }

    public CompletionStage<Void> start(Duration topicCreateTimeout) {
        CompletableFuture<Void> readyFuture = new CompletableFuture<Void>();
        this.executor.submit(() -> {
            try {
                this.consumer = this.consumerFuture.get();
                this.initialize(topicCreateTimeout);
                List<CompletableFuture<Void>> futures = this.partitionReadyFutures();
                CompletableFuture.allOf(futures.toArray(new CompletableFuture[futures.size()])).whenComplete((result, exception) -> {
                    if (exception != null) {
                        log.error("Kafka reader failed to initialize partition", exception);
                        readyFuture.completeExceptionally((Throwable)exception);
                    } else {
                        log.info("Kafka Auth reader initialized on all partitions");
                        readyFuture.complete((Void)result);
                    }
                });
            }
            catch (Throwable e) {
                log.error("Failed to initialize Kafka reader", e);
                this.alive.set(false);
                readyFuture.completeExceptionally(e);
            }
        });
        this.executor.submit(this);
        return readyFuture;
    }

    protected List<CompletableFuture<Void>> partitionReadyFutures() {
        return this.partitionStates.values().stream().map(s -> ((PartitionState)s).readyFuture).collect(Collectors.toList());
    }

    private void initialize(Duration topicCreateTimeout) {
        if (this.allowTopicCreate(this.storeConfig)) {
            this.createAuthTopic(this.storeConfig, this.topic);
        }
        KafkaUtils.waitForTopic(this.topic, this.numPartitions, this.time, topicCreateTimeout, this::describeTopic, null);
        Set<TopicPartition> partitions = IntStream.range(0, this.numPartitions).mapToObj(p -> new TopicPartition(this.topic, p)).collect(Collectors.toSet());
        this.consumer.assign(partitions);
        this.consumer.seekToEnd(partitions);
        Duration timeout = Duration.ofMillis(Integer.MAX_VALUE);
        partitions.forEach(tp -> this.partitionStates.put((TopicPartition)tp, new PartitionState(this.consumer.position(tp, timeout) - 1L)));
        log.debug("auth topic partitions : {}", partitions);
        this.consumer.seekToBeginning(partitions);
    }

    private boolean allowTopicCreate(KafkaStoreConfig configs) {
        Map originalProps = configs.originals();
        if (originalProps.containsKey("confluent.metadata.allow.reader.to.create.auth.topic")) {
            return Boolean.parseBoolean((String)originalProps.get("confluent.metadata.allow.reader.to.create.auth.topic"));
        }
        return false;
    }

    private Set<Integer> describeTopic(String topic) {
        if (!this.alive.get()) {
            throw new RuntimeException("KafkaReader has been shutdown");
        }
        List partitionInfos = this.consumer.partitionsFor(topic);
        if (partitionInfos != null) {
            return partitionInfos.stream().map(PartitionInfo::partition).collect(Collectors.toSet());
        }
        return Collections.emptySet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void createAuthTopic(KafkaStoreConfig storeConfig, String topic) {
        try {
            if (!this.alive.get()) {
                throw new RuntimeException("KafkaReader has been shutdown");
            }
            NewTopic metadataTopic = storeConfig.metadataTopicCreateConfig(topic, this.numPartitions);
            log.info("Creating auth topic {}", (Object)metadataTopic);
            AdminClient adminClient = KafkaAdminClient.create(storeConfig.adminClientConfigs());
            try {
                adminClient.createTopics(Collections.singletonList(metadataTopic)).all().get();
            }
            finally {
                adminClient.close(Duration.ZERO);
            }
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof TopicExistsException) {
                log.debug("Topic was created by different node");
            }
            Throwable cause = e.getCause();
            if (cause instanceof KafkaException) {
                throw (KafkaException)cause;
            }
            throw new KafkaException("Failed to create auth topic in reader " + topic, cause);
        }
        catch (InterruptedException e) {
            throw new InterruptException(e);
        }
    }

    @Override
    public void run() {
        ReaderStartupState readerStartupState = new ReaderStartupState(this.partitionStates);
        while (this.alive.get()) {
            try {
                ConsumerRecords records = this.consumer.poll(Duration.ofHours(1L));
                try {
                    if (this.prepareReaderStartupState) {
                        log.debug("preparing latest auth record entries, size : {}", (Object)readerStartupState.latestRecordMap.size());
                        readerStartupState.processRecords(records);
                        if (readerStartupState.pendingPartitions.isEmpty()) {
                            readerStartupState.latestRecordMap.values().stream().sorted(Comparator.comparingLong(ConsumerRecord::offset)).forEach(this::processConsumerRecord);
                            readerStartupState.newRecords.forEach(this::processConsumerRecord);
                            this.prepareReaderStartupState = false;
                            readerStartupState.clear();
                        }
                    } else {
                        records.forEach(this::processConsumerRecord);
                    }
                    this.statusListener.onReaderSuccess();
                }
                catch (Exception e) {
                    log.error("Unexpected exception while processing records {}", (Object)records, (Object)e);
                    this.fail(e);
                    break;
                }
            }
            catch (WakeupException e) {
                log.trace("Wakeup exception, consumer may be closing", (Throwable)e);
            }
            catch (Throwable exception) {
                log.error("Unexpected exception from consumer poll", exception);
                if (this.statusListener.onReaderFailure()) {
                    this.fail(exception);
                    break;
                }
                if (!(exception instanceof AuthenticationException | exception instanceof AuthorizationException)) continue;
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }
    }

    public CompletableFuture<Void> existingRecordsFuture() {
        return CompletableFuture.allOf((CompletableFuture[])this.partitionStates.values().stream().map(s -> s.existingRecordsFuture).toArray(CompletableFuture[]::new));
    }

    int numPartitions() {
        return this.partitionStates.size();
    }

    public void close(Duration closeTimeout) {
        if (this.alive.getAndSet(false) && this.consumer != null) {
            this.consumer.wakeup();
        }
        this.executor.shutdownNow();
        long endMs = this.time.milliseconds();
        try {
            this.executor.awaitTermination(closeTimeout.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            log.debug("KafkaReader was interrupted while waiting to close");
            throw new InterruptException(e);
        }
        long remainingMs = Math.max(0L, endMs - this.time.milliseconds());
        if (this.consumer != null) {
            this.consumer.close(Duration.ofMillis(remainingMs));
        }
    }

    protected void processConsumerRecord(ConsumerRecord<K, V> record) {
        PartitionState partitionState;
        Object key = record.key();
        Object newValue = record.value();
        Object oldValue = newValue != null ? this.cache.put(key, newValue) : this.cache.remove(key);
        log.debug("Processing new record {}-{}:{} key {} newValue {} oldValue {}", new Object[]{record.topic(), record.partition(), record.offset(), key, newValue, oldValue});
        if (this.consumerListener != null) {
            this.consumerListener.onConsumerRecord(record, oldValue);
        }
        if ((partitionState = this.partitionStates.get(new TopicPartition(record.topic(), record.partition()))) != null) {
            partitionState.onConsume(record.offset(), this.cache.status(record.partition()) == MetadataStoreStatus.INITIALIZED);
        }
    }

    private void fail(Throwable e) {
        IntStream.range(0, this.numPartitions).forEach(p -> this.cache.fail(p, "Metadata reader failed with exception: " + e));
    }

    private static class ReaderStartupState<K, V> {
        private final Map<K, ConsumerRecord<K, V>> latestRecordMap = new HashMap<K, ConsumerRecord<K, V>>();
        private final List<ConsumerRecord<K, V>> newRecords = new ArrayList<ConsumerRecord<K, V>>();
        private final Set<Integer> pendingPartitions;
        private final Map<TopicPartition, PartitionState> partitionStates;

        ReaderStartupState(Map<TopicPartition, PartitionState> partitionStates) {
            this.partitionStates = partitionStates;
            this.pendingPartitions = partitionStates.entrySet().stream().filter(e -> ((PartitionState)e.getValue()).minOffset != -1L).map(e -> ((TopicPartition)e.getKey()).partition()).collect(Collectors.toSet());
        }

        public void processRecords(ConsumerRecords<K, V> records) {
            records.forEach(record -> {
                long minOffset = this.partitionStates.get(new TopicPartition(record.topic(), record.partition())).minOffset;
                if (record.offset() > minOffset) {
                    this.newRecords.add((ConsumerRecord<K, V>)record);
                    this.pendingPartitions.remove(record.partition());
                } else {
                    ConsumerRecord<K, V> oldRecord = this.latestRecordMap.get(record.key());
                    if (oldRecord != null) {
                        if (record.timestamp() >= oldRecord.timestamp()) {
                            this.latestRecordMap.put(record.key(), (ConsumerRecord<Object, ConsumerRecord>)record);
                        }
                    } else {
                        this.latestRecordMap.put(record.key(), (ConsumerRecord<Object, ConsumerRecord>)record);
                    }
                    if (record.offset() == minOffset) {
                        this.pendingPartitions.remove(record.partition());
                    }
                }
            });
        }

        public void clear() {
            this.latestRecordMap.clear();
            this.newRecords.clear();
            this.pendingPartitions.clear();
        }
    }

    protected static class PartitionState {
        private final long minOffset;
        private final CompletableFuture<Void> readyFuture;
        public final CompletableFuture<Void> existingRecordsFuture;
        volatile long currentOffset;

        PartitionState(long offsetAtStartup) {
            this.minOffset = offsetAtStartup;
            this.readyFuture = new CompletableFuture();
            this.existingRecordsFuture = this.minOffset <= 0L ? CompletableFuture.completedFuture(null) : new CompletableFuture();
        }

        public void onConsume(long offset, boolean initialized) {
            this.currentOffset = offset;
            if (this.currentOffset >= this.minOffset) {
                if (this.existingRecordsFuture != null) {
                    this.existingRecordsFuture.complete(null);
                }
                if (!this.readyFuture.isDone() && initialized) {
                    this.readyFuture.complete(null);
                }
            }
        }

        public String toString() {
            return "PartitionState(minOffset=" + this.minOffset + ", currentOffset=" + this.currentOffset + ", existingRecords=" + this.existingRecordsFuture.isDone() + ", ready=" + this.readyFuture.isDone() + ')';
        }
    }
}

