/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller.metrics;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.metrics.ConfluentControllerMetrics;
import org.apache.kafka.controller.metrics.ConfluentControllerMetricsCache;
import org.apache.kafka.controller.metrics.ConfluentControllerMetricsUtils;
import org.apache.kafka.image.ConfigurationDelta;
import org.apache.kafka.image.ConfigurationImage;
import org.apache.kafka.image.ConfigurationsDelta;
import org.apache.kafka.image.ConfigurationsImage;
import org.apache.kafka.image.MetadataImage;
import org.apache.kafka.image.TopicDelta;
import org.apache.kafka.image.TopicImage;
import org.apache.kafka.image.TopicsDelta;
import org.apache.kafka.image.TopicsImage;
import org.apache.kafka.metadata.PartitionRegistration;
import org.apache.kafka.server.config.ServerLogConfigs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfluentControllerMetricsChanges {
    private static final Logger LOG = LoggerFactory.getLogger(ConfluentControllerMetricsChanges.class);
    private final ConfluentControllerMetricsCache cache;
    private final String logPrefix;
    private final Map<String, OptionalInt> topicToMinIsrChanges;
    private final Map<String, TenantChange> tenantChanges;
    private int globalTopicCountChange;
    private int globalPartitionCountChange;
    private int globalOfflinePartitionCountChange;
    private int globalUnderMinIsrCountChange;
    private int preferredReplicaImbalanceCountChange;

    ConfluentControllerMetricsChanges(ConfluentControllerMetricsCache cache) {
        this.cache = cache;
        this.logPrefix = "[ConfluentControllerMetricsChanges id=" + cache.nodeIdString() + "]";
        this.topicToMinIsrChanges = new HashMap<String, OptionalInt>(0);
        this.tenantChanges = new HashMap<String, TenantChange>();
        this.globalTopicCountChange = 0;
        this.globalPartitionCountChange = 0;
        this.globalOfflinePartitionCountChange = 0;
        this.globalUnderMinIsrCountChange = 0;
        this.preferredReplicaImbalanceCountChange = 0;
    }

    int globalTopicCountChange() {
        return this.globalTopicCountChange;
    }

    int globalPartitionCountChange() {
        return this.globalPartitionCountChange;
    }

    int globalOfflinePartitionCountChange() {
        return this.globalOfflinePartitionCountChange;
    }

    int globalUnderMinIsrCountChange() {
        return this.globalUnderMinIsrCountChange;
    }

    int preferredReplicaImbalanceCountChange() {
        return this.preferredReplicaImbalanceCountChange;
    }

    boolean handleConfigsChanges(MetadataImage newImage, ConfigurationsDelta delta) {
        for (Map.Entry<ConfigResource, ConfigurationDelta> entry : delta.changes().entrySet()) {
            if (!this.handleConfigChanges(entry.getKey(), entry.getValue())) continue;
            return true;
        }
        this.handleTopicsWithChangedUMinIsrConfigs(newImage);
        return false;
    }

    boolean handleConfigChanges(ConfigResource configResource, ConfigurationDelta delta) {
        switch (configResource.type()) {
            case BROKER: {
                return (configResource.isDefault() || configResource.name().trim().equals(this.cache.nodeIdString())) && delta.find(ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG) != null;
            }
            case TOPIC: {
                Optional<String> newMinIsrString = delta.find(ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG);
                if (newMinIsrString == null) break;
                ConfluentControllerMetricsUtils.withParsedUnderMinIsrConfig(newMinIsrString, this.cache.faultHandler(), newMinIsr -> this.topicToMinIsrChanges.put(configResource.name(), (OptionalInt)newMinIsr));
                break;
            }
        }
        return false;
    }

    void handleTopicsWithChangedUMinIsrConfigs(MetadataImage newImage) {
        for (String topicName : this.topicToMinIsrChanges.keySet()) {
            this.handleTopicMinIsrChange(newImage, topicName);
        }
    }

    private void handleTopicMinIsrChange(MetadataImage newImage, String topicName) {
        TopicImage prevTopicImage = this.cache.prevImage().topics().getTopic(topicName);
        if (prevTopicImage == null) {
            LOG.trace("{} handleTopicMinIsrChange: skipping newly created topic {}.", (Object)this.logPrefix, (Object)topicName);
            return;
        }
        TopicImage newTopicImage = (TopicImage)newImage.topics().topicsById().get((Object)prevTopicImage.id());
        if (newTopicImage == null) {
            LOG.trace("{} handleTopicMinIsrChange: skipping deleted topic {}.", (Object)this.logPrefix, (Object)topicName);
            return;
        }
        HashSet<Integer> newAndOldPartitions = new HashSet<Integer>(prevTopicImage.partitions().keySet());
        newAndOldPartitions.addAll(newTopicImage.partitions().keySet());
        this.handleTopicChange(newImage, newTopicImage.id(), newAndOldPartitions);
    }

    void handleTopicsChanges(MetadataImage newImage, TopicsDelta topicsDelta) {
        topicsDelta.deletedTopicIds().forEach(id -> this.handleTopicDeletion((Uuid)id));
        topicsDelta.changedTopics().entrySet().forEach(entry -> {
            TopicImage prevTopicImage = (TopicImage)this.cache.prevImage().topics().topicsById().get(entry.getKey());
            TopicImage newTopicImage = (TopicImage)newImage.topics().topicsById().get(entry.getKey());
            if (newTopicImage == null) {
                this.cache.faultHandler().handleFault("Failed to find changed topic " + entry.getKey() + " in new TopicsImage.");
            } else if (prevTopicImage == null || !this.topicToMinIsrChanges.containsKey(newTopicImage.name())) {
                this.handleTopicChange(newImage, newTopicImage.id(), ((TopicDelta)entry.getValue()).partitionChanges().keySet());
            }
        });
    }

    private void handleTopicDeletion(Uuid topicId) {
        TopicImage prevTopic = this.cache.prevImage().topics().getTopic(topicId);
        if (prevTopic == null) {
            this.cache.faultHandler().handleFault("Tried to remove topic " + topicId + ", which does not currently exist.");
            return;
        }
        String tenant = this.cache.topicNameToTenant(prevTopic.name());
        --this.globalTopicCountChange;
        for (PartitionRegistration partition : prevTopic.partitions().values()) {
            this.handlePartitionDeletion(tenant, partition, this.cache.minIsrForTopic(prevTopic.name()));
        }
    }

    private void handleTopicChange(MetadataImage newImage, Uuid topicId, Set<Integer> changedPartitions) {
        TopicImage newTopic = newImage.topics().getTopic(topicId);
        if (newTopic == null) {
            this.cache.faultHandler().handleFault("Tried to apply a change to topic " + topicId + ", which does not exist in the new metadata image.");
            return;
        }
        TopicImage prevTopic = this.cache.prevImage().topics().getTopic(topicId);
        Map<Object, Object> prevPartitions = null;
        if (prevTopic == null) {
            ++this.globalTopicCountChange;
            prevPartitions = Collections.emptyMap();
            if (LOG.isTraceEnabled()) {
                LOG.trace("{} handleTopicChange: creating new topic {}", (Object)this.logPrefix, (Object)newTopic.name());
            }
        } else {
            if (!newTopic.name().equals(prevTopic.name())) {
                this.cache.faultHandler().handleFault("Topic " + prevTopic.id() + " was renamed from " + prevTopic.name() + " to " + newTopic.name() + ", but this renaming is not yet supported.");
                return;
            }
            prevPartitions = prevTopic.partitions();
        }
        String tenant = this.cache.topicNameToTenant(newTopic.name());
        int prevMinIsr = this.cache.minIsrForTopic(newTopic.name());
        int newMinIsr = this.cache.minIsrForTopic(newTopic.name(), this.topicToMinIsrChanges);
        for (int partitionId : changedPartitions) {
            PartitionRegistration prevPartition = (PartitionRegistration)prevPartitions.get(partitionId);
            PartitionRegistration newPartition = newTopic.partitions().get(partitionId);
            if (prevPartition == null) {
                if (newPartition == null) {
                    this.cache.faultHandler().handleFault("Partition " + partitionId + " of topic id " + newTopic.id() + " was part of changedPartitions, but does not exist in old or new images.");
                    continue;
                }
                this.handlePartitionCreation(tenant, newPartition, newMinIsr);
                if (!LOG.isTraceEnabled()) continue;
                LOG.trace("{} handleTopicChange: creating partition {}-{}: {}", new Object[]{this.logPrefix, newTopic.name(), partitionId, newPartition});
                continue;
            }
            if (newPartition == null) {
                this.handlePartitionDeletion(tenant, prevPartition, prevMinIsr);
                if (!LOG.isTraceEnabled()) continue;
                LOG.trace("{} handleTopicChange: deleting partition {}-{}: {}", new Object[]{this.logPrefix, newTopic.name(), partitionId, newPartition});
                continue;
            }
            this.handlePartitionDeletion(tenant, prevPartition, prevMinIsr);
            this.handlePartitionCreation(tenant, newPartition, newMinIsr);
            if (!LOG.isTraceEnabled()) continue;
            LOG.trace("{}: handleTopicChange: changing partition {}-{}: from {} to {}", new Object[]{this.logPrefix, newTopic.name(), partitionId, prevPartition, newPartition});
        }
    }

    void handlePartitionCreation(String tenant, PartitionRegistration partition, int minIsr) {
        ++this.globalPartitionCountChange;
        if (ConfluentControllerMetricsUtils.partitionHasOfflineLeader(partition)) {
            ++this.globalOfflinePartitionCountChange;
        }
        if (partition.isr.length < minIsr) {
            ++this.globalUnderMinIsrCountChange;
        }
        if (ConfluentControllerMetricsUtils.partitionHasImbalancedLeader(partition)) {
            ++this.preferredReplicaImbalanceCountChange;
        }
        if (tenant != null) {
            TenantChange tenantChange = this.tenantChanges.computeIfAbsent(tenant, __ -> new TenantChange());
            tenantChange.partitionCountChange++;
            if (ConfluentControllerMetricsUtils.partitionHasOfflineLeader(partition)) {
                tenantChange.offlinePartitionCountChange++;
            }
            if (partition.isr.length < minIsr) {
                tenantChange.underMinIsrCountChange++;
            }
        }
    }

    void handlePartitionDeletion(String tenant, PartitionRegistration partition, int minIsr) {
        --this.globalPartitionCountChange;
        if (ConfluentControllerMetricsUtils.partitionHasOfflineLeader(partition)) {
            --this.globalOfflinePartitionCountChange;
        }
        if (partition.isr.length < minIsr) {
            --this.globalUnderMinIsrCountChange;
        }
        if (ConfluentControllerMetricsUtils.partitionHasImbalancedLeader(partition)) {
            --this.preferredReplicaImbalanceCountChange;
        }
        if (tenant != null) {
            TenantChange tenantChange = this.tenantChanges.computeIfAbsent(tenant, __ -> new TenantChange());
            tenantChange.partitionCountChange--;
            if (ConfluentControllerMetricsUtils.partitionHasOfflineLeader(partition)) {
                tenantChange.offlinePartitionCountChange--;
            }
            if (partition.isr.length < minIsr) {
                tenantChange.underMinIsrCountChange--;
            }
        }
    }

    void loadImage(MetadataImage newImage) {
        long startTimeNs = Time.SYSTEM.nanoseconds();
        this.cache.clear();
        this.loadConfigurations(newImage.configs());
        this.loadTopics(newImage.topics());
        long endTimeNs = Time.SYSTEM.nanoseconds();
        long durationTimeNs = endTimeNs - startTimeNs;
        LOG.info("{} Finished reloading all Confluent controller metrics in {} ms.", (Object)this.logPrefix, (Object)TimeUnit.NANOSECONDS.toMillis(durationTimeNs));
    }

    void loadConfigurations(ConfigurationsImage image) {
        for (Map.Entry<ConfigResource, ConfigurationImage> entry : image.resourceData().entrySet()) {
            ConfigResource configResource = entry.getKey();
            ConfigurationImage configImage = entry.getValue();
            switch (configResource.type()) {
                case BROKER: {
                    if (configResource.isDefault()) {
                        if (!configImage.data().containsKey(ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG)) break;
                        ConfluentControllerMetricsUtils.withParsedUnderMinIsrConfig(Optional.ofNullable(configImage.data().get(ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG)), this.cache.faultHandler(), underMinIsr -> {
                            this.cache.setClusterLevelMinIsrConfig((OptionalInt)underMinIsr);
                            LOG.info("{} set cluster-level {} to {}.", new Object[]{this.logPrefix, ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG, underMinIsr.orElseGet(() -> -1)});
                        });
                        break;
                    }
                    if (!configResource.name().equals(this.cache.nodeIdString()) || !configImage.data().containsKey(ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG)) break;
                    ConfluentControllerMetricsUtils.withParsedUnderMinIsrConfig(Optional.ofNullable(configImage.data().get(ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG)), this.cache.faultHandler(), underMinIsr -> {
                        this.cache.setNodeLevelMinIsrConfig((OptionalInt)underMinIsr);
                        LOG.info("{} set node-level {} to {}.", new Object[]{this.logPrefix, ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG, underMinIsr.orElseGet(() -> -1)});
                    });
                    break;
                }
                case TOPIC: {
                    if (!configImage.data().containsKey(ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG)) break;
                    ConfluentControllerMetricsUtils.withParsedUnderMinIsrConfig(Optional.ofNullable(configImage.data().get(ServerLogConfigs.MIN_IN_SYNC_REPLICAS_CONFIG)), this.cache.faultHandler(), underMinIsr -> this.cache.setPerTopicMinIsrConfig(configResource.name(), (OptionalInt)underMinIsr));
                    break;
                }
            }
        }
    }

    void loadTopics(TopicsImage image) {
        if (!this.topicToMinIsrChanges.isEmpty()) {
            throw new RuntimeException("Expected topicToMinIsrChanges to be empty when loading full topic state.");
        }
        for (TopicImage topic : image.topicsByName().values()) {
            String tenant = this.cache.topicNameToTenant(topic.name());
            ++this.globalTopicCountChange;
            int minIsr = this.cache.minIsrForTopic(topic.name());
            for (PartitionRegistration partition : topic.partitions().values()) {
                this.handlePartitionCreation(tenant, partition, minIsr);
            }
        }
    }

    void applyDelta(ConfluentControllerMetrics metrics) {
        for (Map.Entry<String, OptionalInt> entry : this.topicToMinIsrChanges.entrySet()) {
            this.cache.setPerTopicMinIsrConfig(entry.getKey(), entry.getValue());
        }
        for (Map.Entry<String, Object> entry : this.tenantChanges.entrySet()) {
            String tenant = entry.getKey();
            TenantChange change = (TenantChange)entry.getValue();
            metrics.addToTenantPartitionMetrics(tenant, change.partitionCountChange, change.offlinePartitionCountChange, change.underMinIsrCountChange);
        }
        if (this.globalTopicCountChange != 0) {
            metrics.addToGlobalTopicCount(this.globalTopicCountChange);
        }
        if (this.globalPartitionCountChange != 0) {
            metrics.addToGlobalPartitionCount(this.globalPartitionCountChange);
        }
        if (this.globalOfflinePartitionCountChange != 0) {
            metrics.addToGlobalOfflinePartitionCount(this.globalOfflinePartitionCountChange);
        }
        if (this.globalUnderMinIsrCountChange != 0) {
            metrics.addToGlobalUnderMinIsrCount(this.globalUnderMinIsrCountChange);
        }
        if (this.preferredReplicaImbalanceCountChange != 0) {
            metrics.addToPreferredReplicaImbalanceCount(this.preferredReplicaImbalanceCountChange);
        }
    }

    void applyFullState(ConfluentControllerMetrics metrics) {
        if (!this.topicToMinIsrChanges.isEmpty()) {
            throw new RuntimeException("Expected topicToMinIsrChanges to be empty when applying full state.");
        }
        HashMap<String, ConfluentControllerMetrics.TenantPartitionMetrics> newTenantPartitionMetricsMap = new HashMap<String, ConfluentControllerMetrics.TenantPartitionMetrics>();
        this.tenantChanges.entrySet().forEach(e -> newTenantPartitionMetricsMap.put((String)e.getKey(), new ConfluentControllerMetrics.TenantPartitionMetrics(((TenantChange)e.getValue()).partitionCountChange, ((TenantChange)e.getValue()).offlinePartitionCountChange, ((TenantChange)e.getValue()).underMinIsrCountChange)));
        metrics.setTenantPartitionMetrics(newTenantPartitionMetricsMap);
        metrics.setGlobalTopicCount(this.globalTopicCountChange);
        metrics.setGlobalPartitionCount(this.globalPartitionCountChange);
        metrics.setGlobalOfflinePartitionCount(this.globalOfflinePartitionCountChange);
        metrics.setGlobalUnderMinIsrCount(this.globalUnderMinIsrCountChange);
        metrics.setPreferredReplicaImbalanceCount(this.preferredReplicaImbalanceCountChange);
    }

    static final class TenantChange {
        private int partitionCountChange = 0;
        private int offlinePartitionCountChange = 0;
        private int underMinIsrCountChange = 0;

        TenantChange() {
        }

        public boolean equals(Object o) {
            if (o == null || !o.getClass().equals(TenantChange.class)) {
                return false;
            }
            TenantChange other = (TenantChange)o;
            return this.partitionCountChange == other.partitionCountChange && this.offlinePartitionCountChange == other.offlinePartitionCountChange && this.underMinIsrCountChange == other.underMinIsrCountChange;
        }

        public int hashCode() {
            return Objects.hash(this.partitionCountChange, this.offlinePartitionCountChange, this.underMinIsrCountChange);
        }

        public String toString() {
            return "TenantChange(partitionCountChange=" + this.partitionCountChange + ", offlinePartitionCountChange=" + this.offlinePartitionCountChange + ", underMinIsrCountChange=" + this.underMinIsrCountChange + ")";
        }
    }
}

