/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.schemaregistry.storage;

import io.confluent.kafka.schemaregistry.exceptions.SchemaRegistryException;
import io.confluent.kafka.schemaregistry.rest.SchemaRegistryConfig;
import io.confluent.kafka.schemaregistry.storage.CloseableIterator;
import io.confluent.kafka.schemaregistry.storage.KafkaSchemaRegistry;
import io.confluent.kafka.schemaregistry.storage.KafkaStoreReaderThread;
import io.confluent.kafka.schemaregistry.storage.Store;
import io.confluent.kafka.schemaregistry.storage.StoreUpdateHandler;
import io.confluent.kafka.schemaregistry.storage.SubjectKey;
import io.confluent.kafka.schemaregistry.storage.exceptions.EntryTooLargeException;
import io.confluent.kafka.schemaregistry.storage.exceptions.SerializationException;
import io.confluent.kafka.schemaregistry.storage.exceptions.StoreException;
import io.confluent.kafka.schemaregistry.storage.exceptions.StoreInitializationException;
import io.confluent.kafka.schemaregistry.storage.exceptions.StoreTimeoutException;
import io.confluent.kafka.schemaregistry.storage.serialization.Serializer;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.Config;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.admin.TopicDescription;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.TopicPartitionInfo;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.errors.RecordTooLargeException;
import org.apache.kafka.common.errors.TopicExistsException;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaStore<K, V>
implements Store<K, V> {
    private static final Logger log = LoggerFactory.getLogger(KafkaStore.class);
    private final String topic;
    private final int desiredReplicationFactor;
    private final String groupId;
    private final StoreUpdateHandler<K, V> storeUpdateHandler;
    private final Serializer<K, V> serializer;
    private final Store<K, V> localStore;
    private final AtomicBoolean initialized = new AtomicBoolean(false);
    private final CountDownLatch initLatch = new CountDownLatch(1);
    private final int initTimeout;
    private final int timeout;
    private final String bootstrapBrokers;
    private final boolean skipSchemaTopicValidation;
    private KafkaProducer<byte[], byte[]> producer;
    private KafkaStoreReaderThread<K, V> kafkaTopicReader;
    private final K noopKey;
    private volatile long lastWrittenOffset = -1L;
    private final SchemaRegistryConfig config;
    private final Lock leaderLock = new ReentrantLock();
    private final Lock lock = new ReentrantLock();

    public KafkaStore(SchemaRegistryConfig config, StoreUpdateHandler<K, V> storeUpdateHandler, Serializer<K, V> serializer, Store<K, V> localStore, K noopKey) throws SchemaRegistryException {
        this.topic = config.getString("kafkastore.topic");
        this.desiredReplicationFactor = config.getInt("kafkastore.topic.replication.factor");
        this.config = config;
        int port = KafkaSchemaRegistry.getInterInstanceListener(config.getListeners(), config.interInstanceListenerName(), config.interInstanceProtocol()).getUri().getPort();
        this.groupId = config.getString("kafkastore.group.id").isEmpty() ? String.format("schema-registry-%s-%d", config.getString("host.name"), port) : config.getString("kafkastore.group.id");
        this.initTimeout = config.getInt("kafkastore.init.timeout.ms");
        this.timeout = config.getInt("kafkastore.timeout.ms");
        this.storeUpdateHandler = storeUpdateHandler;
        this.serializer = serializer;
        this.localStore = localStore;
        this.noopKey = noopKey;
        this.bootstrapBrokers = config.bootstrapBrokers();
        this.skipSchemaTopicValidation = config.getBoolean("kafkastore.topic.skip.validation");
        log.info("Initializing KafkaStore with broker endpoints: {}", (Object)this.bootstrapBrokers);
    }

    @Override
    public void init() throws StoreInitializationException {
        if (this.initialized.get()) {
            throw new StoreInitializationException("Illegal state while initializing store. Store was already initialized");
        }
        this.localStore.init();
        this.createOrVerifySchemaTopic();
        Properties props = new Properties();
        KafkaStore.addSchemaRegistryConfigsToClientProperties(this.config, props);
        props.put("bootstrap.servers", this.bootstrapBrokers);
        props.put("acks", "-1");
        props.put("key.serializer", ByteArraySerializer.class);
        props.put("value.serializer", ByteArraySerializer.class);
        props.put("retries", (Object)0);
        props.put("enable.idempotence", (Object)false);
        this.producer = new KafkaProducer(props);
        this.kafkaTopicReader = new KafkaStoreReaderThread<K, V>(this.bootstrapBrokers, this.topic, this.groupId, this.storeUpdateHandler, this.serializer, this.localStore, (Producer<byte[], byte[]>)this.producer, this.noopKey, this.initialized, this.config);
        HashMap<TopicPartition, Long> checkpoints = new HashMap<TopicPartition, Long>(this.kafkaTopicReader.checkpoints());
        this.kafkaTopicReader.start();
        try {
            this.waitUntilKafkaReaderReachesLastOffset(this.initTimeout);
        }
        catch (StoreException e) {
            throw new StoreInitializationException(e);
        }
        boolean isInitialized = this.initialized.compareAndSet(false, true);
        if (!isInitialized) {
            throw new StoreInitializationException("Illegal state while initializing store. Store was already initialized");
        }
        this.storeUpdateHandler.cacheInitialized(checkpoints);
        this.initLatch.countDown();
    }

    public static void addSchemaRegistryConfigsToClientProperties(SchemaRegistryConfig config, Properties props) {
        props.putAll((Map<?, ?>)config.originalsWithPrefix("kafkastore."));
    }

    private void createOrVerifySchemaTopic() throws StoreInitializationException {
        if (this.skipSchemaTopicValidation) {
            log.info("Skipping auto topic creation and verification");
            return;
        }
        Properties props = new Properties();
        KafkaStore.addSchemaRegistryConfigsToClientProperties(this.config, props);
        props.put("bootstrap.servers", this.bootstrapBrokers);
        try (AdminClient admin = AdminClient.create((Properties)props);){
            Set allTopics = (Set)admin.listTopics().names().get((long)this.initTimeout, TimeUnit.MILLISECONDS);
            if (allTopics.contains(this.topic)) {
                this.verifySchemaTopic(admin);
            } else {
                this.createSchemaTopic(admin);
            }
        }
        catch (TimeoutException e) {
            throw new StoreInitializationException("Timed out trying to create or validate schema topic configuration", e);
        }
        catch (InterruptedException | ExecutionException e) {
            throw new StoreInitializationException("Failed trying to create or validate schema topic configuration", e);
        }
    }

    private void createSchemaTopic(AdminClient admin) throws StoreInitializationException, InterruptedException, ExecutionException, TimeoutException {
        log.info("Creating schemas topic {}", (Object)this.topic);
        int numLiveBrokers = ((Collection)admin.describeCluster().nodes().get((long)this.initTimeout, TimeUnit.MILLISECONDS)).size();
        if (numLiveBrokers <= 0) {
            throw new StoreInitializationException("No live Kafka brokers");
        }
        int schemaTopicReplicationFactor = Math.min(numLiveBrokers, this.desiredReplicationFactor);
        if (schemaTopicReplicationFactor < this.desiredReplicationFactor) {
            log.warn("Creating the schema topic " + this.topic + " using a replication factor of " + schemaTopicReplicationFactor + ", which is less than the desired one of " + this.desiredReplicationFactor + ". If this is a production environment, it's crucial to add more brokers and increase the replication factor of the topic.");
        }
        NewTopic schemaTopicRequest = new NewTopic(this.topic, 1, (short)schemaTopicReplicationFactor);
        HashMap<String, String> topicConfigs = new HashMap<String, String>(this.config.originalsWithPrefix("kafkastore.topic.config."));
        topicConfigs.put("cleanup.policy", "compact");
        schemaTopicRequest.configs(topicConfigs);
        try {
            admin.createTopics(Collections.singleton(schemaTopicRequest)).all().get((long)this.initTimeout, TimeUnit.MILLISECONDS);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof TopicExistsException) {
                this.verifySchemaTopic(admin);
            }
            throw e;
        }
    }

    private void verifySchemaTopic(AdminClient admin) throws StoreInitializationException, InterruptedException, ExecutionException, TimeoutException {
        ConfigResource topicResource;
        Map configs;
        Config topicConfigs;
        String retentionPolicy;
        log.info("Validating schemas topic {}", (Object)this.topic);
        Set<String> topics = Collections.singleton(this.topic);
        Map topicDescription = (Map)admin.describeTopics(topics).all().get((long)this.initTimeout, TimeUnit.MILLISECONDS);
        TopicDescription description = (TopicDescription)topicDescription.get(this.topic);
        int numPartitions = description.partitions().size();
        if (numPartitions != 1) {
            throw new StoreInitializationException("The schema topic " + this.topic + " should have only 1 partition but has " + numPartitions);
        }
        if (((TopicPartitionInfo)description.partitions().get(0)).replicas().size() < this.desiredReplicationFactor) {
            log.warn("The replication factor of the schema topic " + this.topic + " is less than the desired one of " + this.desiredReplicationFactor + ". If this is a production environment, it's crucial to add more brokers and increase the replication factor of the topic.");
        }
        if ((retentionPolicy = (topicConfigs = (Config)(configs = (Map)admin.describeConfigs(Collections.singleton(topicResource = new ConfigResource(ConfigResource.Type.TOPIC, this.topic))).all().get((long)this.initTimeout, TimeUnit.MILLISECONDS)).get(topicResource)).get("cleanup.policy").value()) == null || !"compact".equals(retentionPolicy)) {
            log.error("The retention policy of the schema topic {} is incorrect. You must configure the topic to 'compact' cleanup policy to avoid Kafka deleting your schemas after a week. Refer to Kafka documentation for more details on cleanup policies", (Object)this.topic);
            throw new StoreInitializationException("The retention policy of the schema topic " + this.topic + " is incorrect. Expected cleanup.policy to be 'compact' but it is " + retentionPolicy);
        }
    }

    public void waitUntilKafkaReaderReachesLastOffset(int timeoutMs) throws StoreException {
        long offsetOfLastMessage = this.getLatestOffset(timeoutMs);
        this.waitUntilKafkaReaderReachesOffset(offsetOfLastMessage, timeoutMs);
    }

    public void waitUntilKafkaReaderReachesLastOffset(String subject, int timeoutMs) throws StoreException {
        long lastOffset = this.lastOffset(subject);
        if (lastOffset == -1L) {
            lastOffset = this.getLatestOffset(timeoutMs);
        }
        this.waitUntilKafkaReaderReachesOffset(lastOffset, timeoutMs);
    }

    private void waitUntilKafkaReaderReachesOffset(long offset, int timeoutMs) throws StoreException {
        log.info("Wait to catch up until the offset at {}", (Object)offset);
        this.kafkaTopicReader.waitUntilOffset(offset, timeoutMs, TimeUnit.MILLISECONDS);
        log.info("Reached offset at {}", (Object)offset);
    }

    public void markLastWrittenOffsetInvalid() {
        this.lastWrittenOffset = -1L;
    }

    @Override
    public V get(K key) throws StoreException {
        return this.localStore.get(key);
    }

    @Override
    public V put(K key, V value) throws StoreTimeoutException, StoreException {
        this.assertInitialized();
        if (key == null) {
            throw new StoreException("Key should not be null");
        }
        V oldValue = this.get(key);
        ProducerRecord producerRecord = null;
        try {
            producerRecord = new ProducerRecord(this.topic, Integer.valueOf(0), (Object)this.serializer.serializeKey(key), value == null ? null : this.serializer.serializeValue(value));
        }
        catch (SerializationException e) {
            throw new StoreException("Error serializing schema while creating the Kafka produce record", e);
        }
        boolean knownSuccessfulWrite = false;
        try {
            log.trace("Sending record to KafkaStore topic: {}", (Object)producerRecord);
            Future ack = this.producer.send(producerRecord);
            RecordMetadata recordMetadata = (RecordMetadata)ack.get(this.timeout, TimeUnit.MILLISECONDS);
            log.trace("Waiting for the local store to catch up to offset {}", (Object)recordMetadata.offset());
            this.lastWrittenOffset = recordMetadata.offset();
            if (key instanceof SubjectKey) {
                this.setLastOffset(((SubjectKey)key).getSubject(), recordMetadata.offset());
            }
            this.waitUntilKafkaReaderReachesOffset(recordMetadata.offset(), this.timeout);
            knownSuccessfulWrite = true;
        }
        catch (InterruptedException e) {
            throw new StoreException("Put operation interrupted while waiting for an ack from Kafka", e);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof RecordTooLargeException) {
                throw new EntryTooLargeException("Put operation failed because entry is too large");
            }
            throw new StoreException("Put operation failed while waiting for an ack from Kafka", e);
        }
        catch (TimeoutException e) {
            throw new StoreTimeoutException("Put operation timed out while waiting for an ack from Kafka", e);
        }
        catch (KafkaException ke) {
            throw new StoreException("Put operation to Kafka failed", ke);
        }
        finally {
            if (!knownSuccessfulWrite) {
                this.markLastWrittenOffsetInvalid();
            }
        }
        return oldValue;
    }

    @Override
    public CloseableIterator<V> getAll(K key1, K key2) throws StoreException {
        this.assertInitialized();
        return this.localStore.getAll(key1, key2);
    }

    @Override
    public void putAll(Map<K, V> entries) throws StoreException {
        this.assertInitialized();
        for (Map.Entry<K, V> entry : entries.entrySet()) {
            this.put(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public V delete(K key) throws StoreException {
        this.assertInitialized();
        return this.put(key, null);
    }

    @Override
    public CloseableIterator<K> getAllKeys() throws StoreException {
        this.assertInitialized();
        return this.localStore.getAllKeys();
    }

    @Override
    public void flush() throws StoreException {
        this.localStore.flush();
    }

    @Override
    public void close() {
        try {
            if (this.kafkaTopicReader != null) {
                this.kafkaTopicReader.shutdown();
            }
            if (this.producer != null) {
                this.producer.close();
                log.info("Kafka store producer shut down");
            }
            this.localStore.close();
            if (this.storeUpdateHandler != null) {
                this.storeUpdateHandler.close();
            }
            log.info("Kafka store shut down complete");
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void waitForInit() throws InterruptedException {
        if (this.initLatch.getCount() > 0L) {
            this.initLatch.await();
        }
    }

    public boolean initialized() {
        return this.initialized.get();
    }

    KafkaStoreReaderThread<K, V> getKafkaStoreReaderThread() {
        return this.kafkaTopicReader;
    }

    private void assertInitialized() throws StoreException {
        if (!this.initialized.get()) {
            throw new StoreException("Illegal state. Store not initialized yet");
        }
    }

    private long getLatestOffset(int timeoutMs) throws StoreException {
        ProducerRecord producerRecord = null;
        if (this.lastWrittenOffset >= 0L) {
            return this.lastWrittenOffset;
        }
        try {
            producerRecord = new ProducerRecord(this.topic, Integer.valueOf(0), (Object)this.serializer.serializeKey(this.noopKey), null);
        }
        catch (SerializationException e) {
            throw new StoreException("Failed to serialize noop key.", e);
        }
        try {
            log.trace("Sending Noop record to KafkaStore to find last offset.");
            Future ack = this.producer.send(producerRecord);
            RecordMetadata metadata = (RecordMetadata)ack.get(timeoutMs, TimeUnit.MILLISECONDS);
            this.lastWrittenOffset = metadata.offset();
            log.trace("Noop record's offset is {}", (Object)this.lastWrittenOffset);
            return this.lastWrittenOffset;
        }
        catch (Exception e) {
            throw new StoreException("Failed to write Noop record to kafka store.", e);
        }
    }

    public long lastOffset(String subject) {
        return this.lastWrittenOffset;
    }

    public void setLastOffset(String subject, long lastOffset) {
        this.lastWrittenOffset = lastOffset;
    }

    public Lock leaderLock() {
        return this.leaderLock;
    }

    public Lock lockFor(String subject) {
        return this.lock;
    }
}

