/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.schemaregistry.storage.encoder;

import com.google.common.annotations.VisibleForTesting;
import com.google.crypto.tink.Aead;
import com.google.crypto.tink.KeyTemplate;
import com.google.crypto.tink.KeyTemplates;
import com.google.crypto.tink.KeysetHandle;
import com.google.crypto.tink.KeysetManager;
import com.google.crypto.tink.Registry;
import com.google.crypto.tink.aead.AeadConfig;
import com.google.crypto.tink.proto.AesGcmKey;
import com.google.crypto.tink.subtle.Hkdf;
import com.google.protobuf.ByteString;
import io.confluent.kafka.schemaregistry.rest.SchemaRegistryConfig;
import io.confluent.kafka.schemaregistry.storage.KafkaSchemaRegistry;
import io.confluent.kafka.schemaregistry.storage.MD5;
import io.confluent.kafka.schemaregistry.storage.Metadata;
import io.confluent.kafka.schemaregistry.storage.SchemaRegistry;
import io.confluent.kafka.schemaregistry.storage.SchemaValue;
import io.confluent.kafka.schemaregistry.storage.encoder.KeysetWrapper;
import io.confluent.kafka.schemaregistry.storage.encoder.KeysetWrapperSerde;
import io.confluent.kafka.schemaregistry.utils.QualifiedSubject;
import io.kcache.Cache;
import io.kcache.CacheUpdateHandler;
import io.kcache.KafkaCache;
import io.kcache.KafkaCacheConfig;
import io.kcache.exceptions.CacheInitializationException;
import io.kcache.utils.Caches;
import io.kcache.utils.InMemoryCache;
import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.AbstractMap;
import java.util.Base64;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import org.apache.kafka.common.config.types.Password;
import org.apache.kafka.common.serialization.Serde;
import org.apache.kafka.common.serialization.Serdes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetadataEncoderService
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(MetadataEncoderService.class);
    private static final String AES_GCM_KEY = "type.googleapis.com/google.crypto.tink.AesGcmKey";
    private static final byte[] EMPTY_AAD = new byte[0];
    private static final String KEY_TEMPLATE_NAME = "AES128_GCM";
    private final KafkaSchemaRegistry schemaRegistry;
    private KeyTemplate keyTemplate;
    Cache<String, KeysetWrapper> encoders = null;
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final CountDownLatch initLatch = new CountDownLatch(1);

    public MetadataEncoderService(SchemaRegistry schemaRegistry) {
        try {
            this.schemaRegistry = (KafkaSchemaRegistry)schemaRegistry;
            SchemaRegistryConfig config = schemaRegistry.config();
            String secret = MetadataEncoderService.encoderSecret(config);
            if (secret == null) {
                log.warn("No value specified for {}, sensitive metadata will not be encoded", (Object)"metadata.encoder.secret");
                return;
            }
            String topic = config.getString("metadata.encoder.topic");
            this.keyTemplate = KeyTemplates.get((String)KEY_TEMPLATE_NAME);
            this.encoders = this.createCache((Serde)new Serdes.StringSerde(), new KeysetWrapperSerde(config), topic, null);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Could not instantiate MetadataEncoderService", e);
        }
    }

    @VisibleForTesting
    protected MetadataEncoderService(SchemaRegistry schemaRegistry, Cache<String, KeysetWrapper> encoders) {
        try {
            this.schemaRegistry = (KafkaSchemaRegistry)schemaRegistry;
            SchemaRegistryConfig config = schemaRegistry.config();
            String secret = MetadataEncoderService.encoderSecret(config);
            if (secret == null) {
                log.warn("No value specified for {}, sensitive metadata will not be encoded", (Object)"metadata.encoder.secret");
                return;
            }
            this.keyTemplate = KeyTemplates.get((String)KEY_TEMPLATE_NAME);
            this.encoders = encoders;
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Could not instantiate MetadataEncoderService", e);
        }
    }

    public KafkaSchemaRegistry getSchemaRegistry() {
        return this.schemaRegistry;
    }

    protected static Aead getPrimitive(String secret) throws GeneralSecurityException {
        if (secret == null) {
            throw new IllegalArgumentException("Secret is null");
        }
        byte[] keyBytes = Hkdf.computeHkdf((String)"HmacSha256", (byte[])secret.getBytes(StandardCharsets.UTF_8), null, null, (int)16);
        AesGcmKey key = AesGcmKey.newBuilder().setVersion(0).setKeyValue(ByteString.copyFrom((byte[])keyBytes)).build();
        return (Aead)Registry.getPrimitive((String)AES_GCM_KEY, (ByteString)key.toByteString(), Aead.class);
    }

    protected static String encoderSecret(SchemaRegistryConfig config) {
        Password password = config.getPassword("metadata.encoder.secret");
        String secret = password != null ? password.value() : System.getenv("METADATA_ENCODER_SECRET");
        return secret;
    }

    protected static String encoderOldSecret(SchemaRegistryConfig config) {
        Password password = config.getPassword("metadata.encoder.old.secret");
        String secret = password != null ? password.value() : System.getenv("METADATA_ENCODER_OLD_SECRET");
        return secret;
    }

    protected <K, V> Cache<K, V> createCache(Serde<K> keySerde, Serde<V> valueSerde, String topic, CacheUpdateHandler<K, V> cacheUpdateHandler) throws CacheInitializationException {
        Properties props = this.getKafkaCacheProperties(topic);
        KafkaCacheConfig config = new KafkaCacheConfig((Map)props);
        Cache kafkaCache = Caches.concurrentCache((Cache)new KafkaCache(config, keySerde, valueSerde, cacheUpdateHandler, (Cache)new InMemoryCache()));
        this.getSchemaRegistry().addLeaderChangeListener(isLeader -> {
            if (isLeader.booleanValue()) {
                kafkaCache.reset();
                kafkaCache.sync();
            }
        });
        return kafkaCache;
    }

    private Properties getKafkaCacheProperties(String topic) {
        Properties props = new Properties();
        props.putAll((Map<?, ?>)this.schemaRegistry.config().originalProperties());
        Set<String> keys = props.stringPropertyNames();
        for (String key : keys) {
            String newKey;
            if (!key.startsWith("kafkastore.") || keys.contains(newKey = key.replace("kafkastore", "kafkacache"))) continue;
            props.put(newKey, props.get(key));
        }
        props.put("kafkacache.topic", topic);
        return props;
    }

    public void init() {
        if (this.encoders != null && !this.initialized.get()) {
            this.encoders.init();
            this.maybeRotateSecrets();
            boolean isInitialized = this.initialized.compareAndSet(false, true);
            if (!isInitialized) {
                throw new IllegalStateException("Metadata encoder service was already initialized");
            }
            this.initLatch.countDown();
        }
    }

    private void maybeRotateSecrets() {
        String oldSecret = MetadataEncoderService.encoderOldSecret(this.schemaRegistry.config());
        if (oldSecret != null) {
            log.info("Rotating encoder secret");
            for (String key : this.encoders.keySet()) {
                KeysetWrapper wrapper = (KeysetWrapper)this.encoders.get((Object)key);
                if (!wrapper.isRotationNeeded()) continue;
                try {
                    KeysetHandle handle = wrapper.getKeysetHandle();
                    KeysetHandle rotatedHandle = KeysetManager.withKeysetHandle((KeysetHandle)handle).add(this.keyTemplate).getKeysetHandle();
                    int keyId = rotatedHandle.getAt(rotatedHandle.size()).getId();
                    rotatedHandle = KeysetManager.withKeysetHandle((KeysetHandle)rotatedHandle).setPrimary(keyId).getKeysetHandle();
                    this.encoders.put((Object)key, (Object)new KeysetWrapper(rotatedHandle, false));
                }
                catch (GeneralSecurityException e) {
                    log.error("Could not rotate key for {}", (Object)key, (Object)e);
                }
            }
            log.info("Done rotating encoder secret");
        }
    }

    public void waitForInit() throws InterruptedException {
        this.initLatch.await();
    }

    public KeysetHandle getEncoder(String tenant) {
        if (this.encoders == null) {
            return null;
        }
        KeysetWrapper wrapper = (KeysetWrapper)this.encoders.get((Object)tenant);
        return wrapper != null ? wrapper.getKeysetHandle() : null;
    }

    public void encodeMetadata(SchemaValue schema) {
        if (!this.initialized.get() || schema == null || this.isEncoded(schema)) {
            return;
        }
        this.transformMetadata(schema, true, (aead, value) -> {
            try {
                byte[] ciphertext = aead.encrypt(value.getBytes(StandardCharsets.UTF_8), EMPTY_AAD);
                return Base64.getEncoder().encodeToString(ciphertext);
            }
            catch (GeneralSecurityException e) {
                throw new IllegalStateException("Could not encrypt sensitive metadata", e);
            }
        });
    }

    private boolean isEncoded(SchemaValue schema) {
        if (schema == null || schema.getMetadata() == null || schema.getMetadata().getProperties() == null) {
            return false;
        }
        return schema.getMetadata().getProperties().containsKey("__enc__");
    }

    public void decodeMetadata(SchemaValue schema) {
        if (!this.initialized.get() || schema == null || !this.isEncoded(schema)) {
            return;
        }
        this.transformMetadata(schema, false, (aead, value) -> {
            try {
                byte[] plaintext = aead.decrypt(Base64.getDecoder().decode((String)value), EMPTY_AAD);
                return new String(plaintext, StandardCharsets.UTF_8);
            }
            catch (GeneralSecurityException e) {
                log.error("Could not decrypt sensitive metadata for schema {}", (Object)schema, (Object)e);
                return value;
            }
        });
    }

    private void transformMetadata(SchemaValue schema, boolean isEncode, BiFunction<Aead, String, String> func) {
        Metadata metadata = schema.getMetadata();
        if (metadata == null || metadata.getProperties() == null || metadata.getProperties().isEmpty() || metadata.getSensitive() == null || metadata.getSensitive().isEmpty()) {
            return;
        }
        try {
            String subject = schema.getSubject();
            QualifiedSubject qualifiedSubject = QualifiedSubject.create((String)this.schemaRegistry.tenant(), (String)subject);
            String tenant = qualifiedSubject.getTenant();
            KeysetHandle handle = this.getOrCreateEncoder(tenant);
            Aead aead = (Aead)handle.getPrimitive(Aead.class);
            SortedMap newProperties = metadata.getProperties().entrySet().stream().map(e -> new AbstractMap.SimpleEntry(e.getKey(), metadata.getSensitive().contains(e.getKey()) ? (String)func.apply(aead, (String)e.getValue()) : (String)e.getValue())).collect(Collectors.toMap(AbstractMap.SimpleEntry::getKey, AbstractMap.SimpleEntry::getValue, (e1, e2) -> e2, TreeMap::new));
            if (isEncode) {
                schema.setMd5Bytes(MD5.ofSchema(schema.toSchemaEntity()).bytes());
                newProperties.put("__enc__", "true");
            } else {
                newProperties.remove("__enc__");
            }
            schema.setMetadata(new Metadata(metadata.getTags(), newProperties, metadata.getSensitive()));
        }
        catch (GeneralSecurityException e3) {
            throw new IllegalStateException("Could not encrypt sensitive metadata", e3);
        }
    }

    private KeysetHandle getOrCreateEncoder(String tenant) {
        KeysetWrapper wrapper = (KeysetWrapper)this.encoders.computeIfAbsent((Object)tenant, k -> {
            try {
                KeysetHandle handle = KeysetHandle.generateNew((KeyTemplate)this.keyTemplate);
                return new KeysetWrapper(handle, false);
            }
            catch (GeneralSecurityException e) {
                throw new IllegalStateException("Could not create key template");
            }
        });
        return wrapper.getKeysetHandle();
    }

    @Override
    public void close() {
        log.info("Shutting down MetadataEncoderService");
        if (this.encoders != null) {
            try {
                this.encoders.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    static {
        try {
            AeadConfig.register();
        }
        catch (GeneralSecurityException e) {
            throw new IllegalArgumentException(e);
        }
    }
}

