/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.network;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.common.network.AbstractProxyProtocolEngine;
import org.apache.kafka.common.network.CCloudTrafficType;
import org.apache.kafka.common.network.Mode;
import org.apache.kafka.common.network.ProxyProtocolCommand;
import org.apache.kafka.common.network.ProxyTlv;
import org.apache.kafka.common.network.ProxyTlvType;
import org.apache.kafka.common.network.ProxyTlvValidator;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Utils;

public class ProxyProtocolV2Engine
extends AbstractProxyProtocolEngine {
    private static final String INVALID_PROTOCOL_V2_HEADER = "Invalid Proxy Protocol V2 Header.";
    private static final byte[] PROTOCOL_SIGNATURE = new byte[]{13, 10, 13, 10, 0, 13, 10, 81, 85, 73, 84, 10};
    private boolean protocolSignatureDetected = false;
    private byte protocolByte;
    private int bytesRead;
    private int addressLength;
    private byte[] addressBytes;
    private ProxyProtocolCommand command;
    private final Map<ProxyTlvType, ProxyTlv> tlvs = new HashMap<ProxyTlvType, ProxyTlv>();

    public ProxyProtocolV2Engine(Mode mode, LogContext logContext) {
        super(mode, logContext);
    }

    @Override
    protected void configureClient(Map<String, Object> configs) {
        CCloudTrafficType trafficType;
        this.command = ProxyProtocolV2Engine.getConfiguredCommand(configs);
        if (this.command == ProxyProtocolCommand.PROXY) {
            this.sourceAddress = ProxyProtocolV2Engine.getConfiguredSourceAddress(configs);
            this.sourcePort = ProxyProtocolV2Engine.getConfiguredSourcePort(configs);
        } else {
            this.sourceAddress = InetAddress.getLoopbackAddress();
            this.sourcePort = 0;
        }
        Optional<String> lkcIdOpt = ProxyProtocolV2Engine.getConfiguredLkcId(configs);
        String lkcId = null;
        if (lkcIdOpt.isPresent()) {
            lkcId = lkcIdOpt.get();
            ProxyTlv tlv = ProxyTlv.createForSubtype(ProxyTlvType.LKC_ID, lkcId);
            this.tlvs.put(tlv.type, tlv);
        }
        if ((trafficType = (CCloudTrafficType)((Object)configs.get("confluent.ccloud.traffic.type"))) != null) {
            ProxyTlv tlv = trafficType.tlv();
            this.tlvs.put(tlv.type, tlv);
        }
        this.log.debug("The proxy protocol engine was configured with {}: {}, {}: {}, {}: {}, {}: {}", new Object[]{"confluent.proxy.protocol.client.address", this.sourceAddress, "confluent.proxy.protocol.client.port", this.sourcePort, "confluent.proxy.protocol.client.mode", this.command, "confluent.lkc.id", lkcId});
    }

    protected static ProxyProtocolCommand getConfiguredCommand(Map<String, Object> configs) {
        String command = ProxyProtocolV2Engine.getConfiguredString(configs, "confluent.proxy.protocol.client.mode");
        try {
            return ProxyProtocolCommand.forName(command);
        }
        catch (IllegalArgumentException e) {
            throw ProxyProtocolV2Engine.newConfigException("confluent.proxy.protocol.client.mode", command);
        }
    }

    protected static Optional<String> getConfiguredLkcId(Map<String, Object> configs) {
        String lkcId = (String)configs.get("confluent.lkc.id");
        if (Utils.isBlank(lkcId)) {
            return Optional.empty();
        }
        lkcId = lkcId.trim();
        ProxyProtocolV2Engine.validateLkcId(lkcId);
        return Optional.of(lkcId);
    }

    @Override
    public void processHeaders(ByteBuffer buf) throws IOException {
        if (!this.protocolSignatureDetected) {
            if (buf.remaining() < PROTOCOL_SIGNATURE.length) {
                return;
            }
            for (byte signatureByte : PROTOCOL_SIGNATURE) {
                byte bufferByte = buf.get();
                ++this.bytesRead;
                if (bufferByte == signatureByte) continue;
                if (this.proxyProtocolFallbackEnabled) {
                    buf.position(0);
                    this.proxyHeaderProcessed = true;
                    this.bytesRead = 0;
                    return;
                }
                throw new IOException(INVALID_PROTOCOL_V2_HEADER);
            }
            this.protocolSignatureDetected = true;
        }
        this.parseConnectionInformation(buf);
        while (buf.hasRemaining() && this.bytesRead - 16 < this.addressLength) {
            int currentAddressIndex = this.bytesRead - 16;
            this.addressBytes[currentAddressIndex] = buf.get();
            ++this.bytesRead;
        }
        if (this.bytesRead == 16 + this.addressLength && !this.proxyHeaderProcessed) {
            this.decodeProxyProtocol();
        }
    }

    private void parseConnectionInformation(ByteBuffer buf) throws IOException {
        while (buf.hasRemaining() && this.bytesRead < 16) {
            byte currentByte = buf.get();
            ++this.bytesRead;
            switch (this.bytesRead) {
                case 13: {
                    int commandNibble = currentByte & 0xF;
                    int version = currentByte >> 4;
                    if (version != 2) {
                        this.log.warn("The version (the highest four bits from byte 13 of the header) was {} which is not valid for v2 of the PROXY protocol", (Object)version);
                        throw new IOException(INVALID_PROTOCOL_V2_HEADER);
                    }
                    try {
                        this.command = ProxyProtocolCommand.forValue(commandNibble);
                        break;
                    }
                    catch (IllegalArgumentException e) {
                        this.log.warn("The command (the lowest four bits from byte 13 of the header) was {} which is not valid per the PROXY protocol spec", (Object)commandNibble, (Object)e);
                        throw new IOException(INVALID_PROTOCOL_V2_HEADER);
                    }
                }
                case 14: {
                    this.protocolByte = currentByte;
                    break;
                }
                case 15: {
                    byte byte16 = buf.get();
                    ++this.bytesRead;
                    this.addressLength = (currentByte & 0xFF) << 8 | byte16 & 0xFF;
                    this.addressBytes = new byte[this.addressLength];
                }
            }
        }
    }

    private void decodeProxyProtocol() throws IOException {
        int tlvIndex = 0;
        switch (this.protocolByte) {
            case 17: {
                this.sourceAddress = Inet4Address.getByAddress(Arrays.copyOfRange(this.addressBytes, 0, 4));
                this.sourcePort = (this.addressBytes[8] & 0xFF) << 8 | this.addressBytes[9] & 0xFF;
                this.log.debug("decodeProxyProtocol - PROXY protocol source address/port: {}:{}", (Object)this.sourceAddress, (Object)this.sourcePort);
                tlvIndex = 12;
                break;
            }
            case 33: {
                this.sourceAddress = Inet6Address.getByAddress(Arrays.copyOfRange(this.addressBytes, 0, 16));
                this.sourcePort = (this.addressBytes[32] & 0xFF) << 8 | this.addressBytes[33] & 0xFF;
                this.log.debug("decodeProxyProtocol - PROXY protocol source address/port: {}:{}", (Object)this.sourceAddress, (Object)this.sourcePort);
                tlvIndex = 36;
                break;
            }
            default: {
                if (this.command == ProxyProtocolCommand.LOCAL) break;
                throw new IOException(INVALID_PROTOCOL_V2_HEADER);
            }
        }
        while (tlvIndex + 3 < this.addressBytes.length) {
            ProxyTlv tlv;
            int tlvLength;
            int tlvType = this.addressBytes[tlvIndex] & 0xFF;
            ++tlvIndex;
            if ((tlvIndex += 2) + (tlvLength = (this.addressBytes[tlvIndex] & 0xFF) << 8 | this.addressBytes[tlvIndex + 1] & 0xFF) > this.addressBytes.length) continue;
            byte[] currTlvBytes = Arrays.copyOfRange(this.addressBytes, tlvIndex, tlvIndex + tlvLength);
            tlvIndex += tlvLength;
            if (tlvType == 239) {
                tlv = this.createReservedTlv(currTlvBytes);
            } else {
                ProxyTlvType ptt = new ProxyTlvType(tlvType);
                tlv = new ProxyTlv(ptt, currTlvBytes);
            }
            this.tlvs.put(tlv.type, tlv);
            this.log.debug("Parsed new TLV {}", (Object)tlv);
        }
        this.proxyHeaderProcessed = true;
    }

    private ProxyTlv createReservedTlv(byte[] tlvBytes) throws IOException {
        Optional<Integer> tlvSubtype = Optional.of(Integer.valueOf(tlvBytes[0]));
        Set<ProxyTlvType> registeredTypes = ProxyTlvType.registeredTypes();
        ProxyTlvType ptt = null;
        for (ProxyTlvType registered : registeredTypes) {
            if (registered.type != 239 || !registered.subtype.equals(tlvSubtype)) continue;
            ptt = registered;
            break;
        }
        if (ptt == null) {
            String options = registeredTypes.stream().map(x -> String.format("%s/%s", x.type, x.subtype)).collect(Collectors.joining(", "));
            String message = String.format("No Confluent TLV types found with type %s, subtype %s; known TLV types are: %s", 239, tlvSubtype.get(), options);
            throw new IOException(message);
        }
        ProxyTlvValidator validator = ptt.validator().orElse(null);
        if (validator != null) {
            try {
                validator.validate(tlvBytes);
            }
            catch (Throwable t) {
                this.log.debug("Error validating TLV: {}", (Object)t.getMessage());
            }
        }
        return new ProxyTlv(ptt, tlvBytes);
    }

    @Override
    protected IOException invalidProtocolHeaderException(String s) {
        return new IOException("Invalid Proxy Protocol V2 Header. " + s);
    }

    @Override
    public byte[] emitHeaders(InetAddress destinationAddress, int destinationPort) throws IOException {
        this.validateAddresses(destinationAddress);
        this.validatePorts(destinationPort);
        this.validateProxyMode();
        byte protocolVersionAndCommand = (byte)(this.command == ProxyProtocolCommand.PROXY ? 33 : 32);
        int transportProtocolAndFamily = this.command == ProxyProtocolCommand.LOCAL ? 0 : (destinationAddress instanceof Inet4Address ? 17 : 33);
        byte[] tlvBytes = ProxyProtocolV2Engine.tlvBytes(this.tlvs.values());
        int addressByteLength = this.command == ProxyProtocolCommand.LOCAL ? tlvBytes.length : (destinationAddress instanceof Inet4Address ? 12 + tlvBytes.length : 36 + tlvBytes.length);
        ByteArrayOutputStream header = new ByteArrayOutputStream();
        header.write(PROTOCOL_SIGNATURE);
        header.write(protocolVersionAndCommand);
        header.write(transportProtocolAndFamily);
        header.write(this.shortBytes(addressByteLength));
        if (this.command != ProxyProtocolCommand.LOCAL) {
            header.write(this.sourceAddress.getAddress());
            header.write(destinationAddress.getAddress());
            header.write(this.shortBytes(this.sourcePort));
            header.write(this.shortBytes(destinationPort));
        }
        if (tlvBytes.length > 0) {
            header.write(tlvBytes);
        }
        this.proxyHeaderProcessed = true;
        return header.toByteArray();
    }

    @Override
    public ProxyTlv tlv(ProxyTlvType type) {
        return this.tlvs.get(type);
    }

    @Override
    public ProxyProtocolCommand command() {
        return this.command;
    }

    private void validateProxyMode() throws IOException {
        if (this.command == null) {
            throw this.invalidProtocolHeaderException("The PROXY header could not be emitted because the command was not configured");
        }
    }

    static void validateLkcId(String lkcId) {
        if (!lkcId.startsWith("lkc-")) {
            throw ProxyProtocolV2Engine.newConfigException("confluent.lkc.id", lkcId);
        }
    }

    private byte[] shortBytes(int value) {
        byte[] buf = new byte[]{(byte)(value >> 8 & 0xFF), (byte)(value & 0xFF)};
        return buf;
    }

    static byte[] tlvBytes(ProxyTlv tlv) {
        byte[] value = tlv.rawValue();
        int length = value.length;
        byte[] buf = new byte[3 + length];
        buf[0] = (byte)(tlv.type().type() & 0xFF);
        buf[1] = (byte)(length >> 8 & 0xFF);
        buf[2] = (byte)(length & 0xFF);
        System.arraycopy(value, 0, buf, 3, length);
        return buf;
    }

    static byte[] tlvBytes(Collection<ProxyTlv> tlvs) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        for (ProxyTlv tlv : tlvs) {
            baos.write(ProxyProtocolV2Engine.tlvBytes(tlv));
        }
        return baos.toByteArray();
    }
}

