/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.AbstractMap;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.NavigableMap;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.kafka.common.message.BrokerHeartbeatRequestData;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.controller.BrokerControlState;
import org.apache.kafka.controller.BrokerControlStates;
import org.apache.kafka.metadata.BrokerHeartbeatReply;
import org.apache.kafka.metadata.placement.UsableBroker;
import org.slf4j.Logger;

public class BrokerHeartbeatManager {
    private final Logger log;
    private final Time time;
    private final long sessionTimeoutNs;
    private final HashMap<Integer, BrokerHeartbeatState> brokers;
    private final BrokerHeartbeatStateList unfenced;
    private final TreeSet<BrokerHeartbeatState> active;
    private final Map<Integer, Long> shutdowns;

    BrokerHeartbeatManager(LogContext logContext, Time time, long sessionTimeoutNs) {
        this.log = logContext.logger(BrokerHeartbeatManager.class);
        this.time = time;
        this.sessionTimeoutNs = sessionTimeoutNs;
        this.brokers = new HashMap();
        this.unfenced = new BrokerHeartbeatStateList();
        this.active = new TreeSet<BrokerHeartbeatState>(MetadataOffsetComparator.INSTANCE);
        this.shutdowns = new HashMap<Integer, Long>();
    }

    Time time() {
        return this.time;
    }

    BrokerHeartbeatStateList unfenced() {
        return this.unfenced;
    }

    TreeSet<BrokerHeartbeatState> active() {
        return this.active;
    }

    Collection<BrokerHeartbeatState> brokers() {
        return this.brokers.values();
    }

    NavigableMap<Integer, Long> shutdowns() {
        return new TreeMap<Integer, Long>(this.shutdowns);
    }

    BrokerHeartbeatState brokerHeartbeatState(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker == null) {
            throw new NoSuchElementException("Unable to locate broker " + brokerId);
        }
        return broker;
    }

    void fence(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker != null) {
            this.removeFromActiveAndUnfenced(broker);
            this.shutdowns.remove(brokerId);
        }
    }

    void remove(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.remove(brokerId);
        if (broker != null) {
            this.removeFromActiveAndUnfenced(broker);
            this.shutdowns.remove(brokerId);
        }
    }

    private void removeFromActiveAndUnfenced(BrokerHeartbeatState broker) {
        if (!broker.fenced()) {
            this.unfenced.remove(broker);
        }
        this.active.remove(broker);
    }

    boolean hasValidSession(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker == null) {
            return false;
        }
        return this.hasValidSession(broker);
    }

    private boolean hasValidSession(BrokerHeartbeatState broker) {
        if (broker.fenced()) {
            return false;
        }
        return broker.lastContactNs + this.sessionTimeoutNs >= this.time.nanoseconds();
    }

    void register(int brokerId, boolean fenced) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        long metadataOffset = -1L;
        if (broker == null) {
            broker = new BrokerHeartbeatState(brokerId);
            this.brokers.put(brokerId, broker);
        } else if (broker.fenced() != fenced) {
            metadataOffset = broker.metadataOffset;
        }
        this.touch(brokerId, fenced, metadataOffset);
        this.shutdowns.remove(brokerId);
    }

    void touch(int brokerId, boolean fenced, long metadataOffset) {
        BrokerHeartbeatState broker = this.heartbeatStateOrThrow(brokerId);
        this.removeFromActiveAndUnfenced(broker);
        broker.lastContactNs = this.time.nanoseconds();
        broker.metadataOffset = metadataOffset;
        if (fenced) {
            this.shutdowns.remove(brokerId);
        } else {
            this.unfenced.add(broker);
            if (!this.shutdowns.containsKey(brokerId)) {
                this.active.add(broker);
            }
        }
    }

    Map.Entry<Long, Integer> lowestActiveOffsetAndBroker() {
        Iterator<BrokerHeartbeatState> iterator = this.active.iterator();
        if (!iterator.hasNext()) {
            return new AbstractMap.SimpleImmutableEntry<Long, Integer>(Long.MAX_VALUE, -1);
        }
        BrokerHeartbeatState first = iterator.next();
        return new AbstractMap.SimpleImmutableEntry<Long, Integer>(first.metadataOffset, first.id);
    }

    void updateControlledShutdownOffset(int brokerId, long metadataOffset) {
        BrokerHeartbeatState broker = this.heartbeatStateOrThrow(brokerId);
        if (broker.fenced()) {
            throw new RuntimeException("Cannot update the controlled shutdown state of broker " + brokerId + ", because it is fenced.");
        }
        this.active.remove(broker);
        if (!this.shutdowns.containsKey(brokerId)) {
            this.log.info("Broker {} is entering controlled shutdown.", (Object)brokerId);
        }
        this.shutdowns.put(brokerId, metadataOffset);
        this.log.debug("Updated the controlled shutdown offset for broker {} to {}.", (Object)brokerId, (Object)metadataOffset);
    }

    long nextCheckTimeNs() {
        BrokerHeartbeatState broker = this.unfenced.first();
        if (broker == null) {
            return Long.MAX_VALUE;
        }
        return broker.lastContactNs + this.sessionTimeoutNs;
    }

    Optional<Integer> findOneStaleBroker() {
        BrokerHeartbeatState broker;
        BrokerHeartbeatStateIterator iterator = this.unfenced.iterator();
        if (iterator.hasNext() && !this.hasValidSession(broker = iterator.next())) {
            return Optional.of(broker.id);
        }
        return Optional.empty();
    }

    void handleHeartbeatReply(int brokerId, BrokerHeartbeatReply reply, long metadataOffset) {
        if (reply.isFenced() || reply.shouldShutDown()) {
            BrokerHeartbeatState broker = this.brokers.get(brokerId);
            if (broker != null && !broker.fenced()) {
                this.log.info("Marking broker {} as fenced in heartbeat manager.", (Object)brokerId);
                this.fence(brokerId);
            } else {
                this.log.trace("Nothing to do when handling heartbeat reply for fenced broker {}.", (Object)brokerId);
            }
        } else if (reply.inControlledShutdown()) {
            BrokerHeartbeatState broker = this.brokers.get(brokerId);
            if (broker == null) {
                this.log.info("Clearing controlled shutdown state for untracked broker {}.", (Object)brokerId);
                this.shutdowns.remove(brokerId);
            } else if (!this.shutdowns.containsKey(brokerId)) {
                this.log.info("Marking broker {} as shutting down in heartbeat manager at offset {}", (Object)brokerId, (Object)metadataOffset);
                this.shutdowns.put(brokerId, metadataOffset);
            } else {
                this.log.trace("Nothing to do when handling controlled shutdown heartbeat reply for broker {}.", (Object)brokerId);
            }
        } else {
            this.log.trace("Nothing to do when handling heartbeat reply for broker {}.", (Object)brokerId);
        }
    }

    Iterator<UsableBroker> usableBrokers(Function<Integer, Optional<String>> idToRack, Set<Integer> excludedBrokerIds, Function<Integer, Integer> idToCell) {
        return new UsableBrokerIterator(this.brokers.values().iterator(), idToRack, excludedBrokerIds, idToCell, this.shutdowns);
    }

    BrokerControlState currentBrokerState(BrokerHeartbeatState broker) {
        if (this.shutdowns.containsKey(broker.id)) {
            return BrokerControlState.CONTROLLED_SHUTDOWN;
        }
        if (broker.fenced()) {
            return BrokerControlState.FENCED;
        }
        return BrokerControlState.UNFENCED;
    }

    BrokerControlStates calculateNextBrokerState(int brokerId, BrokerHeartbeatRequestData request, long registerBrokerRecordOffset, Supplier<Boolean> hasLeaderships) {
        BrokerHeartbeatState broker = this.heartbeatStateOrThrow(brokerId);
        BrokerControlState currentState = this.currentBrokerState(broker);
        switch (currentState) {
            case FENCED: {
                if (request.wantShutDown()) {
                    this.log.info("Fenced broker {} has requested and been granted an immediate shutdown.", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                }
                if (!request.wantFence()) {
                    if (request.currentMetadataOffset() >= registerBrokerRecordOffset) {
                        this.log.info("The request from broker {} to unfence has been granted because it has caught up with the offset of its register broker record {}.", (Object)brokerId, (Object)registerBrokerRecordOffset);
                        return new BrokerControlStates(currentState, BrokerControlState.UNFENCED);
                    }
                    if (this.log.isDebugEnabled()) {
                        this.log.debug("The request from broker {} to unfence cannot yet be granted because it has not caught up with the offset of its register broker record {}. It is still at offset {}.", new Object[]{brokerId, registerBrokerRecordOffset, request.currentMetadataOffset()});
                    }
                    return new BrokerControlStates(currentState, BrokerControlState.FENCED);
                }
                return new BrokerControlStates(currentState, BrokerControlState.FENCED);
            }
            case UNFENCED: {
                if (request.wantFence()) {
                    if (request.wantShutDown()) {
                        this.log.info("Unfenced broker {} has requested and been granted an immediate shutdown.", (Object)brokerId);
                        return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                    }
                    this.log.info("Unfenced broker {} has requested and been granted fencing", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.FENCED);
                }
                if (request.wantShutDown()) {
                    if (hasLeaderships.get().booleanValue()) {
                        this.log.info("Unfenced broker {} has requested and been granted a controlled shutdown.", (Object)brokerId);
                        return new BrokerControlStates(currentState, BrokerControlState.CONTROLLED_SHUTDOWN);
                    }
                    this.log.info("Unfenced broker {} has requested and been granted an immediate shutdown.", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                }
                return new BrokerControlStates(currentState, BrokerControlState.UNFENCED);
            }
            case CONTROLLED_SHUTDOWN: {
                if (hasLeaderships.get().booleanValue()) {
                    this.log.info("Broker {} is in controlled shutdown state, but can not shut down because more leaders still need to be moved.", (Object)brokerId);
                    return new BrokerControlStates(currentState, BrokerControlState.CONTROLLED_SHUTDOWN);
                }
                Map.Entry<Long, Integer> lowestActiveOffsetAndBroker = this.lowestActiveOffsetAndBroker();
                long controlledShutdownOffset = this.shutdowns.getOrDefault(brokerId, 0L);
                if (controlledShutdownOffset <= lowestActiveOffsetAndBroker.getKey()) {
                    this.log.info("The request from broker {} to shut down has been granted since the lowest active offset {} is now greater than the broker's controlled shutdown offset {}.", new Object[]{brokerId, lowestActiveOffsetAndBroker.getKey(), controlledShutdownOffset});
                    return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
                }
                this.log.info("The request from broker {} to shut down can not yet be granted because the lowest active offset {} from broker {} is not greater than the broker's shutdown offset {}.", new Object[]{brokerId, lowestActiveOffsetAndBroker.getKey(), lowestActiveOffsetAndBroker.getValue(), controlledShutdownOffset});
                return new BrokerControlStates(currentState, BrokerControlState.CONTROLLED_SHUTDOWN);
            }
        }
        return new BrokerControlStates(currentState, BrokerControlState.SHUTDOWN_NOW);
    }

    private BrokerHeartbeatState heartbeatStateOrThrow(int brokerId) {
        BrokerHeartbeatState broker = this.brokers.get(brokerId);
        if (broker == null) {
            throw new IllegalStateException("Broker " + brokerId + " is not registered.");
        }
        return broker;
    }

    static class UsableBrokerIterator
    implements Iterator<UsableBroker> {
        private final Iterator<BrokerHeartbeatState> iterator;
        private final Function<Integer, Optional<String>> idToRack;
        private final Set<Integer> excludedBrokerIds;
        private final Function<Integer, Integer> idToCell;
        private final Map<Integer, Long> shutdowns;
        private UsableBroker next;

        UsableBrokerIterator(Iterator<BrokerHeartbeatState> iterator, Function<Integer, Optional<String>> idToRack, Set<Integer> excludedBrokerIds, Function<Integer, Integer> idToCell, Map<Integer, Long> shutdowns) {
            this.iterator = iterator;
            this.idToRack = idToRack;
            this.excludedBrokerIds = excludedBrokerIds;
            this.idToCell = idToCell;
            this.shutdowns = shutdowns;
            this.next = null;
        }

        @Override
        public boolean hasNext() {
            BrokerHeartbeatState result;
            if (this.next != null) {
                return true;
            }
            do {
                if (this.iterator.hasNext()) continue;
                return false;
            } while (this.excludedBrokerIds.contains((result = this.iterator.next()).id()));
            Optional<String> rack = this.idToRack.apply(result.id());
            int cellId = this.idToCell.apply(result.id());
            boolean treatAsFenced = result.fenced() || this.shutdowns.containsKey(result.id);
            this.next = new UsableBroker(result.id(), rack, treatAsFenced, cellId);
            return true;
        }

        @Override
        public UsableBroker next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            UsableBroker result = this.next;
            this.next = null;
            return result;
        }
    }

    static class BrokerHeartbeatStateIterator
    implements Iterator<BrokerHeartbeatState> {
        private final BrokerHeartbeatState head;
        private BrokerHeartbeatState cur;

        BrokerHeartbeatStateIterator(BrokerHeartbeatState head) {
            this.head = head;
            this.cur = head;
        }

        @Override
        public boolean hasNext() {
            return this.cur.next != this.head;
        }

        @Override
        public BrokerHeartbeatState next() {
            if (!this.hasNext()) {
                throw new NoSuchElementException();
            }
            BrokerHeartbeatState result = this.cur.next;
            this.cur = this.cur.next;
            return result;
        }
    }

    static class BrokerHeartbeatStateList {
        private final BrokerHeartbeatState head = new BrokerHeartbeatState(-1);

        BrokerHeartbeatStateList() {
            this.head.prev = this.head;
            this.head.next = this.head;
        }

        BrokerHeartbeatState first() {
            BrokerHeartbeatState result = this.head.next;
            return result == this.head ? null : result;
        }

        void add(BrokerHeartbeatState broker) {
            BrokerHeartbeatState cur = this.head.prev;
            while (true) {
                if (cur == this.head || cur.lastContactNs <= broker.lastContactNs) break;
                cur = cur.prev;
            }
            broker.next = cur.next;
            cur.next.prev = broker;
            broker.prev = cur;
            cur.next = broker;
        }

        void remove(BrokerHeartbeatState broker) {
            if (broker.next == null) {
                throw new RuntimeException(broker + " is not in the  list.");
            }
            broker.prev.next = broker.next;
            broker.next.prev = broker.prev;
            broker.prev = null;
            broker.next = null;
        }

        BrokerHeartbeatStateIterator iterator() {
            return new BrokerHeartbeatStateIterator(this.head);
        }
    }

    static class MetadataOffsetComparator
    implements Comparator<BrokerHeartbeatState> {
        static final MetadataOffsetComparator INSTANCE = new MetadataOffsetComparator();

        MetadataOffsetComparator() {
        }

        @Override
        public int compare(BrokerHeartbeatState a, BrokerHeartbeatState b) {
            if (a.metadataOffset < b.metadataOffset) {
                return -1;
            }
            if (a.metadataOffset > b.metadataOffset) {
                return 1;
            }
            if (a.id < b.id) {
                return -1;
            }
            if (a.id > b.id) {
                return 1;
            }
            return 0;
        }
    }

    static class BrokerHeartbeatState {
        private final int id;
        long lastContactNs;
        long metadataOffset;
        private BrokerHeartbeatState prev;
        private BrokerHeartbeatState next;

        BrokerHeartbeatState(int id) {
            this.id = id;
            this.lastContactNs = 0L;
            this.prev = null;
            this.next = null;
            this.metadataOffset = -1L;
        }

        int id() {
            return this.id;
        }

        boolean fenced() {
            return this.prev == null;
        }
    }
}

