/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller.metrics;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import org.apache.kafka.common.config.ConfigResource;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.image.ConfigurationImage;
import org.apache.kafka.image.MetadataDelta;
import org.apache.kafka.image.MetadataImage;
import org.apache.kafka.image.TopicImage;
import org.apache.kafka.image.loader.LoaderManifest;
import org.apache.kafka.image.publisher.MetadataPublisher;
import org.apache.kafka.server.fault.FaultHandler;
import org.apache.kafka.server.immutable.ImmutableSet;
import org.slf4j.Logger;

public class TenantMetricsPublisher
implements MetadataPublisher {
    private final FaultHandler faultHandler;
    protected Map<String, PartitionCountMetricByCleanupPolicy> tenantPartitionCountTracker;
    private Function<String, Boolean> registerTenantGaugeInterface;
    private Function<String, Boolean> removeTenantGaugeInterface;
    private Function<String, Boolean> isTenantOwnedInterface;
    private final Set<String> registeredTenants;
    private Long lastUpdateTime;
    protected static final Long THRESHOLD_IN_MS = 300000L;
    private static final char TENANT_DELIMITER = '_';
    private LogContext logContext = new LogContext();
    private final Logger log;

    public TenantMetricsPublisher(FaultHandler faultHandler) {
        this.faultHandler = faultHandler;
        this.tenantPartitionCountTracker = new ConcurrentHashMap<String, PartitionCountMetricByCleanupPolicy>();
        this.lastUpdateTime = System.currentTimeMillis();
        this.log = this.logContext.logger(TenantMetricsPublisher.class);
        this.registeredTenants = new HashSet<String>();
    }

    @Override
    public String name() {
        return "TenantMetadataPublisher";
    }

    @Override
    public void onMetadataUpdate(MetadataDelta delta, MetadataImage newImage, LoaderManifest manifest) {
        switch (manifest.type()) {
            case LOG_DELTA: 
            case SNAPSHOT: {
                try {
                    this.maybeComputeAndPublishTenantPartitionCountMetric(newImage);
                    break;
                }
                catch (Throwable e) {
                    this.faultHandler.handleFault("Failed to compute Tenant partition count metric " + manifest.provenance(), e);
                }
            }
        }
    }

    public void setTenantRegisterInterface(Function<String, Boolean> registerTenantGaugeInterface) {
        this.registerTenantGaugeInterface = registerTenantGaugeInterface;
    }

    public void setIsTenantOwnedInterface(Function<String, Boolean> isTenantOwnedInterface) {
        this.isTenantOwnedInterface = isTenantOwnedInterface;
    }

    public void setRemoveTenantGaugeInterface(Function<String, Boolean> removeTenantGaugeInterface) {
        this.removeTenantGaugeInterface = removeTenantGaugeInterface;
    }

    protected void setLastUpdateTime(Long time) {
        this.lastUpdateTime = time;
    }

    protected void maybeComputeAndPublishTenantPartitionCountMetric(MetadataImage newImage) {
        Set tenantSet = newImage.topics().topicNamesByTenant().keySet();
        boolean shouldUpdate = this.shouldUpdate();
        if (shouldUpdate) {
            this.lastUpdateTime = System.currentTimeMillis();
            this.log.info("Clearing the tenantPartitionCountTracker");
            this.tenantPartitionCountTracker.clear();
        }
        for (String tenant : tenantSet) {
            Boolean isTenantPresent = this.tenantPartitionCountTracker.containsKey(tenant);
            if (isTenantPresent.booleanValue() && !shouldUpdate) continue;
            this.updatePartitionCountMetricForTenant(tenant, newImage);
        }
        if (shouldUpdate) {
            for (String tenant : tenantSet) {
                this.maybeRegisterTenantGauge(tenant);
            }
        }
    }

    protected void maybeRegisterTenantGauge(String tenant) {
        try {
            Optional<String> originalTenantName = TenantMetricsPublisher.getOriginalTenantName(tenant);
            if (originalTenantName.isPresent() && this.registerTenantGaugeInterface != null && this.isTenantOwnedInterface != null && this.isTenantOwnedInterface.apply(originalTenantName.get()).booleanValue() && !this.registeredTenants.contains(originalTenantName.get())) {
                this.registerTenantGaugeInterface.apply(originalTenantName.get());
                this.registeredTenants.add(originalTenantName.get());
            }
        }
        catch (Exception e) {
            this.log.error("Error registering gauge in tenant metrics publisher for tenant {}", (Object)tenant);
        }
    }

    protected void maybeRemoveTenantGauges() {
        Iterator<String> iterator = this.registeredTenants.iterator();
        while (iterator.hasNext()) {
            String tenant = iterator.next();
            try {
                Optional<String> originalTenantName = TenantMetricsPublisher.getOriginalTenantName(tenant);
                if (!originalTenantName.isPresent() || this.removeTenantGaugeInterface == null || this.isTenantOwnedInterface == null || this.isTenantOwnedInterface.apply(originalTenantName.get()).booleanValue() || !this.registeredTenants.contains(originalTenantName.get())) continue;
                this.removeTenantGaugeInterface.apply(originalTenantName.get());
                iterator.remove();
            }
            catch (Exception e) {
                this.log.error("Error removing gauge in tenant metrics publisher for tenant {}", (Object)tenant);
            }
        }
    }

    protected Set<String> registeredTenants() {
        return Collections.unmodifiableSet(this.registeredTenants);
    }

    protected void updatePartitionCountMetricForTenant(String tenant, MetadataImage newImage) {
        ImmutableSet<String> allTopicsForTenant = newImage.topics().topicsByTenant(tenant);
        Map<ConfigResource, ConfigurationImage> configs = newImage.configs().resourceData();
        HashSet<String> compactedTopics = new HashSet<String>();
        for (Map.Entry<ConfigResource, ConfigurationImage> config : configs.entrySet()) {
            String topicNameInConfig;
            if (!config.getKey().type().equals((Object)ConfigResource.Type.TOPIC) || !allTopicsForTenant.contains(topicNameInConfig = config.getKey().name()) || !config.getValue().data().containsKey("cleanup.policy") || !config.getValue().data().get("cleanup.policy").equals("compact")) continue;
            compactedTopics.add(topicNameInConfig);
        }
        Long compactedPartitionCount = 0L;
        Long nonCompactedPartitionCont = 0L;
        for (String topic : allTopicsForTenant) {
            Integer partitionCount = ((TopicImage)newImage.topics().topicsByName().get((Object)topic)).partitions().size();
            if (compactedTopics.contains(topic)) {
                compactedPartitionCount = compactedPartitionCount + (long)partitionCount.intValue();
                continue;
            }
            nonCompactedPartitionCont = nonCompactedPartitionCont + (long)partitionCount.intValue();
        }
        this.updatePartitionCountMetric(tenant, compactedPartitionCount, nonCompactedPartitionCont);
    }

    protected void updatePartitionCountMetric(String tenant, Long compactedPartitionCount, Long nonCompactedPartitionCont) {
        if (tenant != null) {
            this.tenantPartitionCountTracker.put(tenant, new PartitionCountMetricByCleanupPolicy(compactedPartitionCount, nonCompactedPartitionCont));
        }
    }

    protected String getModifiedTenantName(String tenant) {
        return tenant + '_';
    }

    protected static Optional<String> getOriginalTenantName(String tenant) {
        if (!tenant.isEmpty() && tenant.charAt(tenant.length() - 1) == '_') {
            return Optional.of(tenant.substring(0, tenant.length() - 1));
        }
        return Optional.of(tenant);
    }

    protected boolean shouldUpdate() {
        return this.lastUpdateTime + THRESHOLD_IN_MS < System.currentTimeMillis();
    }

    public PartitionCountMetricByCleanupPolicy getPartitionCountForTenant(String tenant) {
        if (!this.tenantPartitionCountTracker.containsKey(this.getModifiedTenantName(tenant))) {
            throw new RuntimeException("PartitionCount by cleanup policy is not available for given tenant");
        }
        return this.tenantPartitionCountTracker.get(this.getModifiedTenantName(tenant));
    }

    @Override
    public void close() {
        this.tenantPartitionCountTracker.clear();
    }

    public static final class PartitionCountMetricByCleanupPolicy {
        private final Long partitionCountCompacted;
        private final Long partitionCountNonCompacted;

        public PartitionCountMetricByCleanupPolicy() {
            this.partitionCountCompacted = 0L;
            this.partitionCountNonCompacted = 0L;
        }

        PartitionCountMetricByCleanupPolicy(Long partitionCountCompacted, Long partitionCountNonCompacted) {
            this.partitionCountCompacted = partitionCountCompacted;
            this.partitionCountNonCompacted = partitionCountNonCompacted;
        }

        public Long getPartitionCountCompacted() {
            return this.partitionCountCompacted;
        }

        public Long getPartitionCountNonCompacted() {
            return this.partitionCountNonCompacted;
        }

        public String toString() {
            return this.partitionCountCompacted + ":" + this.partitionCountNonCompacted;
        }
    }
}

