/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.audit.telemetry.exporter;

import io.confluent.security.audit.telemetry.exporter.TopicSpec;
import java.io.Closeable;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.CreateTopicsOptions;
import org.apache.kafka.clients.admin.CreateTopicsResult;
import org.apache.kafka.clients.admin.DescribeTopicsOptions;
import org.apache.kafka.clients.admin.DescribeTopicsResult;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.admin.TopicDescription;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.errors.TopicExistsException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TopicManager
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(TopicManager.class);
    private final ConcurrentHashMap<String, Boolean> topicExists;
    private final ConcurrentHashMap<String, TopicSpec> topicMap;
    private final ThreadPoolExecutor reconcileJobExecutor;
    private final AdminClient adminClient;
    private final Map<String, String> defaultTopicConfig;
    private final Integer defaultTopicPartitions;
    private final Integer defaultTopicReplicas;
    private final Integer timeOutMs;
    private Future<Boolean> reconcileFuture;

    public TopicManager(Properties clientProps, Map<String, String> defaultTopicConfig, Integer defaultTopicPartitions, Integer defaultTopicReplicas, Integer timeOutMs, Map<String, TopicSpec> topics) {
        this.adminClient = AdminClient.create((Properties)clientProps);
        this.defaultTopicConfig = defaultTopicConfig;
        this.defaultTopicPartitions = defaultTopicPartitions;
        this.defaultTopicReplicas = defaultTopicReplicas;
        this.timeOutMs = timeOutMs;
        this.reconcileJobExecutor = new ThreadPoolExecutor(1, 1, 30L, TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(1), new ThreadPoolExecutor.AbortPolicy());
        this.reconcileJobExecutor.setThreadFactory(runnable -> {
            Thread thread = new Thread(runnable, "confluent-event-topic-manager-task-scheduler");
            thread.setDaemon(true);
            thread.setUncaughtExceptionHandler((t, e) -> log.error("Uncaught exception in thread '{}':", (Object)t.getName(), (Object)e));
            return thread;
        });
        this.reconcileJobExecutor.allowCoreThreadTimeOut(true);
        this.topicExists = new ConcurrentHashMap();
        this.topicMap = new ConcurrentHashMap();
        topics.entrySet().stream().forEach(t -> {
            this.topicMap.put((String)t.getKey(), (TopicSpec)t.getValue());
            this.topicExists.put((String)t.getKey(), false);
        });
        this.checkTopics();
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Future<Boolean> ensureTopics() {
        try {
            this.reconcileFuture = this.reconcileJobExecutor.submit(this::reconcile);
        }
        catch (RejectedExecutionException rejectedExecutionException) {
            // empty catch block
        }
        return this.reconcileFuture;
    }

    public final Future<Boolean> checkTopics() {
        try {
            this.reconcileFuture = this.reconcileJobExecutor.submit(this::check);
        }
        catch (RejectedExecutionException rejectedExecutionException) {
            // empty catch block
        }
        return this.reconcileFuture;
    }

    private void mergeDefaults(TopicSpec spec) {
        if (spec.partitions() <= 0) {
            spec.setPartitions(this.defaultTopicPartitions);
        }
        if (spec.replicationFactor() <= 0) {
            spec.setReplicationFactor(this.defaultTopicReplicas);
        }
        if (this.defaultTopicConfig != null) {
            this.defaultTopicConfig.entrySet().stream().forEach(e -> spec.config().putIfAbsent((String)e.getKey(), (String)e.getValue()));
        }
    }

    private void updateTopicExistsWithDescribedTopics() {
        DescribeTopicsResult describeResult = this.adminClient.describeTopics((Collection)this.topicMap.keySet(), new DescribeTopicsOptions().timeoutMs(this.timeOutMs));
        for (Map.Entry entry : describeResult.topicNameValues().entrySet()) {
            try {
                TopicDescription description = (TopicDescription)((KafkaFuture)entry.getValue()).get((long)this.timeOutMs.intValue(), TimeUnit.MILLISECONDS);
                this.topicExists.put(description.name(), true);
                log.debug("Event log topic {} ready with {} partitions", entry.getKey(), (Object)description.partitions().size());
            }
            catch (ExecutionException | TimeoutException e) {
                if (e.getCause() instanceof UnknownTopicOrPartitionException) {
                    this.topicExists.put((String)entry.getKey(), false);
                    continue;
                }
                log.error("error while describing topics", (Throwable)e);
            }
            catch (InterruptedException e) {
                log.warn("interrupted while describing event log topic {}", entry.getKey(), (Object)e);
            }
        }
    }

    public final boolean check() {
        this.updateTopicExistsWithDescribedTopics();
        return this.topicExists.values().stream().allMatch(v -> v);
    }

    public boolean reconcile() {
        try {
            this.updateTopicExistsWithDescribedTopics();
            Set missing = this.topicExists.entrySet().stream().filter(e -> (Boolean)e.getValue() == false).map(Map.Entry::getKey).collect(Collectors.toSet());
            List newTopics = missing.stream().map(topicName -> {
                TopicSpec ts = this.topicMap.get(topicName);
                this.mergeDefaults(ts);
                return new NewTopic(topicName, ts.partitions(), (short)ts.replicationFactor()).configs(ts.config());
            }).collect(Collectors.toList());
            CreateTopicsResult createResult = this.adminClient.createTopics(newTopics, new CreateTopicsOptions().timeoutMs(this.timeOutMs));
            for (Map.Entry entry : createResult.values().entrySet()) {
                try {
                    ((KafkaFuture)entry.getValue()).get((long)this.timeOutMs.intValue(), TimeUnit.MILLISECONDS);
                    this.topicExists.put((String)entry.getKey(), true);
                }
                catch (ExecutionException | TimeoutException e2) {
                    if (e2.getCause() instanceof TopicExistsException) {
                        this.topicExists.put((String)entry.getKey(), true);
                        continue;
                    }
                    log.error("error while creating topic " + (String)entry.getKey(), (Throwable)e2);
                }
            }
        }
        catch (InterruptedException e3) {
            log.warn("Event log topic initialization interrupted");
        }
        return this.topicExists.values().stream().allMatch(v -> v);
    }

    public void addTopic(TopicSpec spec) {
        this.topicMap.put(spec.name(), spec);
    }

    public boolean topicExists(String name) {
        return this.topicExists.getOrDefault(name, false);
    }

    public boolean topicManaged(String name) {
        return this.topicMap.containsKey(name);
    }

    public Set<String> managedTopics() {
        return new HashSet<String>(this.topicMap.keySet());
    }

    @Override
    public void close() {
        this.reconcileJobExecutor.shutdownNow();
        this.adminClient.close();
    }

    public static final class Builder {
        private Properties adminClientProperties;
        private Integer defaultTopicPartitions;
        private Integer defaultTopicReplicas;
        private Map<String, String> defaultTopicConfig = new HashMap<String, String>();
        private Integer timeOutMs;
        private Map<String, TopicSpec> topics;

        private Builder() {
        }

        public Builder setDefaultTopicPartitions(Integer defaultTopicPartitions) {
            this.defaultTopicPartitions = defaultTopicPartitions;
            return this;
        }

        public Builder setDefaultTopicReplicas(Integer defaultTopicReplicas) {
            this.defaultTopicReplicas = defaultTopicReplicas;
            return this;
        }

        public Builder setAdminClientProperties(Properties adminClientProperties) {
            this.adminClientProperties = adminClientProperties;
            return this;
        }

        public Builder setDefaultTopicConfig(Map<String, String> defaultTopicConfig) {
            this.defaultTopicConfig = defaultTopicConfig;
            return this;
        }

        public Builder setTimeOutMs(Integer timeOutMs) {
            this.timeOutMs = timeOutMs;
            return this;
        }

        public Builder setTopics(Map<String, TopicSpec> topics) {
            this.topics = topics;
            return this;
        }

        public TopicManager build() {
            Objects.requireNonNull(this.adminClientProperties, "Admin client properties is required.");
            Objects.requireNonNull(this.timeOutMs, "timeout is required");
            return new TopicManager(this.adminClientProperties, this.defaultTopicConfig, this.defaultTopicPartitions, this.defaultTopicReplicas, this.timeOutMs, this.topics);
        }
    }
}

