/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.mtls;

import io.confluent.security.mtls.CertMetadataIdentifier;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509CRL;
import java.security.cert.X509CRLEntry;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertificateMetadata {
    private static final Logger LOG = LoggerFactory.getLogger(CertificateMetadata.class);
    private static final String RFC2253 = "RFC2253";
    private final X509Certificate certificate;
    private final Map<String, String> celVars = new HashMap<String, String>();

    public CertificateMetadata(X509Certificate certificate) {
        this.certificate = certificate;
        this.generateCelVars();
    }

    public Map<String, String> getCelVars() {
        return this.celVars;
    }

    public String getIssuerDn() {
        return this.certificate.getIssuerX500Principal().getName(RFC2253);
    }

    public String getDn() {
        return this.celVars.get(CertMetadataIdentifier.DN.getValue());
    }

    public String getSan() {
        return this.celVars.get(CertMetadataIdentifier.SAN.getValue());
    }

    public String getCn() {
        return this.celVars.get(CertMetadataIdentifier.CN.getValue());
    }

    public String getSnid() {
        return this.celVars.get(CertMetadataIdentifier.SNID.getValue());
    }

    public String getSha1() {
        return this.celVars.get(CertMetadataIdentifier.SHA1.getValue());
    }

    private void generateCelVars() {
        this.celVars.putAll(CertificateMetadata.generateSubjectDnVars(this.certificate));
        this.celVars.put(CertMetadataIdentifier.SAN.getValue(), CertificateMetadata.generateSubjectAlternativeName(this.certificate));
        this.celVars.put(CertMetadataIdentifier.SNID.getValue(), CertificateMetadata.getSerialNumber(this.certificate));
        try {
            this.celVars.put(CertMetadataIdentifier.SHA1.getValue(), CertificateMetadata.calculateSha1Fingerprint(this.certificate));
        }
        catch (Exception e) {
            LOG.error("Failed to calculate SHA-1 fingerprint", (Throwable)e);
        }
    }

    public static String getIssuerDn(X509CRL crlEntry) {
        return crlEntry.getIssuerX500Principal().getName(RFC2253);
    }

    public static String getSerialNumber(X509CRLEntry crlEntry) {
        return CertificateMetadata.toHexStringUpperCase(crlEntry.getSerialNumber());
    }

    static Map<String, String> generateSubjectDnVars(X509Certificate certificate) {
        HashMap<String, String> vars = new HashMap<String, String>();
        String dn = certificate.getSubjectX500Principal().getName(RFC2253);
        vars.put(CertMetadataIdentifier.DN.getValue(), dn);
        try {
            LdapName ldapName = new LdapName(dn);
            ldapName.getRdns().forEach(rdn -> {
                String attribute = rdn.getType().toUpperCase(Locale.ENGLISH);
                String value = Rdn.escapeValue(rdn.getValue());
                if (CertMetadataIdentifier.IDENTIFIERS.contains(attribute)) {
                    vars.putIfAbsent(attribute, value);
                }
            });
        }
        catch (InvalidNameException e) {
            LOG.error("Failed to parse DN: {}", (Object)dn, (Object)e);
        }
        return vars;
    }

    static String generateSubjectAlternativeName(X509Certificate certificate) {
        Collection<List<?>> altNames;
        try {
            altNames = certificate.getSubjectAlternativeNames();
        }
        catch (CertificateParsingException e) {
            LOG.warn("Failed to parse SAN", (Throwable)e);
            return "";
        }
        if (altNames == null || altNames.isEmpty()) {
            return "";
        }
        ArrayList<String> result = new ArrayList<String>();
        for (List<?> altName : altNames) {
            Integer type = altName.size() >= 2 ? (Integer)altName.get(0) : null;
            if (type == null || !(altName.get(1) instanceof String)) continue;
            result.add(String.format("%s:%s", new Object[]{SANType.fromValue(type), altName.get(1)}));
        }
        return String.join((CharSequence)",", result);
    }

    static String getSerialNumber(X509Certificate certificate) {
        return CertificateMetadata.toHexStringUpperCase(certificate.getSerialNumber());
    }

    static String calculateSha1Fingerprint(X509Certificate certificate) throws NoSuchAlgorithmException, CertificateEncodingException {
        MessageDigest md = MessageDigest.getInstance("SHA-1");
        return CertificateMetadata.toHexStringUpperCase(md.digest(certificate.getEncoded()));
    }

    private static String toHexStringUpperCase(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        for (byte b : bytes) {
            sb.append(String.format("%02X", b));
        }
        return sb.toString();
    }

    private static String toHexStringUpperCase(BigInteger serialNumber) {
        return serialNumber.toString(16).toUpperCase(Locale.ENGLISH);
    }

    static enum SANType {
        OTHERNAME(0),
        EMAIL(1),
        DNS(2),
        X400(3),
        DIR(4),
        EDIPARTY(5),
        URI(6),
        IP(7),
        RID(8);

        private static final Map<Integer, SANType> LOOKUP;
        private final int value;

        private SANType(int value) {
            this.value = value;
        }

        int getValue() {
            return this.value;
        }

        static SANType fromValue(int value) {
            SANType result = LOOKUP.get(value);
            if (result == null) {
                throw new IllegalArgumentException("Invalid value: " + value);
            }
            return result;
        }

        static {
            LOOKUP = new HashMap<Integer, SANType>();
            for (SANType item : SANType.values()) {
                LOOKUP.put(item.getValue(), item);
            }
        }
    }
}

