/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.common;

import com.linkedin.kafka.cruisecontrol.common.AdminClientResult;
import com.linkedin.kafka.cruisecontrol.common.KafkaCluster;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.executor.Executor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.kafka.clients.admin.AlterBrokerReplicaExclusionsOptions;
import org.apache.kafka.clients.admin.AlterBrokerReplicaExclusionsResult;
import org.apache.kafka.clients.admin.ConfluentAdmin;
import org.apache.kafka.clients.admin.DescribeBrokerReplicaExclusionsOptions;
import org.apache.kafka.clients.admin.DescribeBrokerReplicaExclusionsResult;
import org.apache.kafka.clients.admin.DescribeClusterOptions;
import org.apache.kafka.clients.admin.DescribeClusterResult;
import org.apache.kafka.clients.admin.DescribeTopicsOptions;
import org.apache.kafka.clients.admin.ExclusionOp;
import org.apache.kafka.clients.admin.ListPartitionReassignmentsResult;
import org.apache.kafka.clients.admin.PartitionReassignment;
import org.apache.kafka.clients.admin.TopicDescription;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.BalancerOperationFailedException;
import org.apache.kafka.common.errors.NoReassignmentInProgressException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class SbkAdminUtils {
    private static final Logger LOG = LoggerFactory.getLogger(SbkAdminUtils.class);
    private ConfluentAdmin adminClient;
    private final int describeTopicsResponseTimeoutMs;
    private final int describeClusterResponseTimeoutMs;
    private final int describeBrokerExclusionTimeoutMs;
    private final int setExclusionDefaultTimeoutMs;

    public SbkAdminUtils(ConfluentAdmin adminClient, KafkaCruiseControlConfig config) {
        this.adminClient = adminClient;
        this.describeTopicsResponseTimeoutMs = config.getInt("describe.topics.response.timeout.ms");
        this.describeClusterResponseTimeoutMs = config.getInt("describe.cluster.response.timeout.ms");
        this.setExclusionDefaultTimeoutMs = config.getInt("broker.replica.exclusion.timeout.ms");
        this.describeBrokerExclusionTimeoutMs = config.getInt("describe.broker.exclusion.timeout.ms");
    }

    public AdminClientResult<KafkaCluster> describeCluster() throws InterruptedException {
        return this.describeCluster(this.describeClusterResponseTimeoutMs);
    }

    public AdminClientResult<KafkaCluster> describeCluster(int timeoutMs) throws InterruptedException {
        AdminClientResult<KafkaCluster> result;
        DescribeClusterOptions options = new DescribeClusterOptions().timeoutMs(Integer.valueOf(timeoutMs));
        try {
            DescribeClusterResult clusterResult = this.adminClient.describeCluster(options);
            Collection nodes = (Collection)clusterResult.nodes().get();
            String clusterId = (String)clusterResult.clusterId().get();
            Node controller = (Node)clusterResult.controller().get();
            Set authorizedOperations = (Set)clusterResult.authorizedOperations().get();
            result = new AdminClientResult<KafkaCluster>(new KafkaCluster(nodes, controller, clusterId, authorizedOperations));
        }
        catch (ExecutionException | KafkaException e) {
            LOG.error("Encountered exception while describing the Kafka cluster", e);
            result = new AdminClientResult<KafkaCluster>(e);
        }
        return result;
    }

    public AdminClientResult<AlterBrokerReplicaExclusionsResult.ExclusionsResult> alterBrokerReplicaExclusions(ExclusionOp op, Collection<Integer> brokerIds) throws InterruptedException {
        Map<Integer, ExclusionOp> exclusionOpMap = brokerIds.stream().collect(Collectors.toMap(id -> id, id -> op));
        try {
            return new AdminClientResult<Object>(this.adminClient.alterBrokerReplicaExclusions(exclusionOpMap, (AlterBrokerReplicaExclusionsOptions)new AlterBrokerReplicaExclusionsOptions().timeoutMs(Integer.valueOf(this.setExclusionDefaultTimeoutMs))).result().get());
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof TimeoutException) {
                LOG.warn("Unable to {} broker exclusions for brokers {} in {} ms", new Object[]{op, brokerIds, this.setExclusionDefaultTimeoutMs, e});
            }
            return new AdminClientResult<AlterBrokerReplicaExclusionsResult.ExclusionsResult>(e.getCause());
        }
    }

    public AdminClientResult<Set<Integer>> describeCurrentlyExcludedBrokers() throws InterruptedException, TimeoutException {
        try {
            DescribeBrokerReplicaExclusionsResult result = this.adminClient.describeBrokerReplicaExclusions((DescribeBrokerReplicaExclusionsOptions)new DescribeBrokerReplicaExclusionsOptions().timeoutMs(Integer.valueOf(this.describeBrokerExclusionTimeoutMs)));
            return new AdminClientResult<Set<Integer>>(((List)result.descriptions().get((long)this.describeBrokerExclusionTimeoutMs, TimeUnit.MILLISECONDS)).stream().map(DescribeBrokerReplicaExclusionsResult.BrokerReplicaExclusionDescription::brokerId).collect(Collectors.toSet()));
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof TimeoutException) {
                LOG.warn("Unable to describe broker exclusions for brokers in {} ms", (Object)this.describeBrokerExclusionTimeoutMs, (Object)e);
            }
            return new AdminClientResult<Set<Integer>>(e.getCause());
        }
    }

    public Map<TopicPartition, ReplicaDescription> getReplicasForPartitions(Collection<TopicPartition> topicPartitions) throws InterruptedException {
        HashMap<TopicPartition, ReplicaDescription> replicaMap = new HashMap<TopicPartition, ReplicaDescription>();
        HashMap<String, Set> partitionsOfInterestByTopic = new HashMap<String, Set>();
        for (TopicPartition tp : topicPartitions) {
            replicaMap.put(tp, ReplicaDescription.ofUnexistingPartition());
            partitionsOfInterestByTopic.computeIfAbsent(tp.topic(), c -> new HashSet()).add(tp);
        }
        Map descriptionFutures = this.adminClient.describeTopics(partitionsOfInterestByTopic.keySet(), new DescribeTopicsOptions().timeoutMs(Integer.valueOf(this.describeTopicsResponseTimeoutMs))).topicNameValues();
        for (Map.Entry topicEntry : descriptionFutures.entrySet()) {
            try {
                TopicDescription td = (TopicDescription)((KafkaFuture)topicEntry.getValue()).get();
                String currentTopic = (String)topicEntry.getKey();
                Set partitionsOfInterest = ((Set)partitionsOfInterestByTopic.get(currentTopic)).stream().map(TopicPartition::partition).collect(Collectors.toSet());
                td.partitions().stream().filter(p -> partitionsOfInterest.contains(p.partition())).forEach(tpi -> {
                    List<Integer> replicaSet = tpi.replicas().stream().map(Node::id).collect(Collectors.toList());
                    replicaMap.put(new TopicPartition(currentTopic, tpi.partition()), ReplicaDescription.ofSuccessfulDescription(replicaSet));
                });
            }
            catch (ExecutionException e) {
                if (e.getCause() instanceof TimeoutException) {
                    LOG.warn("Unable to fetch replica assignments for topics in {} ms", (Object)this.describeTopicsResponseTimeoutMs, (Object)e);
                    return this.failedReplicaDescriptions(topicPartitions, e);
                }
                Set tps = (Set)partitionsOfInterestByTopic.get(topicEntry.getKey());
                LOG.warn("Unable to fetch replica assignments for topic {} (partitions {}).", new Object[]{topicEntry.getKey(), tps, e});
                replicaMap.putAll(this.failedReplicaDescriptions(tps, e));
            }
        }
        return replicaMap;
    }

    private Map<TopicPartition, ReplicaDescription> failedReplicaDescriptions(Collection<TopicPartition> topicPartitions, ExecutionException e) {
        HashMap<TopicPartition, ReplicaDescription> replicaMap = new HashMap<TopicPartition, ReplicaDescription>();
        for (TopicPartition tp : topicPartitions) {
            Throwable throwable = e.getCause() == null ? e : e.getCause();
            replicaMap.put(tp, ReplicaDescription.ofFailedDescription(throwable));
        }
        return replicaMap;
    }

    public int cancelInterBrokerReplicaMovements(List<TopicPartition> partitionReassignmentsToCancel) {
        int numCancelled = 0;
        Optional cancelReassignment = Optional.empty();
        Map<TopicPartition, Optional> partitionsToCancel = partitionReassignmentsToCancel.stream().collect(Collectors.toMap(pr -> pr, pr -> cancelReassignment));
        Map cancellationFutures = this.adminClient.alterPartitionReassignments(partitionsToCancel).values();
        for (Map.Entry futureEntry : cancellationFutures.entrySet()) {
            TopicPartition tp = (TopicPartition)futureEntry.getKey();
            try {
                ((KafkaFuture)futureEntry.getValue()).get();
                ++numCancelled;
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while cancelling partition reassignments.");
                break;
            }
            catch (ExecutionException | ApiException e) {
                if (e.getCause() instanceof NoReassignmentInProgressException || e instanceof NoReassignmentInProgressException) {
                    LOG.debug("Could not cancel reassignment of {} because none was in progress", (Object)tp);
                    continue;
                }
                if (e.getCause() instanceof UnsupportedVersionException) {
                    LOG.warn("Kafka does not support the AlterPartitionReassignments API.Cannot cancel the current partition reassignments.");
                    break;
                }
                LOG.warn("Reassignment cancellation for {} failed.", (Object)tp, (Object)e);
            }
        }
        return numCancelled;
    }

    public Map<TopicPartition, Executor.PartitionReplicas> listTargetReplicasBeingReassigned(Optional<Set<TopicPartition>> partitionsOpt) {
        try {
            ListPartitionReassignmentsResult listPartitionsResult = partitionsOpt.isPresent() ? this.adminClient.listPartitionReassignments(partitionsOpt.get()) : this.adminClient.listPartitionReassignments();
            return ((Map)listPartitionsResult.reassignments().get()).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> {
                PartitionReassignment partitionReassignment = (PartitionReassignment)entry.getValue();
                ArrayList<Integer> targetReplicas = new ArrayList<Integer>(partitionReassignment.replicas());
                targetReplicas.removeAll(partitionReassignment.removingReplicas());
                ArrayList<Integer> targetObservers = new ArrayList<Integer>(partitionReassignment.observers());
                targetObservers.removeAll(partitionReassignment.removingReplicas());
                return new Executor.PartitionReplicas(targetReplicas, targetObservers);
            }));
        }
        catch (Throwable t) {
            LOG.error("Fetching reassigning replicas through the listPartitionReassignments API failed with an exception", t);
            SbkAdminUtils.sneakyThrow(t);
            return null;
        }
    }

    public static <E extends Throwable> void sneakyThrow(Throwable e) throws E {
        throw e;
    }

    public static RuntimeException handleAdminClientException(RuntimeException e) {
        if (e.getCause() instanceof InterruptedException) {
            Thread.currentThread().interrupt();
            throw e;
        }
        if (e.getCause() instanceof ExecutionException) {
            Throwable cause = e.getCause().getCause();
            throw cause instanceof RuntimeException ? (RuntimeException)cause : new RuntimeException(e);
        }
        throw e;
    }

    public void unregisterBrokers(Set<Integer> brokerIdsToUnregister) throws Exception {
        HashSet<Integer> unregisteredBrokers = new HashSet<Integer>();
        for (int brokerId : brokerIdsToUnregister) {
            try {
                this.adminClient.unregisterBroker(brokerId).all().get();
                unregisteredBrokers.add(brokerId);
            }
            catch (ExecutionException ee) {
                Throwable cause = ee.getCause();
                if (cause instanceof UnsupportedVersionException) {
                    LOG.info("Cannot unregister brokers {} because the target cluster does not support the broker unregistration API (expected for ZooKeeper clusters)", brokerIdsToUnregister);
                    return;
                }
                List brokersLeftToUnregister = brokerIdsToUnregister.stream().filter(unregisteredBrokers::contains).collect(Collectors.toList());
                LOG.error("Encountered an exception while unregistering broker {}. Successfully unregistered {} brokers so far ({} were unregistered out of {}). Brokers {} are still left registered.", new Object[]{brokerId, unregisteredBrokers.size(), unregisteredBrokers, brokerIdsToUnregister, brokersLeftToUnregister});
                throw new BalancerOperationFailedException(String.format("Failed to unregister broker %s", brokerId), (Throwable)ee);
            }
        }
        LOG.info("Brokers {} are not longer registered", unregisteredBrokers);
    }

    public static class ReplicaDescription {
        public final List<Integer> replicaSet;
        public final Optional<Throwable> throwableOpt;

        private ReplicaDescription(List<Integer> replicaSet, Optional<Throwable> throwableOpt) {
            this.replicaSet = replicaSet;
            this.throwableOpt = throwableOpt;
        }

        public boolean isFailed() {
            return this.throwableOpt.isPresent();
        }

        public static ReplicaDescription ofUnexistingPartition() {
            return new ReplicaDescription(Collections.emptyList(), Optional.empty());
        }

        public static ReplicaDescription ofSuccessfulDescription(List<Integer> replicaSet) {
            if (replicaSet == null) {
                throw new IllegalArgumentException("replicaSet cannot be null");
            }
            return new ReplicaDescription(replicaSet, Optional.empty());
        }

        public static ReplicaDescription ofFailedDescription(Throwable throwable) {
            if (throwable == null) {
                throw new IllegalArgumentException("Expected throwable to not be null");
            }
            return new ReplicaDescription(Collections.emptyList(), Optional.of(throwable));
        }

        public String toString() {
            return "ReplicaDescription{replicaSet=" + this.replicaSet + ", throwableOpt=" + this.throwableOpt + '}';
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ReplicaDescription that = (ReplicaDescription)o;
            return this.replicaSet.equals(that.replicaSet) && this.throwableOpt.equals(that.throwableOpt);
        }

        public int hashCode() {
            return Objects.hash(this.replicaSet, this.throwableOpt);
        }
    }
}

