/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.monitor.sampling;

import com.linkedin.cruisecontrol.metricdef.MetricDef;
import com.linkedin.cruisecontrol.metricdef.MetricInfo;
import com.linkedin.kafka.cruisecontrol.metricsreporter.metric.RawMetricType;
import com.linkedin.kafka.cruisecontrol.model.ModelUtils;
import com.linkedin.kafka.cruisecontrol.monitor.metricdefinition.KafkaMetricDef;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.BrokerLoad;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.PartitionMetricSample;
import com.linkedin.kafka.cruisecontrol.monitor.sampling.holder.ReplicaMetricSample;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SamplingUtils {
    private static final Logger LOG = LoggerFactory.getLogger(SamplingUtils.class);
    private static final String SKIP_BUILDING_SAMPLE_PREFIX = "Skip generating metric sample for ";
    static final Set<RawMetricType> REPLICA_RAW_METRIC_TYPES = new HashSet<RawMetricType>(Arrays.asList(RawMetricType.TOPIC_FETCH_FROM_FOLLOWER_BYTES_OUT, RawMetricType.TOPIC_FETCH_FROM_FOLLOWER_REQUEST_RATE, RawMetricType.TOPIC_REPLICATION_BYTES_IN));

    private SamplingUtils() {
    }

    static void populateReplicaDistribution(Cluster cluster, Map<Integer, Map<String, Integer>> leaderDistribution, Map<Integer, Map<String, Integer>> followerDistribution) {
        ArrayList offlinePartitions = new ArrayList();
        cluster.topics().forEach(topic -> cluster.partitionsForTopic(topic).forEach(partition -> Arrays.stream(partition.replicas()).filter(replica -> replica != null && !replica.isEmpty()).forEach(replica -> {
            boolean isLeader = false;
            if (partition.leader() == null) {
                offlinePartitions.add(partition);
            } else {
                isLeader = partition.leader().id() == replica.id();
            }
            Map stats = isLeader ? leaderDistribution : followerDistribution;
            Map replicaCountMap = stats.computeIfAbsent(replica.id(), ignored -> new HashMap());
            replicaCountMap.merge(topic, 1, Integer::sum);
        })));
        if (!offlinePartitions.isEmpty()) {
            LOG.warn("Detected {} offline partitions ({}) while building the replica. They are: {}", new Object[]{offlinePartitions.size(), offlinePartitions.stream().map(p -> String.format("%s-%d", p.topic(), p.partition())).collect(Collectors.toList()), offlinePartitions});
        }
    }

    static double estimateCpuUtil(String topic, BrokerLoad brokerLoad, boolean isLeader, int leaderCount, int followerCount, short numCpuCores, double requestContributionWeight, double bytesContributionWeight) {
        BrokerMetrics brokerMetrics = new BrokerMetrics(brokerLoad);
        ReplicaMetrics replicaMetrics = isLeader ? new LeaderReplicaMetrics(brokerLoad, topic, leaderCount) : new FollowerReplicaMetrics(brokerLoad, topic, leaderCount, followerCount);
        return (double)numCpuCores * ModelUtils.estimateCpuUtil(brokerMetrics.brokerCpuUtilization, brokerMetrics.brokerTotalBytesInRate, brokerMetrics.brokerTotalBytesOutRate, brokerMetrics.brokerTotalProduceRate, brokerMetrics.brokerTotalFetchRate, replicaMetrics.totalReplicaBytesInRate(), replicaMetrics.totalReplicaBytesOutRate(), replicaMetrics.replicaProduceRequestRate(), replicaMetrics.replicaFetchRequestRate(), requestContributionWeight, bytesContributionWeight);
    }

    static ReplicaMetricSample buildReplicaMetricSample(PartitionInfo topicPartition, Node replicaNode, int leaderCount, int followerCount, Map<Integer, BrokerLoad> brokerLoadById, long minMetricTimestamp, long maxMetricTimestamp, Map<Integer, Short> cachedNumCoresByBroker, double requestContributionWeight, double bytesContributionWeight) {
        int replicaNodeId = replicaNode.id();
        boolean isLeader = false;
        if (topicPartition.leader() == null) {
            LOG.debug("The topic partition {} appears to be offline - it has no leader.", (Object)topicPartition);
        } else {
            isLeader = replicaNodeId == topicPartition.leader().id();
        }
        boolean isLeaderFinal = isLeader;
        ReplicaMetricSample rms = new ReplicaMetricSample(replicaNodeId, topicPartition, isLeader);
        BrokerLoad brokerLoad = brokerLoadById.get(replicaNodeId);
        rms.open(minMetricTimestamp);
        REPLICA_RAW_METRIC_TYPES.forEach(rawMetricType -> {
            int contributors = rawMetricType.equals((Object)RawMetricType.TOPIC_REPLICATION_BYTES_IN) ? followerCount : leaderCount + followerCount;
            double sampleValue = brokerLoad.topicMetrics(topicPartition.topic(), (RawMetricType)((Object)rawMetricType)) / (double)contributors;
            for (KafkaMetricDef metricDef : KafkaMetricDef.kafkaMetricDefsForRawMetricType(rawMetricType)) {
                double valueToRecord = !isLeaderFinal || metricDef.equals((Object)KafkaMetricDef.RACK_BASED_CONSUME_BYTES_OUT) ? sampleValue : 0.0;
                rms.record(KafkaMetricDef.replicaMetricDef().metricInfo(metricDef), valueToRecord);
            }
        });
        Short numCpuCores = cachedNumCoresByBroker.get(replicaNodeId);
        double estimatedCpuUtil = SamplingUtils.estimateCpuUtil(topicPartition.topic(), brokerLoad, isLeader, leaderCount, followerCount, numCpuCores, requestContributionWeight, bytesContributionWeight);
        rms.record(KafkaMetricDef.replicaMetricDef().metricInfo(KafkaMetricDef.CPU_USAGE), estimatedCpuUtil);
        rms.close(maxMetricTimestamp);
        return rms;
    }

    static PartitionMetricSample buildPartitionMetricSample(Map<Integer, Map<String, Integer>> leaderDistribution, PartitionInfo topicPartition, Map<Integer, BrokerLoad> brokerLoadById, long minMetricTimestamp, long maxMetricTimestamp) {
        int leaderId = topicPartition.leader().id();
        int numLeaders = leaderDistribution.get(leaderId).get(topicPartition.topic());
        BrokerLoad brokerLoad = brokerLoadById.get(leaderId);
        PartitionMetricSample pms = new PartitionMetricSample(leaderId, new TopicPartition(topicPartition.topic(), topicPartition.partition()));
        pms.open(minMetricTimestamp);
        MetricDef partitionMetricDef = KafkaMetricDef.partitionMetricDef();
        for (RawMetricType rawMetricType : RawMetricType.topicMetricTypes()) {
            if (REPLICA_RAW_METRIC_TYPES.contains((Object)rawMetricType)) continue;
            double sampleValue = numLeaders == 0 ? 0.0 : brokerLoad.topicMetrics(topicPartition.topic(), rawMetricType) / (double)numLeaders;
            for (KafkaMetricDef kMetricDef : KafkaMetricDef.kafkaMetricDefsForRawMetricType(rawMetricType)) {
                if (kMetricDef.equals((Object)KafkaMetricDef.RACK_LESS_CONSUME_BYTES_OUT) && numLeaders != 0) {
                    double topicFFFBytesOut = brokerLoad.topicMetrics(topicPartition.topic(), RawMetricType.TOPIC_FETCH_FROM_FOLLOWER_BYTES_OUT);
                    sampleValue = (brokerLoad.topicMetrics(topicPartition.topic(), rawMetricType) - topicFFFBytesOut) / (double)numLeaders;
                }
                MetricInfo metricInfo = partitionMetricDef.metricInfo(kMetricDef);
                pms.record(metricInfo, sampleValue);
            }
        }
        double partitionSize = brokerLoad.partitionMetric(topicPartition.topic(), topicPartition.partition(), RawMetricType.PARTITION_SIZE);
        pms.record(partitionMetricDef.metricInfo(KafkaMetricDef.DISK_USAGE), partitionSize);
        pms.close(maxMetricTimestamp);
        return pms;
    }

    static boolean skipBuildingMetricSample(PartitionInfo topicPartition, Node replicaNode, Map<Integer, BrokerLoad> brokerLoadMap, Map<Integer, Short> cachedNumCoresByBroker) {
        if (replicaNode == null) {
            LOG.trace("Partition {} replica has been assigned to null node.", (Object)topicPartition);
            return true;
        }
        int replicaNodeId = replicaNode.id();
        BrokerLoad brokerLoad = brokerLoadMap.get(replicaNodeId);
        if (brokerLoad == null || !brokerLoad.brokerMetricAvailable(RawMetricType.BROKER_CPU_UTIL)) {
            LOG.debug("{}partition {} because {} metric for broker {} is unavailable.", new Object[]{SKIP_BUILDING_SAMPLE_PREFIX, topicPartition, RawMetricType.BROKER_CPU_UTIL, replicaNodeId});
            return true;
        }
        if (cachedNumCoresByBroker.get(replicaNodeId) == null) {
            LOG.debug("{}partition {} because the number of CPU cores of its leader broker {} is unavailable. Please ensure that either the broker capacity config resolver provides the number of CPU cores without estimation or allow CPU capacity estimation during sampling (i.e. set {} to true).", new Object[]{SKIP_BUILDING_SAMPLE_PREFIX, topicPartition, replicaNodeId, "sampling.allow.cpu.capacity.estimation"});
            return true;
        }
        if (!brokerLoad.topicMetricsAvailable(topicPartition.topic())) {
            LOG.debug("{}partition {} because broker {} has no metric or topic metrics are not available", new Object[]{SKIP_BUILDING_SAMPLE_PREFIX, topicPartition, replicaNodeId});
            return true;
        }
        return false;
    }

    private static class FollowerReplicaMetrics
    implements ReplicaMetrics {
        private final float replicaProduceRequestRate;
        private final float replicaFetchRequestRate;
        private final float totalReplicaBytesInRate;
        private final float totalReplicaBytesOutRate;

        public FollowerReplicaMetrics(BrokerLoad brokerLoad, String topic, int leaderCount, int followerCount) {
            double replicaProduceRequestRate = 0.0;
            double followerFetchRequestRate = brokerLoad.topicMetrics(topic, RawMetricType.TOPIC_FOLLOWER_FETCH_REQUEST_RATE) / (double)followerCount;
            double fetchFromFollowerRequestRate = brokerLoad.topicMetrics(topic, RawMetricType.TOPIC_FETCH_FROM_FOLLOWER_REQUEST_RATE) / (double)(leaderCount + followerCount);
            double replicaProduceBytesInRate = 0.0;
            double replicaReplicationBytesInRate = brokerLoad.topicMetrics(topic, RawMetricType.TOPIC_REPLICATION_BYTES_IN) / (double)followerCount;
            double replicaFetchFromFollowerBytesOutRate = brokerLoad.topicMetrics(topic, RawMetricType.TOPIC_FETCH_FROM_FOLLOWER_BYTES_OUT) / (double)(leaderCount + followerCount);
            double replicaReplicationBytesOutRate = 0.0;
            this.replicaProduceRequestRate = (float)replicaProduceRequestRate;
            this.replicaFetchRequestRate = (float)(followerFetchRequestRate + fetchFromFollowerRequestRate);
            this.totalReplicaBytesInRate = (float)(replicaProduceBytesInRate + replicaReplicationBytesInRate);
            this.totalReplicaBytesOutRate = (float)(replicaFetchFromFollowerBytesOutRate + replicaReplicationBytesOutRate);
        }

        @Override
        public double totalReplicaBytesInRate() {
            return this.totalReplicaBytesInRate;
        }

        @Override
        public double totalReplicaBytesOutRate() {
            return this.totalReplicaBytesOutRate;
        }

        @Override
        public double replicaProduceRequestRate() {
            return this.replicaProduceRequestRate;
        }

        @Override
        public double replicaFetchRequestRate() {
            return this.replicaFetchRequestRate;
        }
    }

    private static class LeaderReplicaMetrics
    implements ReplicaMetrics {
        private final float replicaProduceRequestRate;
        private final float replicaFetchRequestRate;
        private final float totalReplicaBytesInRate;
        private final float totalReplicaBytesOutRate;

        public LeaderReplicaMetrics(BrokerLoad brokerLoad, String topic, int leaderCount) {
            double replicaProduceRequestRate = brokerLoad.topicMetrics(topic, RawMetricType.TOPIC_PRODUCE_REQUEST_RATE) / (double)leaderCount;
            double replicaFetchRequestRate = brokerLoad.topicMetrics(topic, RawMetricType.TOPIC_FETCH_REQUEST_RATE) / (double)leaderCount;
            double replicaFetchFromFollowerRequestRate = 0.0;
            double replicaProduceBytesInRate = brokerLoad.topicMetrics(topic, RawMetricType.TOPIC_BYTES_IN) / (double)leaderCount;
            double replicaReplicationBytesInRate = 0.0;
            double replicaConsumeBytesOutRate = brokerLoad.topicMetrics(topic, RawMetricType.TOPIC_BYTES_OUT) / (double)leaderCount;
            double replicaReplicationBytesOutRate = brokerLoad.topicMetrics(topic, RawMetricType.TOPIC_REPLICATION_BYTES_OUT) / (double)leaderCount;
            double replicaFetchFromFollowerBytesOutRate = 0.0;
            this.replicaProduceRequestRate = (float)replicaProduceRequestRate;
            this.replicaFetchRequestRate = (float)(replicaFetchRequestRate - replicaFetchFromFollowerRequestRate);
            this.totalReplicaBytesInRate = (float)(replicaProduceBytesInRate + replicaReplicationBytesInRate);
            this.totalReplicaBytesOutRate = (float)(replicaConsumeBytesOutRate + replicaReplicationBytesOutRate - replicaFetchFromFollowerBytesOutRate);
        }

        @Override
        public double totalReplicaBytesInRate() {
            return this.totalReplicaBytesInRate;
        }

        @Override
        public double totalReplicaBytesOutRate() {
            return this.totalReplicaBytesOutRate;
        }

        @Override
        public double replicaProduceRequestRate() {
            return this.replicaProduceRequestRate;
        }

        @Override
        public double replicaFetchRequestRate() {
            return this.replicaFetchRequestRate;
        }
    }

    private static interface ReplicaMetrics {
        public double totalReplicaBytesInRate();

        public double totalReplicaBytesOutRate();

        public double replicaProduceRequestRate();

        public double replicaFetchRequestRate();
    }

    private static class BrokerMetrics {
        final double brokerCpuUtilization;
        final double brokerTotalBytesInRate;
        final double brokerTotalBytesOutRate;
        final double brokerTotalProduceRate;
        final double brokerTotalFetchRate;

        public BrokerMetrics(BrokerLoad brokerLoad) {
            this.brokerCpuUtilization = brokerLoad.brokerMetric(RawMetricType.BROKER_CPU_UTIL);
            this.brokerTotalBytesInRate = brokerLoad.brokerMetric(RawMetricType.ALL_TOPIC_BYTES_IN) + brokerLoad.brokerMetric(RawMetricType.ALL_TOPIC_REPLICATION_BYTES_IN);
            this.brokerTotalBytesOutRate = brokerLoad.brokerMetric(RawMetricType.ALL_TOPIC_BYTES_OUT) + brokerLoad.brokerMetric(RawMetricType.ALL_TOPIC_REPLICATION_BYTES_OUT);
            this.brokerTotalProduceRate = brokerLoad.brokerMetric(RawMetricType.ALL_TOPIC_PRODUCE_REQUEST_RATE);
            this.brokerTotalFetchRate = brokerLoad.brokerMetric(RawMetricType.ALL_TOPIC_FETCH_REQUEST_RATE) + brokerLoad.brokerMetric(RawMetricType.ALL_TOPIC_FOLLOWER_FETCH_REQUEST_RATE);
        }
    }
}

