/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.cruisecontrol.analyzer.goals;

import com.linkedin.kafka.cruisecontrol.analyzer.goals.EntityCombinator;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.Goal;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.GoalUtils;
import com.linkedin.kafka.cruisecontrol.exception.OptimizationFailureException;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.Cell;
import com.linkedin.kafka.cruisecontrol.model.ClusterModel;
import com.linkedin.kafka.cruisecontrol.model.Partition;
import com.linkedin.kafka.cruisecontrol.model.Replica;
import com.linkedin.kafka.cruisecontrol.model.ReplicaRelocationContext;
import com.linkedin.kafka.cruisecontrol.model.Tenant;
import io.confluent.cruisecontrol.analyzer.goals.ReplicaRelocator;
import io.confluent.cruisecontrol.analyzer.goals.TopicRebalanceContext;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.stream.Collectors;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CellPartitionRelocator {
    private static final Logger LOG = LoggerFactory.getLogger(CellPartitionRelocator.class);
    private final Cell sourceCell;
    private final String topic;
    private final Tenant tenant;
    private final ClusterModel clusterModel;
    private final Set<Goal> optimizedGoals;
    private final ReplicaRelocator replicaRelocator;
    private int numOfPartitionsToMoveOut;
    private final int maxNumOfPartitionsPerCell;

    public CellPartitionRelocator(Cell sourceCell, Tenant tenant, TopicRebalanceContext topicRebalanceContext, int numOfPartitionsToMoveOut, int maxNumOfPartitionsPerCell) {
        this.sourceCell = sourceCell;
        this.topic = topicRebalanceContext.getTopic();
        this.tenant = tenant;
        this.clusterModel = topicRebalanceContext.getClusterModel();
        this.optimizedGoals = topicRebalanceContext.getOptimizedGoals();
        this.replicaRelocator = topicRebalanceContext.getReplicaRelocator();
        this.numOfPartitionsToMoveOut = numOfPartitionsToMoveOut;
        this.maxNumOfPartitionsPerCell = maxNumOfPartitionsPerCell;
    }

    CellPartitionRelocator(Cell sourceCell, String topic, Tenant tenant, ClusterModel clusterModel, Set<Goal> optimizedGoals, ReplicaRelocator replicaRelocator, int numOfPartitionsToMoveOut, int maxNumOfPartitionsPerCell) {
        this.sourceCell = sourceCell;
        this.topic = topic;
        this.tenant = tenant;
        this.clusterModel = clusterModel;
        this.optimizedGoals = optimizedGoals;
        this.replicaRelocator = replicaRelocator;
        this.numOfPartitionsToMoveOut = numOfPartitionsToMoveOut;
        this.maxNumOfPartitionsPerCell = maxNumOfPartitionsPerCell;
    }

    public boolean movePartitionsOut() throws OptimizationFailureException {
        boolean isSourceNonTenantCell = !this.tenant.cellIds().contains(this.sourceCell.id());
        SortedSet<Cell> candidateCells = this.getCandidateCells();
        HashMap<Integer, Set<TopicPartition>> brokerToImmovablePartitions = new HashMap<Integer, Set<TopicPartition>>();
        SortedSet<Broker> sourceBrokers = this.getSourceBrokers(brokerToImmovablePartitions);
        int totalNumOfCandidatePartitions = this.sourceCell.numTopicLeaderReplicas(this.topic);
        int numOfCandidatePartitionsTried = 0;
        while (this.numOfPartitionsToMoveOut > 0) {
            Broker sourceBroker = sourceBrokers.last();
            Partition candidatePartition = this.getCandidatePartition(sourceBroker, brokerToImmovablePartitions);
            Optional<Cell> destinationCell = this.relocatePartitionToCells(candidatePartition, candidateCells);
            if (!destinationCell.isPresent()) {
                if (isSourceNonTenantCell) {
                    throw new OptimizationFailureException(String.format("Failed to move partition %s of topic %s from broker %d to any of the tenant's %s cells", candidatePartition.topicPartition(), this.topic, sourceBroker.id(), this.tenant.tenantId()));
                }
                brokerToImmovablePartitions.computeIfAbsent(sourceBroker.id(), k -> new HashSet()).add(candidatePartition.topicPartition());
                this.updateSourceBrokers(sourceBrokers, sourceBroker, brokerToImmovablePartitions);
            } else {
                --this.numOfPartitionsToMoveOut;
                this.updateCandidateCells(candidateCells, destinationCell.get());
                this.updateSourceBrokers(sourceBrokers, sourceBroker, brokerToImmovablePartitions);
            }
            if (++numOfCandidatePartitionsTried != totalNumOfCandidatePartitions || this.numOfPartitionsToMoveOut <= 0) continue;
            LOG.info("While balancing tenant {}, failed to move out {} partitions from cell {}", new Object[]{this.tenant.tenantId(), this.numOfPartitionsToMoveOut, this.sourceCell.id()});
            return false;
        }
        return true;
    }

    SortedSet<Cell> getCandidateCells() {
        TreeSet<Cell> candidateCells = new TreeSet<Cell>(Comparator.comparingInt(cell -> cell.numTopicLeaderReplicas(this.topic)).thenComparingInt(Cell::id));
        this.tenant.cellIds().stream().map(this.clusterModel::cell).filter(cell -> cell.numTopicLeaderReplicas(this.topic) < this.maxNumOfPartitionsPerCell).filter(cell -> !cell.equals(this.sourceCell)).forEach(candidateCells::add);
        return candidateCells;
    }

    private SortedSet<Broker> getSourceBrokers(Map<Integer, Set<TopicPartition>> brokerToImmovablePartitions) {
        TreeSet<Broker> sourceBrokers = new TreeSet<Broker>(Comparator.comparingInt(b -> b.numLeaderReplicasOfTopicInBroker(this.topic) - brokerToImmovablePartitions.getOrDefault(b.id(), Collections.emptySet()).size()).thenComparingInt(Broker::id));
        sourceBrokers.addAll(this.sourceCell.brokers());
        return sourceBrokers;
    }

    Partition getCandidatePartition(Broker sourceBroker, Map<Integer, Set<TopicPartition>> brokerToImmovablePartitions) throws OptimizationFailureException {
        Optional<Replica> leaderReplica = sourceBroker.leaderReplicasOfTopicInBroker(this.topic).stream().filter(replica -> !brokerToImmovablePartitions.getOrDefault(sourceBroker.id(), Collections.emptySet()).contains(replica.topicPartition())).findFirst();
        if (!leaderReplica.isPresent()) {
            throw new OptimizationFailureException(String.format("Failed to find partition for topic %s in broker %d for cell %d when %d partitions are yet to be removed", this.topic, sourceBroker.id(), this.sourceCell.id(), this.numOfPartitionsToMoveOut));
        }
        return this.clusterModel.partition(leaderReplica.get().topicPartition());
    }

    private Optional<Cell> relocatePartitionToCells(Partition partition, SortedSet<Cell> candidateCells) {
        for (Cell cell : candidateCells) {
            if (!this.relocatePartitionToCell(partition, cell)) continue;
            return Optional.of(cell);
        }
        return Optional.empty();
    }

    boolean relocatePartitionToCell(Partition partition, Cell destinationCell) {
        List<Replica> replicasToRelocate = partition.replicas().stream().filter(replica -> !replica.broker().cell().equals(destinationCell)).collect(Collectors.toList());
        Set ineligibleBrokers = replicasToRelocate.stream().map(Replica::broker).filter(broker -> !broker.isEligibleSource()).collect(Collectors.toSet());
        if (!ineligibleBrokers.isEmpty()) {
            LOG.debug(String.format("Can't move Partition %s to cell %s as it has its replicas on brokers that are ineligible source %s", partition, destinationCell, ineligibleBrokers));
            return false;
        }
        HashSet<Broker> partitionBrokers = new HashSet<Broker>(partition.partitionBrokers());
        List validBrokers = destinationCell.brokers().stream().filter(broker -> !partitionBrokers.contains(broker)).collect(Collectors.toList());
        List validEligibleBrokers = validBrokers.stream().filter(Broker::isEligibleDestination).sorted(Comparator.comparing(broker -> broker.replicas().size())).collect(Collectors.toList());
        if (validEligibleBrokers.size() < replicasToRelocate.size()) {
            LOG.debug(String.format("Can't move partition %s of topic %s to cell %s as has %d replicas but tenant cell has only %d brokers", partition, this.topic, destinationCell, partitionBrokers.size(), validEligibleBrokers.size()));
            return false;
        }
        Iterable<List<Broker>> candidateBrokersIterable = EntityCombinator.singleEntityListIterable(validEligibleBrokers, replicasToRelocate.size());
        Map<Replica, Broker> replicaMoves = GoalUtils.getPartitionMoves(this.clusterModel, this.optimizedGoals, replicasToRelocate, candidateBrokersIterable);
        if (!replicaMoves.isEmpty()) {
            ReplicaRelocationContext replicaRelocationContext = ReplicaRelocationContext.forPartitionMovementAction(replicaMoves);
            replicaMoves.forEach((replica, broker) -> this.replicaRelocator.relocate((Replica)replica, (Broker)broker, replicaRelocationContext));
            return true;
        }
        return false;
    }

    void updateCandidateCells(SortedSet<Cell> candidateCells, Cell updatedCell) {
        int destinationCellId = updatedCell.id();
        candidateCells.removeIf(cell -> cell.id() == destinationCellId);
        if (updatedCell.numTopicLeaderReplicas(this.topic) < this.maxNumOfPartitionsPerCell) {
            candidateCells.add(updatedCell);
        }
    }

    void updateSourceBrokers(SortedSet<Broker> sourceBrokers, Broker updatedBroker, Map<Integer, Set<TopicPartition>> brokerToImmovablePartitions) {
        int updatedBrokerId = updatedBroker.id();
        sourceBrokers.removeIf(b -> b.id() == updatedBrokerId);
        if (updatedBroker.numLeaderReplicasOfTopicInBroker(this.topic) - brokerToImmovablePartitions.getOrDefault(updatedBroker.id(), Collections.emptySet()).size() > 0) {
            sourceBrokers.add(updatedBroker);
        }
    }
}

