/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.clients.plugins.auth.oauth;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.List;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.net.util.SubnetUtils;
import org.apache.hc.client5.http.DnsResolver;
import org.apache.hc.client5.http.SystemDefaultDnsResolver;
import org.apache.kafka.common.KafkaException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FilteringDnsResolver
implements DnsResolver {
    private static final Logger log = LoggerFactory.getLogger(FilteringDnsResolver.class);
    DnsResolver baseResolver;
    private final boolean filterLocalAddresses;
    private final boolean filterPrivateAddresses;
    private final boolean filterClassEAddresses;
    private final List<SubnetUtils.SubnetInfo> disallowCidrRanges;
    private final List<SubnetUtils.SubnetInfo> allowCidrRanges;
    static final Predicate<InetAddress> LOCAL_ADDRESS_TEST_PREDICATE = address -> address.isAnyLocalAddress() || address.isLoopbackAddress() || address.isLinkLocalAddress();
    static final Predicate<InetAddress> PRIVATE_ADDRESS_TEST_PREDICATE = InetAddress::isSiteLocalAddress;
    private static final SubnetUtils.SubnetInfo CLASS_E_CIDR_RANGE = new SubnetUtils("240.0.0.0/4").getInfo();
    static final Predicate<InetAddress> CLASS_E_ADDRESS_TEST_PREDICATE = address -> {
        if (!(address instanceof Inet4Address)) {
            return false;
        }
        return CLASS_E_CIDR_RANGE.isInRange(address.getHostAddress()) || CLASS_E_CIDR_RANGE.getNetworkAddress().equals(address.getHostAddress()) || CLASS_E_CIDR_RANGE.getBroadcastAddress().equals(address.getHostAddress());
    };

    public FilteringDnsResolver(boolean filterLocalAddresses, boolean filterPrivateAddresses, boolean filterClassEAddresses, List<String> disallowCidrRanges, List<String> allowCidrRanges) {
        this((DnsResolver)new SystemDefaultDnsResolver(), filterLocalAddresses, filterPrivateAddresses, filterClassEAddresses, disallowCidrRanges, allowCidrRanges);
    }

    FilteringDnsResolver(DnsResolver baseResolver, boolean filterLocalAddresses, boolean filterPrivateAddresses, boolean filterClassEAddresses, List<String> disallowCidrRanges, List<String> allowCidrRanges) {
        this.baseResolver = baseResolver;
        this.filterLocalAddresses = filterLocalAddresses;
        this.filterPrivateAddresses = filterPrivateAddresses;
        this.filterClassEAddresses = filterClassEAddresses;
        this.disallowCidrRanges = disallowCidrRanges.stream().filter(cidrRangeString -> {
            try {
                new SubnetUtils(cidrRangeString);
            }
            catch (IllegalArgumentException e) {
                log.warn("Ignoring CIDR range {} from blocklist {} because it isn't a valid CIDR range", cidrRangeString, (Object)disallowCidrRanges);
                return false;
            }
            return true;
        }).map(cidrRangeString -> new SubnetUtils(cidrRangeString).getInfo()).collect(Collectors.toList());
        this.allowCidrRanges = allowCidrRanges.stream().filter(cidrRangeString -> {
            try {
                new SubnetUtils(cidrRangeString);
            }
            catch (IllegalArgumentException e) {
                log.warn("Ignoring CIDR range {} from allowlist {} because it isn't a valid CIDR range", cidrRangeString, (Object)allowCidrRanges);
                return false;
            }
            return true;
        }).map(cidrRangeString -> new SubnetUtils(cidrRangeString).getInfo()).collect(Collectors.toList());
    }

    private boolean noFiltersRestrictions() {
        return !this.filterLocalAddresses && !this.filterPrivateAddresses && !this.filterClassEAddresses;
    }

    private boolean isAllowedCidrRange(InetAddress address) {
        return this.allowCidrRanges.stream().anyMatch(cidrRange -> cidrRange.isInRange(address.getHostAddress()) || cidrRange.getNetworkAddress().equals(address.getHostAddress()) || cidrRange.getBroadcastAddress().equals(address.getHostAddress()));
    }

    private boolean isDisallowedCidrRange(InetAddress address) {
        return this.disallowCidrRanges.stream().anyMatch(cidrRange -> cidrRange.isInRange(address.getHostAddress()) || cidrRange.getNetworkAddress().equals(address.getHostAddress()) || cidrRange.getBroadcastAddress().equals(address.getHostAddress()));
    }

    public InetAddress[] resolve(String host) throws UnknownHostException {
        InetAddress[] inetAddresses = this.baseResolver.resolve(host);
        if (this.noFiltersRestrictions() && this.disallowCidrRanges.isEmpty() && this.allowCidrRanges.isEmpty()) {
            return inetAddresses;
        }
        if (inetAddresses.length > 0) {
            return this.filterAddresses(inetAddresses);
        }
        return inetAddresses;
    }

    public String resolveCanonicalHostname(String host) throws UnknownHostException {
        InetAddress[] resolvedAddresses = this.resolve(host);
        if (resolvedAddresses.length > 0) {
            return resolvedAddresses[0].getCanonicalHostName();
        }
        return host;
    }

    private InetAddress[] filterAddresses(InetAddress[] inetAddresses) {
        InetAddress[] filteredInetAddresses = (InetAddress[])Stream.of(inetAddresses).filter(address -> {
            if (address instanceof Inet6Address) {
                return this.filterV6Address((InetAddress)address);
            }
            return this.filterV4Address((InetAddress)address);
        }).toArray(InetAddress[]::new);
        if (filteredInetAddresses.length == 0) {
            throw new KafkaException(String.format("Unable to connect to the invalid endpoint URL: %s", Arrays.stream(inetAddresses).map(InetAddress::getHostAddress).collect(Collectors.joining(", "))));
        }
        return filteredInetAddresses;
    }

    private boolean filterV6Address(InetAddress address) {
        if (this.filterLocalAddresses && LOCAL_ADDRESS_TEST_PREDICATE.test(address)) {
            log.debug("Filtering out local IPv6 address {}", (Object)address);
            return false;
        }
        if (this.filterPrivateAddresses && PRIVATE_ADDRESS_TEST_PREDICATE.test(address)) {
            log.debug("Filtering out private IPv6 address {}", (Object)address);
            return false;
        }
        return true;
    }

    private boolean filterV4Address(InetAddress address) {
        if (this.isAllowedCidrRange(address)) {
            log.debug("Allowing IP address {} belonging to allow-list CIDR range: {}", (Object)address, this.allowCidrRanges);
            return true;
        }
        if (this.filterLocalAddresses && LOCAL_ADDRESS_TEST_PREDICATE.test(address)) {
            log.debug("Filtering out local IP address {}", (Object)address);
            return false;
        }
        if (this.filterPrivateAddresses && PRIVATE_ADDRESS_TEST_PREDICATE.test(address)) {
            log.debug("Filtering out private IP address {}", (Object)address);
            return false;
        }
        if (this.filterClassEAddresses && CLASS_E_ADDRESS_TEST_PREDICATE.test(address)) {
            log.debug("Filtering out class E IP address {}", (Object)address);
            return false;
        }
        if (this.isDisallowedCidrRange(address)) {
            log.debug("Filtering out IP address {} belonging to block-list CIDR range: {}", (Object)address, this.disallowCidrRanges);
            return false;
        }
        return true;
    }
}

