/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.security.auth.provider.ldap;

import io.confluent.security.auth.metadata.PasswordVerifier;
import io.confluent.security.auth.provider.ldap.LdapConfig;
import io.confluent.security.auth.provider.ldap.LdapContextCreator;
import io.confluent.security.auth.provider.ldap.LdapException;
import io.confluent.security.auth.provider.ldap.LdapGroupManager;
import io.confluent.security.auth.provider.ldap.LdapPrincipalMapping;
import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.concurrent.atomic.AtomicInteger;
import javax.naming.CommunicationException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapContext;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.AppConfigurationEntry;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.security.auth.AuthenticateCallbackHandler;
import org.apache.kafka.common.security.auth.AuthorizationIdProvider;
import org.apache.kafka.common.security.plain.PlainAuthenticateCallback;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LdapAuthenticateCallbackHandler
implements AuthorizationIdProvider,
AuthenticateCallbackHandler,
Closeable {
    private static final Logger log = LoggerFactory.getLogger(LdapAuthenticateCallbackHandler.class);
    private static final String AUTH_FAILED_MESSAGE = "LDAP authentication failed";
    private static final String LDAP_CLOSED_MSG = "LDAP connection has been closed";
    private static final String LDAP_TIMEOUT_MSG = "LDAP response read timed out, timeout used";
    private final SearchControls searchControls;
    private final List<PasswordVerifier> passwordVerifiers;
    private volatile LdapConfig config;
    private volatile UserSearchMode userSearchMode;
    private volatile LdapContextCreator searchContextCreator;
    private volatile LdapContext context;
    private final AtomicInteger ldapSearchCounter = new AtomicInteger(0);

    public LdapAuthenticateCallbackHandler() {
        this.searchControls = new SearchControls();
        this.passwordVerifiers = new ArrayList<PasswordVerifier>();
    }

    public void configure(Map<String, ?> configs, String saslMechanism, List<AppConfigurationEntry> jaasConfigEntries) {
        if (!"PLAIN".equals(saslMechanism)) {
            throw new ConfigException("SASL mechanism not supported: " + saslMechanism);
        }
        this.config = new LdapConfig(configs);
        this.userSearchMode = this.config.userPasswordAttribute != null ? UserSearchMode.PASSWORD_SEARCH : UserSearchMode.DN_SEARCH;
        if (this.config.userNameAttribute == null || this.config.userNameAttribute.isEmpty()) {
            throw new ConfigException("User name attribute not specified");
        }
        this.searchContextCreator = new LdapContextCreator(this.config);
        if (this.userSearchMode == UserSearchMode.DN_SEARCH) {
            this.searchControls.setReturningAttributes(new String[]{this.config.userNameAttribute});
        } else {
            this.searchControls.setReturningAttributes(new String[]{this.config.userNameAttribute, this.config.userPasswordAttribute});
            ServiceLoader<PasswordVerifier> verifiers = ServiceLoader.load(PasswordVerifier.class);
            for (PasswordVerifier verifier2 : verifiers) {
                this.passwordVerifiers.add(verifier2);
            }
            this.passwordVerifiers.add(new DefaultPasswordVerifier());
            this.passwordVerifiers.forEach(verifier -> verifier.configure(configs));
        }
        this.searchControls.setSearchScope(this.config.userSearchScope);
    }

    public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
        NameCallback nameCallback = null;
        PlainAuthenticateCallback authenticateCallback = null;
        for (Callback callback : callbacks) {
            if (callback instanceof NameCallback) {
                nameCallback = (NameCallback)callback;
                continue;
            }
            if (callback instanceof PlainAuthenticateCallback) {
                authenticateCallback = (PlainAuthenticateCallback)callback;
                continue;
            }
            throw new UnsupportedCallbackException(callback);
        }
        String userName = this.username(nameCallback);
        char[] password = this.password(authenticateCallback);
        try {
            boolean authenticated;
            LdapUser ldapUser = this.ldapUser(userName);
            switch (this.userSearchMode) {
                case DN_SEARCH: {
                    authenticated = this.authenticateUsingSimpleBind(ldapUser.dn, password);
                    break;
                }
                case PASSWORD_SEARCH: {
                    authenticated = this.authenticateUsingPasswordSearch(userName, password);
                    break;
                }
                default: {
                    throw new IllegalStateException("Unknown search mode " + (Object)((Object)this.userSearchMode));
                }
            }
            if (this.config.principalMappingMode.equals(LdapPrincipalMapping.LDAP.name)) {
                nameCallback.setName(ldapUser.principal);
            } else {
                nameCallback.setName(userName);
            }
            authenticateCallback.authenticated(authenticated);
        }
        catch (AuthenticationException e) {
            authenticateCallback.authenticated(false);
        }
    }

    private String username(NameCallback callback) {
        String userName = null;
        if (callback != null) {
            userName = callback.getDefaultName();
        }
        if (userName == null || userName.isEmpty()) {
            throw new AuthenticationException("User name not specified");
        }
        return userName;
    }

    private char[] password(PlainAuthenticateCallback callback) {
        char[] password = null;
        if (callback != null) {
            password = callback.password();
        }
        if (password == null || password.length == 0) {
            throw new AuthenticationException("Password not specified");
        }
        return password;
    }

    private boolean authenticateUsingSimpleBind(String userDn, char[] password) {
        Hashtable<String, String> env = new Hashtable<String, String>(this.config.ldapContextEnvironment);
        env.put("java.naming.security.authentication", "simple");
        env.put("java.naming.security.principal", userDn);
        env.put("java.naming.security.credentials", new String(password));
        try {
            InitialDirContext context = new InitialDirContext(env);
            context.close();
            return true;
        }
        catch (NamingException e) {
            if (e instanceof CommunicationException) {
                log.warn("LDAP bind failed for user DN {}", (Object)userDn, (Object)e);
            } else {
                log.trace("LDAP bind failed for user DN {} with specified password", (Object)userDn, (Object)e);
            }
            return false;
        }
    }

    private boolean authenticateUsingPasswordSearch(String userName, char[] password) {
        LdapUser ldapUser = this.ldapUser(userName);
        NameCallback nameCallback = new NameCallback("Name: ");
        nameCallback.setName(userName);
        char[] expectedPassword = new String(ldapUser.password, StandardCharsets.UTF_8).toCharArray();
        for (PasswordVerifier verifier : this.passwordVerifiers) {
            PasswordVerifier.Result result = verifier.verify(expectedPassword, password);
            switch (result) {
                case MATCH: {
                    return true;
                }
                case MISMATCH: {
                    return false;
                }
            }
        }
        return false;
    }

    @Override
    public void close() {
        if (this.context != null) {
            try {
                this.context.close();
            }
            catch (NamingException e) {
                log.error("Failed to close LDAP context", (Throwable)e);
            }
        }
        this.passwordVerifiers.forEach(verifier -> Utils.closeQuietly((AutoCloseable)verifier, (String)"passwordVerifier"));
    }

    private LdapUser ldapUser(String userName) {
        LdapUser userMetadata = Subject.doAs(this.searchContextCreator.subject(), () -> this.searchForLdapUser(userName));
        if (userMetadata == null) {
            throw new AuthenticationException(AUTH_FAILED_MESSAGE);
        }
        return userMetadata;
    }

    private LdapUser searchForLdapUser(String userName) {
        boolean hasContext = this.context != null;
        String debuggingInfo = "";
        if (log.isDebugEnabled()) {
            int searchForLdapUserCounter = this.ldapSearchCounter.getAndIncrement();
            debuggingInfo = this + " Thread:" + Thread.currentThread().getName() + " user:" + userName + " searchCounter:" + searchForLdapUserCounter + " ";
        }
        try {
            if (!hasContext) {
                log.debug("Searching for user with null LDAP context: Must create new context. {}", (Object)debuggingInfo);
                this.context = this.searchContextCreator.createLdapContext();
            }
            try {
                return this.ldapSearch(this.context, userName);
            }
            catch (IOException | NamingException e) {
                if (!this.shouldMakeNewLdapContext(e)) {
                    throw e;
                }
                log.warn("Caught exception searching for LDAP User. {} {}", new Object[]{this.context, debuggingInfo, e});
                if (hasContext) {
                    log.warn("Creating and using new ldapContext. {}", (Object)debuggingInfo);
                    this.context = this.searchContextCreator.createLdapContext();
                    return this.ldapSearch(this.context, userName);
                }
                return null;
            }
        }
        catch (LdapException | IOException | NamingException e) {
            log.error("Failed to obtain user DN for {}. {} {}", new Object[]{userName, this.context, debuggingInfo, e});
            throw new AuthenticationException(AUTH_FAILED_MESSAGE);
        }
    }

    private boolean shouldMakeNewLdapContext(Exception e) {
        String namingMessage;
        if (e instanceof CommunicationException) {
            return true;
        }
        return e instanceof NamingException && (namingMessage = e.getMessage()) != null && (namingMessage.contains(LDAP_CLOSED_MSG) || namingMessage.contains(LDAP_TIMEOUT_MSG));
    }

    private LdapUser ldapSearch(LdapContext context, String userName) throws IOException, NamingException {
        String usernameInLdapRecords;
        Object[] filterArgs = new Object[]{userName};
        log.trace("Searching for user {} with base {} filter {}: [Thread {}]", new Object[]{userName, this.config.userSearchBase, this.config.userDnSearchFilter, Thread.currentThread().getId()});
        NamingEnumeration<SearchResult> enumeration = context.search(this.config.userSearchBase, this.config.userDnSearchFilter, filterArgs, this.searchControls);
        if (!enumeration.hasMore()) {
            log.trace("User not found {} [Thread {}]", (Object)userName, (Object)Thread.currentThread().getId());
            return null;
        }
        SearchResult result = enumeration.next();
        if (enumeration.hasMore()) {
            log.trace("Search found multiple entries for user {} with base {} filter {}: [Thread {}]", new Object[]{userName, this.config.userSearchBase, this.config.userDnSearchFilter, Thread.currentThread().getId()});
            log.error("Found multiple user entries with user name {}", (Object)userName);
            return null;
        }
        Attributes attributes = result.getAttributes();
        Attribute nameAttr = attributes.get(this.config.userNameAttribute);
        if (nameAttr != null) {
            usernameInLdapRecords = LdapGroupManager.attributeValue(nameAttr.get(), this.config.userNameAttributePattern, "", "user name in LDAP Records", LdapConfig.SearchMode.USERS);
            if (usernameInLdapRecords == null) {
                log.trace("Found null user {} with base {} filter {} [Thread {}]", new Object[]{userName, this.config.userSearchBase, this.config.userDnSearchFilter, Thread.currentThread().getId()});
                return null;
            }
        } else {
            log.trace("User name attribute not found in search result [Thread {}]", (Object)Thread.currentThread().getId());
            return null;
        }
        String userDn = result.getNameInNamespace();
        log.trace("Found user {} with base {} filter {}: {} [Thread {}]", new Object[]{userName, this.config.userSearchBase, this.config.userDnSearchFilter, userDn, Thread.currentThread().getId()});
        byte[] password = this.userSearchMode == UserSearchMode.DN_SEARCH ? null : (byte[])result.getAttributes().get(this.config.userPasswordAttribute).get();
        return new LdapUser(userDn, password, usernameInLdapRecords);
    }

    public String toString() {
        return this.getClass().getSimpleName() + "@" + Integer.toHexString(this.hashCode());
    }

    private static class DefaultPasswordVerifier
    implements PasswordVerifier {
        private DefaultPasswordVerifier() {
        }

        public void configure(Map<String, ?> configs) {
        }

        @Override
        public PasswordVerifier.Result verify(char[] expectedPassword, char[] actualPassword) {
            log.debug("Verifying passwords using string comparison since no password verifier was found:");
            boolean match = Arrays.equals(expectedPassword, actualPassword);
            return match ? PasswordVerifier.Result.MATCH : PasswordVerifier.Result.MISMATCH;
        }

        @Override
        public void close() throws IOException {
        }
    }

    private static class LdapUser {
        final String dn;
        final byte[] password;
        final String principal;

        LdapUser(String dn, byte[] password, String principal) {
            this.dn = dn;
            this.password = password;
            this.principal = principal;
        }
    }

    static enum UserSearchMode {
        DN_SEARCH,
        PASSWORD_SEARCH;

    }
}

