/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.multitenant;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.confluent.kafka.clients.CloudAdmin;
import io.confluent.kafka.clients.DescribeTenantsOptions;
import io.confluent.kafka.multitenant.KafkaLogicalClusterMetadata;
import io.confluent.kafka.multitenant.schema.TenantContext;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.ConfluentAdmin;
import org.apache.kafka.clients.admin.DeleteClusterLinksOptions;
import org.apache.kafka.clients.admin.DescribeAclsOptions;
import org.apache.kafka.clients.admin.FenceProducersOptions;
import org.apache.kafka.clients.admin.ListClusterLinksOptions;
import org.apache.kafka.clients.admin.ListTopicsOptions;
import org.apache.kafka.clients.admin.ListTransactionsOptions;
import org.apache.kafka.clients.admin.TransactionState;
import org.apache.kafka.clients.admin.internals.ConfluentAdminUtils;
import org.apache.kafka.common.acl.AccessControlEntryFilter;
import org.apache.kafka.common.acl.AclBindingFilter;
import org.apache.kafka.common.acl.AclOperation;
import org.apache.kafka.common.acl.AclPermissionType;
import org.apache.kafka.common.acl.AclState;
import org.apache.kafka.common.config.internals.ConfluentConfigs;
import org.apache.kafka.common.errors.ClusterLinkDisabledException;
import org.apache.kafka.common.errors.InvalidRequestException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.message.DescribeTenantsResponseData;
import org.apache.kafka.common.resource.PatternType;
import org.apache.kafka.common.resource.ResourcePatternFilter;
import org.apache.kafka.common.resource.ResourceType;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TenantLifecycleManager {
    private static final Long CLOSE_TIMEOUT_MS = TimeUnit.SECONDS.toMillis(30L);
    private static final int LIST_METADATA_TIMEOUT_MS = 30000;
    private static final Logger LOG = LoggerFactory.getLogger(TenantLifecycleManager.class);
    private Long deleteDelayMs;
    private int deleteBatchSize;
    private ExecutorService deletionExecutor;
    private CloudAdmin adminClient;
    private AtomicBoolean adminClientCreated = new AtomicBoolean(false);
    private final Time time;
    private final String clientId;
    private Function<String, Admin> adminClientSupplier;
    final Map<String, State> tenantLifecycleState = new ConcurrentHashMap<String, State>();
    private final boolean isCellsEnabled;
    private final boolean enableAclState;

    public TenantLifecycleManager(Map<String, ?> configs, Time time) {
        this.time = time;
        this.clientId = String.format("tenant-lifecycle-manager-admin-%s", configs.get("node.id"));
        Object deleteDelayValue = configs.get("multitenant.tenant.delete.delay");
        this.deleteDelayMs = deleteDelayValue == null ? ConfluentConfigs.MULTITENANT_TENANT_DELETE_DELAY_MS_DEFAULT : Long.valueOf((Long)deleteDelayValue);
        Object deleteTopicBatchSizeValue = configs.get("multitenant.tenant.delete.batch.size");
        this.deleteBatchSize = deleteTopicBatchSizeValue == null ? ConfluentConfigs.MULTITENANT_TENANT_DELETE_BATCH_SIZE_DEFAULT.intValue() : ((Integer)deleteTopicBatchSizeValue).intValue();
        Object isCellsEnabled = configs.get("confluent.cells.enable");
        this.isCellsEnabled = isCellsEnabled == null ? false : (Boolean)isCellsEnabled;
        Object isAclStateEnabled = configs.get("confluent.multitenant.authorizer.enable.acl.state");
        this.enableAclState = isAclStateEnabled == null ? false : (Boolean)isAclStateEnabled;
    }

    public void setAdminSupplierAndCreateClient(Function<String, Admin> adminClientSupplier) {
        this.adminClientSupplier = adminClientSupplier;
        this.createAdminClient();
    }

    public void createAdminClient() {
        if (this.adminClientSupplier != null) {
            try {
                this.adminClient = (CloudAdmin)this.adminClientSupplier.apply(this.clientId);
            }
            catch (Exception e) {
                LOG.warn("Caught exception when creating admin client", (Throwable)e);
            }
        }
        if (this.adminClient != null) {
            LOG.info("Successfully created admin client");
            this.adminClientCreated.compareAndSet(false, true);
        } else {
            LOG.warn("Failed to create admin client, will retry next time the client is invoked");
        }
        this.deletionExecutor = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("tenant-topic-deletion-thread-%d").build());
    }

    TenantLifecycleManager(long deleteDelayMs, Function<String, Admin> adminClientSupplier) {
        this(deleteDelayMs, true, adminClientSupplier, Time.SYSTEM);
    }

    TenantLifecycleManager(long deleteDelayMs, boolean isCellsEnabled, Function<String, Admin> adminClientSupplier, Time time) {
        this.time = time;
        this.clientId = String.format("tenant-lifecycle-manager-admin-%d", time.milliseconds());
        this.deleteDelayMs = deleteDelayMs;
        this.isCellsEnabled = isCellsEnabled;
        this.adminClientSupplier = adminClientSupplier;
        if (this.adminClientSupplier != null) {
            this.adminClient = (CloudAdmin)this.adminClientSupplier.apply(this.clientId);
            if (this.adminClient != null) {
                this.adminClientCreated.compareAndSet(false, true);
            }
        }
        this.deleteBatchSize = ConfluentConfigs.MULTITENANT_TENANT_DELETE_BATCH_SIZE_DEFAULT;
        this.deletionExecutor = Executors.newSingleThreadExecutor(new ThreadFactoryBuilder().setNameFormat("tenant-topic-deletion-thread-%d").build());
        this.enableAclState = false;
    }

    public synchronized void updateTenantState(KafkaLogicalClusterMetadata lcMeta) {
        String lcId = lcMeta.logicalClusterId();
        if (lcMeta.isActive()) {
            State prevState = this.tenantLifecycleState.put(lcId, State.ACTIVE);
            if (prevState == State.DEACTIVATED) {
                LOG.info("Tenant {} was reactivated and will not be deleted", (Object)lcId);
            } else if (prevState == State.DELETE_IN_PROGRESS || prevState == State.DELETED) {
                LOG.warn("Attempted to reactive tenant {} but it was already deleted.", (Object)lcId);
            }
        } else if (this.shouldDelete(lcMeta)) {
            this.tenantLifecycleState.put(lcId, State.DELETE_IN_PROGRESS);
            LOG.warn("Tenant {} was marked for immediate deletion.", (Object)lcId);
        } else if (this.shouldDeactivate(lcMeta)) {
            this.tenantLifecycleState.put(lcId, State.DEACTIVATED);
            LOG.warn("Tenant {} was deactivated and will be deleted in {}.", (Object)lcId, (Object)Duration.ofMillis(this.deleteDelayMs));
        }
    }

    public boolean isClusterActive(String lcId) {
        return this.tenantLifecycleState.get(lcId) == State.ACTIVE;
    }

    public Set<String> inactiveClusters() {
        return this.tenantLifecycleState.entrySet().stream().filter(lc -> lc.getValue() != State.ACTIVE).map(lc -> (String)lc.getKey()).collect(Collectors.toSet());
    }

    public Set<String> deactivatedClusters() {
        return this.tenantLifecycleState.entrySet().stream().filter(lc -> lc.getValue() == State.DEACTIVATED).map(lc -> (String)lc.getKey()).collect(Collectors.toSet());
    }

    Set<String> deleteInProgressClusters() {
        return this.tenantLifecycleState.entrySet().stream().filter(lc -> lc.getValue() == State.DELETE_IN_PROGRESS).map(lc -> (String)lc.getKey()).collect(Collectors.toSet());
    }

    public Set<String> fullyDeletedClusters() {
        return this.tenantLifecycleState.entrySet().stream().filter(lc -> lc.getValue() == State.DELETED).map(lc -> (String)lc.getKey()).collect(Collectors.toSet());
    }

    public Set<String> deletedClusters() {
        return this.tenantLifecycleState.entrySet().stream().filter(lc -> lc.getValue() == State.DELETE_IN_PROGRESS || lc.getValue() == State.DELETED).map(lc -> (String)lc.getKey()).collect(Collectors.toSet());
    }

    public synchronized void deleteTenants() {
        Set<String> deleteInProgressClusters;
        if (!this.adminClientCreated.get()) {
            this.createAdminClient();
            if (!this.adminClientCreated.get()) {
                return;
            }
        }
        if ((deleteInProgressClusters = this.deleteInProgressClusters()).isEmpty()) {
            return;
        }
        LOG.info("Deleting tenants in: {}", deleteInProgressClusters);
        Set<String> tenantsWithNoClusterLinks = this.deleteClusterLinks(deleteInProgressClusters);
        Set<String> tenantsWithNoTransactions = this.abortTransactions(deleteInProgressClusters);
        Sets.SetView tenantsReadyForDeleteTopics = Sets.intersection(tenantsWithNoClusterLinks, tenantsWithNoTransactions);
        Set<String> tenantsWithNoTopics = this.deleteTopics((Set<String>)tenantsReadyForDeleteTopics);
        Set<String> tenantsWithNoACLs = this.deleteAcls(tenantsWithNoClusterLinks);
        Set<String> tenantsWithNoCellMetadata = this.deleteTenantCellMetadata((Set<String>)Sets.intersection(tenantsWithNoACLs, tenantsWithNoTopics));
        for (String tenant : tenantsWithNoCellMetadata) {
            this.tenantLifecycleState.put(tenant, State.DELETED);
        }
    }

    private Set<String> deleteTopics(Set<String> deleteInProgressClusters) {
        List topicsToDelete;
        ImmutableSet tenantsWithNoTopics = new HashSet();
        try {
            ListTopicsOptions listTopicsOptions = new ListTopicsOptions().timeoutMs(Integer.valueOf(30000));
            Set topics = (Set)this.adminClient.listTopics(listTopicsOptions).names().get();
            topicsToDelete = topics.stream().filter(topic -> TenantContext.isTenantPrefixed(topic) && deleteInProgressClusters.contains(TenantContext.extractTenant(topic))).collect(Collectors.toList());
            Set clustersWithTopicsToDelete = topicsToDelete.stream().map(topic -> TenantContext.extractTenant(topic)).collect(Collectors.toSet());
            tenantsWithNoTopics = Sets.difference(deleteInProgressClusters, clustersWithTopicsToDelete).immutableCopy();
            LOG.info("deleting topics {} because they belong to tenants {}", topicsToDelete, clustersWithTopicsToDelete);
        }
        catch (Exception e) {
            LOG.error("Failed to get list of topics in cluster.", (Throwable)e);
            return tenantsWithNoTopics;
        }
        Runnable deleteTopics = () -> {
            List topicBatches = Lists.partition((List)topicsToDelete, (int)this.deleteBatchSize);
            try {
                for (List topicBatch : topicBatches) {
                    try {
                        this.adminClient.deleteTopics((Collection)topicBatch).all().get();
                        LOG.info("Successfully deleted topics {}", (Object)topicBatch);
                    }
                    catch (UnknownTopicOrPartitionException unknownTopicOrPartitionException) {}
                }
            }
            catch (InterruptedException e) {
                LOG.info("Deleting topics of deactivated tenants was interrupted");
            }
            catch (ExecutionException e) {
                LOG.error("Failed to delete topics for tenants {}. We'll try again next time", (Object)deleteInProgressClusters, (Object)e);
            }
        };
        this.deletionExecutor.execute(deleteTopics);
        return tenantsWithNoTopics;
    }

    private Set<String> deleteTenantCellMetadata(Set<String> tenantIds) {
        Set tenantsInCluster;
        if (!this.isCellsEnabled) {
            return tenantIds;
        }
        ImmutableSet tenantsWithNoCellMetadata = new HashSet();
        try {
            DescribeTenantsOptions options = (DescribeTenantsOptions)new DescribeTenantsOptions().timeoutMs(Integer.valueOf(30000));
            tenantsInCluster = ((List)this.adminClient.describeTenants(Collections.emptyList(), options).value().get()).stream().map(DescribeTenantsResponseData.TenantDescription::tenantId).collect(Collectors.toSet());
            tenantsWithNoCellMetadata = Sets.difference(tenantIds, tenantsInCluster).immutableCopy();
            LOG.info("deleting tenant cell metadata {} because they are empty", (Object)tenantsWithNoCellMetadata);
        }
        catch (Exception e) {
            LOG.error("Failed to get list of tenants in cluster.", (Throwable)e);
            return tenantsWithNoCellMetadata;
        }
        Sets.SetView tenantsToDelete = Sets.intersection(tenantIds, tenantsInCluster);
        Runnable deleteTenants = () -> this.lambda$deleteTenantCellMetadata$13((Set)tenantsToDelete);
        this.deletionExecutor.execute(deleteTenants);
        return tenantsWithNoCellMetadata;
    }

    private Set<String> deleteClusterLinks(Set<String> deleteInProgressClusters) {
        List linksToDelete;
        ImmutableSet tenantsWithNoClusterLinks = new HashSet();
        try {
            ListClusterLinksOptions listOptions = (ListClusterLinksOptions)new ListClusterLinksOptions().includeTopics(false).timeoutMs(Integer.valueOf(30000));
            Collection clusterLinks = (Collection)this.adminClient.listClusterLinks(listOptions).result().get();
            linksToDelete = clusterLinks.stream().filter(link -> TenantContext.isTenantPrefixed(link.linkName()) && deleteInProgressClusters.contains(TenantContext.extractTenant(link.linkName()))).map(link -> link.linkName()).collect(Collectors.toList());
            Set clustersWithLinksToDelete = linksToDelete.stream().map(link -> TenantContext.extractTenant(link)).collect(Collectors.toSet());
            tenantsWithNoClusterLinks = Sets.difference(deleteInProgressClusters, clustersWithLinksToDelete).immutableCopy();
            LOG.info("Deleting links {} because they belong to deleted tenants {}", linksToDelete, clustersWithLinksToDelete);
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof ClusterLinkDisabledException) {
                LOG.info("Skip deleting cluster links since cluster link is disabled");
                return deleteInProgressClusters;
            }
            LOG.error("Failed to get list of cluster links in cluster.", (Throwable)e);
            return tenantsWithNoClusterLinks;
        }
        catch (Exception e) {
            LOG.error("Failed to get list of cluster links in cluster.", (Throwable)e);
            return tenantsWithNoClusterLinks;
        }
        if (linksToDelete.isEmpty()) {
            return tenantsWithNoClusterLinks;
        }
        Runnable deleteLinks = () -> {
            List linkBatches = Lists.partition((List)linksToDelete, (int)this.deleteBatchSize);
            DeleteClusterLinksOptions option = new DeleteClusterLinksOptions().force(true);
            try {
                for (List linkBatch : linkBatches) {
                    try {
                        this.adminClient.deleteClusterLinks((Collection)linkBatch, option).all().get();
                        LOG.info("Successfully deleted links {}", (Object)linkBatch);
                    }
                    catch (ExecutionException executionException) {}
                }
            }
            catch (InterruptedException e) {
                LOG.info("Deleting links of deactivated tenants was interrupted");
            }
            catch (Exception e) {
                LOG.error("Failed to delete links for tenants {}. We'll try again next time", (Object)deleteInProgressClusters, (Object)e);
            }
        };
        this.deletionExecutor.execute(deleteLinks);
        return tenantsWithNoClusterLinks;
    }

    private Set<String> abortTransactions(Set<String> deleteInProgressClusters) {
        List transactionsToAbort;
        ImmutableSet tenantsWithNoTransactions = new HashSet();
        try {
            ListTransactionsOptions options = new ListTransactionsOptions().filterStates(Collections.singletonList(TransactionState.ONGOING));
            Collection transactions = (Collection)this.adminClient.listTransactions(options).all().get();
            transactionsToAbort = transactions.stream().filter(txnListing -> TenantContext.isTenantPrefixed(txnListing.transactionalId()) && deleteInProgressClusters.contains(TenantContext.extractTenant(txnListing.transactionalId()))).map(txnListing -> txnListing.transactionalId()).collect(Collectors.toList());
            Set clustersWithTransactionsToAbort = transactionsToAbort.stream().map(txn -> TenantContext.extractTenant(txn)).collect(Collectors.toSet());
            tenantsWithNoTransactions = Sets.difference(deleteInProgressClusters, clustersWithTransactionsToAbort).immutableCopy();
            LOG.info("Aborting transactions with IDs {} because they belong to deleted tenants {}", transactionsToAbort, clustersWithTransactionsToAbort);
        }
        catch (Exception e) {
            LOG.error("Failed to get list of transactions in cluster.", (Throwable)e);
            return tenantsWithNoTransactions;
        }
        if (transactionsToAbort.isEmpty()) {
            return tenantsWithNoTransactions;
        }
        Runnable abortTransactions = () -> {
            List transactionBatches = Lists.partition((List)transactionsToAbort, (int)this.deleteBatchSize);
            FenceProducersOptions option = new FenceProducersOptions();
            try {
                for (List transactionBatch : transactionBatches) {
                    try {
                        this.adminClient.fenceProducers((Collection)transactionBatch, option).all().get();
                        LOG.info("Successfully aborted transactions with transactional IDs {}", (Object)transactionBatch);
                    }
                    catch (ExecutionException e) {
                        LOG.debug("Failed to abort transactions with transactional IDs {} due to error: {}", (Object)transactionBatch, (Object)e.toString());
                    }
                }
            }
            catch (InterruptedException e) {
                LOG.info("Aborting transactions of deactivated tenants was interrupted");
            }
            catch (Exception e) {
                LOG.error("Failed to abort transactions for tenants {}. We'll try again next time", (Object)deleteInProgressClusters, (Object)e);
            }
        };
        this.deletionExecutor.execute(abortTransactions);
        return tenantsWithNoTransactions;
    }

    private Set<String> deleteAcls(Set<String> deleteInProgressClusters) {
        HashSet<String> tenantsWithNoACLs = new HashSet<String>();
        LinkedList<AclBindingFilter> aclFiltersToDelete = new LinkedList<AclBindingFilter>();
        DescribeAclsOptions describeAclsOptions = new DescribeAclsOptions().timeoutMs(Integer.valueOf(30000));
        for (String lc : deleteInProgressClusters) {
            AclBindingFilter tenantFilter = new AclBindingFilter(new ResourcePatternFilter(ResourceType.ANY, lc + "_", PatternType.CONFLUENT_ALL_TENANT_ANY), new AccessControlEntryFilter(null, null, AclOperation.ANY, AclPermissionType.ANY));
            try {
                Collection acls = this.enableAclState ? (Collection)ConfluentAdminUtils.describeAcls((ConfluentAdmin)this.adminClient, (AclBindingFilter)tenantFilter, (DescribeAclsOptions)describeAclsOptions, (AclState)AclState.ANY).values().get() : (Collection)this.adminClient.describeAcls(tenantFilter).values().get();
                if (acls.isEmpty()) {
                    tenantsWithNoACLs.add(lc);
                    continue;
                }
                aclFiltersToDelete.add(tenantFilter);
            }
            catch (Exception e) {
                if (e.getCause() instanceof InvalidRequestException) {
                    LOG.error("Failed to get ACLs for tenants {} because this operation isn't supporting on this physical cluster. We won't retry and will consider deletion of ACLs for all tenants in list complete.", deleteInProgressClusters, (Object)e);
                    tenantsWithNoACLs.addAll(deleteInProgressClusters);
                    return tenantsWithNoACLs;
                }
                LOG.error("Failed to get ACLs for tenants {}. We'll try again next time", deleteInProgressClusters, (Object)e);
                return tenantsWithNoACLs;
            }
        }
        this.adminClient.deleteAcls(aclFiltersToDelete);
        return tenantsWithNoACLs;
    }

    public void close() {
        if (this.deletionExecutor != null) {
            this.deletionExecutor.shutdownNow();
        }
        if (this.adminClient != null) {
            this.adminClient.close(Duration.ofMillis(CLOSE_TIMEOUT_MS));
        }
    }

    private boolean shouldDeactivate(KafkaLogicalClusterMetadata lcMeta) {
        return lcMeta.lifecycleMetadata() != null && lcMeta.lifecycleMetadata().deletionDate() != null && lcMeta.lifecycleMetadata().deletionDate().getTime() < this.time.milliseconds() && this.tenantLifecycleState.getOrDefault(lcMeta.logicalClusterId(), State.ACTIVE) == State.ACTIVE;
    }

    private boolean shouldDelete(KafkaLogicalClusterMetadata lcMeta) {
        Long deleteBeforeTime = this.time.milliseconds() - this.deleteDelayMs;
        return lcMeta.lifecycleMetadata() != null && lcMeta.lifecycleMetadata().deletionDate() != null && lcMeta.lifecycleMetadata().deletionDate().getTime() < deleteBeforeTime && this.tenantLifecycleState.getOrDefault(lcMeta.logicalClusterId(), State.ACTIVE) != State.DELETED;
    }

    ExecutorService deletionExecutor() {
        return this.deletionExecutor;
    }

    private /* synthetic */ void lambda$deleteTenantCellMetadata$13(Set tenantsToDelete) {
        List tenantBatches = Lists.partition(new ArrayList(tenantsToDelete), (int)this.deleteBatchSize);
        try {
            for (List tenantBatch : tenantBatches) {
                this.adminClient.deleteTenants((Collection)tenantBatch).value().get();
                LOG.info("Successfully deleted tenant cell metadata {}", (Object)tenantBatch);
            }
        }
        catch (InterruptedException e) {
            LOG.info("Deleting tenant cell metadata of deactivated tenants was interrupted");
        }
        catch (ExecutionException e) {
            LOG.error("Failed to delete tenant cell metadata for tenants {}. We'll try again next time", (Object)tenantsToDelete, (Object)e.getCause());
        }
    }

    static enum State {
        ACTIVE,
        DEACTIVATED,
        DELETE_IN_PROGRESS,
        DELETED;

    }
}

