/*
 * Decompiled with CFR 0.152.
 */
package kafka.tier.store;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.function.Supplier;
import kafka.tier.exceptions.TierObjectStoreFatalException;
import kafka.tier.exceptions.TierObjectStoreRetriableException;
import kafka.tier.store.TierObjectAttribute;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.TierObjectStoreResponse;
import kafka.tier.store.VersionInformation;
import kafka.tier.store.objects.FragmentType;
import kafka.tier.store.objects.ObjectType;
import kafka.tier.store.objects.metadata.ObjectStoreMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TierObjectStoreFunctionUtils {
    private static final Logger log = LoggerFactory.getLogger(TierObjectStoreFunctionUtils.class);
    static final int MAX_RETRIES = 50;
    static final long MAX_BACKOFF_MS = Duration.ofMinutes(10L).toMillis();

    private static <T> T retryLoop(ObjectStoreOperation<T> function, Supplier<Boolean> isShutdown) throws IOException, InterruptedException {
        int retryCount = 0;
        while (true) {
            if (isShutdown.get().booleanValue()) {
                throw new InterruptedException("Object store call interrupted");
            }
            try {
                return function.run();
            }
            catch (TierObjectStoreRetriableException e) {
                log.warn("Retry object store call after " + retryCount + " attempts", (Throwable)((Object)e));
                if (++retryCount >= 50) {
                    throw new TierObjectStoreFatalException("Failed to run object store call after 50 attempts", (Throwable)((Object)e));
                }
                Thread.sleep(TierObjectStoreFunctionUtils.backOffMs(retryCount));
                continue;
            }
            break;
        }
    }

    public static String putBuffer(Supplier<Boolean> isShutdown, TierObjectStore objectStore, ObjectStoreMetadata metadata, ByteBuffer buffer, ObjectType objectType) throws IOException, InterruptedException {
        return TierObjectStoreFunctionUtils.retryLoop(() -> objectStore.putBuffer(metadata, buffer, objectType), isShutdown);
    }

    public static String putObject(Supplier<Boolean> isShutdown, TierObjectStore objectStore, File file, ObjectStoreMetadata objectMetadata, ObjectType objectType) throws IOException, InterruptedException {
        return TierObjectStoreFunctionUtils.retryLoop(() -> objectStore.putObject(objectMetadata, file, objectType), isShutdown);
    }

    public static TierObjectAttribute objectExists(Supplier<Boolean> isShutdown, TierObjectStore objectStore, ObjectStoreMetadata metadata, ObjectType objectType) throws IOException, InterruptedException {
        return TierObjectStoreFunctionUtils.retryLoop(() -> objectStore.objectExists(metadata, objectType), isShutdown);
    }

    public static TierObjectStoreResponse getObjectStoreFragment(Supplier<Boolean> isShutdown, TierObjectStore objectStore, ObjectStoreMetadata metadata, FragmentType fragmentType) throws IOException, InterruptedException {
        return TierObjectStoreFunctionUtils.retryLoop(() -> objectStore.getObjectStoreFragment(metadata, fragmentType), isShutdown);
    }

    public static Map<String, List<VersionInformation>> listObject(Supplier<Boolean> isShutdown, TierObjectStore objectStore, String keyPrefix, Boolean versionInfo) throws IOException, InterruptedException {
        return TierObjectStoreFunctionUtils.retryLoop(() -> objectStore.listObject(keyPrefix, versionInfo), isShutdown);
    }

    public static long backOffMs(int retryCount) {
        long randomSecond = 1000 + new Random().nextInt(1000);
        return Math.min(MAX_BACKOFF_MS, (long)Math.pow(2.0, retryCount) * randomSecond);
    }

    @FunctionalInterface
    public static interface ObjectStoreOperation<T> {
        public T run() throws IOException;
    }
}

