/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.dekregistry.storage;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Multimaps;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.TreeMultimap;
import com.google.crypto.tink.Aead;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import io.confluent.dekregistry.client.rest.entities.CreateDekRequest;
import io.confluent.dekregistry.client.rest.entities.CreateKekRequest;
import io.confluent.dekregistry.client.rest.entities.Dek;
import io.confluent.dekregistry.client.rest.entities.Kek;
import io.confluent.dekregistry.client.rest.entities.KeyType;
import io.confluent.dekregistry.client.rest.entities.UpdateKekRequest;
import io.confluent.dekregistry.metrics.MetricsManager;
import io.confluent.dekregistry.storage.DataEncryptionKey;
import io.confluent.dekregistry.storage.DataEncryptionKeyId;
import io.confluent.dekregistry.storage.DefaultDekCacheUpdateHandler;
import io.confluent.dekregistry.storage.DekCacheUpdateHandler;
import io.confluent.dekregistry.storage.DekRegistryConfig;
import io.confluent.dekregistry.storage.EncryptionKey;
import io.confluent.dekregistry.storage.EncryptionKeyId;
import io.confluent.dekregistry.storage.KeyEncryptionKey;
import io.confluent.dekregistry.storage.KeyEncryptionKeyId;
import io.confluent.dekregistry.storage.exceptions.AlreadyExistsException;
import io.confluent.dekregistry.storage.exceptions.DekGenerationException;
import io.confluent.dekregistry.storage.exceptions.InvalidKeyException;
import io.confluent.dekregistry.storage.exceptions.KeyNotSoftDeletedException;
import io.confluent.dekregistry.storage.exceptions.KeyReferenceExistsException;
import io.confluent.dekregistry.storage.exceptions.KeySoftDeletedException;
import io.confluent.dekregistry.storage.exceptions.TooManyKeysException;
import io.confluent.dekregistry.storage.serialization.EncryptionKeyIdSerde;
import io.confluent.dekregistry.storage.serialization.EncryptionKeySerde;
import io.confluent.dekregistry.storage.utils.CompositeCacheUpdateHandler;
import io.confluent.dekregistry.web.rest.handlers.EncryptionUpdateRequestHandler;
import io.confluent.kafka.schemaregistry.client.rest.RestService;
import io.confluent.kafka.schemaregistry.client.rest.exceptions.RestClientException;
import io.confluent.kafka.schemaregistry.client.rest.utils.UrlList;
import io.confluent.kafka.schemaregistry.encryption.tink.Cryptor;
import io.confluent.kafka.schemaregistry.encryption.tink.DekFormat;
import io.confluent.kafka.schemaregistry.exceptions.SchemaRegistryException;
import io.confluent.kafka.schemaregistry.exceptions.SchemaRegistryRequestForwardingException;
import io.confluent.kafka.schemaregistry.exceptions.UnknownLeaderException;
import io.confluent.kafka.schemaregistry.rest.handlers.UpdateRequestHandler;
import io.confluent.kafka.schemaregistry.storage.KafkaSchemaRegistry;
import io.confluent.kafka.schemaregistry.storage.Mode;
import io.confluent.kafka.schemaregistry.storage.SchemaRegistry;
import io.confluent.kafka.schemaregistry.utils.JacksonMapper;
import io.confluent.rest.RestConfigException;
import io.confluent.rest.exceptions.RestException;
import io.kcache.Cache;
import io.kcache.CacheUpdateHandler;
import io.kcache.KafkaCache;
import io.kcache.KafkaCacheConfig;
import io.kcache.KeyValue;
import io.kcache.KeyValueIterator;
import io.kcache.exceptions.CacheInitializationException;
import io.kcache.utils.Caches;
import io.kcache.utils.InMemoryCache;
import jakarta.ws.rs.core.UriBuilder;
import java.io.Closeable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import org.apache.kafka.common.serialization.Serde;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class DekRegistry
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(DekRegistry.class);
    public static final String KEY = "dekRegistry";
    public static final int LATEST_VERSION = -1;
    public static final int MIN_VERSION = 1;
    public static final byte[] EMPTY_AAD = new byte[0];
    public static final String X_FORWARD_HEADER = "X-Forward";
    public static final String AWS_KMS = "aws-kms";
    public static final String AZURE_KMS = "azure-kms";
    public static final String GCP_KMS = "gcp-kms";
    private static final String TEST_SUBJECT = "__TEST";
    private static final TypeReference<Kek> KEK_TYPE = new TypeReference<Kek>(){};
    private static final TypeReference<Dek> DEK_TYPE = new TypeReference<Dek>(){};
    private static final TypeReference<Void> VOID_TYPE = new TypeReference<Void>(){};
    private final KafkaSchemaRegistry schemaRegistry;
    private final MetricsManager metricsManager;
    private final DekRegistryConfig config;
    private final int kekSearchDefaultLimit;
    private final int kekSearchMaxLimit;
    private final int dekSubjectSearchDefaultLimit;
    private final int dekSubjectSearchMaxLimit;
    private final int dekVersionSearchDefaultLimit;
    private final int dekVersionSearchMaxLimit;
    final Cache<EncryptionKeyId, EncryptionKey> keys;
    private final SetMultimap<String, KeyEncryptionKeyId> sharedKeys;
    private final Map<DekFormat, Cryptor> cryptors;
    private final Map<String, Lock> tenantToLock = new ConcurrentHashMap<String, Lock>();
    private final AtomicBoolean initialized = new AtomicBoolean();
    private final CountDownLatch initLatch = new CountDownLatch(1);

    @Inject
    public DekRegistry(SchemaRegistry schemaRegistry, MetricsManager metricsManager) {
        try {
            this.schemaRegistry = (KafkaSchemaRegistry)schemaRegistry;
            this.schemaRegistry.properties().put(KEY, this);
            this.schemaRegistry.addUpdateRequestHandler((UpdateRequestHandler)new EncryptionUpdateRequestHandler());
            this.metricsManager = metricsManager;
            this.config = new DekRegistryConfig(schemaRegistry.config().originalProperties());
            this.keys = this.createCache(new EncryptionKeyIdSerde(), new EncryptionKeySerde(), this.config.topic(), this.getCacheUpdateHandler(this.config));
            this.sharedKeys = Multimaps.synchronizedSetMultimap((SetMultimap)TreeMultimap.create());
            this.cryptors = new ConcurrentHashMap<DekFormat, Cryptor>();
            this.kekSearchDefaultLimit = this.config.getInt("kek.search.default.limit");
            this.kekSearchMaxLimit = this.config.getInt("kek.search.max.limit");
            this.dekSubjectSearchDefaultLimit = this.config.getInt("dek.subject.search.default.limit");
            this.dekSubjectSearchMaxLimit = this.config.getInt("dek.subject.search.max.limit");
            this.dekVersionSearchDefaultLimit = this.config.getInt("dek.version.search.default.limit");
            this.dekVersionSearchMaxLimit = this.config.getInt("dek.version.search.max.limit");
        }
        catch (RestConfigException e) {
            throw new IllegalArgumentException("Could not instantiate DekRegistry", e);
        }
    }

    public KafkaSchemaRegistry getSchemaRegistry() {
        return this.schemaRegistry;
    }

    public MetricsManager getMetricsManager() {
        return this.metricsManager;
    }

    protected <K, V> Cache<K, V> createCache(Serde<K> keySerde, Serde<V> valueSerde, String topic, CacheUpdateHandler<K, V> cacheUpdateHandler) throws CacheInitializationException {
        Properties props = this.getKafkaCacheProperties(topic);
        KafkaCacheConfig config = new KafkaCacheConfig((Map)props);
        Cache kafkaCache = Caches.concurrentCache((Cache)new KafkaCache(config, keySerde, valueSerde, cacheUpdateHandler, (Cache)new InMemoryCache()));
        this.getSchemaRegistry().addLeaderChangeListener(isLeader -> {
            if (isLeader.booleanValue()) {
                kafkaCache.reset();
                kafkaCache.sync();
            }
        });
        return kafkaCache;
    }

    private Properties getKafkaCacheProperties(String topic) {
        Properties props = new Properties();
        props.putAll((Map<?, ?>)this.schemaRegistry.config().originalProperties());
        Set<String> keys = props.stringPropertyNames();
        for (String key : keys) {
            String newKey;
            if (!key.startsWith("kafkastore.") || keys.contains(newKey = key.replace("kafkastore", "kafkacache"))) continue;
            props.put(newKey, props.get(key));
        }
        props.put("kafkacache.topic", topic);
        return props;
    }

    private CacheUpdateHandler<EncryptionKeyId, EncryptionKey> getCacheUpdateHandler(DekRegistryConfig config) {
        Map handlerConfigs = config.originalsWithPrefix("dek.registry.update.handlers.");
        handlerConfigs.put(KEY, this);
        List customCacheHandlers = config.getConfiguredInstances("dek.registry.update.handlers", DekCacheUpdateHandler.class, handlerConfigs);
        DefaultDekCacheUpdateHandler cacheHandler = new DefaultDekCacheUpdateHandler(this);
        for (DekCacheUpdateHandler customCacheHandler : customCacheHandlers) {
            log.info("Registering custom cache handler: {}", (Object)customCacheHandler.getClass().getName());
        }
        customCacheHandlers.add(cacheHandler);
        return new CompositeCacheUpdateHandler<EncryptionKeyId, EncryptionKey>(customCacheHandlers);
    }

    public Cache<EncryptionKeyId, EncryptionKey> keys() {
        return this.keys;
    }

    public DekRegistryConfig config() {
        return this.config;
    }

    protected SetMultimap<String, KeyEncryptionKeyId> getSharedKeys() {
        return this.sharedKeys;
    }

    protected Cryptor getCryptor(DekFormat dekFormat) {
        return this.cryptors.computeIfAbsent(dekFormat, k -> {
            try {
                return new Cryptor(dekFormat);
            }
            catch (GeneralSecurityException e) {
                log.error("Invalid format {}", (Object)dekFormat);
                throw new IllegalArgumentException("Invalid format " + String.valueOf(dekFormat), e);
            }
        });
    }

    @PostConstruct
    public void init() {
        if (!this.initialized.get()) {
            this.keys.init();
            boolean isInitialized = this.initialized.compareAndSet(false, true);
            if (!isInitialized) {
                throw new IllegalStateException("DekRegistry was already initialized");
            }
            this.initLatch.countDown();
        }
    }

    public void waitForInit() throws InterruptedException {
        this.initLatch.await();
    }

    public boolean initialized() {
        return this.initialized.get();
    }

    public boolean isLeader() {
        return this.schemaRegistry.isLeader();
    }

    private boolean isLeader(Map<String, String> headerProperties) {
        String forwardHeader = headerProperties.get(X_FORWARD_HEADER);
        return this.isLeader() && (forwardHeader == null || Boolean.parseBoolean(forwardHeader));
    }

    protected Lock lockFor(String tenant) {
        return this.tenantToLock.computeIfAbsent(tenant, k -> new ReentrantLock());
    }

    private void lock(String tenant, Map<String, String> headerProperties) {
        String forwardHeader = headerProperties.get(X_FORWARD_HEADER);
        if (forwardHeader == null || Boolean.parseBoolean(forwardHeader)) {
            this.lockFor(tenant).lock();
        }
    }

    private void unlock(String tenant) {
        if (((ReentrantLock)this.lockFor(tenant)).isHeldByCurrentThread()) {
            this.lockFor(tenant).unlock();
        }
    }

    public List<String> getKekNames(List<String> subjectPrefix, boolean lookupDeleted) {
        String tenant = this.schemaRegistry.tenant();
        if (subjectPrefix == null || subjectPrefix.isEmpty()) {
            return this.getKeks(tenant, lookupDeleted).stream().map(kv -> ((KeyEncryptionKeyId)kv.key).getName()).collect(Collectors.toList());
        }
        return this.getDeks(tenant, lookupDeleted).stream().filter(kv -> subjectPrefix.stream().anyMatch(prefix -> ((DataEncryptionKeyId)kv.key).getSubject().startsWith((String)prefix))).map(kv -> ((DataEncryptionKeyId)kv.key).getKekName()).sorted().distinct().collect(Collectors.toList());
    }

    protected List<KeyValue<EncryptionKeyId, EncryptionKey>> getKeks(String tenant, boolean lookupDeleted) {
        ArrayList<KeyValue<EncryptionKeyId, EncryptionKey>> result = new ArrayList<KeyValue<EncryptionKeyId, EncryptionKey>>();
        KeyEncryptionKeyId key1 = new KeyEncryptionKeyId(tenant, String.valueOf('\u0000'));
        KeyEncryptionKeyId key2 = new KeyEncryptionKeyId(tenant, String.valueOf('\uffff'));
        try (KeyValueIterator iter = this.keys().range((Object)key1, true, (Object)key2, false);){
            while (iter.hasNext()) {
                KeyValue kv = (KeyValue)iter.next();
                if (((EncryptionKey)kv.value).isDeleted() && !lookupDeleted) continue;
                result.add((KeyValue<EncryptionKeyId, EncryptionKey>)kv);
            }
        }
        return result;
    }

    public KeyEncryptionKey getKek(String name, boolean lookupDeleted) {
        String tenant = this.schemaRegistry.tenant();
        KeyEncryptionKeyId keyId = new KeyEncryptionKeyId(tenant, name);
        KeyEncryptionKey key = (KeyEncryptionKey)this.keys.get((Object)keyId);
        if (key != null && (!key.isDeleted() || lookupDeleted)) {
            return key;
        }
        return null;
    }

    public Kek toKekEntity(KeyEncryptionKey kek) {
        return kek.toKekEntity();
    }

    public List<String> getDekSubjects(String kekName, boolean lookupDeleted) {
        String tenant = this.schemaRegistry.tenant();
        return this.getDeks(tenant, kekName, lookupDeleted).stream().map(kv -> ((DataEncryptionKeyId)kv.key).getSubject()).sorted().distinct().collect(Collectors.toList());
    }

    public List<Integer> getDekVersions(String kekName, String subject, DekFormat algorithm, boolean lookupDeleted) {
        String tenant = this.schemaRegistry.tenant();
        return this.getDeks(tenant, kekName, subject, algorithm, lookupDeleted).stream().map(kv -> ((DataEncryptionKeyId)kv.key).getVersion()).collect(Collectors.toList());
    }

    protected List<KeyValue<EncryptionKeyId, EncryptionKey>> getDeks(String tenant, boolean lookupDeleted) {
        return this.getDeks(tenant, String.valueOf('\u0000'), String.valueOf('\uffff'), lookupDeleted);
    }

    protected List<KeyValue<EncryptionKeyId, EncryptionKey>> getDeks(String tenant, String kekName, boolean lookupDeleted) {
        return this.getDeks(tenant, kekName, kekName, lookupDeleted);
    }

    protected List<KeyValue<EncryptionKeyId, EncryptionKey>> getDeks(String tenant, String minKekName, String maxKekName, boolean lookupDeleted) {
        ArrayList<KeyValue<EncryptionKeyId, EncryptionKey>> result = new ArrayList<KeyValue<EncryptionKeyId, EncryptionKey>>();
        DataEncryptionKeyId key1 = new DataEncryptionKeyId(tenant, minKekName, ":.:", DekFormat.AES128_GCM, 1);
        DataEncryptionKeyId key2 = new DataEncryptionKeyId(tenant, maxKekName, ":.\uffff:", DekFormat.AES256_SIV, Integer.MAX_VALUE);
        try (KeyValueIterator iter = this.keys().range((Object)key1, true, (Object)key2, false);){
            while (iter.hasNext()) {
                KeyValue kv = (KeyValue)iter.next();
                if (((EncryptionKey)kv.value).isDeleted() && !lookupDeleted) continue;
                result.add((KeyValue<EncryptionKeyId, EncryptionKey>)kv);
            }
        }
        return result;
    }

    protected List<KeyValue<EncryptionKeyId, EncryptionKey>> getDeks(String tenant, String kekName, String subject, DekFormat algorithm, boolean lookupDeleted) {
        if (algorithm == null) {
            algorithm = DekFormat.AES256_GCM;
        }
        ArrayList<KeyValue<EncryptionKeyId, EncryptionKey>> result = new ArrayList<KeyValue<EncryptionKeyId, EncryptionKey>>();
        DataEncryptionKeyId key1 = new DataEncryptionKeyId(tenant, kekName, subject, algorithm, 1);
        DataEncryptionKeyId key2 = new DataEncryptionKeyId(tenant, kekName, subject, algorithm, Integer.MAX_VALUE);
        try (KeyValueIterator iter = this.keys().range((Object)key1, true, (Object)key2, false);){
            while (iter.hasNext()) {
                KeyValue kv = (KeyValue)iter.next();
                DataEncryptionKeyId key = (DataEncryptionKeyId)kv.key;
                if (((EncryptionKey)kv.value).isDeleted() && !lookupDeleted) continue;
                result.add((KeyValue<EncryptionKeyId, EncryptionKey>)kv);
            }
        }
        return result;
    }

    public DataEncryptionKey getLatestDek(String kekName, String subject, DekFormat algorithm, boolean lookupDeleted) throws SchemaRegistryException {
        return this.getLatestDek(kekName, subject, algorithm, lookupDeleted, true);
    }

    public DataEncryptionKey getLatestDek(String kekName, String subject, DekFormat algorithm, boolean lookupDeleted, boolean maybeGenerateRawDek) throws SchemaRegistryException {
        String tenant = this.schemaRegistry.tenant();
        List<KeyValue<EncryptionKeyId, EncryptionKey>> deks = this.getDeks(tenant, kekName, subject, algorithm, lookupDeleted);
        Collections.reverse(deks);
        if (deks.isEmpty()) {
            return null;
        }
        DataEncryptionKey dek = (DataEncryptionKey)deks.get((int)0).value;
        return maybeGenerateRawDek ? this.maybeGenerateRawDek(dek) : dek;
    }

    public DataEncryptionKey getDek(String kekName, String subject, int version, DekFormat algorithm, boolean lookupDeleted) throws SchemaRegistryException {
        DataEncryptionKeyId keyId;
        DataEncryptionKey key;
        if (version == -1) {
            return this.getLatestDek(kekName, subject, algorithm, lookupDeleted);
        }
        String tenant = this.schemaRegistry.tenant();
        if (algorithm == null) {
            algorithm = DekFormat.AES256_GCM;
        }
        if ((key = (DataEncryptionKey)this.keys.get((Object)(keyId = new DataEncryptionKeyId(tenant, kekName, subject, algorithm, version)))) != null && (!key.isDeleted() || lookupDeleted)) {
            return this.maybeGenerateRawDek(key);
        }
        return null;
    }

    private DataEncryptionKey maybeGenerateRawDek(DataEncryptionKey key) throws SchemaRegistryException {
        KeyEncryptionKey kek = this.getKek(key.getKekName(), true);
        if (kek.isShared()) {
            key = this.generateRawDek(kek, key);
        }
        return key;
    }

    public Kek createKekOrForward(CreateKekRequest request, Map<String, String> headerProperties) throws SchemaRegistryException {
        String tenant = this.schemaRegistry.tenant();
        this.lock(tenant, headerProperties);
        try {
            if (this.isLeader(headerProperties)) {
                Kek kek = this.toKekEntity(this.createKek(request));
                return kek;
            }
            if (this.schemaRegistry.leaderIdentity() != null) {
                Kek kek = this.forwardCreateKekRequestToLeader(request, headerProperties);
                return kek;
            }
            throw new UnknownLeaderException("Request failed since leader is unknown");
        }
        finally {
            this.unlock(tenant);
        }
    }

    private Kek forwardCreateKekRequestToLeader(CreateKekRequest request, Map<String, String> headerProperties) throws SchemaRegistryRequestForwardingException {
        RestService leaderRestService = this.schemaRegistry.leaderRestService();
        UrlList baseUrl = leaderRestService.getBaseUrls();
        UriBuilder builder = UriBuilder.fromPath((String)"/dek-registry/v1/keks");
        String path = builder.build(new Object[0]).toString();
        log.debug(String.format("Forwarding create key request to %s", baseUrl));
        try {
            return (Kek)leaderRestService.httpRequest(path, "POST", DekRegistry.toJson(request), headerProperties, KEK_TYPE);
        }
        catch (IOException e) {
            throw new SchemaRegistryRequestForwardingException(String.format("Unexpected error while forwarding the create key request to %s", baseUrl), (Throwable)e);
        }
        catch (RestClientException e) {
            throw new RestException(e.getMessage(), e.getStatus(), e.getErrorCode(), (Throwable)e);
        }
    }

    public KeyEncryptionKey createKek(CreateKekRequest request) throws SchemaRegistryException {
        this.keys.sync();
        String tenant = this.schemaRegistry.tenant();
        if (this.metricsManager.getKeyCount(tenant, KeyType.KEK) > (long)this.config.maxKeys()) {
            throw new TooManyKeysException(KeyType.KEK.name());
        }
        String kmsType = this.normalizeKmsType(request.getKmsType());
        TreeMap<String, String> kmsProps = request.getKmsProps() != null ? new TreeMap<String, String>(request.getKmsProps()) : Collections.emptySortedMap();
        KeyEncryptionKey key = new KeyEncryptionKey(request.getName(), kmsType, request.getKmsKeyId(), kmsProps, request.getDoc(), request.isShared(), request.isDeleted());
        KeyEncryptionKeyId keyId = new KeyEncryptionKeyId(tenant, request.getName());
        KeyEncryptionKey oldKey = (KeyEncryptionKey)this.keys.get((Object)keyId);
        if (!(oldKey == null || request.isDeleted() != oldKey.isDeleted() && oldKey.isEquivalent(key))) {
            throw new AlreadyExistsException(request.getName());
        }
        this.keys.put((Object)keyId, (Object)key);
        key = (KeyEncryptionKey)this.keys.get((Object)keyId);
        return key;
    }

    private String normalizeKmsType(String kmsType) {
        String type = kmsType.toLowerCase(Locale.ROOT);
        if (type.startsWith("aws")) {
            return AWS_KMS;
        }
        if (type.startsWith("azure")) {
            return AZURE_KMS;
        }
        if (type.startsWith("gcp")) {
            return GCP_KMS;
        }
        return kmsType;
    }

    public void testKek(KeyEncryptionKey kek) throws SchemaRegistryException {
        DataEncryptionKey key = new DataEncryptionKey(kek.getName(), TEST_SUBJECT, DekFormat.AES256_GCM, 1, null, false);
        if (!kek.isShared()) {
            throw new InvalidKeyException("shared");
        }
        this.generateEncryptedDek(kek, key);
    }

    public Dek createDekOrForward(String kekName, CreateDekRequest request, Map<String, String> headerProperties) throws SchemaRegistryException {
        String tenant = this.schemaRegistry.tenant();
        this.lock(tenant, headerProperties);
        try {
            if (this.isLeader(headerProperties)) {
                Dek dek = this.createDek(kekName, request).toDekEntity();
                return dek;
            }
            if (this.schemaRegistry.leaderIdentity() != null) {
                Dek dek = this.forwardCreateDekRequestToLeader(kekName, request, headerProperties);
                return dek;
            }
            throw new UnknownLeaderException("Request failed since leader is unknown");
        }
        finally {
            this.unlock(tenant);
        }
    }

    private Dek forwardCreateDekRequestToLeader(String kekName, CreateDekRequest request, Map<String, String> headerProperties) throws SchemaRegistryRequestForwardingException {
        RestService leaderRestService = this.schemaRegistry.leaderRestService();
        UrlList baseUrl = leaderRestService.getBaseUrls();
        UriBuilder builder = UriBuilder.fromPath((String)"/dek-registry/v1/keks/{name}/deks/{subject}");
        String path = builder.build(new Object[]{kekName, request.getSubject()}).toString();
        log.debug(String.format("Forwarding create key request to %s", baseUrl));
        try {
            return (Dek)leaderRestService.httpRequest(path, "POST", DekRegistry.toJson(request), headerProperties, DEK_TYPE);
        }
        catch (IOException e) {
            throw new SchemaRegistryRequestForwardingException(String.format("Unexpected error while forwarding the create key request to %s", baseUrl), (Throwable)e);
        }
        catch (RestClientException e) {
            throw new RestException(e.getMessage(), e.getStatus(), e.getErrorCode(), (Throwable)e);
        }
    }

    public DataEncryptionKey createDek(String kekName, CreateDekRequest request) throws SchemaRegistryException {
        Mode mode;
        DataEncryptionKeyId keyId;
        DataEncryptionKey oldKey;
        this.keys.sync();
        String tenant = this.schemaRegistry.tenant();
        if (this.metricsManager.getKeyCount(tenant, KeyType.DEK) > (long)this.config.maxKeys()) {
            throw new TooManyKeysException(KeyType.DEK.name());
        }
        DekFormat algorithm = request.getAlgorithm() != null ? request.getAlgorithm() : DekFormat.AES256_GCM;
        int version = request.getVersion() != null ? request.getVersion() : 1;
        DataEncryptionKey key = new DataEncryptionKey(kekName, request.getSubject(), algorithm, version, request.getEncryptedKeyMaterial(), request.isDeleted());
        KeyEncryptionKey kek = this.getKek(key.getKekName(), true);
        if (key.getEncryptedKeyMaterial() == null) {
            if (kek.isShared()) {
                key = this.generateEncryptedDek(kek, key);
            } else {
                throw new InvalidKeyException("encryptedKeyMaterial");
            }
        }
        if (!((oldKey = (DataEncryptionKey)this.keys.get((Object)(keyId = new DataEncryptionKeyId(tenant, kekName, request.getSubject(), algorithm, version)))) == null || request.isDeleted() != oldKey.isDeleted() && oldKey.isEquivalent(key))) {
            throw new AlreadyExistsException(request.getSubject());
        }
        this.keys.put((Object)keyId, (Object)key);
        key = (DataEncryptionKey)this.keys.get((Object)keyId);
        if (kek.isShared() && (mode = this.schemaRegistry.getModeInScope(request.getSubject())) != Mode.IMPORT) {
            key = this.generateRawDek(kek, key);
        }
        return key;
    }

    protected DataEncryptionKey generateEncryptedDek(KeyEncryptionKey kek, DataEncryptionKey key) throws DekGenerationException {
        try {
            Aead aead = this.getAead(kek);
            byte[] rawDek = this.getCryptor(key.getAlgorithm()).generateKey();
            byte[] encryptedDek = aead.encrypt(rawDek, EMPTY_AAD);
            String encryptedDekStr = new String(Base64.getEncoder().encode(encryptedDek), StandardCharsets.UTF_8);
            key = new DataEncryptionKey(key.getKekName(), key.getSubject(), key.getAlgorithm(), key.getVersion(), encryptedDekStr, key.isDeleted());
            return key;
        }
        catch (GeneralSecurityException e) {
            String msg = "Could not generate encrypted dek for " + key.getSubject();
            log.error(msg, (Throwable)e);
            msg = msg + ": " + e.getMessage();
            Throwable cause = e.getCause();
            if (cause != null) {
                msg = msg + ": " + cause.getMessage();
            }
            throw new DekGenerationException(msg);
        }
    }

    protected DataEncryptionKey generateRawDek(KeyEncryptionKey kek, DataEncryptionKey key) throws DekGenerationException {
        try {
            Aead aead = this.getAead(kek);
            byte[] encryptedDek = Base64.getDecoder().decode(key.getEncryptedKeyMaterial().getBytes(StandardCharsets.UTF_8));
            byte[] rawDek = aead.decrypt(encryptedDek, EMPTY_AAD);
            String rawDekStr = new String(Base64.getEncoder().encode(rawDek), StandardCharsets.UTF_8);
            DataEncryptionKey newKey = new DataEncryptionKey(key.getKekName(), key.getSubject(), key.getAlgorithm(), key.getVersion(), key.getEncryptedKeyMaterial(), key.isDeleted());
            newKey.setKeyMaterial(rawDekStr);
            newKey.setTimestamp(key.getTimestamp());
            return newKey;
        }
        catch (GeneralSecurityException e) {
            String msg = "Could not generate raw dek for " + key.getSubject();
            log.error(msg, (Throwable)e);
            msg = msg + ": " + e.getMessage();
            Throwable cause = e.getCause();
            if (cause != null) {
                msg = msg + ": " + cause.getMessage();
            }
            throw new DekGenerationException(msg);
        }
    }

    protected Aead getAead(KeyEncryptionKey kek) throws GeneralSecurityException {
        return this.toKekEntity(kek).toAead(this.config.originals());
    }

    public Kek putKekOrForward(String name, UpdateKekRequest request, Map<String, String> headerProperties) throws SchemaRegistryException {
        String tenant = this.schemaRegistry.tenant();
        this.lock(tenant, headerProperties);
        try {
            if (this.isLeader(headerProperties)) {
                KeyEncryptionKey kek = this.putKek(name, request);
                Kek kek2 = kek != null ? this.toKekEntity(kek) : null;
                return kek2;
            }
            if (this.schemaRegistry.leaderIdentity() != null) {
                Kek kek = this.forwardPutKekRequestToLeader(name, request, headerProperties);
                return kek;
            }
            throw new UnknownLeaderException("Request failed since leader is unknown");
        }
        finally {
            this.unlock(tenant);
        }
    }

    private Kek forwardPutKekRequestToLeader(String name, UpdateKekRequest request, Map<String, String> headerProperties) throws SchemaRegistryRequestForwardingException {
        RestService leaderRestService = this.schemaRegistry.leaderRestService();
        UrlList baseUrl = leaderRestService.getBaseUrls();
        UriBuilder builder = UriBuilder.fromPath((String)"/dek-registry/v1/keks/{name}");
        String path = builder.build(new Object[]{name}).toString();
        log.debug(String.format("Forwarding put key request to %s", baseUrl));
        try {
            return (Kek)leaderRestService.httpRequest(path, "PUT", DekRegistry.toJson(request), headerProperties, KEK_TYPE);
        }
        catch (IOException e) {
            throw new SchemaRegistryRequestForwardingException(String.format("Unexpected error while forwarding the put key request to %s", baseUrl), (Throwable)e);
        }
        catch (RestClientException e) {
            throw new RestException(e.getMessage(), e.getStatus(), e.getErrorCode(), (Throwable)e);
        }
    }

    public KeyEncryptionKey putKek(String name, UpdateKekRequest request) throws SchemaRegistryException {
        this.keys.sync();
        String tenant = this.schemaRegistry.tenant();
        KeyEncryptionKeyId keyId = new KeyEncryptionKeyId(tenant, name);
        KeyEncryptionKey key = (KeyEncryptionKey)this.keys.get((Object)keyId);
        if (key == null || key.isDeleted()) {
            return null;
        }
        TreeMap<String, String> kmsProps = request.getOptionalKmsProps() != null ? (request.getOptionalKmsProps().isPresent() ? new TreeMap<String, String>(request.getKmsProps()) : null) : key.getKmsProps();
        String doc = request.getOptionalDoc() != null ? request.getDoc() : key.getDoc();
        boolean shared = request.isOptionalShared() != null ? request.isShared().booleanValue() : key.isShared();
        KeyEncryptionKey newKey = new KeyEncryptionKey(name, key.getKmsType(), key.getKmsKeyId(), kmsProps, doc, shared, false);
        if (newKey.isEquivalent(key)) {
            return key;
        }
        this.keys.put((Object)keyId, (Object)newKey);
        newKey = (KeyEncryptionKey)this.keys.get((Object)keyId);
        return newKey;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteKekOrForward(String name, boolean permanentDelete, Map<String, String> headerProperties) throws SchemaRegistryException {
        block5: {
            String tenant = this.schemaRegistry.tenant();
            this.lock(tenant, headerProperties);
            try {
                if (this.isLeader(headerProperties)) {
                    this.deleteKek(name, permanentDelete);
                    break block5;
                }
                if (this.schemaRegistry.leaderIdentity() != null) {
                    this.forwardDeleteKekRequestToLeader(name, permanentDelete, headerProperties);
                    break block5;
                }
                throw new UnknownLeaderException("Request failed since leader is unknown");
            }
            finally {
                this.unlock(tenant);
            }
        }
    }

    private void forwardDeleteKekRequestToLeader(String name, boolean permanentDelete, Map<String, String> headerProperties) throws SchemaRegistryRequestForwardingException {
        RestService leaderRestService = this.schemaRegistry.leaderRestService();
        UrlList baseUrl = leaderRestService.getBaseUrls();
        UriBuilder builder = UriBuilder.fromPath((String)"/dek-registry/v1/keks/{name}").queryParam("permanent", new Object[]{permanentDelete});
        String path = builder.build(new Object[]{name}).toString();
        log.debug(String.format("Forwarding delete key request to %s", baseUrl));
        try {
            leaderRestService.httpRequest(path, "DELETE", null, headerProperties, VOID_TYPE);
        }
        catch (IOException e) {
            throw new SchemaRegistryRequestForwardingException(String.format("Unexpected error while forwarding the delete key request to %s", baseUrl), (Throwable)e);
        }
        catch (RestClientException e) {
            throw new RestException(e.getMessage(), e.getStatus(), e.getErrorCode(), (Throwable)e);
        }
    }

    public void deleteKek(String name, boolean permanentDelete) throws SchemaRegistryException {
        this.keys.sync();
        String tenant = this.schemaRegistry.tenant();
        if (!this.getDeks(tenant, name, permanentDelete).isEmpty()) {
            throw new KeyReferenceExistsException(name);
        }
        KeyEncryptionKeyId keyId = new KeyEncryptionKeyId(tenant, name);
        KeyEncryptionKey oldKey = (KeyEncryptionKey)this.keys.get((Object)keyId);
        if (oldKey == null) {
            return;
        }
        if (permanentDelete) {
            if (!oldKey.isDeleted()) {
                throw new KeyNotSoftDeletedException(name);
            }
            this.keys.remove((Object)keyId);
        } else if (!oldKey.isDeleted()) {
            KeyEncryptionKey newKey = new KeyEncryptionKey(name, oldKey.getKmsType(), oldKey.getKmsKeyId(), oldKey.getKmsProps(), oldKey.getDoc(), oldKey.isShared(), true);
            this.keys.put((Object)keyId, (Object)newKey);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteDekOrForward(String name, String subject, DekFormat algorithm, boolean permanentDelete, Map<String, String> headerProperties) throws SchemaRegistryException {
        block5: {
            String tenant = this.schemaRegistry.tenant();
            this.lock(tenant, headerProperties);
            try {
                if (this.isLeader(headerProperties)) {
                    this.deleteDek(name, subject, algorithm, permanentDelete);
                    break block5;
                }
                if (this.schemaRegistry.leaderIdentity() != null) {
                    this.forwardDeleteDekRequestToLeader(name, subject, algorithm, permanentDelete, headerProperties);
                    break block5;
                }
                throw new UnknownLeaderException("Request failed since leader is unknown");
            }
            finally {
                this.unlock(tenant);
            }
        }
    }

    private void forwardDeleteDekRequestToLeader(String name, String subject, DekFormat algorithm, boolean permanentDelete, Map<String, String> headerProperties) throws SchemaRegistryRequestForwardingException {
        RestService leaderRestService = this.schemaRegistry.leaderRestService();
        UrlList baseUrl = leaderRestService.getBaseUrls();
        UriBuilder builder = UriBuilder.fromPath((String)"/dek-registry/v1/keks/{name}/deks/{subject}").queryParam("permanent", new Object[]{permanentDelete});
        if (algorithm != null) {
            builder = builder.queryParam("algorithm", new Object[]{algorithm.name()});
        }
        String path = builder.build(new Object[]{name, subject}).toString();
        log.debug(String.format("Forwarding delete key request to %s", baseUrl));
        try {
            leaderRestService.httpRequest(path, "DELETE", null, headerProperties, VOID_TYPE);
        }
        catch (IOException e) {
            throw new SchemaRegistryRequestForwardingException(String.format("Unexpected error while forwarding the delete key request to %s", baseUrl), (Throwable)e);
        }
        catch (RestClientException e) {
            throw new RestException(e.getMessage(), e.getStatus(), e.getErrorCode(), (Throwable)e);
        }
    }

    public void deleteDek(String name, String subject, DekFormat algorithm, boolean permanentDelete) throws SchemaRegistryException {
        this.keys.sync();
        String tenant = this.schemaRegistry.tenant();
        List<KeyValue<EncryptionKeyId, EncryptionKey>> deks = this.getDeks(tenant, name, subject, algorithm, permanentDelete);
        if (permanentDelete) {
            for (KeyValue<EncryptionKeyId, EncryptionKey> dek : deks) {
                if (((EncryptionKey)dek.value).isDeleted()) continue;
                DataEncryptionKeyId keyId = (DataEncryptionKeyId)dek.key;
                throw new KeyNotSoftDeletedException(keyId.getSubject());
            }
            for (KeyValue<EncryptionKeyId, EncryptionKey> dek : deks) {
                DataEncryptionKeyId keyId = (DataEncryptionKeyId)dek.key;
                this.keys.remove((Object)keyId);
            }
        } else {
            for (KeyValue<EncryptionKeyId, EncryptionKey> dek : deks) {
                if (((EncryptionKey)dek.value).isDeleted()) continue;
                DataEncryptionKeyId id = (DataEncryptionKeyId)dek.key;
                DataEncryptionKey oldKey = (DataEncryptionKey)dek.value;
                DataEncryptionKey newKey = new DataEncryptionKey(name, oldKey.getSubject(), oldKey.getAlgorithm(), oldKey.getVersion(), oldKey.getEncryptedKeyMaterial(), true);
                this.keys.put((Object)id, (Object)newKey);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteDekVersionOrForward(String name, String subject, int version, DekFormat algorithm, boolean permanentDelete, Map<String, String> headerProperties) throws SchemaRegistryException {
        block5: {
            String tenant = this.schemaRegistry.tenant();
            this.lock(tenant, headerProperties);
            try {
                if (this.isLeader(headerProperties)) {
                    this.deleteDekVersion(name, subject, version, algorithm, permanentDelete);
                    break block5;
                }
                if (this.schemaRegistry.leaderIdentity() != null) {
                    this.forwardDeleteDekVersionRequestToLeader(name, subject, version, algorithm, permanentDelete, headerProperties);
                    break block5;
                }
                throw new UnknownLeaderException("Request failed since leader is unknown");
            }
            finally {
                this.unlock(tenant);
            }
        }
    }

    private void forwardDeleteDekVersionRequestToLeader(String name, String subject, int version, DekFormat algorithm, boolean permanentDelete, Map<String, String> headerProperties) throws SchemaRegistryRequestForwardingException {
        RestService leaderRestService = this.schemaRegistry.leaderRestService();
        UrlList baseUrl = leaderRestService.getBaseUrls();
        UriBuilder builder = UriBuilder.fromPath((String)"/dek-registry/v1/keks/{name}/deks/{subject}/versions/{version}").queryParam("permanent", new Object[]{permanentDelete});
        if (algorithm != null) {
            builder = builder.queryParam("algorithm", new Object[]{algorithm.name()});
        }
        String path = builder.build(new Object[]{name, subject, version}).toString();
        log.debug(String.format("Forwarding delete key version request to %s", baseUrl));
        try {
            leaderRestService.httpRequest(path, "DELETE", null, headerProperties, VOID_TYPE);
        }
        catch (IOException e) {
            throw new SchemaRegistryRequestForwardingException(String.format("Unexpected error while forwarding the delete key version request to %s", baseUrl), (Throwable)e);
        }
        catch (RestClientException e) {
            throw new RestException(e.getMessage(), e.getStatus(), e.getErrorCode(), (Throwable)e);
        }
    }

    public void deleteDekVersion(String name, String subject, int version, DekFormat algorithm, boolean permanentDelete) throws SchemaRegistryException {
        String tenant;
        DataEncryptionKeyId id;
        DataEncryptionKey key;
        this.keys.sync();
        if (algorithm == null) {
            algorithm = DekFormat.AES256_GCM;
        }
        if ((key = (DataEncryptionKey)this.keys.get((Object)(id = new DataEncryptionKeyId(tenant = this.schemaRegistry.tenant(), name, subject, algorithm, version)))) == null) {
            return;
        }
        if (permanentDelete) {
            if (!key.isDeleted()) {
                throw new KeyNotSoftDeletedException(subject);
            }
            this.keys.remove((Object)id);
        } else if (!key.isDeleted()) {
            DataEncryptionKey newKey = new DataEncryptionKey(name, key.getSubject(), key.getAlgorithm(), key.getVersion(), key.getEncryptedKeyMaterial(), true);
            this.keys.put((Object)id, (Object)newKey);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void undeleteKekOrForward(String name, Map<String, String> headerProperties) throws SchemaRegistryException {
        block5: {
            String tenant = this.schemaRegistry.tenant();
            this.lock(tenant, headerProperties);
            try {
                if (this.isLeader(headerProperties)) {
                    this.undeleteKek(name);
                    break block5;
                }
                if (this.schemaRegistry.leaderIdentity() != null) {
                    this.forwardUndeleteKekRequestToLeader(name, headerProperties);
                    break block5;
                }
                throw new UnknownLeaderException("Request failed since leader is unknown");
            }
            finally {
                this.unlock(tenant);
            }
        }
    }

    private void forwardUndeleteKekRequestToLeader(String name, Map<String, String> headerProperties) throws SchemaRegistryRequestForwardingException {
        RestService leaderRestService = this.schemaRegistry.leaderRestService();
        UrlList baseUrl = leaderRestService.getBaseUrls();
        UriBuilder builder = UriBuilder.fromPath((String)"/dek-registry/v1/keks/{name}/undelete");
        String path = builder.build(new Object[]{name}).toString();
        log.debug(String.format("Forwarding undelete key request to %s", baseUrl));
        try {
            leaderRestService.httpRequest(path, "POST", null, headerProperties, VOID_TYPE);
        }
        catch (IOException e) {
            throw new SchemaRegistryRequestForwardingException(String.format("Unexpected error while forwarding the undelete key request to %s", baseUrl), (Throwable)e);
        }
        catch (RestClientException e) {
            throw new RestException(e.getMessage(), e.getStatus(), e.getErrorCode(), (Throwable)e);
        }
    }

    public void undeleteKek(String name) throws SchemaRegistryException {
        this.keys.sync();
        String tenant = this.schemaRegistry.tenant();
        KeyEncryptionKeyId keyId = new KeyEncryptionKeyId(tenant, name);
        KeyEncryptionKey key = (KeyEncryptionKey)this.keys.get((Object)keyId);
        if (key == null) {
            return;
        }
        if (key.isDeleted()) {
            KeyEncryptionKey newKey = new KeyEncryptionKey(name, key.getKmsType(), key.getKmsKeyId(), key.getKmsProps(), key.getDoc(), key.isShared(), false);
            this.keys.put((Object)keyId, (Object)newKey);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void undeleteDekOrForward(String name, String subject, DekFormat algorithm, Map<String, String> headerProperties) throws SchemaRegistryException {
        block5: {
            String tenant = this.schemaRegistry.tenant();
            this.lock(tenant, headerProperties);
            try {
                if (this.isLeader(headerProperties)) {
                    this.undeleteDek(name, subject, algorithm);
                    break block5;
                }
                if (this.schemaRegistry.leaderIdentity() != null) {
                    this.forwardUndeleteDekRequestToLeader(name, subject, algorithm, headerProperties);
                    break block5;
                }
                throw new UnknownLeaderException("Request failed since leader is unknown");
            }
            finally {
                this.unlock(tenant);
            }
        }
    }

    private void forwardUndeleteDekRequestToLeader(String name, String subject, DekFormat algorithm, Map<String, String> headerProperties) throws SchemaRegistryRequestForwardingException {
        RestService leaderRestService = this.schemaRegistry.leaderRestService();
        UrlList baseUrl = leaderRestService.getBaseUrls();
        UriBuilder builder = UriBuilder.fromPath((String)"/dek-registry/v1/keks/{name}/deks/{subject}/undelete");
        if (algorithm != null) {
            builder = builder.queryParam("algorithm", new Object[]{algorithm.name()});
        }
        String path = builder.build(new Object[]{name, subject}).toString();
        log.debug(String.format("Forwarding undelete key request to %s", baseUrl));
        try {
            leaderRestService.httpRequest(path, "POST", null, headerProperties, VOID_TYPE);
        }
        catch (IOException e) {
            throw new SchemaRegistryRequestForwardingException(String.format("Unexpected error while forwarding the undelete key request to %s", baseUrl), (Throwable)e);
        }
        catch (RestClientException e) {
            throw new RestException(e.getMessage(), e.getStatus(), e.getErrorCode(), (Throwable)e);
        }
    }

    public void undeleteDek(String name, String subject, DekFormat algorithm) throws SchemaRegistryException {
        this.keys.sync();
        String tenant = this.schemaRegistry.tenant();
        KeyEncryptionKeyId keyId = new KeyEncryptionKeyId(tenant, name);
        KeyEncryptionKey key = (KeyEncryptionKey)this.keys.get((Object)keyId);
        if (key == null) {
            return;
        }
        if (key.isDeleted()) {
            throw new KeySoftDeletedException(name);
        }
        List<KeyValue<EncryptionKeyId, EncryptionKey>> deks = this.getDeks(tenant, name, subject, algorithm, true);
        for (KeyValue<EncryptionKeyId, EncryptionKey> dek : deks) {
            DataEncryptionKeyId id = (DataEncryptionKeyId)dek.key;
            DataEncryptionKey oldKey = (DataEncryptionKey)dek.value;
            if (!oldKey.isDeleted()) continue;
            DataEncryptionKey newKey = new DataEncryptionKey(name, oldKey.getSubject(), oldKey.getAlgorithm(), oldKey.getVersion(), oldKey.getEncryptedKeyMaterial(), false);
            this.keys.put((Object)id, (Object)newKey);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void undeleteDekVersionOrForward(String name, String subject, int version, DekFormat algorithm, Map<String, String> headerProperties) throws SchemaRegistryException {
        block5: {
            String tenant = this.schemaRegistry.tenant();
            this.lock(tenant, headerProperties);
            try {
                if (this.isLeader(headerProperties)) {
                    this.undeleteDekVersion(name, subject, version, algorithm);
                    break block5;
                }
                if (this.schemaRegistry.leaderIdentity() != null) {
                    this.forwardUndeleteDekVersionRequestToLeader(name, subject, version, algorithm, headerProperties);
                    break block5;
                }
                throw new UnknownLeaderException("Request failed since leader is unknown");
            }
            finally {
                this.unlock(tenant);
            }
        }
    }

    private void forwardUndeleteDekVersionRequestToLeader(String name, String subject, int version, DekFormat algorithm, Map<String, String> headerProperties) throws SchemaRegistryRequestForwardingException {
        RestService leaderRestService = this.schemaRegistry.leaderRestService();
        UrlList baseUrl = leaderRestService.getBaseUrls();
        UriBuilder builder = UriBuilder.fromPath((String)"/dek-registry/v1/keks/{name}/deks/{subject}/versions/{version}/undelete");
        if (algorithm != null) {
            builder = builder.queryParam("algorithm", new Object[]{algorithm.name()});
        }
        String path = builder.build(new Object[]{name, subject, version}).toString();
        log.debug(String.format("Forwarding undelete key version request to %s", baseUrl));
        try {
            leaderRestService.httpRequest(path, "POST", null, headerProperties, VOID_TYPE);
        }
        catch (IOException e) {
            throw new SchemaRegistryRequestForwardingException(String.format("Unexpected error while forwarding the undelete key version request to %s", baseUrl), (Throwable)e);
        }
        catch (RestClientException e) {
            throw new RestException(e.getMessage(), e.getStatus(), e.getErrorCode(), (Throwable)e);
        }
    }

    public void undeleteDekVersion(String name, String subject, int version, DekFormat algorithm) throws SchemaRegistryException {
        String tenant;
        KeyEncryptionKeyId keyId;
        KeyEncryptionKey key;
        this.keys.sync();
        if (algorithm == null) {
            algorithm = DekFormat.AES256_GCM;
        }
        if ((key = (KeyEncryptionKey)this.keys.get((Object)(keyId = new KeyEncryptionKeyId(tenant = this.schemaRegistry.tenant(), name)))) == null) {
            return;
        }
        if (key.isDeleted()) {
            throw new KeySoftDeletedException(name);
        }
        DataEncryptionKeyId id = new DataEncryptionKeyId(tenant, name, subject, algorithm, version);
        DataEncryptionKey oldKey = (DataEncryptionKey)this.keys.get((Object)id);
        if (oldKey == null) {
            return;
        }
        if (oldKey.isDeleted()) {
            DataEncryptionKey newKey = new DataEncryptionKey(name, oldKey.getSubject(), oldKey.getAlgorithm(), oldKey.getVersion(), oldKey.getEncryptedKeyMaterial(), false);
            this.keys.put((Object)id, (Object)newKey);
        }
    }

    public int normalizeLimit(int suppliedLimit, int defaultLimit, int maxLimit) {
        int limit = defaultLimit;
        if (suppliedLimit > 0 && suppliedLimit <= maxLimit) {
            limit = suppliedLimit;
        }
        return limit;
    }

    public int normalizeKekLimit(int suppliedLimit) {
        return this.normalizeLimit(suppliedLimit, this.kekSearchDefaultLimit, this.kekSearchMaxLimit);
    }

    public int normalizeDekSubjectLimit(int suppliedLimit) {
        return this.normalizeLimit(suppliedLimit, this.dekSubjectSearchDefaultLimit, this.dekSubjectSearchMaxLimit);
    }

    public int normalizeDekVersionLimit(int suppliedLimit) {
        return this.normalizeLimit(suppliedLimit, this.dekVersionSearchDefaultLimit, this.dekVersionSearchMaxLimit);
    }

    @Override
    @PreDestroy
    public void close() throws IOException {
        log.info("Shutting down dek registry");
        if (this.keys != null) {
            this.keys.close();
        }
    }

    private static byte[] toJson(Object o) throws JsonProcessingException {
        return JacksonMapper.INSTANCE.writeValueAsBytes(o);
    }
}

