/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.schemaregistry.storage;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OffsetCheckpoint
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(OffsetCheckpoint.class);
    public static final String CHECKPOINT_FILE_NAME = ".checkpoint";
    public static final String LOCK_FILE_NAME = ".lock";
    private static final Pattern WHITESPACE_MINIMUM_ONCE = Pattern.compile("\\s+");
    protected final File file;
    private final Object lock;
    private FileChannel channel;
    private FileLock fileLock;
    private int version;

    public OffsetCheckpoint(String checkpointDir, int version, String topic) throws IOException {
        File baseDir = this.baseDir(checkpointDir, topic);
        this.file = new File(baseDir, CHECKPOINT_FILE_NAME);
        this.lock = new Object();
        this.setUpLockFile(baseDir);
        this.version = version;
    }

    protected void setUpLockFile(File baseDir) throws IOException {
        File lockFile = new File(baseDir, LOCK_FILE_NAME);
        FileChannel channel = FileChannel.open(lockFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE);
        FileLock fileLock = this.tryLock(channel);
        if (fileLock == null) {
            channel.close();
            throw new IOException("Could not obtain file lock");
        }
        this.channel = channel;
        this.fileLock = fileLock;
    }

    protected File baseDir(String checkpointDir, String topic) throws IOException {
        File dir = new File(checkpointDir, topic);
        if (!dir.exists() && !dir.mkdirs() || !dir.isDirectory() || !dir.canWrite()) {
            throw new IOException(String.format("Cannot access or write to directory [%s]", dir.getPath()));
        }
        return dir;
    }

    private FileLock tryLock(FileChannel channel) throws IOException {
        try {
            return channel.tryLock();
        }
        catch (OverlappingFileLockException e) {
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void write(Map<TopicPartition, Long> offsets) throws IOException {
        if (this.version < 0) {
            return;
        }
        if (offsets.isEmpty()) {
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            File temp = new File(this.file.getAbsolutePath() + ".tmp");
            LOG.trace("Writing tmp checkpoint file {}", (Object)temp.getAbsolutePath());
            FileOutputStream fileOutputStream = new FileOutputStream(temp);
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)fileOutputStream, StandardCharsets.UTF_8));){
                OffsetCheckpoint.writeIntLine(writer, this.version);
                OffsetCheckpoint.writeIntLine(writer, offsets.size());
                for (Map.Entry<TopicPartition, Long> entry : offsets.entrySet()) {
                    TopicPartition tp = entry.getKey();
                    Long offset = entry.getValue();
                    if (offset >= 0L) {
                        OffsetCheckpoint.writeEntry(writer, tp, offset);
                        continue;
                    }
                    LOG.warn("Received offset={} to write to checkpoint file for {}", (Object)offset, (Object)tp);
                }
                writer.flush();
                fileOutputStream.getFD().sync();
            }
            LOG.trace("Swapping tmp checkpoint file {} {}", (Object)temp.toPath(), (Object)this.file.toPath());
            Utils.atomicMoveWithFallback((Path)temp.toPath(), (Path)this.file.toPath());
        }
    }

    static void writeIntLine(BufferedWriter writer, int number) throws IOException {
        writer.write(Integer.toString(number));
        writer.newLine();
    }

    static void writeEntry(BufferedWriter writer, TopicPartition part, long offset) throws IOException {
        writer.write(part.topic());
        writer.write(32);
        writer.write(Integer.toString(part.partition()));
        writer.write(32);
        writer.write(Long.toString(offset));
        writer.newLine();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Map<TopicPartition, Long> read() throws IOException {
        Object object = this.lock;
        synchronized (object) {
            try {
                int oldVersion;
                block17: {
                    String line;
                    HashMap<TopicPartition, Long> offsets;
                    int expectedSize;
                    BufferedReader reader;
                    block18: {
                        reader = Files.newBufferedReader(this.file.toPath());
                        oldVersion = OffsetCheckpoint.readInt(reader);
                        if (oldVersion != this.version) break block17;
                        expectedSize = OffsetCheckpoint.readInt(reader);
                        offsets = new HashMap<TopicPartition, Long>();
                        line = reader.readLine();
                        break block18;
                        finally {
                            if (reader != null) {
                                reader.close();
                            }
                        }
                    }
                    while (line != null) {
                        String[] pieces = WHITESPACE_MINIMUM_ONCE.split(line);
                        if (pieces.length != 3) {
                            throw new IOException(String.format("Malformed line in offset checkpoint file: '%s'.", line));
                        }
                        String topic = pieces[0];
                        int partition = Integer.parseInt(pieces[1]);
                        TopicPartition tp = new TopicPartition(topic, partition);
                        long offset = Long.parseLong(pieces[2]);
                        if (offset >= 0L) {
                            offsets.put(tp, offset);
                        } else {
                            LOG.warn("Read offset={} from checkpoint file for {}", (Object)offset, (Object)tp);
                        }
                        line = reader.readLine();
                    }
                    if (offsets.size() != expectedSize) {
                        throw new IOException(String.format("Expected %d entries but found only %d", expectedSize, offsets.size()));
                    }
                    HashMap<TopicPartition, Long> hashMap = offsets;
                    return hashMap;
                }
                LOG.warn("Old offset checkpoint version: {}", (Object)oldVersion);
            }
            catch (NoSuchFileException noSuchFileException) {
                // empty catch block
            }
            return Collections.emptyMap();
        }
    }

    static int readInt(BufferedReader reader) throws IOException {
        String line = reader.readLine();
        if (line == null) {
            throw new EOFException("File ended prematurely.");
        }
        return Integer.parseInt(line);
    }

    @Override
    public void close() throws IOException {
        if (this.fileLock == null) {
            return;
        }
        this.fileLock.release();
        this.channel.close();
        this.fileLock = null;
        this.channel = null;
    }

    public void delete() throws IOException {
        Files.deleteIfExists(this.file.toPath());
    }

    public String toString() {
        return this.file.getAbsolutePath();
    }
}

