/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.cruisecontrol.analyzer.history;

import com.linkedin.kafka.cruisecontrol.common.KafkaCruiseControlThreadFactory;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import io.confluent.cruisecontrol.analyzer.history.AbstractEntityEventHistory;
import io.confluent.cruisecontrol.analyzer.history.EntityMovement;
import io.confluent.cruisecontrol.analyzer.history.EntityMovementHistoryListener;
import io.confluent.cruisecontrol.analyzer.history.EventHistoryPool;
import io.confluent.cruisecontrol.analyzer.history.SuspendedEntity;
import io.confluent.cruisecontrol.analyzer.history.SuspendedTenant;
import io.confluent.cruisecontrol.analyzer.history.SuspendedTopicPartition;
import io.confluent.cruisecontrol.analyzer.history.TenantMovement;
import io.confluent.cruisecontrol.analyzer.history.TopicPartitionMovement;
import java.util.Collection;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.concurrent.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class EntityMovementHistory<M extends EntityMovement>
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(EntityMovementHistory.class);
    private static final long TERMINATION_TIMEOUT_MS = 60000L;
    private final AtomicLong epoch;
    private final int entityMaximumMovements;
    private final long entitySuspensionMs;
    private final EventHistoryPool<M> recentEntityMovements;
    private final EventHistoryPool<SuspendedTopicPartition> recentSuspendedTopicPartitions;
    private final EventHistoryPool<SuspendedTenant> recentSuspendedTenants;
    private final Collection<String> suspendedEntities;
    private final ExecutorService entityMovementsCleanerExecutor;
    private final ExecutorService suspendedEntityCleanerExecutor;
    private final ScheduledExecutorService monitorExecutor;
    private static final long MONITOR_DELAY_MS = TimeUnit.MINUTES.toMillis(10L);
    private final Collection<EntityMovementHistoryListener<M>> entityMovementsListeners;
    private final Collection<EntityMovementHistoryListener<SuspendedTopicPartition>> suspendedTopicPartitionsListeners;
    private final Collection<EntityMovementHistoryListener<SuspendedTenant>> suspendedTenantsListeners;
    private final ConcurrentMap<Object, Integer> numberOfMovementsByEntity;
    private final InterruptiblePoller entityMovementsCleaner;
    private final InterruptiblePoller suspendedTopicPartitionsCleaner;
    private final InterruptiblePoller suspendedTenantsCleaner;
    private final Runnable entityMovementHistoryMonitor;
    private final EntityType entityType;

    public static EntityMovementHistory<TopicPartitionMovement> createTopicPartitionHistory(KafkaCruiseControlConfig config) {
        return new EntityMovementHistory<TopicPartitionMovement>(config, EntityType.TOPIC_PARTITION);
    }

    public static EntityMovementHistory<TenantMovement> createTenantHistory(KafkaCruiseControlConfig config) {
        return new EntityMovementHistory<TenantMovement>(config, EntityType.TENANT);
    }

    private EntityMovementHistory(KafkaCruiseControlConfig config, EntityType entityType) {
        this.entityType = entityType;
        this.entityMaximumMovements = entityType == EntityType.TOPIC_PARTITION ? config.getInt("topic.partition.maximum.movements") : config.getInt("tenant.maximum.movements");
        this.entitySuspensionMs = entityType == EntityType.TOPIC_PARTITION ? config.getLong("topic.partition.suspension.ms") : config.getLong("tenant.suspension.ms");
        this.recentEntityMovements = new EventHistoryPool();
        this.recentSuspendedTopicPartitions = new EventHistoryPool();
        this.recentSuspendedTenants = new EventHistoryPool();
        this.suspendedEntities = ConcurrentHashMap.newKeySet();
        this.entityMovementsCleanerExecutor = Executors.newSingleThreadExecutor(new KafkaCruiseControlThreadFactory("EntityMovementsCleaner", true, LOG));
        this.suspendedEntityCleanerExecutor = Executors.newSingleThreadExecutor(new KafkaCruiseControlThreadFactory("SuspendedEntitiesCleaner", true, LOG));
        this.monitorExecutor = Executors.newSingleThreadScheduledExecutor(new KafkaCruiseControlThreadFactory("EntitySuspensionHistoryMonitor", true, LOG));
        this.entityMovementsListeners = new ConcurrentLinkedQueue<EntityMovementHistoryListener<M>>();
        this.suspendedTopicPartitionsListeners = new ConcurrentLinkedQueue<EntityMovementHistoryListener<SuspendedTopicPartition>>();
        this.suspendedTenantsListeners = new ConcurrentLinkedQueue<EntityMovementHistoryListener<SuspendedTenant>>();
        this.numberOfMovementsByEntity = new ConcurrentHashMap<Object, Integer>();
        this.epoch = new AtomicLong(0L);
        this.entityMovementsCleaner = InterruptiblePoller.of(() -> {
            try {
                EntityMovement entityMovement = (EntityMovement)this.recentEntityMovements.takeExpired();
                Object entity = entityMovement.entity();
                this.numberOfMovementsByEntity.compute(entity, (topicPartition, oldNumber) -> {
                    if (oldNumber == null) {
                        return -1;
                    }
                    if (oldNumber == 1) {
                        return null;
                    }
                    return oldNumber - 1;
                });
                long currentEpoch = this.epoch.get();
                long entityEpoch = entityMovement.epoch();
                if (currentEpoch <= entityEpoch) {
                    if (currentEpoch < entityEpoch) {
                        LOG.warn("EntitySuspensionHistory has a smaller epoch ({}) than the one from Entity ({})", (Object)currentEpoch, (Object)entityEpoch);
                    }
                    for (EntityMovementHistoryListener listener : this.entityMovementsListeners) {
                        EntityMovementHistory.safeExecute(() -> listener.onExpiredHistory(entityMovement));
                    }
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        });
        this.suspendedTopicPartitionsCleaner = this.suspendedEntitiesCleaner(this.recentSuspendedTopicPartitions, this.suspendedTopicPartitionsListeners);
        this.suspendedTenantsCleaner = this.suspendedEntitiesCleaner(this.recentSuspendedTenants, this.suspendedTenantsListeners);
        this.entityMovementHistoryMonitor = () -> LOG.info("Current numbers of repeated movements generated for {} {}(s) are {}", new Object[]{this.numberOfMovementsByEntity.size(), entityType, this.numberOfMovementsByEntity});
        this.startCleaners();
        this.startMonitors();
    }

    private <S extends SuspendedEntity> InterruptiblePoller suspendedEntitiesCleaner(EventHistoryPool<S> recentSuspendedEntities, Collection<EntityMovementHistoryListener<S>> suspendedEntitiesListeners) {
        return InterruptiblePoller.of(() -> {
            try {
                SuspendedEntity suspendedEntity = (SuspendedEntity)recentSuspendedEntities.takeExpired();
                long currentEpoch = this.epoch.get();
                long epoch = suspendedEntity.epoch();
                if (currentEpoch <= suspendedEntity.epoch()) {
                    if (currentEpoch < epoch) {
                        LOG.warn("EntityMovementHistory has a smaller epoch ({}) than the one from SuspendedEntity ({})", (Object)currentEpoch, (Object)epoch);
                    }
                    this.suspendedEntities.remove(suspendedEntity.entityId());
                    for (EntityMovementHistoryListener listener : suspendedEntitiesListeners) {
                        EntityMovementHistory.safeExecute(() -> listener.onExpiredHistory(suspendedEntity));
                    }
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        });
    }

    private void startCleaners() {
        this.entityMovementsCleanerExecutor.execute(this.entityMovementsCleaner);
        if (this.entityType == EntityType.TOPIC_PARTITION) {
            this.suspendedEntityCleanerExecutor.execute(this.suspendedTopicPartitionsCleaner);
        } else {
            this.suspendedEntityCleanerExecutor.execute(this.suspendedTenantsCleaner);
        }
    }

    private void startMonitors() {
        this.monitorExecutor.scheduleAtFixedRate(this.entityMovementHistoryMonitor, MONITOR_DELAY_MS, MONITOR_DELAY_MS, TimeUnit.MILLISECONDS);
    }

    public void addEntityMovementListener(EntityMovementHistoryListener<M> entityMovementListener) {
        this.entityMovementsListeners.add(entityMovementListener);
    }

    public void addSuspendedTopicPartitionListener(EntityMovementHistoryListener<SuspendedTopicPartition> suspendedTopicPartitionListener) {
        if (!this.entityType.equals((Object)EntityType.TOPIC_PARTITION)) {
            throw new IllegalStateException("Can't add topic partition listener to EntityMovementHistory with entity type " + String.valueOf((Object)this.entityType));
        }
        this.suspendedTopicPartitionsListeners.add(suspendedTopicPartitionListener);
    }

    public void addSuspendedTenantListener(EntityMovementHistoryListener<SuspendedTenant> suspendedTenantListener) {
        if (!this.entityType.equals((Object)EntityType.TENANT)) {
            throw new IllegalStateException("Can't add tenant listener to EntityMovementHistory with entity type " + String.valueOf((Object)this.entityType));
        }
        this.suspendedTenantsListeners.add(suspendedTenantListener);
    }

    public void record(M entityMovement) {
        long currentEpoch = this.epoch.get();
        boolean added = this.recentEntityMovements.add(entityMovement);
        if (added) {
            Object entity = ((AbstractEntityEventHistory)entityMovement).entity();
            int newNumber = this.numberOfMovementsByEntity.compute(((AbstractEntityEventHistory)entityMovement).entity(), (entityObject, oldNumber) -> oldNumber == null ? 1 : oldNumber + 1);
            for (EntityMovementHistoryListener listener : this.entityMovementsListeners) {
                EntityMovementHistory.safeExecute(() -> listener.onNewHistory(entityMovement));
            }
            this.maybeUpdateSuspendedEntity(entity, newNumber, currentEpoch);
        } else {
            LOG.debug("Received EntityMovements with outdated epoch: {}, current epoch: {}", entityMovement, (Object)currentEpoch);
        }
    }

    private void maybeUpdateSuspendedEntity(Object entity, int numberOfMovements, long epoch) {
        if (numberOfMovements <= this.entityMaximumMovements) {
            return;
        }
        if (this.entityType.equals((Object)EntityType.TOPIC_PARTITION)) {
            SuspendedTopicPartition suspendedTopicPartition = new SuspendedTopicPartition(entity, this.entitySuspensionMs, epoch);
            this.updateSuspendedEntity(suspendedTopicPartition, this.recentSuspendedTopicPartitions, this.suspendedTopicPartitionsListeners);
        } else if (this.entityType.equals((Object)EntityType.TENANT)) {
            SuspendedTenant suspendedTenant = new SuspendedTenant(entity, this.entitySuspensionMs, epoch);
            this.updateSuspendedEntity(suspendedTenant, this.recentSuspendedTenants, this.suspendedTenantsListeners);
        }
    }

    private <S extends SuspendedEntity> void updateSuspendedEntity(S suspendedEntity, EventHistoryPool<S> recentSuspendedEntities, Collection<EntityMovementHistoryListener<S>> suspendedEntitiesListeners) {
        boolean updated = recentSuspendedEntities.update(suspendedEntity);
        if (updated) {
            this.suspendedEntities.add(suspendedEntity.entityId());
            for (EntityMovementHistoryListener listener : suspendedEntitiesListeners) {
                EntityMovementHistory.safeExecute(() -> listener.onNewHistory(suspendedEntity));
            }
        }
    }

    public boolean isEntitySuspended(String entityId) {
        return this.suspendedEntities.contains(entityId);
    }

    public long currentEpoch() {
        return this.epoch.get();
    }

    public void clear() {
        block4: {
            long newEpoch;
            block3: {
                newEpoch = this.epoch.incrementAndGet();
                this.recentEntityMovements.newEpoch(newEpoch);
                this.recentSuspendedTopicPartitions.newEpoch(newEpoch);
                this.recentSuspendedTenants.newEpoch(newEpoch);
                this.suspendedEntities.clear();
                for (EntityMovementHistoryListener<M> entityMovementHistoryListener : this.entityMovementsListeners) {
                    EntityMovementHistory.safeExecute(() -> listener.onUpdatedEpoch(newEpoch));
                }
                if (!this.entityType.equals((Object)EntityType.TOPIC_PARTITION)) break block3;
                for (EntityMovementHistoryListener<Object> entityMovementHistoryListener : this.suspendedTopicPartitionsListeners) {
                    EntityMovementHistory.safeExecute(() -> listener.onUpdatedEpoch(newEpoch));
                }
                break block4;
            }
            if (!this.entityType.equals((Object)EntityType.TENANT)) break block4;
            for (EntityMovementHistoryListener<Object> entityMovementHistoryListener : this.suspendedTenantsListeners) {
                EntityMovementHistory.safeExecute(() -> listener.onUpdatedEpoch(newEpoch));
            }
        }
    }

    public int numberOfMovements(Object entity) {
        return this.numberOfMovementsByEntity.getOrDefault(entity, 0);
    }

    @Override
    public void close() throws Exception {
        this.entityMovementsCleanerExecutor.shutdownNow();
        this.suspendedEntityCleanerExecutor.shutdownNow();
        this.monitorExecutor.shutdownNow();
        boolean entityMovementsCleanerTerminated = false;
        boolean suspendedEntityCleanerTerminated = false;
        boolean monitorExecutorTerminated = false;
        try {
            entityMovementsCleanerTerminated = this.entityMovementsCleanerExecutor.awaitTermination(60000L, TimeUnit.MILLISECONDS);
            suspendedEntityCleanerTerminated = this.suspendedEntityCleanerExecutor.awaitTermination(60000L, TimeUnit.MILLISECONDS);
            monitorExecutorTerminated = this.monitorExecutor.awaitTermination(60000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (Exception e) {
            LOG.error("Exception is thrown while waiting for the termination of cleaners", (Throwable)e);
            throw e;
        }
        finally {
            if (!entityMovementsCleanerTerminated) {
                LOG.error("EntityMovementsCleaner didn't terminate within {} ms", (Object)60000L);
            }
            if (!suspendedEntityCleanerTerminated) {
                LOG.error("SuspendedEntitiesCleaner didn't terminate within {} ms", (Object)60000L);
            }
            if (!monitorExecutorTerminated) {
                LOG.error("MonitorExecutor didn't terminate within {} ms", (Object)60000L);
            }
        }
    }

    private static void safeExecute(Runnable task) {
        try {
            task.run();
        }
        catch (Exception e) {
            LOG.error("Exception were caught during EntityMovementHistory notifications", (Throwable)e);
        }
    }

    public static enum EntityType {
        TOPIC_PARTITION,
        TENANT;

    }

    static class InterruptiblePoller
    implements Runnable {
        private final Runnable task;

        private InterruptiblePoller(Runnable task) {
            this.task = task;
        }

        public static InterruptiblePoller of(Runnable task) {
            return new InterruptiblePoller(task);
        }

        @Override
        public void run() {
            block9: {
                block7: {
                    Throwable thrown = null;
                    try {
                        while (!Thread.interrupted()) {
                            this.task.run();
                        }
                        if (thrown == null) break block7;
                    }
                    catch (Throwable e) {
                        block8: {
                            try {
                                thrown = e;
                                if (thrown == null) break block8;
                            }
                            catch (Throwable throwable) {
                                if (thrown != null) {
                                    LOG.error(String.format("Thread %s exited abnormally", Thread.currentThread().getName()), thrown);
                                } else {
                                    LOG.info("Thread {} exited", (Object)Thread.currentThread().getName());
                                }
                                throw throwable;
                            }
                            LOG.error(String.format("Thread %s exited abnormally", Thread.currentThread().getName()), thrown);
                        }
                        LOG.info("Thread {} exited", (Object)Thread.currentThread().getName());
                    }
                    LOG.error(String.format("Thread %s exited abnormally", Thread.currentThread().getName()), thrown);
                    break block9;
                }
                LOG.info("Thread {} exited", (Object)Thread.currentThread().getName());
            }
        }
    }
}

