/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.detector;

import com.linkedin.cruisecontrol.detector.Anomaly;
import com.linkedin.kafka.cruisecontrol.common.CellResource;
import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.config.KafkaCruiseControlConfig;
import com.linkedin.kafka.cruisecontrol.detector.CellOverload;
import com.linkedin.kafka.cruisecontrol.detector.CellOverloadPlan;
import com.linkedin.kafka.cruisecontrol.detector.ResourceUtilizationDetector;
import com.linkedin.kafka.cruisecontrol.detector.utils.CellOverloadOccurrence;
import com.linkedin.kafka.cruisecontrol.detector.utils.CellOverloadOccurrenceRecorder;
import com.linkedin.kafka.cruisecontrol.executor.TenantProposal;
import com.linkedin.kafka.cruisecontrol.model.Cell;
import com.linkedin.kafka.cruisecontrol.model.CellResourceLoad;
import com.linkedin.kafka.cruisecontrol.model.ClusterModel;
import com.linkedin.kafka.cruisecontrol.model.OverloadedCell;
import com.linkedin.kafka.cruisecontrol.model.Tenant;
import com.linkedin.kafka.cruisecontrol.model.view.CellTenantView;
import com.linkedin.kafka.cruisecontrol.model.view.ClusterModelCellView;
import io.confluent.cruisecontrol.analyzer.history.EntityMovementHistory;
import io.confluent.cruisecontrol.analyzer.history.EntityMovementLogger;
import io.confluent.cruisecontrol.analyzer.history.TenantMovement;
import io.confluent.databalancer.metrics.CellOverloadMetrics;
import io.confluent.databalancer.metrics.internals.TenantUtilizationMetricHandler;
import io.confluent.kafka.clients.CloudAdmin;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Queue;
import java.util.Random;
import java.util.Set;
import java.util.SortedSet;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.kafka.common.PartitionPlacementStrategy;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CellOverloadDetector
implements ResourceUtilizationDetector {
    private static final int NO_TARGET_CELL_ID = -100;
    private static final Logger LOG = LoggerFactory.getLogger(CellOverloadDetector.class);
    private static final double HIGH_CPU_UTILIZATION = 0.5;
    private static final String HIGH_CPU_UTILIZATION_PERCENTAGE = String.format("%.0f%%", 50.0);
    private static final String CELL_LOAD_UPPER_BOUND_METRIC_NAME = "cell-load-upper-bound";
    private final CloudAdmin cloudAdmin;
    private final Supplier<Boolean> shouldSkipAnomalyDetection;
    private final Queue<Anomaly> anomalies;
    private final CellOverloadOccurrence cellOverloadOccurrence;
    private final CellOverloadOccurrenceRecorder cellOverloadOccurrenceRecorder;
    private final TenantUtilizationMetricHandler tenantUtilizationMetricHandler;
    private final EntityMovementHistory<TenantMovement> tenantMovementHistory;
    private final Time time;
    private final double cellLoadUpperBound;
    private final long cellOverloadDetectionIntervalMs;
    private final long maxReplicasPerBroker;
    private final long tenantSuspensionMs;
    private final Random random;
    private Set<Integer> managedCellIds;
    private long nextDetectionTimeMs;

    public CellOverloadDetector(KafkaCruiseControlConfig kccConfig, CloudAdmin cloudAdmin, Supplier<Boolean> shouldSkipAnomalyDetection, Queue<Anomaly> anomalies, CellOverloadOccurrenceRecorder cellOverloadOccurrenceRecorder, CellOverloadMetrics cellOverloadMetrics, TenantUtilizationMetricHandler tenantUtilizationMetricHandler, Time time) {
        this(cloudAdmin, shouldSkipAnomalyDetection, anomalies, cellOverloadOccurrenceRecorder.cellOverloadOccurrence(), cellOverloadOccurrenceRecorder, cellOverloadMetrics, tenantUtilizationMetricHandler, CellOverloadDetector.createEntityMovementHistory(kccConfig), time, kccConfig.getDouble("cell.load.upper.bound"), kccConfig.getLong("cell.overload.detection.interval.ms"), kccConfig.getLong("max.replicas"), kccConfig.getLong("tenant.suspension.ms"));
    }

    private static EntityMovementHistory<TenantMovement> createEntityMovementHistory(KafkaCruiseControlConfig kccConfig) {
        EntityMovementLogger entityMovementLogger = new EntityMovementLogger();
        EntityMovementHistory<TenantMovement> tenantMovementHistory = EntityMovementHistory.createTenantHistory(kccConfig);
        tenantMovementHistory.addEntityMovementListener(entityMovementLogger.tenantMovementListener());
        tenantMovementHistory.addSuspendedTenantListener(entityMovementLogger.suspendedTenantListener());
        return tenantMovementHistory;
    }

    CellOverloadDetector(CloudAdmin cloudAdmin, Supplier<Boolean> shouldSkipAnomalyDetection, Queue<Anomaly> anomalies, CellOverloadOccurrence cellOverloadOccurrence, CellOverloadOccurrenceRecorder cellOverloadOccurrenceRecorder, CellOverloadMetrics cellOverloadMetrics, TenantUtilizationMetricHandler tenantUtilizationMetricHandler, EntityMovementHistory<TenantMovement> tenantMovementHistory, Time time, double cellLoadUpperBound, long cellOverloadDetectionIntervalMs, long maxReplicasPerBroker, long tenantSuspensionMs) {
        this.cloudAdmin = cloudAdmin;
        this.shouldSkipAnomalyDetection = shouldSkipAnomalyDetection;
        this.anomalies = anomalies;
        this.cellOverloadOccurrence = cellOverloadOccurrence;
        this.cellOverloadOccurrenceRecorder = cellOverloadOccurrenceRecorder;
        this.tenantUtilizationMetricHandler = tenantUtilizationMetricHandler;
        this.tenantMovementHistory = tenantMovementHistory;
        this.time = time;
        this.cellLoadUpperBound = cellLoadUpperBound;
        this.cellOverloadDetectionIntervalMs = cellOverloadDetectionIntervalMs;
        this.maxReplicasPerBroker = maxReplicasPerBroker;
        this.tenantSuspensionMs = tenantSuspensionMs;
        this.random = new Random();
        this.nextDetectionTimeMs = time.hiResClockMs();
        cellOverloadMetrics.newGauge(CELL_LOAD_UPPER_BOUND_METRIC_NAME, () -> cellLoadUpperBound);
    }

    @Override
    public void detectResourceUtilization(ClusterModel clusterModel) {
        this.tenantUtilizationMetricHandler.refreshTenantStripeCountMetrics(clusterModel);
        boolean shouldContinue = this.isNextDetectionTimeReached();
        if (!shouldContinue) {
            return;
        }
        if (this.shouldSkipAnomalyDetection.get().booleanValue()) {
            return;
        }
        this.runOnce(clusterModel);
    }

    @Override
    public void close() {
        try {
            this.tenantMovementHistory.close();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    Optional<ClusterModelCellView> runOnce(ClusterModel clusterModel) {
        Optional<ClusterModelCellView> cellViewOpt = this.preProcess(clusterModel);
        if (!cellViewOpt.isPresent()) {
            return Optional.empty();
        }
        ClusterModelCellView cellView = cellViewOpt.get();
        Optional<OverloadedCellDetectionResult> overloadedCellDetectionResultOpt = this.detectOverloadedCells(cellView);
        if (!overloadedCellDetectionResultOpt.isPresent()) {
            return Optional.of(cellView);
        }
        Collection<CellOverloadPlan> cellOverloadPlans = this.balanceOverloadedCells(cellView, overloadedCellDetectionResultOpt.get());
        this.reportCellOverloadPlans(cellOverloadPlans, cellView);
        return Optional.of(cellView);
    }

    private boolean isNextDetectionTimeReached() {
        long now = this.time.hiResClockMs();
        long remainingTimeMs = this.nextDetectionTimeMs - now;
        return remainingTimeMs <= 0L;
    }

    private Optional<ClusterModelCellView> preProcess(ClusterModel clusterModel) {
        try {
            this.mayPersistManagedCells(clusterModel);
        }
        catch (InterruptedException ex) {
            LOG.info("CellOverloadDetector is shutdown during the pre-process of a cluster model.");
            Thread.currentThread().interrupt();
            return Optional.empty();
        }
        this.tenantUtilizationMetricHandler.update(clusterModel);
        if (clusterModel.hasReassigningPartitions()) {
            LOG.debug("CellOverloadDetector is skipping detection because there are ongoing partition reassignments.");
            return Optional.empty();
        }
        if (clusterModel.replicasAreInExpectedCell()) {
            return Optional.of(new ClusterModelCellView(clusterModel));
        }
        LOG.warn("CellOverloadDetector detected replicas of some tenants not in their designated cell and will skip detecting cell overload.");
        return Optional.empty();
    }

    private void mayPersistManagedCells(ClusterModel clusterModel) throws InterruptedException {
        Collection<Cell> cells = clusterModel.cells();
        Set<Integer> updatedCellIds = cells.stream().map(Cell::id).collect(Collectors.toSet());
        if (!updatedCellIds.equals(this.managedCellIds)) {
            LOG.info("CellOverloadDetector is updating managed cells from {} to {}", this.managedCellIds, updatedCellIds);
            this.managedCellIds = updatedCellIds;
            this.cellOverloadOccurrenceRecorder.persistManagedCellIds(updatedCellIds);
        }
    }

    private void scheduleNextDetection() {
        long now = this.time.hiResClockMs();
        double factor = this.random.nextDouble() + 0.5;
        long timeUntilNextDetectionMs = (long)((double)this.cellOverloadDetectionIntervalMs * factor);
        this.nextDetectionTimeMs = now + timeUntilNextDetectionMs;
        long minutesUntilNextDetection = TimeUnit.MILLISECONDS.toMinutes(timeUntilNextDetectionMs);
        long secondsUntilNextDetection = TimeUnit.MILLISECONDS.toSeconds(timeUntilNextDetectionMs);
        long secondsUntilNextDetectionFromMinutes = TimeUnit.MINUTES.toSeconds(minutesUntilNextDetection);
        LOG.info("CellOverloadDetector will start detecting cell overload now. The next detection is expected to happen in {} min, {} sec.", (Object)minutesUntilNextDetection, (Object)(secondsUntilNextDetection - secondsUntilNextDetectionFromMinutes));
    }

    private Optional<OverloadedCellDetectionResult> detectOverloadedCells(ClusterModelCellView cellView) {
        this.scheduleNextDetection();
        Collection<CellTenantView> cells = cellView.cells(cell -> cell.state().isEligibleSource());
        HashSet<Integer> underloadedCellIds = new HashSet<Integer>();
        ArrayList<OverloadedCell> consistentOverloadedCells = new ArrayList<OverloadedCell>();
        try {
            for (CellTenantView cell2 : cells) {
                OverloadedCell overloadedCell = cell2.detectOverload(this.cellLoadUpperBound, this.maxReplicasPerBroker);
                if (overloadedCell.isOverloaded()) {
                    LOG.info("CellOverloadDetector detected overloaded cell {} and recorded 1 occurrence. Cell Resource Utilization: {}", (Object)cell2.id(), (Object)overloadedCell);
                    this.cellOverloadOccurrenceRecorder.addOccurrence(cell2.id());
                } else {
                    long occurrences = this.cellOverloadOccurrence.occurrences(cell2.id());
                    if (occurrences > 0L) {
                        LOG.info("CellOverloadDetector detected underloaded cell {} and deleted 1 occurrence: current occurrences {} -> {}", new Object[]{cell2.id(), occurrences, occurrences - 1L});
                        this.cellOverloadOccurrenceRecorder.deleteOccurrence(cell2.id());
                    }
                    if (occurrences <= 1L && cell2.state().isEligibleDestination()) {
                        underloadedCellIds.add(cell2.id());
                    }
                }
                if (!this.cellOverloadOccurrence.isOverloaded(cell2.id())) continue;
                LOG.info("CellOverloadDetector detected a consistently overloaded cell {}", (Object)overloadedCell);
                consistentOverloadedCells.add(overloadedCell);
            }
        }
        catch (InterruptedException ex) {
            LOG.info("CellOverloadDetector is shutdown during the detection of overloaded cells.");
            Thread.currentThread().interrupt();
            return Optional.empty();
        }
        if (consistentOverloadedCells.isEmpty()) {
            LOG.debug("CellOverloadDetector didn't detect any consistently overloaded cells.");
            return Optional.empty();
        }
        LOG.info("CellOverloadDetector detected consistent overloaded cells: {}", consistentOverloadedCells);
        return Optional.of(new OverloadedCellDetectionResult(underloadedCellIds, consistentOverloadedCells));
    }

    private Collection<CellOverloadPlan> balanceOverloadedCells(ClusterModelCellView cellView, OverloadedCellDetectionResult overloadedCellDetectionResult) {
        long now = this.time.milliseconds();
        LOG.info("CellOverloadDetector starts to balance overloaded cells at {}", (Object)now);
        Set<Integer> underloadedCellIds = overloadedCellDetectionResult.underloadedCellIds;
        Collection<OverloadedCell> consistentOverloadedCells = overloadedCellDetectionResult.consistentOverloadedCells;
        return consistentOverloadedCells.stream().map(overloadedCell -> this.balanceOverloadedCell(cellView, (OverloadedCell)overloadedCell, underloadedCellIds)).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
    }

    private Optional<CellOverloadPlan> balanceOverloadedCell(ClusterModelCellView cellView, OverloadedCell overloadedCell, Set<Integer> underloadedCellIds) {
        LOG.debug("CellOverloadDetector is balancing {}", (Object)overloadedCell);
        CellTenantView cell = overloadedCell.cell();
        double cpuCapacity = cell.capacity().eligibleDestinationCapacityFor(Resource.CPU);
        SortedSet<CellResourceLoad> overloadedResources = overloadedCell.overloadedResources();
        if (overloadedResources.isEmpty()) {
            LOG.warn("CellOverloadDetector detected overloaded cell {} with no overloaded resources.", (Object)overloadedCell.cell());
            return Optional.empty();
        }
        CellResource mostOverloadedResource = overloadedResources.last().cellResource();
        List<Tenant> tenantsSortedByOverloadedResource = cell.tenantsSortedByCellResource(mostOverloadedResource);
        CellOverloadPlan.Builder cellOverloadPlanBuilder = CellOverloadPlan.builder(cell.id());
        int numHighCpuTenants = 0;
        int numInvalidPolicyTenants = 0;
        int numSuspendedTenants = 0;
        for (Tenant tenant : tenantsSortedByOverloadedResource) {
            double cpuUsageByCell = tenant.utilization(cell.id()).eligibleSourceUtilization().map(load -> load.expectedUtilizationFor(Resource.CPU)).orElse(0.0);
            if (cpuUsageByCell / cpuCapacity > 0.5) {
                LOG.debug("CellOverloadDetector skipped Tenant {} because its CPU usage is over {} as part of alleviating resource utilization for cell {}.", new Object[]{HIGH_CPU_UTILIZATION_PERCENTAGE, tenant.tenantId(), cell.id()});
                ++numHighCpuTenants;
                continue;
            }
            if (tenant.placementPolicy() != PartitionPlacementStrategy.TENANT_IN_CELL) {
                LOG.debug("CellOverloadDetector skipped Tenant {} because it has partition placement strategy {}", (Object)tenant.tenantId(), (Object)tenant.placementPolicy());
                ++numInvalidPolicyTenants;
                continue;
            }
            if (this.isTenantSuspended(tenant)) {
                LOG.debug("CellOverloadDetector skipped Tenant {} because it is moved recently as part of alleviating resource utilization for cell {}.", (Object)tenant.tenantId(), (Object)cell.id());
                ++numSuspendedTenants;
                continue;
            }
            Optional<TenantProposal> tenantProposalOpt = this.moveTenant(cell.id(), cellView, underloadedCellIds, tenant);
            if (!tenantProposalOpt.isPresent()) continue;
            TenantProposal tenantProposal = tenantProposalOpt.get();
            this.logTenantProposal(tenantProposal, cell, tenant);
            cellOverloadPlanBuilder.addTenantProposal(tenant, tenantProposal);
            this.tenantMovementHistory.record(new TenantMovement(tenant, tenantProposal.oldCellId(), tenantProposal.newCellId(), "CellOverloadDetector", this.tenantSuspensionMs, this.tenantMovementEpoch()));
            OverloadedCell overloadedCellAfterTenantMove = cell.detectOverload(this.cellLoadUpperBound, this.maxReplicasPerBroker);
            if (overloadedCellAfterTenantMove.isOverloaded()) continue;
            cellOverloadPlanBuilder.willFixCellOverload(true);
            break;
        }
        LOG.info("CellOverloadDetector skipped {} tenants due to high CPU utilization, {} tenants due to invalid partition placement policy and {} tenants due to oscillation prevention.", new Object[]{numHighCpuTenants, numInvalidPolicyTenants, numSuspendedTenants});
        CellOverloadPlan cellOverloadPlan = cellOverloadPlanBuilder.build();
        if (cellOverloadPlan.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(cellOverloadPlan);
    }

    private void logTenantProposal(TenantProposal tenantProposal, CellTenantView cell, Tenant tenant) {
        if (LOG.isDebugEnabled()) {
            String tenantProposalLog = String.format("CellOverloadDetector generated a TenantProposal %s as part of alleviating resource utilization for cell %d", tenantProposal, cell.id()) + System.lineSeparator() + "Load of the tenant being moved: " + String.valueOf(tenant.load(cell.id())) + System.lineSeparator() + "Load of the cell after the tenant is moved: " + String.valueOf(cell.load());
            LOG.debug(tenantProposalLog);
        }
    }

    private Optional<TenantProposal> moveTenant(int sourceCellID, ClusterModelCellView cellView, Set<Integer> underloadedCellIds, Tenant tenant) {
        double minimumCellLoad = Double.MAX_VALUE;
        int targetCellID = -100;
        for (int destinationCellID : underloadedCellIds) {
            if (tenant.cellIds().contains(destinationCellID)) continue;
            ClusterModelCellView clonedCellView = new ClusterModelCellView(cellView);
            clonedCellView.relocateTenant(tenant.tenantId(), sourceCellID, destinationCellID);
            double destinationCellLoad = clonedCellView.cell(destinationCellID).cellUtilRatio(this.maxReplicasPerBroker);
            if (destinationCellLoad < this.cellLoadUpperBound && destinationCellLoad < minimumCellLoad) {
                LOG.debug("CellOverloadDetector found a possible destination cell for tenant {}: cell {}. After the move, the load of the destination cell will be {}.", new Object[]{tenant.tenantId(), destinationCellID, destinationCellLoad});
                minimumCellLoad = destinationCellLoad;
                targetCellID = destinationCellID;
            }
            clonedCellView.relocateTenant(tenant.tenantId(), destinationCellID, sourceCellID);
        }
        if (targetCellID != -100) {
            LOG.debug("CellOverloadDetector found a destination cell for tenant {}: cell {}.", (Object)tenant.tenantId(), (Object)targetCellID);
            TenantProposal tenantProposal = new TenantProposal(tenant, sourceCellID, targetCellID);
            cellView.apply(tenantProposal);
            return Optional.of(tenantProposal);
        }
        LOG.debug("CellOverloadDetector didn't find a destination cell for tenant {}.", (Object)tenant.tenantId());
        return Optional.empty();
    }

    private void reportCellOverloadPlans(Collection<CellOverloadPlan> cellOverloadPlans, ClusterModelCellView postOptimizationCellView) {
        if (!cellOverloadPlans.isEmpty()) {
            CellOverload cellOverload = new CellOverload(this.cloudAdmin, this.cellOverloadOccurrenceRecorder, cellOverloadPlans, postOptimizationCellView, this.cellLoadUpperBound, this.maxReplicasPerBroker);
            this.anomalies.add(cellOverload);
            LOG.info("CellOverloadDetector generated plans to fix CellOverload:\n{}", cellOverloadPlans);
        }
    }

    boolean isTenantSuspended(Tenant tenant) {
        return this.tenantMovementHistory.isEntitySuspended(tenant.tenantId());
    }

    private long tenantMovementEpoch() {
        return this.tenantMovementHistory.currentEpoch();
    }

    private static class OverloadedCellDetectionResult {
        private final Set<Integer> underloadedCellIds;
        private final Collection<OverloadedCell> consistentOverloadedCells;

        private OverloadedCellDetectionResult(Set<Integer> underloadedCellIds, Collection<OverloadedCell> consistentOverloadedCells) {
            this.underloadedCellIds = underloadedCellIds;
            this.consistentOverloadedCells = consistentOverloadedCells;
        }
    }
}

