/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafkarest.controllers;

import com.google.common.collect.ImmutableList;
import io.confluent.kafkarest.controllers.ConsumerGroupOffsetsManager;
import io.confluent.kafkarest.entities.v3.ListConsumerGroupOffsetsResultsData;
import io.confluent.kafkarest.entities.v3.TopicPartitionOffsetAndMetaData;
import io.confluent.kafkarest.entities.v3.UpdateConsumerGroupOffsetsRequest;
import io.confluent.kafkarest.entities.v3.UpdateConsumerGroupOffsetsResponse;
import io.confluent.kafkarest.exceptions.ConsumerGroupOffsetsUtils;
import jakarta.annotation.Nonnull;
import jakarta.inject.Inject;
import jakarta.ws.rs.core.Response;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import org.apache.kafka.clients.admin.Admin;
import org.apache.kafka.clients.admin.AlterConsumerGroupOffsetsResult;
import org.apache.kafka.clients.admin.ListConsumerGroupOffsetsResult;
import org.apache.kafka.clients.admin.ListOffsetsResult;
import org.apache.kafka.clients.admin.OffsetSpec;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsumerGroupOffsetsManagerImpl
implements ConsumerGroupOffsetsManager {
    private final Admin adminClient;
    private static final Logger log = LoggerFactory.getLogger(ConsumerGroupOffsetsManagerImpl.class);

    @Inject
    public ConsumerGroupOffsetsManagerImpl(Admin adminClient) {
        this.adminClient = Objects.requireNonNull(adminClient);
    }

    @Override
    public CompletableFuture<ListConsumerGroupOffsetsResultsData> getConsumerGroupOffsets(String consumerGroupId) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                ListConsumerGroupOffsetsResult listConsumerGroupOffsetsResult = this.adminClient.listConsumerGroupOffsets(consumerGroupId);
                ListConsumerGroupOffsetsResultsData resultData = this.convertToListConsumerGroupOffsetsResultData(listConsumerGroupOffsetsResult, consumerGroupId);
                return resultData;
            }
            catch (KafkaException ex) {
                log.debug(String.format("Got an exception after attempting to get a consumer group id %s offsets: %s", consumerGroupId, ex.getMessage()));
                throw ex;
            }
        });
    }

    @Override
    public CompletableFuture<UpdateConsumerGroupOffsetsResponse> updateConsumerGroupOffsets(@Nonnull String consumerGroupId, @Nonnull UpdateConsumerGroupOffsetsRequest request) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                HashMap<TopicPartition, UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse> tpErrorResponses = new HashMap<TopicPartition, UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse>();
                Map<TopicPartition, OffsetAndMetadata> tpResetOffsets = this.prepareResetOffsets(consumerGroupId, request, tpErrorResponses);
                log.debug("Reset-offset for group: {} are {}", (Object)consumerGroupId, tpResetOffsets);
                log.debug("Partition errors after preparing reset-offsets for group: {} are {}", (Object)consumerGroupId, tpErrorResponses);
                AlterConsumerGroupOffsetsResult alterResult = this.adminClient.alterConsumerGroupOffsets(consumerGroupId, tpResetOffsets);
                return this.prepareUpdateConsumerGroupOffsetsResponse(consumerGroupId, request, alterResult, tpErrorResponses);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private UpdateConsumerGroupOffsetsResponse prepareUpdateConsumerGroupOffsetsResponse(String consumerGroupId, UpdateConsumerGroupOffsetsRequest request, AlterConsumerGroupOffsetsResult alterResult, Map<TopicPartition, UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse> tpErrorResponses) throws InterruptedException {
        ArrayList<UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse> tpResponses = new ArrayList<UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse>();
        for (UpdateConsumerGroupOffsetsRequest.TopicPartitionOffsetData tpRequest : request.getTopicPartitionOffsetRequests()) {
            TopicPartition tp = new TopicPartition(tpRequest.getTopicName(), tpRequest.getPartitionId().intValue());
            if (tpErrorResponses.containsKey(tp)) {
                tpResponses.add(tpErrorResponses.get(tp));
                continue;
            }
            KafkaFuture tpFuture = alterResult.partitionResult(tp);
            try {
                tpFuture.get();
                tpResponses.add(UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse.create(tp.topic(), tp.partition(), Response.Status.OK.getStatusCode(), ""));
            }
            catch (ExecutionException ex) {
                ConsumerGroupOffsetsUtils.Error restError = ConsumerGroupOffsetsUtils.mapAlterOffsetExecutionExceptionToError(ex, tp, consumerGroupId);
                tpResponses.add(UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse.create(tp.topic(), tp.partition(), restError.errorCode(), restError.errorMessage()));
            }
        }
        return UpdateConsumerGroupOffsetsResponse.create(tpResponses);
    }

    private Map<TopicPartition, OffsetAndMetadata> prepareResetOffsets(String consumerGroupId, UpdateConsumerGroupOffsetsRequest request, Map<TopicPartition, UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse> tpErrorResponses) throws InterruptedException {
        ImmutableList<UpdateConsumerGroupOffsetsRequest.TopicPartitionOffsetData> tpOffsetRequests = request.getTopicPartitionOffsetRequests();
        HashMap<TopicPartition, OffsetSpec> fetchStartTps = new HashMap<TopicPartition, OffsetSpec>();
        HashMap<TopicPartition, OffsetSpec> fetchEndTps = new HashMap<TopicPartition, OffsetSpec>();
        for (UpdateConsumerGroupOffsetsRequest.TopicPartitionOffsetData tpOffsetData : tpOffsetRequests) {
            UpdateConsumerGroupOffsetsRequest.OffsetType offsetType = tpOffsetData.getOffsetData().getOffsetType();
            TopicPartition tp = new TopicPartition(tpOffsetData.getTopicName(), tpOffsetData.getPartitionId().intValue());
            if (offsetType.equals((Object)UpdateConsumerGroupOffsetsRequest.OffsetType.OFFSET)) {
                fetchStartTps.put(tp, OffsetSpec.earliest());
                fetchEndTps.put(tp, OffsetSpec.latest());
                continue;
            }
            if (offsetType.equals((Object)UpdateConsumerGroupOffsetsRequest.OffsetType.LATEST)) {
                fetchEndTps.put(tp, OffsetSpec.latest());
                continue;
            }
            if (offsetType.equals((Object)UpdateConsumerGroupOffsetsRequest.OffsetType.EARLIEST)) {
                fetchStartTps.put(tp, OffsetSpec.earliest());
                continue;
            }
            tpErrorResponses.put(tp, UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse.create(tp.topic(), tp.partition(), Response.Status.BAD_REQUEST.getStatusCode(), String.format("%s offset-type is not supported for reset.", new Object[]{offsetType})));
        }
        ListOffsetsResult startOffsetsResult = this.adminClient.listOffsets(fetchStartTps);
        Map<TopicPartition, Long> startOffsets = this.handleListOffsetsResult(consumerGroupId, fetchStartTps.keySet(), startOffsetsResult, tpErrorResponses);
        ListOffsetsResult endOffsetsResult = this.adminClient.listOffsets(fetchEndTps);
        Map<TopicPartition, Long> endOffsets = this.handleListOffsetsResult(consumerGroupId, fetchEndTps.keySet(), endOffsetsResult, tpErrorResponses);
        log.debug("Start-offsets for group: {} are {}", (Object)consumerGroupId, startOffsets);
        log.debug("End-offsets for group: {} are {}", (Object)consumerGroupId, endOffsets);
        log.debug("Partitions with errors while listing offsets for group: {} are {}", (Object)consumerGroupId, tpErrorResponses);
        HashMap<TopicPartition, OffsetAndMetadata> tpResetOffsets = new HashMap<TopicPartition, OffsetAndMetadata>();
        for (UpdateConsumerGroupOffsetsRequest.TopicPartitionOffsetData tpOffsetData : tpOffsetRequests) {
            UpdateConsumerGroupOffsetsRequest.OffsetType offsetType = tpOffsetData.getOffsetData().getOffsetType();
            TopicPartition tp = new TopicPartition(tpOffsetData.getTopicName(), tpOffsetData.getPartitionId().intValue());
            if (tpErrorResponses.containsKey(tp)) continue;
            if (offsetType == UpdateConsumerGroupOffsetsRequest.OffsetType.OFFSET) {
                Long offset = tpOffsetData.getOffsetData().getOffset();
                if (!this.isOffsetValidForPartition(tp, offset, startOffsets, endOffsets, tpErrorResponses)) continue;
                tpResetOffsets.put(tp, new OffsetAndMetadata(offset.longValue()));
                continue;
            }
            if (offsetType == UpdateConsumerGroupOffsetsRequest.OffsetType.LATEST) {
                tpResetOffsets.put(tp, new OffsetAndMetadata(endOffsets.get(tp).longValue()));
                continue;
            }
            tpResetOffsets.put(tp, new OffsetAndMetadata(startOffsets.get(tp).longValue()));
        }
        return tpResetOffsets;
    }

    private Map<TopicPartition, Long> handleListOffsetsResult(String consumerGroupId, Set<TopicPartition> partitions, ListOffsetsResult result, Map<TopicPartition, UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse> tpErrorResponses) throws InterruptedException {
        HashMap<TopicPartition, Long> offsets = new HashMap<TopicPartition, Long>();
        for (TopicPartition tp : partitions) {
            try {
                ListOffsetsResult.ListOffsetsResultInfo info = (ListOffsetsResult.ListOffsetsResultInfo)result.partitionResult(tp).get();
                offsets.put(tp, info.offset());
            }
            catch (ExecutionException e) {
                ConsumerGroupOffsetsUtils.Error restError = ConsumerGroupOffsetsUtils.mapListOffsetExecutionExceptionToError(e, tp, consumerGroupId);
                tpErrorResponses.put(tp, UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse.create(tp.topic(), tp.partition(), restError.errorCode(), restError.errorMessage()));
            }
        }
        return offsets;
    }

    private boolean isOffsetValidForPartition(TopicPartition tp, Long offset, Map<TopicPartition, Long> startOffsets, Map<TopicPartition, Long> endOffsets, Map<TopicPartition, UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse> tpErrorResponses) {
        Long startOffset = Objects.requireNonNull(startOffsets.get(tp));
        Long endOffset = Objects.requireNonNull(endOffsets.get(tp));
        String errorMessage = "";
        if (offset < startOffset) {
            errorMessage = String.format("For %s, new offset %d is less than start-offset %d.", tp, offset, startOffset);
        } else if (offset > endOffset) {
            errorMessage = String.format("For %s, new offset %d is greater than end-offset %d.", tp, offset, endOffset);
        }
        if (!errorMessage.isEmpty()) {
            tpErrorResponses.put(tp, UpdateConsumerGroupOffsetsResponse.TopicPartitionResponse.create(tp.topic(), tp.partition(), Response.Status.BAD_REQUEST.getStatusCode(), errorMessage));
            return false;
        }
        return true;
    }

    private ListConsumerGroupOffsetsResultsData convertToListConsumerGroupOffsetsResultData(ListConsumerGroupOffsetsResult result, String consumerGroupId) {
        ArrayList<TopicPartitionOffsetAndMetaData> offsets = new ArrayList<TopicPartitionOffsetAndMetaData>();
        try {
            Map topicPartitionOffsetMap = (Map)result.partitionsToOffsetAndMetadata().get();
            for (Map.Entry entry : topicPartitionOffsetMap.entrySet()) {
                offsets.add(TopicPartitionOffsetAndMetaData.fromOffsetAndMetadata((TopicPartition)entry.getKey(), (OffsetAndMetadata)entry.getValue()));
            }
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e.getCause());
        }
        return ListConsumerGroupOffsetsResultsData.builder().setOffsets(offsets).setConsumerGroupId(consumerGroupId).build();
    }
}

