/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.kafka.schemaregistry.security.license;

import io.confluent.common.security.license.LicenseEventListener;
import io.confluent.kafka.schemaregistry.security.license.LicensePrivilegeManager;
import io.confluent.kafka.schemaregistry.security.license.entities.V1License;
import io.confluent.kafka.schemaregistry.storage.SchemaRegistry;
import io.confluent.license.InvalidLicenseException;
import io.confluent.license.License;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SchemaRegistryLicenseListener
implements LicenseEventListener {
    private static final Logger log = LoggerFactory.getLogger(SchemaRegistryLicenseListener.class);
    private final SchemaRegistry schemaRegistry;
    private final AtomicReference<List<License>> cachedLicenses = new AtomicReference(new ArrayList());

    public SchemaRegistryLicenseListener(SchemaRegistry schemaRegistry) {
        this.schemaRegistry = schemaRegistry;
        log.info("Initialized SchemaRegistryLicenseAttributionListener");
    }

    public void onLicenseValidated(List<License> licenses) {
        try {
            log.debug("Processing license validation event for licenses {}", licenses);
            List<License> validLicenses = this.filterValidLicenses(licenses);
            List<License> currentLicenses = this.cachedLicenses.get();
            if (!this.areLicenseListsEqual(validLicenses, currentLicenses)) {
                log.info("License attributes changed. Updating schema registry properties. Previous: {}, New: {}", currentLicenses, validLicenses);
                this.cachedLicenses.set(new ArrayList<License>(validLicenses));
                LicensePrivilegeManager.logLicenseMessages(licenses);
                List<V1License> v1Licens = this.convertToV1Licenses(validLicenses);
                this.updateSchemaRegistryProperties(v1Licens);
                log.info("Successfully updated schema registry deployment attributes: {}", validLicenses);
            } else {
                log.debug("License attributes unchanged: {}", currentLicenses);
            }
        }
        catch (Exception e) {
            log.error("Failed to update schema registry deployment attributes", (Throwable)e);
        }
    }

    public void onLicenseValidationFailure(InvalidLicenseException exception) {
        log.warn("License validation failed.", (Throwable)exception);
    }

    public void updateSchemaRegistryProperties(List<V1License> licenses) {
        try {
            if (licenses == null) {
                log.warn("Licenses list is null, skipping property update");
                return;
            }
            if (this.schemaRegistry != null && this.schemaRegistry.properties() != null) {
                log.info("Attempting to update schema registry properties with {} v1License entities", (Object)licenses.size());
                Map srProps = this.schemaRegistry.properties();
                SchemaRegistryLicenseListener.updateDeploymentAttributesProp(srProps, licenses);
            } else {
                log.warn("SchemaRegistry is null, cannot update properties");
            }
        }
        catch (Exception e) {
            log.error("Failed to update schema registry properties", (Throwable)e);
            throw e;
        }
    }

    static void updateDeploymentAttributesProp(Map<String, Object> schemaRegistryProps, List<V1License> newLicenses) {
        if (schemaRegistryProps == null) {
            log.info("Schema registry properties map is null, cannot update deployment attributes");
            return;
        }
        try {
            schemaRegistryProps.compute("schema.registry.metadata.deployment.attributes", (key, existingValue) -> SchemaRegistryLicenseListener.mergeDeploymentAttributes(existingValue, newLicenses));
            log.debug("Updated deployment attributes: {}", newLicenses);
        }
        catch (Exception e) {
            log.error("Failed to update deployment attributes property", (Throwable)e);
        }
    }

    private List<License> filterValidLicenses(List<License> licenses) {
        ArrayList<License> validLicenses = new ArrayList<License>();
        if (licenses == null || licenses.isEmpty()) {
            return validLicenses;
        }
        for (License license : licenses) {
            if (license != null && license.isValid()) {
                validLicenses.add(license);
                log.debug("Added valid license: {}", (Object)license);
                continue;
            }
            log.debug("Skipping invalid license: {}", (Object)license);
        }
        log.debug("Filtered {} valid licenses from {} total licenses", (Object)validLicenses.size(), (Object)licenses.size());
        return validLicenses;
    }

    private List<V1License> convertToV1Licenses(List<License> licenses) {
        ArrayList<V1License> v1Licens = new ArrayList<V1License>();
        if (licenses == null || licenses.isEmpty()) {
            return v1Licens;
        }
        for (License license : licenses) {
            try {
                V1License v1LicenseEntity = new V1License(license);
                v1Licens.add(v1LicenseEntity);
                log.debug("Converted license to v1License: {}", (Object)v1LicenseEntity);
            }
            catch (Exception e) {
                log.warn("Could not convert license to v1License: {}", (Object)license, (Object)e);
            }
        }
        log.debug("Converted {} licenses to {} v1License entities", (Object)licenses.size(), (Object)v1Licens.size());
        return v1Licens;
    }

    private boolean areLicenseListsEqual(List<License> list1, List<License> list2) {
        if (list1 == null && list2 == null) {
            return true;
        }
        if (list1 == null || list2 == null) {
            return false;
        }
        if (list1.size() != list2.size()) {
            return false;
        }
        HashSet<String> signatures1 = new HashSet<String>();
        HashSet<String> signatures2 = new HashSet<String>();
        for (License license : list1) {
            signatures1.add(this.getLicenseSignature(license));
        }
        for (License license : list2) {
            signatures2.add(this.getLicenseSignature(license));
        }
        return signatures1.equals(signatures2);
    }

    private String getLicenseSignature(License license) {
        if (license == null) {
            return "null";
        }
        try {
            return String.valueOf(license.type()) + ":" + String.valueOf(license.deploymentScope());
        }
        catch (Exception e) {
            log.warn("Could not create signature for license: {}", (Object)license, (Object)e);
            return "error:" + license.toString();
        }
    }

    static List<Object> mergeDeploymentAttributes(Object existingValue, List<V1License> newLicenses) {
        HashSet<String> existingStrProps = new HashSet<String>();
        HashSet<Object> existingNonStrProps = new HashSet<Object>();
        SchemaRegistryLicenseListener.processExistingAttributes(existingValue, existingStrProps, existingNonStrProps);
        int originalSize = existingStrProps.size();
        SchemaRegistryLicenseListener.addNewLicenseAttributes(newLicenses, existingStrProps);
        ArrayList<Object> result = new ArrayList<Object>(existingStrProps);
        result.addAll(existingNonStrProps);
        log.debug("Merged attributes: existing={}, new={}, total={}, duplicates_avoided={}", new Object[]{originalSize, newLicenses != null ? newLicenses.size() : 0, result.size(), originalSize + (newLicenses != null ? newLicenses.size() : 0) - result.size()});
        return result;
    }

    /*
     * Enabled aggressive block sorting
     */
    static void processExistingAttributes(Object existingValue, Set<String> existingStrProps, Set<Object> existingObjetProps) {
        if (existingValue == null) {
            log.debug("No existing deployment attributes found, creating new list");
            return;
        }
        if (!(existingValue instanceof List)) {
            log.warn("Existing deployment attributes property has unexpected type: {}", existingValue.getClass());
            throw new IllegalArgumentException("Existing deployment attributes property has unexpected type: " + String.valueOf(existingValue.getClass()));
        }
        List existingList = (List)existingValue;
        Iterator iterator = existingList.iterator();
        while (true) {
            if (!iterator.hasNext()) {
                log.debug("Found {} existing deployment attributes", (Object)existingList.size());
                return;
            }
            Object item = iterator.next();
            if (item instanceof String) {
                existingStrProps.add((String)item);
                continue;
            }
            existingObjetProps.add(item);
        }
    }

    static void addNewLicenseAttributes(List<V1License> newLicenses, Set<String> existingStrProps) {
        if (newLicenses != null) {
            for (V1License license : newLicenses) {
                if (license == null) continue;
                try {
                    String attributeString = license.toAttributeString();
                    existingStrProps.add(attributeString);
                }
                catch (Exception e) {
                    log.warn("Could not convert v1License to attribute string: {}", (Object)license, (Object)e);
                }
            }
        }
    }

    public List<License> getCachedLicenses() {
        return new ArrayList<License>((Collection)this.cachedLicenses.get());
    }

    public Set<String> getCachedAttributes() {
        List<License> licenses = this.cachedLicenses.get();
        HashSet<String> attributes = new HashSet<String>();
        List<V1License> v1Licens = this.convertToV1Licenses(licenses);
        for (V1License license : v1Licens) {
            try {
                attributes.add(license.toAttributeString());
            }
            catch (Exception e) {
                log.warn("Could not convert cached license to attribute string: {}", (Object)license, (Object)e);
            }
        }
        return attributes;
    }
}

