/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.schema.manager;

import com.google.common.annotations.VisibleForTesting;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import io.confluent.dekregistry.client.rest.entities.KeyType;
import io.confluent.dekregistry.storage.DataEncryptionKey;
import io.confluent.dekregistry.storage.DataEncryptionKeyId;
import io.confluent.dekregistry.storage.DekRegistry;
import io.confluent.dekregistry.storage.EncryptionKey;
import io.confluent.dekregistry.storage.EncryptionKeyId;
import io.confluent.dekregistry.storage.KeyEncryptionKey;
import io.confluent.dekregistry.storage.KeyEncryptionKeyId;
import io.confluent.kafka.schemaregistry.encryption.tink.DekFormat;
import io.confluent.kafka.schemaregistry.storage.CloseableIterator;
import io.confluent.kafka.schemaregistry.storage.ConfigKey;
import io.confluent.kafka.schemaregistry.storage.ConfigValue;
import io.confluent.kafka.schemaregistry.storage.DeleteSubjectKey;
import io.confluent.kafka.schemaregistry.storage.DeleteSubjectValue;
import io.confluent.kafka.schemaregistry.storage.KafkaSchemaRegistry;
import io.confluent.kafka.schemaregistry.storage.SchemaKey;
import io.confluent.kafka.schemaregistry.storage.SchemaRegistry;
import io.confluent.kafka.schemaregistry.storage.SchemaRegistryKey;
import io.confluent.kafka.schemaregistry.storage.SchemaRegistryValue;
import io.confluent.kafka.schemaregistry.storage.SchemaValue;
import io.confluent.kafka.schemaregistry.storage.SubjectKey;
import io.confluent.kafka.schemaregistry.storage.exceptions.StoreException;
import io.confluent.kafka.schemaregistry.utils.QualifiedSubject;
import io.confluent.schema.manager.SchemaCacheManagerConfig;
import io.confluent.schema.manager.web.rest.entities.Config;
import io.confluent.schema.manager.web.rest.entities.Dek;
import io.confluent.schema.manager.web.rest.entities.DeleteSubject;
import io.confluent.schema.manager.web.rest.entities.Kek;
import io.confluent.schema.manager.web.rest.entities.Schema;
import io.confluent.schema.manager.web.rest.entities.SchemaRegistryEvent;
import io.kcache.KeyValue;
import io.kcache.KeyValueIterator;
import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.annotation.PreDestroy;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class SchemaCacheManager
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(SchemaCacheManager.class);
    private final ExecutorService executorService = Executors.newFixedThreadPool(1);
    private final ConcurrentHashMap<String, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>>> schemaCachePerContext;
    private final ConcurrentHashMap<String, ConcurrentSkipListMap<Long, List<EncryptionKeyId>>> encryptionKeyCachePerContext;
    private final KafkaSchemaRegistry schemaRegistry;
    private final DekRegistry dekRegistry;
    private final long cacheExpiryIntervalSeconds;
    private final long cacheThresholdLimitMins;
    private ScheduledExecutorService scheduledExpiryExecutorService;

    @Inject
    public SchemaCacheManager(SchemaRegistry schemaRegistry) {
        this(schemaRegistry, SchemaCacheManager.getCacheExpiryInterval((KafkaSchemaRegistry)schemaRegistry), SchemaCacheManager.getCacheThresholdLimit((KafkaSchemaRegistry)schemaRegistry));
    }

    @VisibleForTesting
    public SchemaCacheManager(SchemaRegistry schemaRegistry, long cacheExpiryInterval) {
        this(schemaRegistry, cacheExpiryInterval, cacheExpiryInterval);
    }

    @VisibleForTesting
    public SchemaCacheManager(SchemaRegistry schemaRegistry, long cacheExpiryInterval, long cacheThresholdLimitMins) {
        this.schemaRegistry = (KafkaSchemaRegistry)schemaRegistry;
        this.dekRegistry = (DekRegistry)schemaRegistry.properties().get("dekRegistry");
        this.cacheExpiryIntervalSeconds = cacheExpiryInterval;
        this.cacheThresholdLimitMins = cacheThresholdLimitMins;
        this.schemaCachePerContext = new ConcurrentHashMap();
        this.encryptionKeyCachePerContext = new ConcurrentHashMap();
        this.scheduledExpiryExecutorService = Executors.newScheduledThreadPool(1);
        this.scheduledExpiryExecutorService.scheduleAtFixedRate(this::expireCacheEntries, cacheExpiryInterval, cacheExpiryInterval, TimeUnit.SECONDS);
    }

    @VisibleForTesting
    public int getSchemaCacheSize(String contextName) {
        ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache = this.schemaCachePerContext.get(contextName);
        return cache != null ? cache.size() : -1;
    }

    @VisibleForTesting
    public int getEncryptionKeyCacheSize(String contextName) {
        ConcurrentSkipListMap<Long, List<EncryptionKeyId>> cache = this.encryptionKeyCachePerContext.get(contextName);
        return cache != null ? cache.size() : -1;
    }

    @VisibleForTesting
    public long getExpiryIntervalSeconds() {
        return this.cacheExpiryIntervalSeconds;
    }

    @VisibleForTesting
    public long getCacheExpiryThresholdLimitMins() {
        return this.cacheThresholdLimitMins;
    }

    private void expireCacheEntries() {
        int newSize;
        long expiryThreshold;
        int oldSize;
        ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache;
        for (Map.Entry<String, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>>> entry : this.schemaCachePerContext.entrySet()) {
            cache = entry.getValue();
            oldSize = cache.size();
            if (!cache.isEmpty()) {
                long latestTimestamp = cache.lastKey();
                expiryThreshold = latestTimestamp - TimeUnit.MINUTES.toMillis(this.cacheThresholdLimitMins);
                cache.headMap((Object)expiryThreshold).clear();
            }
            newSize = cache.size();
            log.info("Expired {} entries from cache for context: {}, old size: {}, new size: {}", new Object[]{oldSize - newSize, entry.getKey(), oldSize, newSize});
        }
        for (Map.Entry<String, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>>> entry : this.encryptionKeyCachePerContext.entrySet()) {
            cache = entry.getValue();
            oldSize = cache.size();
            if (!cache.isEmpty()) {
                long latestTimestamp = cache.lastKey();
                expiryThreshold = latestTimestamp - TimeUnit.MINUTES.toMillis(this.cacheThresholdLimitMins);
                cache.headMap((Object)expiryThreshold).clear();
            }
            newSize = cache.size();
            log.info("Expired {} entries from encryption key cache for context: {}, old size: {}, new size: {}", new Object[]{oldSize - newSize, entry.getKey(), oldSize, newSize});
        }
    }

    private String constructCacheKey(String contextName) {
        String tenant = this.schemaRegistry.tenant();
        if ("default".equals(tenant) || contextName.startsWith(tenant + "_")) {
            return contextName;
        }
        return tenant + "_" + contextName;
    }

    public List<SchemaRegistryEvent> getSchemas(String contextName, long ts, long limit) throws StoreException {
        String key = this.constructCacheKey(contextName);
        log.debug("using cache key for getSchemas: {}", (Object)key);
        String tenantPrefix = this.getTenantPrefix(this.schemaRegistry.tenant());
        if (this.schemaCachePerContext.containsKey(key) && this.schemaCachePerContext.get(key).firstKey() > ts) {
            this.schemaCachePerContext.remove(key);
        }
        ConcurrentSkipListMap contextCache = this.schemaCachePerContext.computeIfAbsent(key, k -> {
            this.executorService.submit(() -> {
                try {
                    this.initializeCache(key, tenantPrefix, contextName);
                }
                catch (StoreException e) {
                    this.schemaCachePerContext.remove(key);
                    throw new RuntimeException(e);
                }
            });
            return new ConcurrentSkipListMap();
        });
        return this.serveSchemasFromCache(contextCache, ts, limit);
    }

    private void initializeCache(String key, String tenantPrefix, String contextName) throws StoreException {
        ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache = new ConcurrentSkipListMap<Long, List<SchemaRegistryKey>>();
        KeyValue<String, String> range = this.getRange(tenantPrefix, contextName);
        String start = (String)range.key;
        String end = (String)range.value;
        cache.put(0L, new ArrayList());
        this.loadConfigsIntoCache(start, end, cache, contextName);
        this.loadDeletesIntoCache(start, end, cache, contextName);
        this.loadSchemasIntoCache(start, end, cache, contextName);
        if (!cache.isEmpty()) {
            this.schemaCachePerContext.put(key, cache);
        }
    }

    public List<SchemaRegistryEvent> getKeys(String contextName, long ts, long limit) throws StoreException {
        String key = this.constructCacheKey(contextName);
        log.debug("using cache key for getKeys: {}", (Object)key);
        ConcurrentSkipListMap contextCache = this.encryptionKeyCachePerContext.computeIfAbsent(key, k -> {
            List kekNames = this.dekRegistry.getKekNames(null, true);
            String tenant = this.schemaRegistry.tenant();
            this.executorService.submit(() -> {
                try {
                    this.initializeEncryptionKeysCache(key, contextName, tenant, kekNames);
                }
                catch (StoreException e) {
                    this.encryptionKeyCachePerContext.remove(key);
                    throw new RuntimeException(e);
                }
            });
            return new ConcurrentSkipListMap();
        });
        return this.serveKeysFromCache(contextCache, ts, limit);
    }

    private List<EncryptionKey> getKeyValuesFromKeys(List<EncryptionKeyId> keys) {
        ArrayList<EncryptionKey> values = new ArrayList<EncryptionKey>();
        for (EncryptionKeyId key : keys) {
            EncryptionKey dek = (EncryptionKey)this.dekRegistry.keys().get((Object)key);
            if (!dek.getType().equals((Object)KeyType.DEK)) continue;
            KeyEncryptionKey kek = this.dekRegistry.getKek(((DataEncryptionKey)dek).getKekName(), true);
            values.add((EncryptionKey)kek);
            values.add(dek);
        }
        return values;
    }

    private String removeDefaultContext(String subjectOrPrefix) {
        String ctx = ":.:";
        return subjectOrPrefix.startsWith(ctx) ? subjectOrPrefix.substring(ctx.length()) : subjectOrPrefix;
    }

    private Schema translateSchema(SchemaKey key, SchemaValue value) {
        if (value == null) {
            Schema schema = new Schema(key.getSubject(), key.getVersion(), null, null, true);
            schema.setPermanentDeleted(true);
            return schema;
        }
        return new Schema(value.getSubject(), value.getVersion(), value.getId(), value.getMd5(), value.getSchemaType(), value.getReferences(), value.getMetadata(), value.getRuleSet(), value.getSchema(), value.isDeleted());
    }

    private Config translateConfig(ConfigKey key, ConfigValue value) {
        if (value == null) {
            Config config = new Config(key.getSubject(), null);
            config.setPermanentDeleted(true);
            return config;
        }
        return new Config(value.getSubject(), value.getAlias(), value.isNormalize(), value.isValidateFields(), value.isValidateRules(), value.getCompatibilityLevel(), value.getCompatibilityGroup(), value.getDefaultMetadata(), value.getOverrideMetadata(), value.getDefaultRuleSet(), value.getOverrideRuleSet());
    }

    private DeleteSubject translateDeleteSubject(DeleteSubjectKey key, DeleteSubjectValue value) {
        return new DeleteSubject(key.getSubject(), value.getVersion());
    }

    private SchemaRegistryEvent getSREventFromSRKey(SchemaRegistryKey key) throws StoreException {
        SchemaRegistryValue value = (SchemaRegistryValue)this.schemaRegistry.getLookupCache().get((Object)key);
        SchemaRegistryEvent event = null;
        switch (key.getKeyType()) {
            case SCHEMA: {
                event = this.translateSchema((SchemaKey)key, (SchemaValue)value);
                break;
            }
            case CONFIG: {
                event = this.translateConfig((ConfigKey)key, (ConfigValue)value);
                break;
            }
            case DELETE_SUBJECT: {
                event = this.translateDeleteSubject((DeleteSubjectKey)key, (DeleteSubjectValue)value);
                break;
            }
            default: {
                log.info("Ignoring key type {}", (Object)key.getKeyType());
            }
        }
        return event;
    }

    private SchemaRegistryEvent translateDek(DataEncryptionKeyId key, DataEncryptionKey value) {
        if (value == null) {
            Dek dek = new Dek(key.getKekName(), key.getSubject(), key.getVersion(), null, null, null, true);
            dek.setPermanentDeleted(true);
            return dek;
        }
        return new Dek(value.getKekName(), value.getSubject(), value.getVersion(), value.getAlgorithm(), value.getEncryptedKeyMaterial(), value.getKeyMaterial(), value.isDeleted());
    }

    private SchemaRegistryEvent translateKek(KeyEncryptionKeyId key, KeyEncryptionKey value) {
        if (value == null) {
            Kek kek = new Kek(key.getName(), null, null, null, null, false, true);
            kek.setPermanentDeleted(true);
            return kek;
        }
        return new Kek(value.getName(), value.getKmsType(), value.getKmsKeyId(), value.getKmsProps(), value.getDoc(), value.isShared(), value.isDeleted());
    }

    private List<SchemaRegistryEvent> getSREventFromEncryptionKey(EncryptionKeyId key) {
        EncryptionKey value = (EncryptionKey)this.dekRegistry.keys().get((Object)key);
        ArrayList<SchemaRegistryEvent> events = new ArrayList<SchemaRegistryEvent>();
        switch (key.getType()) {
            case DEK: {
                KeyEncryptionKey kek = this.dekRegistry.getKek(((DataEncryptionKeyId)key).getKekName(), true);
                SchemaRegistryEvent kekEvent = this.translateKek(null, kek);
                events.add(kekEvent);
                SchemaRegistryEvent dekEvent = this.translateDek((DataEncryptionKeyId)key, (DataEncryptionKey)value);
                events.add(dekEvent);
                break;
            }
            case KEK: {
                events.add(this.translateKek((KeyEncryptionKeyId)key, (KeyEncryptionKey)value));
                break;
            }
            default: {
                log.info("Ignoring encryption key type {}", (Object)key.getType());
            }
        }
        return events;
    }

    private List<SchemaRegistryEvent> serveSchemasFromCache(ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache, long ts, long limit) throws StoreException {
        Iterator iterator = cache.tailMap((Object)ts, true).entrySet().iterator();
        ArrayList<SchemaRegistryEvent> schemas = new ArrayList<SchemaRegistryEvent>();
        block0: while (iterator.hasNext() && limit > 0L) {
            Map.Entry entry = iterator.next();
            if ((Long)entry.getKey() < ts) continue;
            long entryTs = (Long)entry.getKey();
            for (SchemaRegistryKey key : (List)entry.getValue()) {
                SchemaRegistryEvent event = this.getSREventFromSRKey(key);
                event.setTimestamp(entryTs);
                schemas.add(event);
                if (--limit != 0L) continue;
                continue block0;
            }
        }
        return schemas;
    }

    private List<SchemaRegistryEvent> serveKeysFromCache(ConcurrentSkipListMap<Long, List<EncryptionKeyId>> cache, long ts, long limit) throws StoreException {
        Iterator iterator = cache.tailMap((Object)ts, true).entrySet().iterator();
        ArrayList<SchemaRegistryEvent> keys = new ArrayList<SchemaRegistryEvent>();
        block0: while (iterator.hasNext() && limit > 0L) {
            Map.Entry entry = iterator.next();
            if ((Long)entry.getKey() < ts) continue;
            for (EncryptionKeyId key : (List)entry.getValue()) {
                List<SchemaRegistryEvent> events = this.getSREventFromEncryptionKey(key);
                events.forEach(e -> e.setTimestamp((Long)entry.getKey()));
                keys.addAll(events);
                if (--limit != 0L) continue;
                continue block0;
            }
        }
        return keys;
    }

    @VisibleForTesting
    public boolean isCacheInitializedForKey(String key) {
        ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache = this.schemaCachePerContext.get(key);
        return cache != null && !cache.isEmpty();
    }

    private String getTenantPrefix(String tenant) {
        return tenant.equals("default") ? "" : tenant + "_";
    }

    private KeyValue<String, String> getRange(String tenantPrefix, String contextName) {
        String end;
        Object start;
        boolean isDefaultContext = contextName.equals(":.:");
        if (isDefaultContext) {
            start = tenantPrefix;
            end = tenantPrefix + "\uffff";
        } else if (contextName.equals(":*:")) {
            start = tenantPrefix + ":.:";
            end = tenantPrefix + ":.\uffff:";
        } else {
            start = tenantPrefix + contextName;
            end = (String)start + "\uffff";
        }
        return new KeyValue(start, (Object)end);
    }

    private KeyValue<SchemaKey, SchemaKey> getSchemaRange(String start, String end) {
        SchemaKey key1 = new SchemaKey(start, 1);
        SchemaKey key2 = new SchemaKey(end, Integer.MAX_VALUE);
        return new KeyValue((Object)key1, (Object)key2);
    }

    private KeyValue<ConfigKey, ConfigKey> getConfigsRange(String start, String end) {
        ConfigKey key1 = new ConfigKey(start);
        ConfigKey key2 = new ConfigKey(end);
        return new KeyValue((Object)key1, (Object)key2);
    }

    private KeyValue<DeleteSubjectKey, DeleteSubjectKey> getDeletesRange(String start, String end) {
        DeleteSubjectKey key1 = new DeleteSubjectKey(start);
        DeleteSubjectKey key2 = new DeleteSubjectKey(end);
        return new KeyValue((Object)key1, (Object)key2);
    }

    private void addSchemasRangeToCache(SchemaKey start, SchemaKey end, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache, String contextName) throws StoreException {
        try (CloseableIterator schemas = this.schemaRegistry.getLookupCache().getAll((Object)start, (Object)end);){
            while (schemas.hasNext()) {
                SchemaValue schemaValue = (SchemaValue)schemas.next();
                SchemaKey key = new SchemaKey(schemaValue.getSubject(), schemaValue.getVersion().intValue());
                if (!this.matches(contextName, schemaValue.getSubject())) continue;
                this.addSchemaRegistryKeyToCache((SchemaRegistryKey)key, schemaValue.getTimestamp(), cache);
            }
        }
    }

    private void addConfigsRangeToCache(ConfigKey start, ConfigKey end, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache, String contextName) throws StoreException {
        try (CloseableIterator configs = this.schemaRegistry.getLookupCache().getAll((Object)start, (Object)end);){
            while (configs.hasNext()) {
                ConfigValue configValue = (ConfigValue)configs.next();
                ConfigKey key = new ConfigKey(configValue.getSubject());
                if (!this.matches(contextName, configValue.getSubject())) continue;
                this.addSchemaRegistryKeyToCache((SchemaRegistryKey)key, configValue.getTimestamp(), cache);
            }
        }
    }

    private void addDeletesRangeToCache(DeleteSubjectKey start, DeleteSubjectKey end, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache, String contextName) throws StoreException {
        try (CloseableIterator deletes = this.schemaRegistry.getLookupCache().getAll((Object)start, (Object)end);){
            while (deletes.hasNext()) {
                DeleteSubjectValue deleteSubjectValue = (DeleteSubjectValue)deletes.next();
                DeleteSubjectKey key = new DeleteSubjectKey(deleteSubjectValue.getSubject());
                if (!this.matches(contextName, deleteSubjectValue.getSubject())) continue;
                this.addSchemaRegistryKeyToCache((SchemaRegistryKey)key, deleteSubjectValue.getTimestamp(), cache);
            }
        }
    }

    private void loadSchemasIntoCache(String start, String end, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache, String contextName) throws StoreException {
        KeyValue<SchemaKey, SchemaKey> schemasRange = this.getSchemaRange(start, end);
        SchemaKey key1 = (SchemaKey)schemasRange.key;
        SchemaKey key2 = (SchemaKey)schemasRange.value;
        this.addSchemasRangeToCache(key1, key2, cache, contextName);
    }

    private void loadConfigsIntoCache(String start, String end, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache, String contextName) throws StoreException {
        KeyValue<ConfigKey, ConfigKey> configsRange = this.getConfigsRange(start, end);
        ConfigKey key1 = (ConfigKey)configsRange.key;
        ConfigKey key2 = (ConfigKey)configsRange.value;
        this.addConfigsRangeToCache(key1, key2, cache, contextName);
    }

    private void loadDeletesIntoCache(String start, String end, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache, String contextName) throws StoreException {
        KeyValue<DeleteSubjectKey, DeleteSubjectKey> deletesRange = this.getDeletesRange(start, end);
        DeleteSubjectKey key1 = (DeleteSubjectKey)deletesRange.key;
        DeleteSubjectKey key2 = (DeleteSubjectKey)deletesRange.value;
        this.addDeletesRangeToCache(key1, key2, cache, contextName);
    }

    private void initializeEncryptionKeysCache(String key, String contextName, String tenant, List<String> kekNames) throws StoreException {
        ConcurrentSkipListMap<Long, List<EncryptionKeyId>> cache = new ConcurrentSkipListMap<Long, List<EncryptionKeyId>>();
        String tenantPrefix = this.getTenantPrefix(tenant);
        KeyValue<String, String> range = this.getRange(tenantPrefix, contextName);
        String start = (String)range.key;
        String end = (String)range.value;
        if (this.dekRegistry == null) {
            return;
        }
        for (String kekName : kekNames) {
            KeyValueIterator dekKIterator;
            DataEncryptionKeyId key1 = new DataEncryptionKeyId(tenant, kekName, start, DekFormat.AES128_GCM, 1);
            DataEncryptionKeyId key2 = new DataEncryptionKeyId(tenant, kekName, end, DekFormat.AES256_SIV, Integer.MAX_VALUE);
            KeyValueIterator deks1 = dekKIterator = this.dekRegistry.keys().range((Object)key1, true, (Object)key2, false);
            try {
                while (deks1.hasNext()) {
                    KeyValue value = (KeyValue)deks1.next();
                    DataEncryptionKey dekKey = (DataEncryptionKey)value.value;
                    if (!((EncryptionKeyId)value.key).getType().equals((Object)KeyType.DEK) || !this.matches(contextName, dekKey.getSubject())) continue;
                    this.addEncryptionKeyIdToCache((EncryptionKeyId)value.key, ((EncryptionKey)value.value).getTimestamp(), cache);
                }
            }
            finally {
                if (deks1 == null) continue;
                deks1.close();
            }
        }
        if (!cache.isEmpty()) {
            this.encryptionKeyCachePerContext.put(key, cache);
        }
    }

    private void addSchemaRegistryKeyToCache(SchemaRegistryKey key, long timestamp, ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache) {
        cache.compute(timestamp, (k, v) -> {
            if (v == null) {
                ArrayList<SchemaRegistryKey> schemaKeys = new ArrayList<SchemaRegistryKey>();
                schemaKeys.add(key);
                return schemaKeys;
            }
            v.add(key);
            return v;
        });
    }

    private void addEncryptionKeyIdToCache(EncryptionKeyId key, long timestamp, ConcurrentSkipListMap<Long, List<EncryptionKeyId>> cache) {
        cache.compute(timestamp, (k, v) -> {
            if (v == null) {
                ArrayList<EncryptionKeyId> encryptionKeyIds = new ArrayList<EncryptionKeyId>();
                encryptionKeyIds.add(key);
                return encryptionKeyIds;
            }
            v.add(key);
            return v;
        });
    }

    void updateCache(SchemaRegistryKey key, SchemaRegistryValue value, long timestamp) {
        if (key instanceof SubjectKey) {
            ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache;
            String allContextsKey;
            ConcurrentSkipListMap<Long, List<SchemaRegistryKey>> cache2;
            String defaultCacheKey;
            String subject = ((SubjectKey)key).getSubject();
            Object context = ".";
            if (!StringUtils.isEmpty((CharSequence)subject)) {
                context = this.getContextNameFromSubject(subject);
            }
            if (((String)context).equals(".")) {
                context = ":.:";
            }
            if ((defaultCacheKey = this.schemaRegistry.tenant() + "_").equals(context)) {
                context = (String)context + ":.:";
            }
            log.debug("update schema cache using key: {}", context);
            if (this.schemaCachePerContext.containsKey(context) && ((cache2 = this.schemaCachePerContext.get(context)).isEmpty() || timestamp >= cache2.lastKey())) {
                this.addSchemaRegistryKeyToCache(key, timestamp, cache2);
            }
            if (this.schemaCachePerContext.containsKey(allContextsKey = this.constructCacheKey(":*:")) && ((cache = this.schemaCachePerContext.get(allContextsKey)).isEmpty() || timestamp >= cache.lastKey())) {
                this.addSchemaRegistryKeyToCache(key, timestamp, cache);
            }
        }
    }

    boolean prefixMatch(String prefix, String subject) {
        if (StringUtils.isEmpty((CharSequence)prefix) || StringUtils.isEmpty((CharSequence)subject)) {
            return false;
        }
        if (prefix.equals(":*:")) {
            return true;
        }
        if (prefix.endsWith(":.:")) {
            return !subject.startsWith(":.");
        }
        return subject.startsWith(prefix);
    }

    void updateKeysCache(EncryptionKeyId key, EncryptionKey value, long timestamp) {
        if (key.getType().equals((Object)KeyType.KEK)) {
            KeyEncryptionKeyId kekId = (KeyEncryptionKeyId)key;
            KeyEncryptionKey kek = (KeyEncryptionKey)value;
            List dekSubjects = this.dekRegistry.getDekSubjects(kekId.getName(), true);
            for (String contextKey : this.encryptionKeyCachePerContext.keySet()) {
                if (!dekSubjects.stream().anyMatch(s -> this.prefixMatch(contextKey, (String)s)) && kek != null && !kek.isDeleted()) continue;
                this.addEncryptionKeyIdToCache(key, timestamp, this.encryptionKeyCachePerContext.get(contextKey));
            }
        } else if (key.getType().equals((Object)KeyType.DEK)) {
            ConcurrentSkipListMap<Long, List<EncryptionKeyId>> cache;
            String allContextsKey;
            ConcurrentSkipListMap<Long, List<EncryptionKeyId>> cache2;
            DataEncryptionKeyId dekKey = (DataEncryptionKeyId)key;
            String subject = dekKey.getSubject();
            String context = this.getContextNameFromSubject(subject);
            if (context.equals(".")) {
                context = ":.:";
            }
            Object tenantPrefix = "";
            if (!key.getTenant().equals("default")) {
                tenantPrefix = key.getTenant() + "_";
            }
            String cacheKey = (String)tenantPrefix + context;
            log.debug("update key cache using key: {}", (Object)cacheKey);
            if (this.encryptionKeyCachePerContext.containsKey(cacheKey) && ((cache2 = this.encryptionKeyCachePerContext.get(cacheKey)).isEmpty() || timestamp >= cache2.lastKey())) {
                this.addEncryptionKeyIdToCache(key, timestamp, cache2);
            }
            if (this.schemaCachePerContext.containsKey(allContextsKey = this.constructCacheKey(":*:")) && ((cache = this.encryptionKeyCachePerContext.get(allContextsKey)).isEmpty() || timestamp >= cache.lastKey())) {
                this.addEncryptionKeyIdToCache(key, timestamp, cache);
            }
        }
    }

    private String getContextNameFromSubject(String subject) {
        String qualifiedContext = QualifiedSubject.create((String)this.schemaRegistry.tenant(), (String)subject).toQualifiedContext();
        if (StringUtils.isEmpty((CharSequence)qualifiedContext) || qualifiedContext.equals(".")) {
            return ".";
        }
        return qualifiedContext;
    }

    protected boolean matches(String contextName, String subject) {
        boolean isAllContexts = contextName.endsWith(":*:");
        if (isAllContexts) {
            return true;
        }
        boolean isDefaultContextPrefix = contextName.endsWith(":.:");
        if (isDefaultContextPrefix) {
            return !subject.startsWith(":.");
        }
        return subject.startsWith(contextName);
    }

    private static long getCacheExpiryInterval(KafkaSchemaRegistry schemaRegistry) {
        try {
            SchemaCacheManagerConfig config = new SchemaCacheManagerConfig(schemaRegistry.config().originalProperties());
            return config.schemaCacheManagerExpiryIntervalSeconds();
        }
        catch (Exception e) {
            log.warn("Failed to get cache expiry interval from configuration, using default: {}", (Object)3600, (Object)e);
            return 3600L;
        }
    }

    private static long getCacheThresholdLimit(KafkaSchemaRegistry schemaRegistry) {
        try {
            SchemaCacheManagerConfig config = new SchemaCacheManagerConfig(schemaRegistry.config().originalProperties());
            return config.schemaCacheManagerCacheThresholdLimitMins();
        }
        catch (Exception e) {
            log.warn("Failed to get cache threshold limit from configuration, using default: {}", (Object)10, (Object)e);
            return 10L;
        }
    }

    public long getBulkContextApiLimit() {
        try {
            SchemaCacheManagerConfig config = new SchemaCacheManagerConfig(this.schemaRegistry.config().originalProperties());
            return config.schemaCacheManagerBulkContextApiLimit();
        }
        catch (Exception e) {
            log.warn("Failed to get bulk context API limit from configuration, using default: {}", (Object)100, (Object)e);
            return 100L;
        }
    }

    @Override
    @PreDestroy
    public void close() throws IOException {
        log.info("Shutting down Schema Cache manager");
        if (this.scheduledExpiryExecutorService != null) {
            this.scheduledExpiryExecutorService.shutdown();
        }
    }
}

