/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.server;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.cache.Cache;
import org.apache.kafka.common.cache.LRUCache;
import org.apache.kafka.common.cache.SynchronizedCache;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.InvalidRequestException;
import org.apache.kafka.common.errors.TelemetryTooLargeException;
import org.apache.kafka.common.errors.ThrottlingQuotaExceededException;
import org.apache.kafka.common.errors.UnknownSubscriptionIdException;
import org.apache.kafka.common.errors.UnsupportedCompressionTypeException;
import org.apache.kafka.common.message.GetTelemetrySubscriptionsResponseData;
import org.apache.kafka.common.message.PushTelemetryResponseData;
import org.apache.kafka.common.metrics.CompoundStat;
import org.apache.kafka.common.metrics.Measurable;
import org.apache.kafka.common.metrics.MeasurableStat;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Max;
import org.apache.kafka.common.metrics.stats.Meter;
import org.apache.kafka.common.metrics.stats.SampledStat;
import org.apache.kafka.common.metrics.stats.WindowedCount;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.requests.GetTelemetrySubscriptionsRequest;
import org.apache.kafka.common.requests.GetTelemetrySubscriptionsResponse;
import org.apache.kafka.common.requests.PushTelemetryRequest;
import org.apache.kafka.common.requests.PushTelemetryResponse;
import org.apache.kafka.common.requests.RequestContext;
import org.apache.kafka.common.utils.Crc32C;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.metrics.ClientMetricsConfigs;
import org.apache.kafka.server.metrics.ClientMetricsInstance;
import org.apache.kafka.server.metrics.ClientMetricsInstanceMetadata;
import org.apache.kafka.server.metrics.ClientMetricsReceiverPlugin;
import org.apache.kafka.server.network.ConnectionDisconnectListener;
import org.apache.kafka.server.util.timer.SystemTimer;
import org.apache.kafka.server.util.timer.SystemTimerReaper;
import org.apache.kafka.server.util.timer.Timer;
import org.apache.kafka.server.util.timer.TimerTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClientMetricsManager
implements AutoCloseable {
    public static final String CLIENT_METRICS_REAPER_THREAD_NAME = "client-metrics-reaper";
    private static final Logger log = LoggerFactory.getLogger(ClientMetricsManager.class);
    private static final List<Byte> SUPPORTED_COMPRESSION_TYPES = List.of(Byte.valueOf(CompressionType.ZSTD.id), Byte.valueOf(CompressionType.LZ4.id), Byte.valueOf(CompressionType.GZIP.id), Byte.valueOf(CompressionType.SNAPPY.id));
    private static final int DEFAULT_CACHE_EXPIRY_MS = 60000;
    protected static final String DEFAULT_SUBSCRIPTION_PREFIX = "default-";
    private final ClientMetricsReceiverPlugin receiverPlugin;
    private final Cache<Uuid, ClientMetricsInstance> clientInstanceCache;
    private final Map<String, Uuid> clientConnectionIdMap;
    private final Timer expirationTimer;
    private final Map<String, SubscriptionInfo> subscriptionMap;
    private final int clientTelemetryMaxBytes;
    private final boolean deltaTemporality;
    private final Time time;
    private final int cacheExpiryMs;
    private final AtomicLong lastCacheErrorLogMs;
    private final Metrics metrics;
    private final ClientMetricsStats clientMetricsStats;
    private final ConnectionDisconnectListener connectionDisconnectListener;
    private final AtomicInteger subscriptionUpdateVersion;

    public ClientMetricsManager(ClientMetricsReceiverPlugin receiverPlugin, int clientTelemetryMaxBytes, Time time, Properties defaultConfigs, int clientInstanceCacheSize, Metrics metrics, boolean deltaTemporality) {
        this(receiverPlugin, clientTelemetryMaxBytes, time, 60000, defaultConfigs, clientInstanceCacheSize, metrics, deltaTemporality);
    }

    ClientMetricsManager(ClientMetricsReceiverPlugin receiverPlugin, int clientTelemetryMaxBytes, Time time, int cacheExpiryMs, Properties defaultConfigs, int clientInstanceCacheSize, Metrics metrics, boolean deltaTemporality) {
        this.receiverPlugin = receiverPlugin;
        this.subscriptionMap = new ConcurrentHashMap<String, SubscriptionInfo>();
        this.subscriptionUpdateVersion = new AtomicInteger(0);
        this.clientInstanceCache = new SynchronizedCache((Cache)new LRUCache(clientInstanceCacheSize));
        this.clientConnectionIdMap = new ConcurrentHashMap<String, Uuid>();
        this.expirationTimer = new SystemTimerReaper(CLIENT_METRICS_REAPER_THREAD_NAME, (Timer)new SystemTimer("client-metrics"));
        this.clientTelemetryMaxBytes = clientTelemetryMaxBytes;
        this.time = time;
        this.cacheExpiryMs = cacheExpiryMs;
        this.lastCacheErrorLogMs = new AtomicLong(0L);
        this.metrics = metrics;
        this.clientMetricsStats = new ClientMetricsStats();
        this.connectionDisconnectListener = new ClientConnectionDisconnectListener();
        this.deltaTemporality = deltaTemporality;
        this.initializeDefaultSubscription(defaultConfigs);
    }

    public Set<String> listClientMetricsResources() {
        return this.subscriptionMap.keySet();
    }

    public void updateSubscription(String subscriptionName, Properties properties) {
        ClientMetricsConfigs.validate(subscriptionName, properties);
        if (properties.isEmpty()) {
            if (this.subscriptionMap.containsKey(subscriptionName)) {
                log.info("Removing subscription [{}] from the subscription map", (Object)subscriptionName);
                this.subscriptionMap.remove(subscriptionName);
                this.subscriptionUpdateVersion.incrementAndGet();
            }
            return;
        }
        this.updateClientSubscription(subscriptionName, new ClientMetricsConfigs(properties));
        this.subscriptionUpdateVersion.incrementAndGet();
    }

    public GetTelemetrySubscriptionsResponse processGetTelemetrySubscriptionRequest(GetTelemetrySubscriptionsRequest request, RequestContext requestContext) {
        long now = this.time.milliseconds();
        Uuid clientInstanceId = Optional.ofNullable(request.data().clientInstanceId()).filter(id -> !id.equals((Object)Uuid.ZERO_UUID)).orElse(this.generateNewClientId());
        ClientMetricsInstance clientInstance = this.clientInstance(clientInstanceId, requestContext);
        try {
            this.validateGetRequest(request, clientInstance, now);
        }
        catch (ApiException exception) {
            return request.getErrorResponse(0, (Throwable)exception);
        }
        clientInstance.lastKnownError(Errors.NONE);
        return this.createGetSubscriptionResponse(clientInstanceId, clientInstance);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public PushTelemetryResponse processPushTelemetryRequest(PushTelemetryRequest request, RequestContext requestContext) {
        Uuid clientInstanceId = request.data().clientInstanceId();
        if (clientInstanceId == null || Uuid.RESERVED.contains(clientInstanceId)) {
            String msg = String.format("Invalid request from the client [%s], invalid client instance id", clientInstanceId);
            return request.getErrorResponse(0, (Throwable)new InvalidRequestException(msg));
        }
        long now = this.time.milliseconds();
        ClientMetricsInstance clientInstance = this.clientInstance(clientInstanceId, requestContext);
        try {
            this.validatePushRequest(request, clientInstance, now);
        }
        catch (ApiException exception) {
            log.debug("Error validating push telemetry request from client [{}]", (Object)clientInstanceId, (Object)exception);
            clientInstance.lastKnownError(Errors.forException((Throwable)exception));
            PushTelemetryResponse pushTelemetryResponse = request.getErrorResponse(0, (Throwable)exception);
            return pushTelemetryResponse;
        }
        finally {
            clientInstance.terminating(request.data().terminating());
        }
        ByteBuffer metrics = request.data().metrics();
        if (metrics != null && metrics.limit() > 0) {
            try {
                long exportTimeStartMs = this.time.hiResClockMs();
                this.receiverPlugin.exportMetrics(requestContext, request);
                this.clientMetricsStats.recordPluginExport(clientInstanceId, this.time.hiResClockMs() - exportTimeStartMs);
            }
            catch (Throwable exception) {
                this.clientMetricsStats.recordPluginErrorCount(clientInstanceId);
                clientInstance.lastKnownError(Errors.INVALID_RECORD);
                log.error("Error exporting client metrics to the plugin for client instance id: {}", (Object)clientInstanceId, (Object)exception);
                return request.errorResponse(0, Errors.INVALID_RECORD);
            }
        }
        clientInstance.lastKnownError(Errors.NONE);
        return new PushTelemetryResponse(new PushTelemetryResponseData());
    }

    public boolean isTelemetryReceiverConfigured() {
        return !this.receiverPlugin.isEmpty();
    }

    public ConnectionDisconnectListener connectionDisconnectListener() {
        return this.connectionDisconnectListener;
    }

    @Override
    public void close() throws Exception {
        this.subscriptionMap.clear();
        this.expirationTimer.close();
        this.clientMetricsStats.unregisterMetrics();
    }

    private void initializeDefaultSubscription(Properties defaultConfigs) {
        try {
            List<String> metricsList = ClientMetricsManager.extractListFromString(defaultConfigs.getOrDefault((Object)"metrics", "").toString());
            List<String> intervalMsList = ClientMetricsManager.extractListFromString(defaultConfigs.getOrDefault((Object)"interval.ms", "").toString());
            List<String> matchList = ClientMetricsManager.extractListFromString(defaultConfigs.getOrDefault((Object)"match", "").toString());
            int matchListSize = matchList.size();
            if (metricsList.size() != intervalMsList.size()) {
                throw new Exception("Size of the Interval Ms List and Metrics List must be same");
            }
            for (int i = 0; i < metricsList.size(); ++i) {
                Properties prop = new Properties();
                prop.put("metrics", metricsList.get(i));
                prop.put("interval.ms", intervalMsList.get(i));
                if (i <= matchListSize - 1 && !matchList.get(i).isEmpty()) {
                    prop.put("match", matchList.get(i));
                }
                this.updateSubscription(DEFAULT_SUBSCRIPTION_PREFIX + i, prop);
            }
        }
        catch (Exception e) {
            log.error("Error initializing default subscription for client metrics.", (Throwable)e);
        }
    }

    static List<String> extractListFromString(String config) {
        if (config.isEmpty()) {
            return Collections.emptyList();
        }
        return Arrays.stream(config.split(";")).map(String::trim).map(s -> s.replaceAll("^\"|\"$", "")).collect(Collectors.toList());
    }

    private void updateClientSubscription(String subscriptionName, ClientMetricsConfigs configs) {
        List metrics = configs.getList("metrics");
        int pushInterval = configs.getInt("interval.ms");
        List clientMatchPattern = configs.getList("match");
        SubscriptionInfo newSubscription = new SubscriptionInfo(subscriptionName, metrics, pushInterval, ClientMetricsConfigs.parseMatchingPatterns(clientMatchPattern));
        this.subscriptionMap.put(subscriptionName, newSubscription);
    }

    private Uuid generateNewClientId() {
        Uuid id = Uuid.randomUuid();
        while (this.clientInstanceCache.get((Object)id) != null) {
            id = Uuid.randomUuid();
        }
        return id;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ClientMetricsInstance clientInstance(Uuid clientInstanceId, RequestContext requestContext) {
        ClientMetricsManager clientMetricsManager;
        ClientMetricsInstance clientInstance = (ClientMetricsInstance)this.clientInstanceCache.get((Object)clientInstanceId);
        if (clientInstance == null) {
            clientMetricsManager = this;
            synchronized (clientMetricsManager) {
                clientInstance = (ClientMetricsInstance)this.clientInstanceCache.get((Object)clientInstanceId);
                if (clientInstance != null) {
                    return clientInstance;
                }
                ClientMetricsInstanceMetadata instanceMetadata = new ClientMetricsInstanceMetadata(clientInstanceId, requestContext);
                clientInstance = this.createClientInstanceAndUpdateCache(clientInstanceId, instanceMetadata, requestContext.connectionId());
            }
        }
        if (clientInstance.subscriptionVersion() < this.subscriptionUpdateVersion.get()) {
            clientMetricsManager = this;
            synchronized (clientMetricsManager) {
                clientInstance = (ClientMetricsInstance)this.clientInstanceCache.get((Object)clientInstanceId);
                if (clientInstance.subscriptionVersion() >= this.subscriptionUpdateVersion.get()) {
                    return clientInstance;
                }
                clientInstance.cancelExpirationTimerTask();
                clientInstance = this.createClientInstanceAndUpdateCache(clientInstanceId, clientInstance.instanceMetadata(), requestContext.connectionId());
            }
        }
        long expirationTimeMs = Math.max(this.cacheExpiryMs, clientInstance.pushIntervalMs() * 3);
        ExpirationTimerTask timerTask = new ExpirationTimerTask(clientInstanceId, requestContext.connectionId(), expirationTimeMs);
        clientInstance.updateExpirationTimerTask(timerTask);
        this.expirationTimer.add((TimerTask)timerTask);
        return clientInstance;
    }

    private ClientMetricsInstance createClientInstanceAndUpdateCache(Uuid clientInstanceId, ClientMetricsInstanceMetadata instanceMetadata, String connectionId) {
        ClientMetricsInstance clientInstance = this.createClientInstance(clientInstanceId, instanceMetadata);
        this.clientMetricsStats.maybeAddClientInstanceMetrics(clientInstanceId);
        this.clientInstanceCache.put((Object)clientInstanceId, (Object)clientInstance);
        this.clientConnectionIdMap.put(connectionId, clientInstanceId);
        return clientInstance;
    }

    private ClientMetricsInstance createClientInstance(Uuid clientInstanceId, ClientMetricsInstanceMetadata instanceMetadata) {
        int pushIntervalMs = 300000;
        HashSet<String> subscribedMetrics = new HashSet<String>();
        boolean allMetricsSubscribed = false;
        int currentSubscriptionVersion = this.subscriptionUpdateVersion.get();
        for (SubscriptionInfo info : this.subscriptionMap.values()) {
            if (!instanceMetadata.isMatch(info.matchPattern())) continue;
            allMetricsSubscribed = allMetricsSubscribed || info.metrics().contains("*");
            subscribedMetrics.addAll(info.metrics());
            pushIntervalMs = Math.min(pushIntervalMs, info.intervalMs());
        }
        if (allMetricsSubscribed) {
            subscribedMetrics.clear();
            subscribedMetrics.add("*");
        }
        int subscriptionId = this.computeSubscriptionId(subscribedMetrics, pushIntervalMs, clientInstanceId);
        return new ClientMetricsInstance(clientInstanceId, instanceMetadata, subscriptionId, currentSubscriptionVersion, subscribedMetrics, pushIntervalMs);
    }

    private int computeSubscriptionId(Set<String> metrics, int pushIntervalMs, Uuid clientInstanceId) {
        byte[] metricsBytes = (metrics.toString() + pushIntervalMs).getBytes(StandardCharsets.UTF_8);
        long computedCrc = Crc32C.compute((byte[])metricsBytes, (int)0, (int)metricsBytes.length);
        return (int)computedCrc ^ clientInstanceId.hashCode();
    }

    private GetTelemetrySubscriptionsResponse createGetSubscriptionResponse(Uuid clientInstanceId, ClientMetricsInstance clientInstance) {
        GetTelemetrySubscriptionsResponseData data = new GetTelemetrySubscriptionsResponseData().setClientInstanceId(clientInstanceId).setSubscriptionId(clientInstance.subscriptionId()).setRequestedMetrics(new ArrayList<String>(clientInstance.metrics())).setAcceptedCompressionTypes(SUPPORTED_COMPRESSION_TYPES).setPushIntervalMs(clientInstance.pushIntervalMs()).setTelemetryMaxBytes(this.clientTelemetryMaxBytes).setDeltaTemporality(this.deltaTemporality).setErrorCode(Errors.NONE.code());
        return new GetTelemetrySubscriptionsResponse(data);
    }

    private void validateGetRequest(GetTelemetrySubscriptionsRequest request, ClientMetricsInstance clientInstance, long timestamp) {
        if (!clientInstance.maybeUpdateGetRequestTimestamp(timestamp) && clientInstance.lastKnownError() != Errors.UNKNOWN_SUBSCRIPTION_ID && clientInstance.lastKnownError() != Errors.UNSUPPORTED_COMPRESSION_TYPE) {
            this.clientMetricsStats.recordThrottleCount(clientInstance.clientInstanceId());
            String msg = String.format("Request from the client [%s] arrived before the next push interval time", request.data().clientInstanceId());
            throw new ThrottlingQuotaExceededException(msg);
        }
    }

    private void validatePushRequest(PushTelemetryRequest request, ClientMetricsInstance clientInstance, long timestamp) {
        if (clientInstance.terminating()) {
            String msg = String.format("Client [%s] sent the previous request with state terminating to TRUE, can not acceptany requests after that", request.data().clientInstanceId());
            throw new InvalidRequestException(msg);
        }
        if (!clientInstance.maybeUpdatePushRequestTimestamp(timestamp) && !request.data().terminating()) {
            this.clientMetricsStats.recordThrottleCount(clientInstance.clientInstanceId());
            String msg = String.format("Request from the client [%s] arrived before the next push interval time", request.data().clientInstanceId());
            throw new ThrottlingQuotaExceededException(msg);
        }
        if (request.data().subscriptionId() != clientInstance.subscriptionId()) {
            this.clientMetricsStats.recordUnknownSubscriptionCount();
            String msg = String.format("Unknown client subscription id for the client [%s]", request.data().clientInstanceId());
            throw new UnknownSubscriptionIdException(msg);
        }
        if (!ClientMetricsManager.isSupportedCompressionType(request.data().compressionType())) {
            String msg = String.format("Unknown compression type [%s] is received in telemetry request from [%s]", request.data().compressionType(), request.data().clientInstanceId());
            throw new UnsupportedCompressionTypeException(msg);
        }
        if (request.data().metrics() != null && request.data().metrics().limit() > this.clientTelemetryMaxBytes) {
            String msg = String.format("Telemetry request from [%s] is larger than the maximum allowed size [%s]", request.data().clientInstanceId(), this.clientTelemetryMaxBytes);
            throw new TelemetryTooLargeException(msg);
        }
    }

    private static boolean isSupportedCompressionType(int id) {
        try {
            CompressionType.forId((int)id);
            return true;
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }

    SubscriptionInfo subscriptionInfo(String subscriptionName) {
        return this.subscriptionMap.get(subscriptionName);
    }

    Collection<SubscriptionInfo> subscriptions() {
        return Collections.unmodifiableCollection(this.subscriptionMap.values());
    }

    ClientMetricsInstance clientInstance(Uuid clientInstanceId) {
        return (ClientMetricsInstance)this.clientInstanceCache.get((Object)clientInstanceId);
    }

    long clientInstanceCacheSize() {
        return this.clientInstanceCache.size();
    }

    int subscriptionUpdateVersion() {
        return this.subscriptionUpdateVersion.get();
    }

    Timer expirationTimer() {
        return this.expirationTimer;
    }

    Map<String, Uuid> clientConnectionIdMap() {
        return this.clientConnectionIdMap;
    }

    final class ClientMetricsStats {
        private static final String GROUP_NAME = "client-metrics";
        static final String INSTANCE_COUNT = "instance-count";
        static final String UNKNOWN_SUBSCRIPTION_REQUEST = "unknown-subscription-request";
        static final String THROTTLE = "throttle";
        static final String PLUGIN_EXPORT = "plugin-export";
        static final String PLUGIN_ERROR = "plugin-error";
        static final String PLUGIN_EXPORT_TIME = "plugin-export-time";
        private final Set<String> sensorsName = ConcurrentHashMap.newKeySet();
        private final List<MetricName> registeredMetricNames = new ArrayList<MetricName>();
        private final Set<String> instanceSensors = Stream.of("throttle", "plugin-export", "plugin-error").collect(Collectors.toSet());

        ClientMetricsStats() {
            Measurable instanceCount = (config, now) -> ClientMetricsManager.this.clientInstanceCache.size();
            MetricName instanceCountMetric = ClientMetricsManager.this.metrics.metricName(INSTANCE_COUNT, GROUP_NAME, "The current number of client metrics instances being managed by the broker");
            ClientMetricsManager.this.metrics.addMetric(instanceCountMetric, instanceCount);
            this.registeredMetricNames.add(instanceCountMetric);
            Sensor unknownSubscriptionRequestCountSensor = ClientMetricsManager.this.metrics.sensor(UNKNOWN_SUBSCRIPTION_REQUEST);
            unknownSubscriptionRequestCountSensor.add((CompoundStat)this.createMeter(ClientMetricsManager.this.metrics, (SampledStat)new WindowedCount(), UNKNOWN_SUBSCRIPTION_REQUEST, Collections.emptyMap()));
            this.sensorsName.add(unknownSubscriptionRequestCountSensor.name());
        }

        public void maybeAddClientInstanceMetrics(Uuid clientInstanceId) {
            if (ClientMetricsManager.this.metrics.getSensor("plugin-export-" + String.valueOf(clientInstanceId)) != null) {
                return;
            }
            Map<String, String> tags = Collections.singletonMap("client_instance_id", clientInstanceId.toString());
            Sensor throttleCount = ClientMetricsManager.this.metrics.sensor("throttle-" + String.valueOf(clientInstanceId));
            throttleCount.add((CompoundStat)this.createMeter(ClientMetricsManager.this.metrics, (SampledStat)new WindowedCount(), THROTTLE, tags));
            this.sensorsName.add(throttleCount.name());
            Sensor pluginExport = ClientMetricsManager.this.metrics.sensor("plugin-export-" + String.valueOf(clientInstanceId));
            pluginExport.add((CompoundStat)this.createMeter(ClientMetricsManager.this.metrics, (SampledStat)new WindowedCount(), PLUGIN_EXPORT, tags));
            pluginExport.add(ClientMetricsManager.this.metrics.metricName("plugin-export-time-avg", GROUP_NAME, "Average time broker spent in invoking plugin exportMetrics call", tags), (MeasurableStat)new Avg());
            pluginExport.add(ClientMetricsManager.this.metrics.metricName("plugin-export-time-max", GROUP_NAME, "Maximum time broker spent in invoking plugin exportMetrics call", tags), (MeasurableStat)new Max());
            this.sensorsName.add(pluginExport.name());
            Sensor pluginErrorCount = ClientMetricsManager.this.metrics.sensor("plugin-error-" + String.valueOf(clientInstanceId));
            pluginErrorCount.add((CompoundStat)this.createMeter(ClientMetricsManager.this.metrics, (SampledStat)new WindowedCount(), PLUGIN_ERROR, tags));
            this.sensorsName.add(pluginErrorCount.name());
        }

        public void recordUnknownSubscriptionCount() {
            this.record(UNKNOWN_SUBSCRIPTION_REQUEST);
        }

        public void recordThrottleCount(Uuid clientInstanceId) {
            this.record(THROTTLE, clientInstanceId);
        }

        public void recordPluginExport(Uuid clientInstanceId, long timeMs) {
            this.record(PLUGIN_EXPORT, clientInstanceId, timeMs);
        }

        public void recordPluginErrorCount(Uuid clientInstanceId) {
            this.record(PLUGIN_ERROR, clientInstanceId);
        }

        public void unregisterClientInstanceMetrics(Uuid clientInstanceId) {
            for (String name : this.instanceSensors) {
                String sensorName = name + "-" + String.valueOf(clientInstanceId);
                ClientMetricsManager.this.metrics.removeSensor(sensorName);
                this.sensorsName.remove(sensorName);
            }
        }

        public void unregisterMetrics() {
            for (MetricName metricName : this.registeredMetricNames) {
                ClientMetricsManager.this.metrics.removeMetric(metricName);
            }
            for (String name : this.sensorsName) {
                ClientMetricsManager.this.metrics.removeSensor(name);
            }
            this.sensorsName.clear();
        }

        private Meter createMeter(Metrics metrics, SampledStat stat, String name, Map<String, String> metricTags) {
            MetricName rateMetricName = metrics.metricName(name + "-rate", GROUP_NAME, String.format("The number of %s per second", name), metricTags);
            MetricName totalMetricName = metrics.metricName(name + "-count", GROUP_NAME, String.format("The total number of %s", name), metricTags);
            return new Meter(stat, rateMetricName, totalMetricName);
        }

        private void record(String metricName) {
            this.record(metricName, null, 1L);
        }

        private void record(String metricName, Uuid clientInstanceId) {
            this.record(metricName, clientInstanceId, 1L);
        }

        private void record(String metricName, Uuid clientInstanceId, long value) {
            Object sensorName = clientInstanceId != null ? metricName + "-" + String.valueOf(clientInstanceId) : metricName;
            Sensor sensor = ClientMetricsManager.this.metrics.getSensor((String)sensorName);
            if (sensor != null) {
                sensor.record((double)value);
            }
        }
    }

    private final class ClientConnectionDisconnectListener
    implements ConnectionDisconnectListener {
        private ClientConnectionDisconnectListener() {
        }

        public void onDisconnect(String connectionId) {
            log.trace("Removing client connection id [{}] from the client instance cache", (Object)connectionId);
            Uuid clientInstanceId = ClientMetricsManager.this.clientConnectionIdMap.remove(connectionId);
            if (clientInstanceId == null) {
                log.trace("Client connection id [{}] is not found in the client instance cache", (Object)connectionId);
                return;
            }
            ClientMetricsManager.this.clientMetricsStats.unregisterClientInstanceMetrics(clientInstanceId);
            ClientMetricsInstance clientInstance = (ClientMetricsInstance)ClientMetricsManager.this.clientInstanceCache.get((Object)clientInstanceId);
            if (clientInstance != null) {
                clientInstance.cancelExpirationTimerTask();
                ClientMetricsManager.this.clientInstanceCache.remove((Object)clientInstanceId);
            }
        }
    }

    public static class SubscriptionInfo {
        private final String name;
        private final Set<String> metrics;
        private final int intervalMs;
        private final Map<String, Pattern> matchPattern;

        public SubscriptionInfo(String name, List<String> metrics, int intervalMs, Map<String, Pattern> matchPattern) {
            this.name = name;
            this.metrics = new HashSet<String>(metrics);
            this.intervalMs = intervalMs;
            this.matchPattern = matchPattern;
        }

        public String name() {
            return this.name;
        }

        public Set<String> metrics() {
            return this.metrics;
        }

        public int intervalMs() {
            return this.intervalMs;
        }

        public Map<String, Pattern> matchPattern() {
            return this.matchPattern;
        }
    }

    private final class ExpirationTimerTask
    extends TimerTask {
        private static final long CACHE_ERROR_LOG_INTERVAL_MS = 300000L;
        private final Uuid clientInstanceId;
        private final String connectionId;

        private ExpirationTimerTask(Uuid clientInstanceId, String connectionId, long delayMs) {
            super(delayMs);
            this.clientInstanceId = clientInstanceId;
            this.connectionId = connectionId;
        }

        public void run() {
            log.trace("Expiration timer task run for client instance id: {}, after delay ms: {}", (Object)this.clientInstanceId, (Object)this.delayMs);
            ClientMetricsManager.this.clientMetricsStats.unregisterClientInstanceMetrics(this.clientInstanceId);
            ClientMetricsManager.this.clientConnectionIdMap.remove(this.connectionId);
            if (!ClientMetricsManager.this.clientInstanceCache.remove((Object)this.clientInstanceId)) {
                long lastErrorMs = ClientMetricsManager.this.lastCacheErrorLogMs.get();
                if (ClientMetricsManager.this.time.milliseconds() - lastErrorMs > 300000L && ClientMetricsManager.this.lastCacheErrorLogMs.compareAndSet(lastErrorMs, ClientMetricsManager.this.time.milliseconds())) {
                    log.warn("Client metrics instance cache cannot find the client instance id: {}. The cache must be at capacity, size: {}. Connection map size: {}", new Object[]{this.clientInstanceId, ClientMetricsManager.this.clientInstanceCache.size(), ClientMetricsManager.this.clientConnectionIdMap.size()});
                }
            }
        }
    }
}

