/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.rbacapi.services;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import io.confluent.mds.DynamicConfigurator;
import io.confluent.rbacapi.entities.ClusterInfo;
import io.confluent.rbacapi.exceptions.ClusterRegistryConflictException;
import io.confluent.rbacapi.exceptions.ClusterRegistryGenericClientErrorException;
import io.confluent.rbacapi.exceptions.ClusterRegistryNoAccessException;
import io.confluent.rbacapi.exceptions.ClusterRegistryNotFoundException;
import io.confluent.rbacapi.exceptions.ClusterRegistryUpdateException;
import io.confluent.rbacapi.exceptions.ClusterRegistryVerifyException;
import io.confluent.rbacapi.services.ClusterRegistryGatekeeper;
import io.confluent.rbacapi.services.ClusterRegistryList;
import io.confluent.rbacapi.utils.ClusterType;
import io.confluent.rbacapi.validation.v1.V1ValidationUtil;
import io.confluent.security.authorizer.Scope;
import jakarta.validation.ConstraintViolation;
import jakarta.validation.ConstraintViolationException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.clients.admin.AlterConfigsResult;
import org.apache.kafka.clients.admin.ConfigEntry;
import org.apache.kafka.common.Reconfigurable;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.security.auth.KafkaPrincipal;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterRegistryService
implements Reconfigurable {
    private static final Set<String> CLUSTER_REGISTRY_CONFIG_SET = ImmutableSet.of((Object)"confluent.metadata.server.cluster.registry.clusters");
    private final ObjectMapper objectMapper;
    private final ClusterRegistryGatekeeper clusterRegistryGatekeeper;
    private final DynamicConfigurator dynamicConfigurator;
    private static final Logger log = LoggerFactory.getLogger(ClusterRegistryService.class);
    private static final V1ValidationUtil validationUtil = new V1ValidationUtil();
    private long dynamicConfigUpdateTimeoutMs = 29000L;
    private volatile ClusterRegistryList volatileClusterRegistryList = new ClusterRegistryList(Collections.emptyList());

    public ClusterRegistryService(ObjectMapper objectMapper, ClusterRegistryGatekeeper clusterRegistryGatekeeper, DynamicConfigurator dynamicConfigurator) {
        this.objectMapper = objectMapper;
        this.clusterRegistryGatekeeper = clusterRegistryGatekeeper;
        this.dynamicConfigurator = dynamicConfigurator;
    }

    public List<ClusterInfo> getClusters(KafkaPrincipal callingPrincipal, ClusterType clusterType) {
        ClusterRegistryList clusterRegistryList = this.volatileClusterRegistryList;
        List<ClusterInfo> clusterInfos = clusterRegistryList.clustersByType(clusterType);
        return this.clusterRegistryGatekeeper.filterClusterInfosBasedOnReadAuthorization(callingPrincipal, clusterInfos);
    }

    public ClusterInfo getNamedCluster(KafkaPrincipal callingPrincipal, String clusterName) {
        List<ClusterInfo> temp;
        ClusterRegistryList clusterRegistryList = this.volatileClusterRegistryList;
        Optional<ClusterInfo> clusterInfo = clusterRegistryList.clusterByName(clusterName);
        if (clusterInfo.isPresent() && (temp = this.clusterRegistryGatekeeper.filterClusterInfosBasedOnReadAuthorization(callingPrincipal, Collections.singletonList(clusterInfo.get()))).size() == 1) {
            return temp.get(0);
        }
        throw new ClusterRegistryNotFoundException(clusterName);
    }

    public Scope lookupScopeByClusterName(String clusterName, KafkaPrincipal callingPrincipal) {
        ClusterInfo clusterInfo = this.getNamedCluster(callingPrincipal, clusterName);
        if (clusterInfo == null) {
            return null;
        }
        return clusterInfo.getScope();
    }

    public String lookupClusterNameByScope(Scope scope, KafkaPrincipal callingPrincipal) {
        List<ClusterInfo> clusterListForPrincipal = this.getClusters(callingPrincipal, ClusterType.NOT_SPECIFIED);
        ClusterInfo clusterInfo = clusterListForPrincipal.stream().filter(c -> c.getScope().equals((Object)scope)).findFirst().orElse(null);
        if (clusterInfo == null) {
            return null;
        }
        return clusterInfo.getClusterName();
    }

    public synchronized void overwriteClusters(KafkaPrincipal callingPrincipal, List<ClusterInfo> clusters) {
        ClusterRegistryList clusterRegistryList = this.volatileClusterRegistryList;
        for (ClusterInfo clusterInfo2 : clusters) {
            try {
                validationUtil.verifyClusterInfo(clusterInfo2);
            }
            catch (ConstraintViolationException cve) {
                String message = cve.getConstraintViolations().stream().map(ConstraintViolation::getMessage).collect(Collectors.joining(","));
                throw new ClusterRegistryGenericClientErrorException(message);
            }
        }
        for (ClusterInfo clusterInfo3 : clusters) {
            if (this.clusterRegistryGatekeeper.canWrite(callingPrincipal, clusterInfo3.getScope())) continue;
            throw new ClusterRegistryNoAccessException(clusterInfo3.getClusterName());
        }
        for (ClusterInfo clusterInfo4 : clusters) {
            String existingName;
            Scope existingScope;
            Optional<ClusterInfo> existingClusterWithSameName = clusterRegistryList.clusterByName(clusterInfo4.getClusterName());
            if (existingClusterWithSameName.isPresent() && !(existingScope = existingClusterWithSameName.get().getScope()).equals((Object)clusterInfo4.getScope())) {
                throw new ClusterRegistryConflictException(String.format("Cluster with name %s already exists with a different scope. Please register with a different name OR unregister the other cluster.", clusterInfo4.getClusterName()));
            }
            Optional<ClusterInfo> existingClusterWithSameScope = clusterRegistryList.clustersByScope(clusterInfo4.getScope());
            if (!existingClusterWithSameScope.isPresent() || (existingName = existingClusterWithSameScope.get().getClusterName()).equals(clusterInfo4.getClusterName())) continue;
            throw new ClusterRegistryConflictException(String.format("Scope %s is already registered with name %s. Please unregister cluster name %s and try again.", clusterInfo4.getScope(), existingName, existingName));
        }
        try {
            ArrayList<ClusterInfo> existingClusters = new ArrayList<ClusterInfo>(clusterRegistryList.clusters());
            for (ClusterInfo cluster : clusters) {
                existingClusters.removeIf(clusterInfo -> clusterInfo.getClusterName().equals(cluster.getClusterName()));
            }
            existingClusters.addAll(clusters);
            ClusterRegistryList clusterRegistryList2 = new ClusterRegistryList(existingClusters);
            this.updateDynamicConfig(clusterRegistryList2);
        }
        catch (ConstraintViolationException e) {
            String string = e.getConstraintViolations().stream().map(ConstraintViolation::getMessage).collect(Collectors.joining(","));
            throw new ClusterRegistryGenericClientErrorException(string);
        }
    }

    public synchronized void deleteNamedCluster(KafkaPrincipal callingPrincipal, String clusterName) {
        ClusterRegistryList clusterRegistryList = this.volatileClusterRegistryList;
        Optional<ClusterInfo> clusterInfo = clusterRegistryList.clusterByName(clusterName);
        if (!clusterInfo.isPresent()) {
            return;
        }
        if (!this.clusterRegistryGatekeeper.canWrite(callingPrincipal, clusterInfo.get().getScope())) {
            throw new ClusterRegistryNoAccessException(clusterName);
        }
        ArrayList<ClusterInfo> clusters = new ArrayList<ClusterInfo>(clusterRegistryList.clusters());
        clusters.removeIf(ci -> ci.getClusterName().equals(clusterName));
        this.updateDynamicConfig(new ClusterRegistryList(clusters));
    }

    public Set<String> reconfigurableConfigs() {
        return CLUSTER_REGISTRY_CONFIG_SET;
    }

    public void configure(Map<String, ?> configs) {
        String clusterRegString = this.extractClusterRegString(configs.get("cluster.registry.clusters"));
        this.volatileClusterRegistryList = this.parseJson(clusterRegString);
    }

    public void validateReconfiguration(Map<String, ?> configs) throws ConfigException {
        String clusterRegString = this.extractClusterRegString(configs.get("confluent.metadata.server.cluster.registry.clusters"));
        this.parseJson(clusterRegString);
    }

    public void reconfigure(Map<String, ?> configs) {
        String clusterRegString = this.extractClusterRegString(configs.get("confluent.metadata.server.cluster.registry.clusters"));
        this.volatileClusterRegistryList = this.parseJson(clusterRegString);
    }

    private String extractClusterRegString(Object clusterRegConfigObj) {
        if (clusterRegConfigObj instanceof String) {
            String clusterRegString = (String)clusterRegConfigObj;
            if (StringUtils.isBlank((CharSequence)clusterRegString)) {
                throw new ConfigException("cluster.registry.clusters", clusterRegConfigObj, "Expected a non-blank JSON String");
            }
            return clusterRegString;
        }
        throw new ConfigException("cluster.registry.clusters", clusterRegConfigObj, "Expected a non-null JSON String");
    }

    private ClusterRegistryList parseJson(String clusterRegJson) {
        try {
            List clusters = (List)this.objectMapper.readValue(clusterRegJson, (TypeReference)new TypeReference<List<ClusterInfo>>(){});
            return new ClusterRegistryList(clusters);
        }
        catch (IOException e) {
            throw new ConfigException("cluster.registry.clusters", (Object)clusterRegJson, e.getMessage());
        }
        catch (ConstraintViolationException e) {
            throw new ConfigException("cluster.registry.clusters", (Object)clusterRegJson, e.getConstraintViolations().stream().map(ConstraintViolation::getMessage).collect(Collectors.joining(",")));
        }
    }

    private void updateDynamicConfig(ClusterRegistryList clusterRegistryList) {
        if (this.volatileClusterRegistryList.equals(clusterRegistryList)) {
            log.debug("Update to cluster registry is a no-op. Exiting early.");
            return;
        }
        try {
            String configValue = this.objectMapper.writeValueAsString(clusterRegistryList.clusters());
            ConfigEntry configEntry = new ConfigEntry("confluent.metadata.server.cluster.registry.clusters", configValue);
            AlterConfigsResult alterConfigsResult = this.dynamicConfigurator.setClusterConfig(Collections.singleton(configEntry));
            long writeStarted = System.currentTimeMillis();
            try {
                alterConfigsResult.all().get(this.dynamicConfigUpdateTimeoutMs, TimeUnit.MILLISECONDS);
            }
            catch (TimeoutException e) {
                log.debug("Timeout for updates to dynamic config.", (Throwable)e);
                if (this.volatileClusterRegistryList.equals(clusterRegistryList)) {
                    log.debug("Local cluster registry state updated by a dynamic config reconfigure.");
                    return;
                }
                throw new ClusterRegistryUpdateException("Dynamic config update failed becauseof a timeout in future.get().", e);
            }
            catch (InterruptedException | ExecutionException e) {
                throw new ClusterRegistryUpdateException("Dynamic config update failed.", e);
            }
            long writeCompleted = System.currentTimeMillis();
            try {
                long pollingBudget = this.dynamicConfigUpdateTimeoutMs - (writeCompleted - writeStarted);
                this.pollForBackgroundUpdateToDynamicConfig(clusterRegistryList, pollingBudget);
            }
            catch (TimeoutException e) {
                throw new ClusterRegistryVerifyException("Timeout in verifying that local Cluster Registry state was updated by a dynamic config reconfigure.", e);
            }
        }
        catch (JsonProcessingException ignored) {
            log.debug("Unexpected JSON error in ClusterRegistry update.", (Throwable)ignored);
        }
    }

    private void pollForBackgroundUpdateToDynamicConfig(ClusterRegistryList expectedClusterRegistryList, long timeoutMs) throws TimeoutException {
        long start = System.currentTimeMillis();
        while (System.currentTimeMillis() - start < timeoutMs && !this.volatileClusterRegistryList.equals(expectedClusterRegistryList)) {
            try {
                Thread.sleep(20L);
            }
            catch (InterruptedException interruptedException) {}
        }
        if (!this.volatileClusterRegistryList.equals(expectedClusterRegistryList)) {
            throw new TimeoutException(String.format("Timed out after %dms trying to verify Cluster Registry update.", timeoutMs));
        }
    }

    @VisibleForTesting
    public List<ClusterInfo> getAllClusters() {
        return this.volatileClusterRegistryList.clusters();
    }

    @VisibleForTesting
    public void resetClusters() {
        this.volatileClusterRegistryList = new ClusterRegistryList(Collections.emptyList());
    }

    @VisibleForTesting
    public void setDynamicConfigUpdateTimeoutMs(long dynamicConfigUpdateTimeoutMs) {
        this.dynamicConfigUpdateTimeoutMs = dynamicConfigUpdateTimeoutMs;
    }

    public List<ClusterInfo> getUnrestrictedClusters(ClusterType clusterType) {
        return this.volatileClusterRegistryList.clustersByType(clusterType);
    }
}

