/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.connect.replicator;

import com.google.common.annotations.VisibleForTesting;
import io.confluent.connect.replicator.DeadlineManager;
import io.confluent.connect.replicator.KafkaConfigs;
import io.confluent.connect.replicator.ReplicatorSourceTaskConfig;
import io.confluent.connect.replicator.Translator;
import io.confluent.connect.replicator.metrics.ConfluentReplicatorMetrics;
import io.confluent.connect.replicator.metrics.ConfluentReplicatorTaskMetricsGroup;
import io.confluent.connect.replicator.offsets.ConsumerOffsetsTopicCommitter;
import io.confluent.connect.replicator.offsets.ConsumerOffsetsTranslator;
import io.confluent.connect.replicator.offsets.ConsumerTimestampsCommitter;
import io.confluent.connect.replicator.offsets.OffsetManager;
import io.confluent.connect.replicator.schemas.SchemaTranslator;
import io.confluent.connect.replicator.util.NewReplicatorAdminClient;
import io.confluent.connect.replicator.util.ReplicatorAdminClient;
import io.confluent.connect.replicator.util.TopicMetadata;
import io.confluent.connect.replicator.util.TranslatorMonitor;
import io.confluent.connect.replicator.util.Utils;
import io.confluent.connect.replicator.util.Version;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetOutOfRangeException;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.errors.CorruptRecordException;
import org.apache.kafka.common.errors.InvalidConfigurationException;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.header.Header;
import org.apache.kafka.common.header.Headers;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.SchemaAndValue;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.header.ConnectHeaders;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.apache.kafka.connect.source.SourceTaskContext;
import org.apache.kafka.connect.storage.Converter;
import org.apache.kafka.connect.storage.HeaderConverter;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplicatorSourceTask
extends SourceTask {
    private static final Logger log = LoggerFactory.getLogger(ReplicatorSourceTask.class);
    public static final String REPLICATOR_ID_HEADER = "__replicator_id";
    public static final Pattern PROVENANCE_HEADER_PATTERN = Pattern.compile("([^,]+),([^,]+),([^,]+)");
    public static final Pattern FILTER_OVERRIDE_PATTERN = Pattern.compile("([^,]+),([^,]+),([^-]*)-([^;]*);?");
    private String taskId;
    private ReplicatorSourceTaskConfig config;
    private Converter sourceKeyConverter;
    private Converter sourceValueConverter;
    private ReplicatorAdminClient sourceAdminClient;
    private ReplicatorAdminClient destAdminClient;
    private final TranslatorMonitor translatorMonitor;
    private String sourceClusterId;
    private String destClusterId;
    DeadlineManager deadlineManager = new DeadlineManager();
    private final Set<String> sourceTopicsNeedingExpansion = new HashSet<String>();
    private final Set<String> managedSourceTopics = new HashSet<String>();
    private volatile Consumer<byte[], byte[]> consumer;
    private Map<String, Translator> translators = new HashMap<String, Translator>();
    private ConsumerTimestampsCommitter timestampsCommitter;
    private ConsumerOffsetsTopicCommitter offsetTopicCommitter;
    private volatile boolean isStarting = false;
    private HeaderConverter converter;
    private List<FilterOverride> filterOverrides = new ArrayList<FilterOverride>();
    private ConfluentReplicatorTaskMetricsGroup replicatorTaskMetricsGroup;
    private OffsetManager offsetManager = null;

    public ReplicatorSourceTask() {
        this(null, null, null, Time.SYSTEM, null, null, null, null, new TranslatorMonitor(10000L), null, null, null);
    }

    ReplicatorSourceTask(ReplicatorSourceTaskConfig config, SourceTaskContext context, String taskId, Time time, Consumer<byte[], byte[]> consumer, ConsumerOffsetsTranslator offsetsReplicator, ReplicatorAdminClient sourceAdminClient, ReplicatorAdminClient destAdminClient, TranslatorMonitor translatorMonitor, Converter sourceKeyConverter, Converter sourceValueConverter, ConfluentReplicatorTaskMetricsGroup metricsGroup) {
        this(config, context, taskId, time, consumer, offsetsReplicator, sourceAdminClient, destAdminClient, translatorMonitor, sourceKeyConverter, sourceValueConverter, null, metricsGroup);
    }

    ReplicatorSourceTask(ReplicatorSourceTaskConfig config, SourceTaskContext context, String taskId, Time time, Consumer<byte[], byte[]> consumer, ConsumerOffsetsTranslator offsetsReplicator, ReplicatorAdminClient sourceAdminClient, ReplicatorAdminClient destAdminClient, TranslatorMonitor translatorMonitor, Converter sourceKeyConverter, Converter sourceValueConverter, ConsumerTimestampsCommitter timestampsCommiter, ConfluentReplicatorTaskMetricsGroup metricsGroup) {
        this.config = config;
        this.context = context;
        this.taskId = taskId;
        this.deadlineManager.setTime(time);
        this.consumer = consumer;
        if (offsetsReplicator != null) {
            this.translators.put(offsetsReplicator.topic(), offsetsReplicator);
        }
        this.sourceAdminClient = sourceAdminClient;
        this.destAdminClient = destAdminClient;
        this.translatorMonitor = translatorMonitor;
        this.sourceKeyConverter = sourceKeyConverter;
        this.sourceValueConverter = sourceValueConverter;
        this.timestampsCommitter = timestampsCommiter;
        this.replicatorTaskMetricsGroup = metricsGroup;
        this.offsetManager = new OffsetManager(config, consumer, context, this.deadlineManager);
    }

    ReplicatorSourceTask(ReplicatorSourceTaskConfig config, SourceTaskContext context, String taskId, Time time, Consumer<byte[], byte[]> consumer, ConsumerOffsetsTranslator offsetsReplicator, ReplicatorAdminClient sourceAdminClient, ReplicatorAdminClient destAdminClient, TranslatorMonitor translatorMonitor, Converter sourceKeyConverter, Converter sourceValueConverter, ConsumerTimestampsCommitter timestampsCommiter, ConsumerOffsetsTopicCommitter offsetTopicCommitter, ConfluentReplicatorTaskMetricsGroup metricsGroup) {
        this.config = config;
        this.context = context;
        this.taskId = taskId;
        this.deadlineManager.setTime(time);
        this.consumer = consumer;
        if (offsetsReplicator != null) {
            this.translators.put(offsetsReplicator.topic(), offsetsReplicator);
        }
        this.sourceAdminClient = sourceAdminClient;
        this.destAdminClient = destAdminClient;
        this.translatorMonitor = translatorMonitor;
        this.sourceKeyConverter = sourceKeyConverter;
        this.sourceValueConverter = sourceValueConverter;
        this.timestampsCommitter = timestampsCommiter;
        this.offsetTopicCommitter = offsetTopicCommitter;
        this.replicatorTaskMetricsGroup = metricsGroup;
        this.offsetManager = new OffsetManager(config, consumer, context, this.deadlineManager);
    }

    ReplicatorSourceTask(ReplicatorSourceTaskConfig config, SourceTaskContext context, String taskId, Time time, Consumer<byte[], byte[]> consumer, ConsumerOffsetsTranslator offsetsReplicator, ReplicatorAdminClient sourceAdminClient, ReplicatorAdminClient destAdminClient, TranslatorMonitor translatorMonitor, SchemaTranslator schemaTranslator, Converter sourceKeyConverter, Converter sourceValueConverter, ConsumerTimestampsCommitter timestampsCommiter, ConsumerOffsetsTopicCommitter offsetTopicCommitter, ConfluentReplicatorTaskMetricsGroup metricsGroup) {
        this.config = config;
        this.context = context;
        this.taskId = taskId;
        this.deadlineManager.setTime(time);
        this.consumer = consumer;
        if (offsetsReplicator != null) {
            this.translators.put(offsetsReplicator.topic(), offsetsReplicator);
        }
        if (schemaTranslator != null) {
            this.translators.put(schemaTranslator.topic(), schemaTranslator);
        }
        this.sourceAdminClient = sourceAdminClient;
        this.destAdminClient = destAdminClient;
        this.translatorMonitor = translatorMonitor;
        this.sourceKeyConverter = sourceKeyConverter;
        this.sourceValueConverter = sourceValueConverter;
        this.timestampsCommitter = timestampsCommiter;
        this.offsetTopicCommitter = offsetTopicCommitter;
        this.replicatorTaskMetricsGroup = metricsGroup;
        this.offsetManager = new OffsetManager(config, consumer, context, this.deadlineManager);
    }

    public void initialize(SourceTaskContext context) {
        log.debug("Initializing SourceTaskContext for ReplicatorSourceTask");
        super.initialize(context);
        this.offsetManager.setContext(context);
    }

    public String version() {
        return Version.getVersion();
    }

    public synchronized void start(Map<String, String> props) {
        try {
            log.debug("Gathering configs for Replicator source task");
            this.config = this.taskConfig(props);
            this.offsetManager.setConfig(this.config);
        }
        catch (ConfigException e) {
            throw new ConnectException("Failed to start Kafka replicator task due to configuration error", (Throwable)e);
        }
        this.taskId = this.config.getTaskId();
        log.info("Starting Replicator source task {}", (Object)this.taskId);
        this.sourceKeyConverter = this.config.getSourceKeyConverter();
        this.sourceValueConverter = this.config.getSourceValueConverter();
        log.debug("Creating destination admin client...");
        this.destAdminClient = this.createAdminClient(this.destAdminClient, this.config::dstAdminClientConfig);
        log.debug("Creating source admin client...");
        this.sourceAdminClient = this.createAdminClient(this.sourceAdminClient, this.config::srcAdminClientConfig);
        this.setClusterIds();
        if (this.config.isProvenanceHeaderEnabled()) {
            log.debug("provenance.header.enable is set to true. Setting up filter overrides");
            String overrides = this.config.getString("provenance.header.filter.overrides");
            ReplicatorSourceTask.parseFilterOverrides(overrides, this.filterOverrides);
        }
        log.debug("Building source consumer for Replicator source task {}", (Object)this.taskId);
        this.consumer = this.buildSourceConsumer(this.config);
        log.debug("Building Consumer Offsets Translator for Replicator source task {}", (Object)this.taskId);
        ConsumerOffsetsTranslator offsetsReplicator = new ConsumerOffsetsTranslator(this.config.originalsStrings(), this.taskId, this.deadlineManager.getTime(), this.config.getOffsetTranslatorBatchPeriodMs(), this.config.getOffsetTranslatorBatchSize());
        this.translators.put(offsetsReplicator.topic(), offsetsReplicator);
        this.offsetManager.setConsumer(this.consumer);
        if (this.config.isOffsetTimestampsCommitEnabled()) {
            log.debug("offset.timestamps.commit is set to true. Building the Consumer Timestamps Committer");
            String groupId = props.get("src.consumer.group.id");
            if (groupId == null) {
                groupId = this.config.getName();
            }
            Map srcConfigs = this.config.originalsWithPrefix(KafkaConfigs.KafkaCluster.SOURCE.prefix());
            this.timestampsCommitter = new ConsumerTimestampsCommitter(groupId, srcConfigs, this.sourceAdminClient);
        }
        if (this.config.isOffsetTopicCommitEnabled()) {
            log.debug("offset.topic.commit is set to true. Building the Consumer Offsets Topic Committer");
            int offsetTopicCommitPeriod = this.config.getOffsetTopicCommitBatchPeriodMs();
            this.offsetTopicCommitter = new ConsumerOffsetsTopicCommitter(this.consumer, offsetTopicCommitPeriod > -1 && this.config.isProvenanceHeaderEnabled() && this.config.getTopicPreservePartitions(), this.deadlineManager.getTime(), offsetTopicCommitPeriod);
        }
        if (this.config.getSchemaRegistryTopic() != null) {
            log.info("Registering schema translator for topic {}", (Object)this.config.getSchemaRegistryTopic());
            this.translators.put(this.config.getSchemaRegistryTopic(), new SchemaTranslator(this.config, this.deadlineManager.getTime()));
        }
        this.converter = this.config.getSourceHeaderConverter();
        log.debug("Fetching source assignments for Replicator source task {}", (Object)this.taskId);
        List sourceAssignment = this.config.getPartitionAssignment().partitions();
        this.doStart(sourceAssignment);
        log.info("Started kafka replicator task {} replicating topic partitions {}", (Object)this.taskId, (Object)sourceAssignment);
        log.info("Setting up metrics recording for task {}...", (Object)this.taskId);
        if (this.replicatorTaskMetricsGroup == null) {
            this.replicatorTaskMetricsGroup = new ConfluentReplicatorTaskMetricsGroup(this.config, this.taskId, sourceAssignment, new ConfluentReplicatorMetrics(this.taskId, this.deadlineManager.getTime()), this.sourceClusterId, this.destClusterId, this.config.getName());
            this.replicatorTaskMetricsGroup.setupMetrics();
        }
        log.info("Successfully set up metrics recording for task {}", (Object)this.taskId);
        log.info("Successfully started up Replicator source task {}", (Object)this.taskId);
    }

    @VisibleForTesting
    protected void setClusterIds() {
        try {
            this.sourceClusterId = this.sourceAdminClient.clusterId();
        }
        catch (Exception e) {
            log.error("Failed to obtain source cluster ID", (Throwable)e);
        }
        if (this.sourceClusterId == null) {
            throw new ConnectException("Failed to obtain source cluster ID, please restart the source Kafka cluster");
        }
        try {
            this.destClusterId = this.destAdminClient.clusterId();
        }
        catch (Exception e) {
            log.error("Failed to obtain destination cluster ID", (Throwable)e);
        }
        if (this.destClusterId == null) {
            throw new ConnectException("Failed to obtain destination cluster ID, please restart the destination Kafka cluster");
        }
        if (this.sourceClusterId.equals(this.destClusterId)) {
            log.warn("The source and destination cluster IDs match. This is normal when replicating to different topics in the same cluster. Otherwise, check your source and destination cluster properties.");
        }
        log.info("Source cluster ID: {}", (Object)this.sourceClusterId);
        log.info("Destination cluster ID: {}", (Object)this.destClusterId);
    }

    protected static void parseFilterOverrides(String overrides, List<FilterOverride> filterOverrides) {
        if (overrides != null && !overrides.isEmpty()) {
            Matcher matcher = FILTER_OVERRIDE_PATTERN.matcher(overrides);
            while (matcher.find()) {
                filterOverrides.add(new FilterOverride(matcher));
            }
        }
    }

    synchronized void doStart(Collection<TopicPartition> sourceAssignment) {
        this.isStarting = true;
        try {
            log.debug("Assigning the consumer of Replicator source task {} to the following source assignment {}", (Object)this.taskId, sourceAssignment);
            this.consumer.assign(sourceAssignment);
            this.watchTopicsForAssignedPartitions(sourceAssignment);
            this.initializeAssignedPartitions(sourceAssignment);
        }
        finally {
            this.isStarting = false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized List<SourceRecord> poll() {
        log.debug("Beginning poll for Replicator source task {}", (Object)this.taskId);
        try {
            if (this.offsetTopicCommitter != null) {
                this.offsetTopicCommitter.checkCommit();
            }
            log.debug("Retrying topic expansion if needed...");
            this.retryTopicExpansionIfNeeded();
            log.debug("Updating topic configs if needed...");
            this.updateTopicConfigsIfNeeded();
            log.debug("Attempting to resume any paused partitions...");
            this.maybeResumePartitions();
            long now = this.deadlineManager.getMilliSeconds();
            long maxWaitTimeMillis = this.getMaxPollWaitTimeMillis(now);
            ConsumerRecords records = ConsumerRecords.empty();
            if (!this.consumer.assignment().isEmpty()) {
                records = this.consumer.poll(Duration.ofMillis(maxWaitTimeMillis));
            }
            if (log.isDebugEnabled()) {
                Set<String> topics = ReplicatorSourceTask.topicNamesFor((ConsumerRecords<byte[], byte[]>)records);
                log.debug("Read {} records from {} topics: {}", new Object[]{records.count(), topics.size(), topics});
            }
            boolean preservePartitions = this.config.getTopicPreservePartitions();
            ArrayList<SourceRecord> sourceRecords = new ArrayList<SourceRecord>(records.count());
            int filteredRecords = 0;
            for (TopicPartition topicPartition : records.partitions()) {
                if (this.skipRecordReplication(topicPartition, (ConsumerRecords<byte[], byte[]>)records)) {
                    log.trace("Skipping record replication for topic partition {} because it's part of the __consumer_timestamps topic or part of a schema topic", (Object)topicPartition);
                    continue;
                }
                String sourceTopic = topicPartition.topic();
                String destTopic = ReplicatorSourceTask.toDestTopic(sourceTopic, this.config);
                int partition = topicPartition.partition();
                Map connectSourcePartition = Utils.toConnectPartition((String)sourceTopic, (int)partition);
                long numPerPartitionRecordsToConnect = 0L;
                for (ConsumerRecord record : records.records(topicPartition)) {
                    if (this.config.isProvenanceHeaderEnabled() && ReplicatorSourceTask.shouldFilterRecord((ConsumerRecord<byte[], byte[]>)record, this.destClusterId, destTopic, this.filterOverrides)) {
                        log.trace("Filtering record from topic partition {} due to provenance headers", (Object)topicPartition);
                        ++filteredRecords;
                        if (this.offsetTopicCommitter == null) continue;
                        this.offsetTopicCommitter.maybeCommitFilteredRecordByReplicator(record.topic(), record.partition(), record.offset());
                        continue;
                    }
                    ++numPerPartitionRecordsToConnect;
                    Map connectOffset = Utils.toConnectOffset((long)record.offset());
                    Pair<SchemaAndValue, SchemaAndValue> converted = this.convertKeyValue((ConsumerRecord<byte[], byte[]>)record, sourceTopic);
                    Long timestamp = ReplicatorSourceTask.timestampFromRecord((ConsumerRecord<byte[], byte[]>)record);
                    Integer destPartition = preservePartitions ? Integer.valueOf(partition) : null;
                    ConnectHeaders connectHeaders = ReplicatorSourceTask.toConnectHeaders(this.sourceClusterId, sourceTopic, (ConsumerRecord<byte[], byte[]>)record, this.converter, this.config.isProvenanceHeaderEnabled(), now);
                    sourceRecords.add(new SourceRecord(connectSourcePartition, connectOffset, destTopic, destPartition, ((SchemaAndValue)converted.getKey()).schema(), ((SchemaAndValue)converted.getKey()).value(), ((SchemaAndValue)converted.getValue()).schema(), ((SchemaAndValue)converted.getValue()).value(), timestamp, (Iterable)connectHeaders));
                }
                if (this.offsetTopicCommitter == null) continue;
                this.offsetTopicCommitter.updateNumUncommittedRecords(topicPartition, numPerPartitionRecordsToConnect);
            }
            if (log.isDebugEnabled()) {
                log.debug("Replicating {} records{}, filtering {} records due to provenance headers", new Object[]{sourceRecords.size(), this.config.isProvenanceHeaderEnabled() ? " and adding provenance headers" : "", filteredRecords});
            }
            ArrayList<SourceRecord> arrayList = sourceRecords;
            return arrayList;
        }
        catch (OffsetOutOfRangeException e) {
            Map outOfRangePartitions = e.offsetOutOfRangePartitions();
            log.warn("Consumer from source cluster detected out of range partitions: {}", (Object)outOfRangePartitions);
            this.offsetManager.seekToBeginning(outOfRangePartitions.keySet());
            List<SourceRecord> list = Collections.emptyList();
            return list;
        }
        catch (WakeupException e) {
            log.debug("Kafka replicator task {} woken up", (Object)this.taskId);
            List<SourceRecord> list = Collections.emptyList();
            return list;
        }
        finally {
            this.translateCollectedRecords();
        }
    }

    long getMaxPollWaitTimeMillis(long now) {
        List<Long> deadlines = this.translators.values().stream().map(Translator::nextDeadline).collect(Collectors.toList());
        deadlines.add(this.deadlineManager.get(DeadlineManager.DeadlineType.RETRY_TOPIC_EXPANSION));
        deadlines.add(this.deadlineManager.get(DeadlineManager.DeadlineType.TOPIC_CONFIG_CHECK));
        deadlines.add(this.deadlineManager.get(DeadlineManager.DeadlineType.TRY_RESUME));
        long nextDeadline = Utils.nextDeadline((Long[])deadlines.toArray(new Long[deadlines.size()]));
        long maxWaitTimeMillis = Math.max(0L, nextDeadline - now);
        maxWaitTimeMillis = Math.min(maxWaitTimeMillis, (long)this.config.getConsumerPollTimeoutIntervalMs());
        log.debug("Polling for records, waiting at most {} ms", (Object)maxWaitTimeMillis);
        return maxWaitTimeMillis;
    }

    private Pair<SchemaAndValue, SchemaAndValue> convertKeyValue(ConsumerRecord<byte[], byte[]> record, String sourceTopic) {
        SchemaAndValue value;
        SchemaAndValue key;
        try {
            key = this.sourceKeyConverter.toConnectData(sourceTopic, (byte[])record.key());
        }
        catch (Exception e) {
            log.error("Failed to convert source record key under topic {}", (Object)sourceTopic, (Object)e);
            throw e;
        }
        finally {
            try {
                SchemaAndValue value2 = this.sourceValueConverter.toConnectData(sourceTopic, (byte[])record.value());
            }
            catch (Exception e) {
                log.error("Failed to convert source record value under topic {}", (Object)sourceTopic, (Object)e);
                throw e;
            }
        }
        return Pair.of((Object)key, (Object)value);
    }

    void translateCollectedRecords() {
        for (Translator translator : this.translators.values()) {
            if (translator.isDestinationReady()) {
                List<ConsumerRecord<byte[], byte[]>> translatedRecords = translator.translateCollectedRecords();
                if (translatedRecords.size() > 0) {
                    log.debug("Translated " + translatedRecords.size() + " collected records with translator: " + translator.getClass().getName());
                }
                if (this.offsetTopicCommitter == null) continue;
                this.offsetTopicCommitter.commitRecords(translatedRecords);
                continue;
            }
            log.debug("Pausing partitions for __consumer_timestamps and/or schema topics since destination is not ready");
            List<OffsetManager.TopicPartitionInfo> partitionInfos = this.consumer.assignment().stream().filter(sourcePartition -> translator.topic().equals(sourcePartition.topic())).map(partition -> new OffsetManager.TopicPartitionInfo((TopicPartition)partition, "partition " + String.valueOf(ReplicatorSourceTask.toDestPartition(partition, this.config)), true)).collect(Collectors.toList());
            if (translator.seekToBeginningOnPause()) {
                partitionInfos.forEach(item -> log.trace("Seeking to the beginning and pausing source partition {} since destination {} is not ready yet", (Object)item.getSourcePartition(), (Object)item.getDestinationId()));
                if (!partitionInfos.isEmpty()) {
                    this.offsetManager.seekToBeginning(partitionInfos.stream().map(item -> item.getSourcePartition()).collect(Collectors.toList()));
                }
            } else {
                partitionInfos.forEach(item -> log.trace("Pausing source partition {} since destination {} is not ready yet", (Object)item.getSourcePartition(), (Object)item.getDestinationId()));
            }
            this.offsetManager.pauseSourcePartitions(partitionInfos);
        }
    }

    private boolean skipRecordReplication(TopicPartition topicPartition, ConsumerRecords<byte[], byte[]> records) {
        String sourceTopic = topicPartition.topic();
        Translator translator = this.translators.get(sourceTopic);
        if (translator != null) {
            List consumerRecords = records.records(topicPartition);
            translator.collect(consumerRecords);
            return true;
        }
        return false;
    }

    protected static Set<String> topicNamesFor(ConsumerRecords<byte[], byte[]> records) {
        return records.isEmpty() ? Collections.emptySet() : records.partitions().stream().map(TopicPartition::topic).collect(Collectors.toSet());
    }

    protected static Long timestampFromRecord(ConsumerRecord<byte[], byte[]> record) {
        Long timestamp;
        if (record.timestamp() >= 0L) {
            timestamp = record.timestamp();
        } else if (record.timestamp() == -1L) {
            timestamp = null;
        } else {
            throw new CorruptRecordException(String.format("Invalid Record timestamp: %d", record.timestamp()));
        }
        return timestamp;
    }

    protected static boolean shouldFilterRecord(ConsumerRecord<byte[], byte[]> record, String destClusterId, String destTopic, List<FilterOverride> filterOverrides) {
        boolean foundFilteredHeaders = false;
        Iterable headers = record.headers().headers(REPLICATOR_ID_HEADER);
        for (Header header : headers) {
            ProvenanceHeader provenanceHeader = ReplicatorSourceTask.parseProvenanceHeader(header.value(), record);
            if (provenanceHeader.isValid()) {
                if (!destClusterId.equals(provenanceHeader.clusterId()) || !destTopic.equals(provenanceHeader.topic())) continue;
                if (ReplicatorSourceTask.matchesFilterOverride(provenanceHeader, filterOverrides)) {
                    log.trace("No candidate filtered headers matched an override");
                    return false;
                }
                log.trace("Found candidate filtered header {}", (Object)provenanceHeader);
                foundFilteredHeaders = true;
                continue;
            }
            if (ReplicatorSourceTask.matchesFilterOverride(provenanceHeader, filterOverrides)) {
                log.trace("No candidate filtered headers matched an override");
                return false;
            }
            log.trace("Found candidate filtered header {}", (Object)provenanceHeader);
            foundFilteredHeaders = true;
        }
        return foundFilteredHeaders;
    }

    protected static boolean matchesFilterOverride(ProvenanceHeader header, List<FilterOverride> filterOverrides) {
        for (FilterOverride override : filterOverrides) {
            if (!override.matches(header)) continue;
            log.trace("Candidate filtered header {} matches override {}", (Object)header, (Object)override);
            return true;
        }
        return false;
    }

    protected static ProvenanceHeader parseProvenanceHeader(byte[] headerValue, ConsumerRecord<byte[], byte[]> record) {
        String value = new String(headerValue, StandardCharsets.UTF_8);
        if (value.length() > 1000) {
            throw new InvalidConfigurationException("Provenance header pattern is too long; it must be less than 1000 characters.");
        }
        Matcher m = PROVENANCE_HEADER_PATTERN.matcher(value);
        if (m.matches()) {
            long ts;
            String clusterId = m.group(1);
            String topic = m.group(2);
            try {
                ts = Long.parseLong(m.group(3));
            }
            catch (NumberFormatException e) {
                ts = record.timestamp();
            }
            return new ProvenanceHeader(clusterId, topic, ts, true);
        }
        return new ProvenanceHeader(value, record.topic(), record.timestamp(), false);
    }

    protected static byte[] formatProvenanceHeader(String clusterId, String topic, Long ts) {
        StringBuilder sb = new StringBuilder();
        sb.append(clusterId).append(",").append(topic).append(",").append(ts);
        return sb.toString().getBytes(StandardCharsets.UTF_8);
    }

    protected static ConnectHeaders toConnectHeaders(String sourceClusterId, String sourceTopic, ConsumerRecord<byte[], byte[]> record, HeaderConverter converter, boolean isProvenanceHeaderEnabled, long nowInMillis) {
        log.trace("Creating a Connect header for new Source Record...");
        Headers headers = record.headers();
        ConnectHeaders connectHeaders = new ConnectHeaders();
        if (headers != null) {
            for (Header origHeader : headers) {
                ProvenanceHeader provenanceHeader;
                if (isProvenanceHeaderEnabled && REPLICATOR_ID_HEADER.equals(origHeader.key()) && (provenanceHeader = ReplicatorSourceTask.parseProvenanceHeader(origHeader.value(), record)).isValid() && sourceClusterId.equals(provenanceHeader.clusterId()) && sourceTopic.equals(provenanceHeader.topic())) continue;
                connectHeaders.add(origHeader.key(), converter.toConnectHeader(sourceTopic, origHeader.key(), origHeader.value()));
            }
        }
        if (isProvenanceHeaderEnabled) {
            connectHeaders.add(REPLICATOR_ID_HEADER, (Object)ReplicatorSourceTask.formatProvenanceHeader(sourceClusterId, sourceTopic, nowInMillis), Schema.BYTES_SCHEMA);
        }
        return connectHeaders;
    }

    private void maybeResumePartitions() {
        long now = this.deadlineManager.getMilliSeconds();
        Long tryResumeDeadline = this.deadlineManager.get(DeadlineManager.DeadlineType.TRY_RESUME);
        if (tryResumeDeadline == null) {
            log.debug("No partitions to resume");
            return;
        }
        if (tryResumeDeadline > now) {
            log.debug("Resuming at {} (in {} ms)", (Object)tryResumeDeadline, (Object)(tryResumeDeadline - now));
            return;
        }
        for (TopicPartition sourcePartition : this.consumer.paused()) {
            Translator translator = this.translators.get(sourcePartition.topic());
            if (translator != null && translator.isDestinationReady()) {
                log.debug("Resuming paused partition {} since destination is now ready", (Object)sourcePartition);
                this.consumer.resume(Collections.singleton(sourcePartition));
                continue;
            }
            TopicPartition destPartition = ReplicatorSourceTask.toDestPartition(sourcePartition, this.config);
            if (!this.destAdminClient.partitionExists(destPartition)) continue;
            log.debug("Resuming paused partition {} since partition {} now exists in the destination cluster", (Object)sourcePartition, (Object)destPartition);
            this.consumer.resume(Collections.singleton(sourcePartition));
        }
        if (!this.consumer.paused().isEmpty()) {
            tryResumeDeadline = this.deadlineManager.getMilliSeconds() + 5000L;
            log.debug("Setting resume deadline to {} (in {} ms)", (Object)tryResumeDeadline, (Object)5000L);
            this.deadlineManager.set(DeadlineManager.DeadlineType.TRY_RESUME, tryResumeDeadline);
        } else {
            this.deadlineManager.set(DeadlineManager.DeadlineType.TRY_RESUME, null);
        }
    }

    private void watchTopicsForAssignedPartitions(Collection<TopicPartition> sourceAssignment) {
        HashSet<String> interestedDestTopics = new HashSet<String>();
        ArrayList<Translator> interestedTranslators = new ArrayList<Translator>();
        for (TopicPartition sourcePartition : sourceAssignment) {
            Translator translator = this.translators.get(sourcePartition.topic());
            if (translator != null) {
                interestedTranslators.add(translator);
                continue;
            }
            interestedDestTopics.add(ReplicatorSourceTask.toDestTopic(sourcePartition.topic(), this.config));
        }
        this.translatorMonitor.setInterestedTranslators(interestedTranslators, () -> {
            if (this.deadlineManager.get(DeadlineManager.DeadlineType.TRY_RESUME) == null) {
                this.wakeupConsumer();
            }
        });
        this.destAdminClient.setInterestedTopics(interestedDestTopics, this::wakeupConsumer);
    }

    private void wakeupConsumer() {
        log.debug("Waking up consumer because there has been a change in metadata on the source topics");
        this.deadlineManager.set(DeadlineManager.DeadlineType.TRY_RESUME, this.deadlineManager.getMilliSeconds());
        if (!this.isStarting) {
            this.consumer.wakeup();
        }
    }

    private void initializeAssignedPartitions(Collection<TopicPartition> sourceAssignment) {
        log.debug("Initializing source partitions for Replicator source task {}", (Object)this.taskId);
        this.offsetManager.initializePartitionOffset(sourceAssignment, this.translators, this.destAdminClient);
        for (TopicPartition sourcePartition : sourceAssignment) {
            if (this.translators.get(sourcePartition.topic()) != null || sourcePartition.partition() != 0) continue;
            String sourceTopic = sourcePartition.topic();
            if (this.isDestTopicExpansionNeeded(sourceTopic)) {
                this.sourceTopicsNeedingExpansion.add(sourceTopic);
            }
            if (!this.config.getTopicConfigSync()) continue;
            this.managedSourceTopics.add(sourceTopic);
        }
    }

    public static TopicPartition toDestPartition(TopicPartition sourcePartition, ReplicatorSourceTaskConfig config) {
        return new TopicPartition(ReplicatorSourceTask.toDestTopic(sourcePartition.topic(), config), sourcePartition.partition());
    }

    private static String toDestTopic(String sourceTopic, ReplicatorSourceTaskConfig config) {
        return Utils.renameTopic((String)config.getTopicRenameFormat(), (String)sourceTopic);
    }

    private void retryTopicExpansionIfNeeded() {
        if (this.sourceTopicsNeedingExpansion.isEmpty()) {
            this.deadlineManager.set(DeadlineManager.DeadlineType.RETRY_TOPIC_EXPANSION, null);
            log.debug("No topic expansion was needed!");
            return;
        }
        Long retryTopicExpansionDeadline = this.deadlineManager.get(DeadlineManager.DeadlineType.RETRY_TOPIC_EXPANSION);
        if (retryTopicExpansionDeadline == null || retryTopicExpansionDeadline < this.deadlineManager.getMilliSeconds()) {
            HashSet<String> sourceTopicsCreatedOrExpanded = new HashSet<String>();
            for (String sourceTopic : this.sourceTopicsNeedingExpansion) {
                if (!this.maybeCreateOrExpandDestTopic(sourceTopic)) continue;
                sourceTopicsCreatedOrExpanded.add(sourceTopic);
            }
            log.debug("The following topics were expanded: {}", sourceTopicsCreatedOrExpanded);
            this.sourceTopicsNeedingExpansion.removeAll(sourceTopicsCreatedOrExpanded);
            this.deadlineManager.set(DeadlineManager.DeadlineType.RETRY_TOPIC_EXPANSION, this.deadlineManager.getMilliSeconds() + (long)this.config.getTopicCreateBackoffMs());
        }
    }

    private void overrideTimestampType(Properties sourceTopicConfig) {
        sourceTopicConfig.setProperty("message.timestamp.type", this.config.getTopicTimestampType());
    }

    private void updateTopicConfigsIfNeeded() {
        if (this.managedSourceTopics.isEmpty()) {
            this.deadlineManager.set(DeadlineManager.DeadlineType.TOPIC_CONFIG_CHECK, null);
            log.debug("No topic config update needed!");
            return;
        }
        Long topicConfigCheckDeadline = this.deadlineManager.get(DeadlineManager.DeadlineType.TOPIC_CONFIG_CHECK);
        long now = this.deadlineManager.getMilliSeconds();
        if (topicConfigCheckDeadline != null && now < topicConfigCheckDeadline) {
            log.debug("Checking for config update at {} (in {} ms)", (Object)now, (Object)(topicConfigCheckDeadline - now));
            return;
        }
        log.debug("Verifying topic configuration for topics {} for replicator task {}", this.managedSourceTopics, (Object)this.taskId);
        for (String sourceTopic : this.managedSourceTopics) {
            String destTopic = ReplicatorSourceTask.toDestTopic(sourceTopic, this.config);
            if (!this.destAdminClient.topicExists(destTopic)) continue;
            String configuringTopic = null;
            String clusterLocation = null;
            try {
                configuringTopic = sourceTopic;
                clusterLocation = "source";
                Properties sourceTopicConfig = this.sourceAdminClient.topicConfig(sourceTopic);
                this.overrideTimestampType(sourceTopicConfig);
                configuringTopic = destTopic;
                clusterLocation = "destination";
                Properties destTopicConfig = this.destAdminClient.topicConfig(destTopic);
                if (sourceTopicConfig.equals(destTopicConfig)) continue;
                log.info("Updating configuration of destination topic {} with properties {}", (Object)destTopic, (Object)sourceTopicConfig);
                this.destAdminClient.changeTopicConfig(destTopic, sourceTopicConfig);
            }
            catch (InterruptedException | RuntimeException | ExecutionException e) {
                log.warn("Failed topic configuration check for {} topic {} on task {} exception {}. Will retry later.", new Object[]{clusterLocation, configuringTopic, this.taskId, e.toString()});
            }
        }
        this.deadlineManager.set(DeadlineManager.DeadlineType.TOPIC_CONFIG_CHECK, this.deadlineManager.getMilliSeconds() + (long)this.config.getTopicConfigSyncIntervalMs());
    }

    private boolean isDestTopicExpansionNeeded(String sourceTopic) {
        boolean autoCreateTopics = this.config.getTopicAutoCreate();
        boolean preservePartitions = this.config.getTopicPreservePartitions();
        if (!autoCreateTopics && !preservePartitions) {
            return false;
        }
        List sourcePartitionMetadata = this.consumer.partitionsFor(sourceTopic);
        if (sourcePartitionMetadata == null) {
            return false;
        }
        int numSourcePartitions = sourcePartitionMetadata.size();
        TopicMetadata destTopicMetadata = this.destAdminClient.topicMetadata(ReplicatorSourceTask.toDestTopic(sourceTopic, this.config));
        if (destTopicMetadata == null) {
            return autoCreateTopics;
        }
        return preservePartitions && numSourcePartitions > destTopicMetadata.numPartitions();
    }

    private boolean maybeCreateOrExpandDestTopic(String sourceTopic) {
        String destTopic = ReplicatorSourceTask.toDestTopic(sourceTopic, this.config);
        List sourcePartitionMetadata = this.consumer.partitionsFor(sourceTopic);
        int numSourcePartitions = sourcePartitionMetadata.size();
        log.info("Fetching destination topic metadata for destination topic " + destTopic);
        TopicMetadata destTopicMetadata = this.destAdminClient.topicMetadata(destTopic);
        if (this.config.getTopicAutoCreate() && destTopicMetadata == null) {
            int aliveBrokers;
            int replicationFactor;
            boolean useSourceReplicationFactor;
            Properties sourceTopicConfig;
            try {
                log.info("Fetching source topic configs for source topic " + sourceTopic);
                sourceTopicConfig = this.sourceAdminClient.topicConfig(sourceTopic);
            }
            catch (InterruptedException | RuntimeException | ExecutionException e) {
                log.warn("Encountered exception when trying to fetch source topic configs. Unable to create destination topic {} at this time: ", (Object)destTopic, (Object)e);
                return false;
            }
            this.overrideTimestampType(sourceTopicConfig);
            boolean bl = useSourceReplicationFactor = this.config.getDestTopicReplicationFactor() == 0;
            if (useSourceReplicationFactor) {
                replicationFactor = ((PartitionInfo)sourcePartitionMetadata.get(0)).replicas().length;
            } else {
                replicationFactor = this.config.getDestTopicReplicationFactor();
                log.info("Using user-specified replication factor of {} for creating destination topic {}", (Object)replicationFactor, (Object)destTopic);
            }
            try {
                log.debug("Determining number of alive brokers in destination cluster.");
                aliveBrokers = this.destAdminClient.aliveBrokers();
            }
            catch (InterruptedException | RuntimeException | ExecutionException e) {
                log.warn("Encountered exception when trying to determine number of alive brokers in destination cluster. Unable to create destination topic {} at this time: ", (Object)destTopic, (Object)e);
                return false;
            }
            if (replicationFactor > aliveBrokers) {
                log.warn("Unable to create topic {} in the destination cluster because " + (useSourceReplicationFactor ? "the source replication factor " : " the specified dest.topic.replication.factor=") + "{} is greater than the number of alive brokers {}. This could be a transient issue because some brokers may go down or get disconnected at the destination cluster. If the number of brokers deployed at the destination cluster is greater than the replication factor, the system will fix by itself. Otherwise, the issue can be fixed by adding additional brokers to the destination cluster or setting '{}' flag in replicator's config to meet the number of brokers at the destination cluster.", new Object[]{destTopic, replicationFactor, aliveBrokers, "dest.topic.replication.factor"});
                return false;
            }
            try {
                log.info("Creating topic {} in destination cluster with {} partitions and replication factor {}", new Object[]{destTopic, numSourcePartitions, replicationFactor});
                return this.destAdminClient.createTopic(destTopic, numSourcePartitions, (short)replicationFactor, sourceTopicConfig);
            }
            catch (InterruptedException | RuntimeException | ExecutionException e) {
                log.warn("Encountered exception when trying to create destination topic {}: ", (Object)destTopic, (Object)e);
                return false;
            }
        }
        if (this.config.getTopicPreservePartitions() && destTopicMetadata != null && numSourcePartitions > destTopicMetadata.numPartitions()) {
            try {
                log.info("Increasing number of partitions of topic {} from {} to {} in the destination cluster", new Object[]{destTopic, destTopicMetadata.numPartitions(), numSourcePartitions});
                this.destAdminClient.addPartitions(destTopic, numSourcePartitions);
                return true;
            }
            catch (InterruptedException | RuntimeException | ExecutionException e) {
                log.warn("Encountered exception when trying to increase number of partitions for destination topic {} ", (Object)destTopic, (Object)e);
                return false;
            }
        }
        return false;
    }

    public void commitRecord(SourceRecord record, RecordMetadata metadata) throws InterruptedException {
        if (metadata != null) {
            log.trace("Committing source record with record metadata {}", (Object)metadata.toString());
        }
        if (this.timestampsCommitter != null) {
            if (metadata != null) {
                log.trace("Committing consumer timestamps with record metadata {}", (Object)metadata.toString());
            }
            this.timestampsCommitter.commitRecord(record);
        }
        if (this.offsetTopicCommitter != null) {
            if (metadata != null) {
                log.trace("Committing offsets with record metadata {}", (Object)metadata.toString());
            }
            this.offsetTopicCommitter.commitRecord(record, metadata);
        }
        if (record != null) {
            this.replicatorTaskMetricsGroup.recordTaskTopicPartitionMetrics(record, metadata);
        }
    }

    public void commit() throws InterruptedException {
        if (this.timestampsCommitter != null) {
            this.timestampsCommitter.commit();
        }
        if (this.offsetTopicCommitter != null) {
            this.offsetTopicCommitter.commit();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() {
        log.info("Closing kafka replicator task {}", (Object)this.taskId);
        if (this.consumer != null) {
            this.consumer.wakeup();
            ReplicatorSourceTask replicatorSourceTask = this;
            synchronized (replicatorSourceTask) {
                AtomicReference firstException = new AtomicReference();
                log.debug("Closing source admin client");
                Utils.closeQuietly((AutoCloseable)this.sourceAdminClient, (String)"source admin client", firstException);
                log.debug("Closing destination admin client");
                Utils.closeQuietly((AutoCloseable)this.destAdminClient, (String)"destination admin client", firstException);
                log.debug("Closing consumer");
                Utils.closeQuietly(this.consumer, (String)"consumer", firstException);
                if (this.timestampsCommitter != null) {
                    log.debug("Stopping timestamps committer");
                    this.timestampsCommitter.stop();
                }
                if (this.translatorMonitor != null) {
                    log.debug("Closing translator monitor");
                    this.translatorMonitor.close();
                }
            }
        }
        log.info("Shutting down metrics recording for task {}", (Object)this.taskId);
        if (this.replicatorTaskMetricsGroup != null) {
            this.replicatorTaskMetricsGroup.stopMetrics();
        }
    }

    private Consumer<byte[], byte[]> buildSourceConsumer(ReplicatorSourceTaskConfig config) {
        if (this.consumer != null) {
            return this.consumer;
        }
        return ReplicatorSourceTask.createConsumerHelper(config.getSourceConsumerConfigs(), config.getName(), this.taskId);
    }

    @NotNull
    public static Consumer<byte[], byte[]> createConsumerHelper(Map<String, ?> config, String name, String taskId) {
        HashMap<String, Object> consumerConfig = new HashMap<String, Object>();
        consumerConfig.putAll(config);
        if (!consumerConfig.containsKey("group.id")) {
            consumerConfig.put("group.id", name);
        }
        if (!consumerConfig.containsKey("client.id")) {
            consumerConfig.put("client.id", taskId);
        }
        consumerConfig.put("enable.auto.commit", false);
        consumerConfig.put("auto.offset.reset", "none");
        consumerConfig.put("allow.auto.create.topics", "false");
        log.debug("Initializing consumer with client id {} and group id {}", consumerConfig.get("client.id"), consumerConfig.get("group.id"));
        log.debug("Initializing Replicator Task Connector in Group");
        return new KafkaConsumer(consumerConfig, (Deserializer)new ByteArrayDeserializer(), (Deserializer)new ByteArrayDeserializer());
    }

    private ReplicatorSourceTaskConfig taskConfig(Map<String, String> props) {
        return this.config != null ? this.config : new ReplicatorSourceTaskConfig(props);
    }

    private ReplicatorAdminClient createAdminClient(ReplicatorAdminClient adminClient, Supplier<Map<String, Object>> adminClientConfig) {
        if (adminClient != null) {
            return adminClient;
        }
        Time time = this.deadlineManager.getTime();
        return new NewReplicatorAdminClient(adminClientConfig.get(), time, 30000L, this.taskId);
    }

    @VisibleForTesting
    protected ReplicatorAdminClient getSourceAdminClient() {
        return this.sourceAdminClient;
    }

    @VisibleForTesting
    protected ReplicatorAdminClient getDestinationAdminClient() {
        return this.destAdminClient;
    }

    protected static class FilterOverride {
        private Pattern clusterId;
        private Pattern topic;
        private long startTsInclusive;
        private long endTsExclusive;

        public FilterOverride(Matcher matcher) {
            this.clusterId = Pattern.compile(matcher.group(1));
            this.topic = Pattern.compile(matcher.group(2));
            try {
                this.startTsInclusive = Long.parseLong(matcher.group(3));
            }
            catch (NumberFormatException e) {
                this.startTsInclusive = 0L;
            }
            try {
                this.endTsExclusive = Long.parseLong(matcher.group(4));
            }
            catch (NumberFormatException e) {
                this.endTsExclusive = Long.MAX_VALUE;
            }
        }

        public Pattern clusterId() {
            return this.clusterId;
        }

        public Pattern topic() {
            return this.topic;
        }

        public long startTsInclusive() {
            return this.startTsInclusive;
        }

        public long endTsExclusive() {
            return this.endTsExclusive;
        }

        public boolean matches(ProvenanceHeader value) {
            if (value.topic.length() > 1000) {
                throw new InvalidConfigurationException("Topic length is too long; it must be less than 1000 characters.");
            }
            if (value.clusterId.length() > 1000) {
                throw new InvalidConfigurationException("Cluster ID length is too long; it must be less than 1000 characters.");
            }
            Long ts = value.ts();
            boolean tsMatches = ts == null || ts == -1L || ts >= this.startTsInclusive && ts < this.endTsExclusive;
            return tsMatches && (value.topic() == null || this.topic.matcher(value.topic()).matches()) && this.clusterId.matcher(value.clusterId()).matches();
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            FilterOverride that = (FilterOverride)o;
            return Objects.equals(this.clusterId, that.clusterId) && Objects.equals(this.topic, that.topic) && Objects.equals(this.startTsInclusive, that.startTsInclusive) && Objects.equals(this.endTsExclusive, that.endTsExclusive);
        }

        public int hashCode() {
            return Objects.hash(this.clusterId, this.topic, this.startTsInclusive, this.endTsExclusive);
        }

        public String toString() {
            return String.valueOf(this.clusterId) + "," + String.valueOf(this.topic) + "," + this.startTsInclusive + "-" + this.endTsExclusive;
        }
    }

    protected static class ProvenanceHeader {
        private String clusterId;
        private String topic;
        private Long ts;
        private boolean valid;

        public ProvenanceHeader(String clusterId, String topic, Long ts, boolean valid) {
            this.clusterId = clusterId;
            this.topic = topic;
            this.ts = ts;
            this.valid = valid;
        }

        public String clusterId() {
            return this.clusterId;
        }

        public String topic() {
            return this.topic;
        }

        public Long ts() {
            return this.ts;
        }

        public boolean isValid() {
            return this.valid;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ProvenanceHeader that = (ProvenanceHeader)o;
            return this.valid == that.valid && Objects.equals(this.clusterId, that.clusterId) && Objects.equals(this.topic, that.topic) && Objects.equals(this.ts, that.ts);
        }

        public int hashCode() {
            return Objects.hash(this.clusterId, this.topic, this.ts, this.valid);
        }

        public String toString() {
            return this.clusterId + "," + this.topic + "," + this.ts;
        }
    }
}

