/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.telemetry.collector;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.sun.management.OperatingSystemMXBean;
import com.sun.management.UnixOperatingSystemMXBean;
import io.confluent.telemetry.Context;
import io.confluent.telemetry.MetricKey;
import io.confluent.telemetry.collector.ConfluentMetricNamingConvention;
import io.confluent.telemetry.collector.LastValueTracker;
import io.confluent.telemetry.collector.MetricsCollector;
import io.confluent.telemetry.emitter.Emitter;
import io.confluent.telemetry.metrics.SinglePointMetric;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import java.time.Clock;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JvmMetricsCollector
implements MetricsCollector {
    private static final Logger log = LoggerFactory.getLogger(JvmMetricsCollector.class);
    public static final String SYSTEM_DOMAIN = "io.confluent.system";
    private final Map<String, String> labels;
    private final MemoryMXBean memoryMXBean = ManagementFactory.getMemoryMXBean();
    private final java.lang.management.OperatingSystemMXBean osBean = ManagementFactory.getOperatingSystemMXBean();
    private final LastValueTracker<Long> lastValueTracker = new LastValueTracker();
    private final Clock clock;
    private final Supplier<? extends Iterable<GarbageCollectorMXBean>> gcMxBeanSupplier;

    private JvmMetricsCollector(Context ctx) {
        this(ctx, Clock.systemUTC(), ManagementFactory::getGarbageCollectorMXBeans);
    }

    @VisibleForTesting
    JvmMetricsCollector(Context ctx, Clock clock, Supplier<? extends Iterable<GarbageCollectorMXBean>> gcMxBeanSupplier) {
        this.labels = this.createLabels(ctx);
        this.clock = clock;
        this.gcMxBeanSupplier = gcMxBeanSupplier;
    }

    @Override
    public void collect(Emitter emitter) {
        this.collectSystemMetrics(emitter);
        this.collectMemoryMetrics(emitter);
        this.collectGarbageCollectionMetrics(emitter);
    }

    private void collectSystemMetrics(Emitter emitter) {
        this.emitMetrics(emitter, this.getOSMetrics(), "jvm/os");
    }

    private Map<String, Number> getOSMetrics() {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        if (this.osBean instanceof OperatingSystemMXBean) {
            OperatingSystemMXBean osMXBean = (OperatingSystemMXBean)this.osBean;
            builder.put((Object)"FreePhysicalMemorySize", (Object)osMXBean.getFreeMemorySize()).put((Object)"TotalPhysicalMemorySize", (Object)osMXBean.getTotalMemorySize()).put((Object)"ProcessCpuLoad", (Object)osMXBean.getProcessCpuLoad()).put((Object)"ProcessCpuTime", (Object)osMXBean.getProcessCpuTime()).put((Object)"SystemCpuLoad", (Object)osMXBean.getCpuLoad());
            if (osMXBean instanceof UnixOperatingSystemMXBean) {
                UnixOperatingSystemMXBean unixOsMXBean = (UnixOperatingSystemMXBean)osMXBean;
                builder.put((Object)"OpenFileDescriptorCount", (Object)unixOsMXBean.getOpenFileDescriptorCount()).put((Object)"MaxFileDescriptorCount", (Object)unixOsMXBean.getMaxFileDescriptorCount());
            }
        }
        builder.put((Object)"SystemLoadAverage", (Object)this.osBean.getSystemLoadAverage());
        return builder.build();
    }

    private void collectMemoryMetrics(Emitter emitter) {
        this.emitMemoryMetrics(emitter, this.memoryMXBean.getHeapMemoryUsage(), "jvm/mem/heap");
        this.emitMemoryMetrics(emitter, this.memoryMXBean.getNonHeapMemoryUsage(), "jvm/mem/nonheap");
    }

    private void emitMemoryMetrics(Emitter emitter, MemoryUsage heapMemoryUsage, String metricGroup) {
        this.emitMetrics(emitter, (Map<String, Number>)ImmutableMap.of((Object)"Committed", (Object)heapMemoryUsage.getCommitted(), (Object)"Used", (Object)heapMemoryUsage.getUsed()), metricGroup);
    }

    private void collectGarbageCollectionMetrics(Emitter emitter) {
        for (GarbageCollectorMXBean gcBean : this.gcMxBeanSupplier.get()) {
            ImmutableMap metricLabels = ImmutableMap.builder().putAll(this.labels).put((Object)"gc_name", (Object)gcBean.getName()).build();
            MetricKey countMetric = new MetricKey(ConfluentMetricNamingConvention.fullMetricName(SYSTEM_DOMAIN, "jvm/gc/collections", "count/delta"), (Map<String, String>)metricLabels);
            emitter.maybeEmitMetric(this.calculateDelta(countMetric, gcBean.getCollectionCount()));
            MetricKey timeMetric = new MetricKey(ConfluentMetricNamingConvention.fullMetricName(SYSTEM_DOMAIN, "jvm/gc/collections", "time/delta"), (Map<String, String>)metricLabels);
            emitter.maybeEmitMetric(this.calculateDelta(timeMetric, gcBean.getCollectionTime()));
        }
    }

    private SinglePointMetric calculateDelta(MetricKey metricKey, long currentValue) {
        Instant now = Instant.now(this.clock);
        return this.lastValueTracker.getAndSet(metricKey, now, currentValue).map(i -> SinglePointMetric.deltaSum(metricKey, currentValue - (Long)i.getValue(), now, i.getIntervalStart())).orElse(SinglePointMetric.deltaSum(metricKey, currentValue, now, now));
    }

    private void emitMetrics(Emitter emitter, Map<String, Number> metrics, String metricGroup) {
        for (Map.Entry<String, Number> attribute : metrics.entrySet()) {
            String metricKey = attribute.getKey();
            Number metricValue = attribute.getValue();
            String name = ConfluentMetricNamingConvention.fullMetricName(SYSTEM_DOMAIN, metricGroup, metricKey);
            MetricKey key = new MetricKey(name, this.labels);
            if (!emitter.shouldEmitMetric(key)) continue;
            Instant now = Instant.now();
            emitter.emitMetric(SinglePointMetric.gauge(key, metricValue, now));
        }
    }

    private Map<String, String> createLabels(Context context) {
        java.lang.management.OperatingSystemMXBean osBean = ManagementFactory.getOperatingSystemMXBean();
        HashMap<String, String> labels = new HashMap<String, String>();
        labels.put("os.name", osBean.getName());
        labels.put("os.version", osBean.getVersion());
        labels.put("os.arch", osBean.getArch());
        labels.put("os.processors", String.valueOf(osBean.getAvailableProcessors()));
        if (context.isDebugEnabled()) {
            labels.put("library", "none");
        }
        return labels;
    }

    public String toString() {
        return JvmMetricsCollector.class.getCanonicalName();
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private Context context;

        private Builder() {
        }

        public Builder setContext(Context context) {
            this.context = context;
            return this;
        }

        public JvmMetricsCollector build() {
            Objects.requireNonNull(this.context);
            return new JvmMetricsCollector(this.context);
        }
    }
}

