/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.telemetry.config.remote.polling.kubernetes;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.confluent.telemetry.config.remote.polling.PollingRemoteConfigurationSource;
import io.confluent.telemetry.config.remote.polling.kubernetes.ConfigSet;
import io.confluent.telemetry.config.remote.polling.kubernetes.KubernetesRemoteConfigurationConfig;
import io.confluent.telemetry.config.remote.polling.kubernetes.KubernetesResourceNotUpdated;
import io.confluent.telemetry.config.remote.polling.kubernetes.V1Object;
import io.confluent.telemetry.config.remote.polling.kubernetes.V1ObjectMeta;
import io.confluent.telemetry.config.v2.remote.RemoteConfiguration;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Paths;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManagerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KubernetesRemoteConfigurationSource
extends PollingRemoteConfigurationSource {
    private static final Logger log = LoggerFactory.getLogger(KubernetesRemoteConfigurationSource.class);
    private static final String KUBERNETES_SERVICE_HOST_ENV_VAR = "KUBERNETES_SERVICE_HOST";
    private static final String KUBERNETES_SERVICE_PORT_ENV_VAR = "KUBERNETES_SERVICE_PORT";
    private static final String SECRETS_PATH = "secrets";
    private static final String CONFIGMAPS_PATH = "configmaps";
    private static final String SERVICE_ACCOUNT_TOKEN_PATH = "/var/run/secrets/kubernetes.io/serviceaccount/token";
    private static final String CERTIFICATE_AUTHORITY = "/var/run/secrets/kubernetes.io/serviceaccount/ca.crt";
    private static final String REPORTER_VERSION = RemoteConfiguration.getSchemaVersion();
    private final ObjectMapper mapper = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false).configure(DeserializationFeature.FAIL_ON_MISSING_CREATOR_PROPERTIES, true).configure(DeserializationFeature.FAIL_ON_NULL_CREATOR_PROPERTIES, true);
    private List<String> latestResourceVersions;
    private final List<URL> clusterOverrideEndpoints;
    private final List<URL> defaultEndpoints;
    private final SSLSocketFactory sslSocketFactory;
    private String serviceAccountToken;
    private final int connectTimeoutMs;
    private final int readTimeoutMs;

    public KubernetesRemoteConfigurationSource(KubernetesRemoteConfigurationConfig config, Optional<String> physicalClusterId, Consumer<RemoteConfiguration> callback) throws CertificateException, NoSuchAlgorithmException, KeyStoreException, IOException, KeyManagementException {
        super(config.getRefreshInterval(), callback);
        Preconditions.checkArgument((config.getConnectTimeout() > 0 ? 1 : 0) != 0, (Object)"connect timeout must be greater than 0");
        Preconditions.checkArgument((config.getReadTimeout() > 0 ? 1 : 0) != 0, (Object)"read timeout must be greater than 0");
        ArrayList<URL> clusterOverrideEndpoints = new ArrayList<URL>();
        ArrayList<URL> defaultEndpoints = new ArrayList<URL>();
        if (physicalClusterId.isPresent()) {
            clusterOverrideEndpoints.add(new URL(KubernetesRemoteConfigurationSource.parseEndpointFromEnv(config, physicalClusterId.get(), SECRETS_PATH)));
            clusterOverrideEndpoints.add(new URL(KubernetesRemoteConfigurationSource.parseEndpointFromEnv(config, physicalClusterId.get(), CONFIGMAPS_PATH)));
        }
        defaultEndpoints.add(new URL(KubernetesRemoteConfigurationSource.parseDefaultEndpointFromEnv(config, SECRETS_PATH)));
        defaultEndpoints.add(new URL(KubernetesRemoteConfigurationSource.parseDefaultEndpointFromEnv(config, CONFIGMAPS_PATH)));
        this.clusterOverrideEndpoints = clusterOverrideEndpoints;
        this.defaultEndpoints = defaultEndpoints;
        this.sslSocketFactory = KubernetesRemoteConfigurationSource.buildSSLSocketFactory();
        this.serviceAccountToken = KubernetesRemoteConfigurationSource.getKubeServiceAccountToken();
        this.connectTimeoutMs = config.getConnectTimeout();
        this.readTimeoutMs = config.getReadTimeout();
    }

    @VisibleForTesting
    KubernetesRemoteConfigurationSource(KubernetesRemoteConfigurationConfig config, List<URL> clusterOverrideEndpoints, List<URL> defaultEndpoints, Consumer<RemoteConfiguration> callback) throws NoSuchAlgorithmException, KeyStoreException, KeyManagementException, IOException, CertificateException {
        super(config.getRefreshInterval(), callback);
        this.clusterOverrideEndpoints = clusterOverrideEndpoints;
        this.defaultEndpoints = defaultEndpoints;
        this.sslSocketFactory = KubernetesRemoteConfigurationSource.buildSSLSocketFactory();
        this.serviceAccountToken = KubernetesRemoteConfigurationSource.getKubeServiceAccountToken();
        this.connectTimeoutMs = config.getConnectTimeout();
        this.readTimeoutMs = config.getReadTimeout();
    }

    @VisibleForTesting
    static SSLSocketFactory buildSSLSocketFactory() throws CertificateException, IOException, KeyStoreException, NoSuchAlgorithmException, KeyManagementException {
        KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
        keyStore.load(null, null);
        try (BufferedInputStream caInput = new BufferedInputStream(Files.newInputStream(Paths.get(CERTIFICATE_AUTHORITY, new String[0]), new OpenOption[0]));){
            CertificateFactory ca = CertificateFactory.getInstance("X.509");
            int certificateNumber = 0;
            while (((InputStream)caInput).available() > 0) {
                keyStore.setCertificateEntry("ca" + certificateNumber, ca.generateCertificate(caInput));
                ++certificateNumber;
            }
        }
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
        tmf.init(keyStore);
        SSLContext context = SSLContext.getInstance("TLS");
        context.init(null, tmf.getTrustManagers(), null);
        return context.getSocketFactory();
    }

    @Override
    protected Optional<RemoteConfiguration> requestConfig() {
        Optional<RemoteConfiguration> remoteConfig = Optional.empty();
        try {
            if (!this.clusterOverrideEndpoints.isEmpty()) {
                remoteConfig = this.requestAndMerge(this.clusterOverrideEndpoints);
            }
            if (!remoteConfig.isPresent() && !this.defaultEndpoints.isEmpty()) {
                remoteConfig = this.requestAndMerge(this.defaultEndpoints);
            }
        }
        catch (KubernetesResourceNotUpdated e) {
            return Optional.empty();
        }
        return remoteConfig;
    }

    private Optional<RemoteConfiguration> requestAndMerge(List<URL> endpoints) throws KubernetesResourceNotUpdated {
        List<V1Object> objects = endpoints.parallelStream().map(this::requestConfig).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        if (objects.isEmpty()) {
            return Optional.empty();
        }
        if (this.resourceNotUpdated(objects)) {
            throw new KubernetesResourceNotUpdated();
        }
        return this.mergeAndResolve(objects);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Optional<V1Object> requestConfig(URL endpoint) {
        try {
            HttpURLConnection connection = (HttpURLConnection)endpoint.openConnection();
            connection.setRequestMethod("GET");
            connection.setConnectTimeout(this.connectTimeoutMs);
            connection.setReadTimeout(this.readTimeoutMs);
            connection.setRequestProperty("Authorization", "Bearer " + this.serviceAccountToken);
            ((HttpsURLConnection)connection).setSSLSocketFactory(this.sslSocketFactory);
            int responseCode = connection.getResponseCode();
            switch (responseCode) {
                case 404: {
                    return Optional.empty();
                }
                case 200: {
                    try (InputStream is = connection.getInputStream();){
                        Optional<V1Object> optional = Optional.of((V1Object)this.mapper.readValue(is, V1Object.class));
                        return optional;
                    }
                }
                case 401: {
                    this.serviceAccountToken = KubernetesRemoteConfigurationSource.getKubeServiceAccountToken();
                    log.warn("Auth denied for Secret/ConfigMap. Refreshing service account token.");
                    return Optional.empty();
                }
            }
            try (InputStream is = connection.getErrorStream();){
                log.error("Error querying for Secret/ConfigMap. Status code: {} with body: {}", (Object)responseCode, (Object)new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8)).lines().collect(Collectors.joining("\n")));
                return Optional.empty();
            }
        }
        catch (IOException e) {
            log.error("Error requesting remote config from k8s source. Endpoint: {}", (Object)endpoint, (Object)e);
            return Optional.empty();
        }
    }

    private Optional<RemoteConfiguration> mergeAndResolve(List<V1Object> objects) {
        Optional<ConfigSet> mergedConfigs = this.mergeConfigs(objects);
        return mergedConfigs.flatMap(configs -> {
            Optional<RemoteConfiguration> resolvedConfig = configs.getCompatibleConfigVersion(REPORTER_VERSION);
            log.info("Propagating config: {}", (Object)((V1Object)objects.get(0)).getMetadata().getName());
            return resolvedConfig;
        });
    }

    private boolean resourceNotUpdated(List<V1Object> objects) {
        List resourceVersions = objects.stream().map(V1Object::getMetadata).filter(Objects::nonNull).map(V1ObjectMeta::getResourceVersion).filter(Objects::nonNull).sorted().collect(Collectors.toList());
        if (!Objects.equals(resourceVersions.size(), objects.size())) {
            log.error("Missing resource version(s) for Secret/ConfigMap. Skipping remote configuration.");
            return true;
        }
        if (this.latestResourceVersions == null || !this.latestResourceVersions.equals(resourceVersions)) {
            this.latestResourceVersions = resourceVersions;
            return false;
        }
        return true;
    }

    private Optional<ConfigSet> mergeConfigs(List<V1Object> objects) {
        try {
            ConfigSet mergedConfigs = ConfigSet.parseConfigYaml(objects.get(0));
            for (int i = 1; i < objects.size(); ++i) {
                ConfigSet configs = ConfigSet.parseConfigYaml(objects.get(i));
                mergedConfigs = mergedConfigs.merge(configs);
            }
            return Optional.of(mergedConfigs);
        }
        catch (JsonProcessingException | IllegalArgumentException | IllegalStateException e) {
            log.error("Can't parse Secret/ConfigMap yaml. Reason: {}", (Object)e.getMessage(), (Object)e);
            return Optional.empty();
        }
    }

    public static String parseEndpointFromEnv(KubernetesRemoteConfigurationConfig config, String objectName, String objectPath) {
        String path = String.format("/api/v1/namespaces/%s/%s/%s", config.getNamespace(), objectPath, objectName);
        String kubeServiceHost = System.getenv(KUBERNETES_SERVICE_HOST_ENV_VAR);
        if (Objects.isNull(kubeServiceHost)) {
            return "https://kubernetes.default.svc" + path;
        }
        return "https://" + kubeServiceHost + ":" + System.getenv(KUBERNETES_SERVICE_PORT_ENV_VAR) + path;
    }

    public static String parseDefaultEndpointFromEnv(KubernetesRemoteConfigurationConfig config, String objectPath) {
        return KubernetesRemoteConfigurationSource.parseEndpointFromEnv(config, config.getConfigMapNameConfig(), objectPath);
    }

    @VisibleForTesting
    static String getKubeServiceAccountToken() throws IOException {
        return new String(Files.readAllBytes(Paths.get(SERVICE_ACCOUNT_TOKEN_PATH, new String[0])));
    }

    @VisibleForTesting
    void setLatestResourceVersions(List<String> resourceVersions) {
        this.latestResourceVersions = resourceVersions;
    }

    @VisibleForTesting
    List<String> getLatestResourceVersions() {
        return this.latestResourceVersions;
    }
}

