/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.rest.server;

import com.google.common.annotations.VisibleForTesting;
import io.confluent.ksql.util.KsqlException;
import io.confluent.ksql.util.Pair;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class BackupReplayFile
implements Closeable {
    private static final Logger LOGGER = LogManager.getLogger(BackupReplayFile.class);
    private static final String KEY_VALUE_SEPARATOR_STR = ":";
    private static final String NEW_LINE_SEPARATOR_STR = "\n";
    private static final String TMP_SUFFIX = ".tmp";
    private static final String DIRTY_SUFFIX = ".dirty";
    private static final byte[] KEY_VALUE_SEPARATOR_BYTES = ":".getBytes(StandardCharsets.UTF_8);
    private static final byte[] NEW_LINE_SEPARATOR_BYTES = "\n".getBytes(StandardCharsets.UTF_8);
    private final File file;
    private final boolean writable;
    private final Filesystem filesystem;

    public static BackupReplayFile readOnly(File file) throws IOException {
        return new BackupReplayFile(file, false, BackupReplayFile.filesystemImpl());
    }

    public static BackupReplayFile writable(File file) throws IOException {
        return new BackupReplayFile(file, true, BackupReplayFile.filesystemImpl());
    }

    @VisibleForTesting
    BackupReplayFile(File file, boolean write, Filesystem filesystem) throws IOException {
        this.file = Objects.requireNonNull(file, "file");
        this.filesystem = Objects.requireNonNull(filesystem, "filesystem");
        this.writable = write;
        if (write) {
            this.initFiles();
        }
    }

    private static FileOutputStream createWriter(File file, Filesystem filesystem) {
        try {
            return filesystem.outputStream(file, true);
        }
        catch (FileNotFoundException e) {
            throw new KsqlException(String.format("Failed to create/open replay file: %s", file.getAbsolutePath()), (Throwable)e);
        }
    }

    public File getFile() {
        return this.file;
    }

    public String getPath() {
        return this.file.getAbsolutePath();
    }

    private static void appendRecordToFile(ConsumerRecord<byte[], byte[]> record, File file, Filesystem filesystem) throws IOException {
        try (FileOutputStream writer = BackupReplayFile.createWriter(file, filesystem);){
            writer.write((byte[])record.key());
            writer.write(KEY_VALUE_SEPARATOR_BYTES);
            writer.write((byte[])record.value());
            writer.write(NEW_LINE_SEPARATOR_BYTES);
        }
    }

    public void write(ConsumerRecord<byte[], byte[]> record) throws IOException {
        if (!this.writable) {
            throw new IOException("Write permission denied.");
        }
        File dirty = this.dirty(this.file);
        File tmp = this.tmp(this.file);
        BackupReplayFile.appendRecordToFile(record, dirty, this.filesystem);
        Files.createLink(tmp.toPath(), this.file.toPath());
        Files.move(dirty.toPath(), this.file.toPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        Files.move(tmp.toPath(), dirty.toPath(), new CopyOption[0]);
        BackupReplayFile.appendRecordToFile(record, dirty, this.filesystem);
    }

    public List<Pair<byte[], byte[]>> readRecords() throws IOException {
        ArrayList<Pair<byte[], byte[]>> commands = new ArrayList<Pair<byte[], byte[]>>();
        for (String line : Files.readAllLines(this.getFile().toPath(), StandardCharsets.UTF_8)) {
            String commandId = line.substring(0, line.indexOf(KEY_VALUE_SEPARATOR_STR));
            String command = line.substring(line.indexOf(KEY_VALUE_SEPARATOR_STR) + 1);
            commands.add((Pair<byte[], byte[]>)new Pair((Object)commandId.getBytes(StandardCharsets.UTF_8), (Object)command.getBytes(StandardCharsets.UTF_8)));
        }
        return commands;
    }

    @Override
    public void close() {
    }

    private void initFiles() throws IOException {
        Path dirtyFile = this.dirty(this.file).toPath();
        Files.deleteIfExists(dirtyFile);
        Files.deleteIfExists(this.tmp(this.file).toPath());
        if (this.file.createNewFile()) {
            LOGGER.info("created new backup replay file {}", (Object)this.file.getAbsolutePath());
        }
        Files.copy(this.file.toPath(), this.dirty(this.file).toPath(), StandardCopyOption.REPLACE_EXISTING);
    }

    private File dirty(File file) {
        return new File(file.getPath() + DIRTY_SUFFIX);
    }

    private File tmp(File file) {
        return new File(file.getPath() + TMP_SUFFIX);
    }

    private static Filesystem filesystemImpl() {
        return FileOutputStream::new;
    }

    @VisibleForTesting
    static interface Filesystem {
        public FileOutputStream outputStream(File var1, boolean var2) throws FileNotFoundException;
    }
}

