/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.rest.server;

import com.google.common.annotations.VisibleForTesting;
import io.confluent.ksql.rest.server.BackupReplayFile;
import io.confluent.ksql.rest.server.CommandTopicBackup;
import io.confluent.ksql.rest.server.CommandTopicMigrationUtil;
import io.confluent.ksql.rest.server.computation.InternalTopicSerdes;
import io.confluent.ksql.rest.server.resources.CommandTopicCorruptionException;
import io.confluent.ksql.services.KafkaTopicClient;
import io.confluent.ksql.util.KsqlServerException;
import io.confluent.ksql.util.Pair;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Supplier;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class CommandTopicBackupImpl
implements CommandTopicBackup {
    private static final Logger LOG = LogManager.getLogger(CommandTopicBackupImpl.class);
    private static final String PREFIX = "backup_";
    private final File backupLocation;
    private final String topicName;
    private final Supplier<Long> ticker;
    private BackupReplayFile replayFile;
    private List<Pair<byte[], byte[]>> latestReplay;
    private int latestReplayIdx;
    private boolean corruptionDetected;
    private KafkaTopicClient kafkaTopicClient;

    public CommandTopicBackupImpl(String location, String topicName, KafkaTopicClient kafkaTopicClient) {
        this(location, topicName, System::currentTimeMillis, kafkaTopicClient);
    }

    @VisibleForTesting
    CommandTopicBackupImpl(String location, String topicName, Supplier<Long> ticker, KafkaTopicClient kafkaTopicClient) {
        this.backupLocation = new File(Objects.requireNonNull(location, "location"));
        this.topicName = Objects.requireNonNull(topicName, "topicName");
        this.ticker = Objects.requireNonNull(ticker, "ticker");
        this.kafkaTopicClient = Objects.requireNonNull(kafkaTopicClient, "kafkaTopicClient");
        CommandTopicBackupImpl.ensureDirectoryExists(this.backupLocation);
    }

    @Override
    public void initialize() {
        this.replayFile = this.openOrCreateReplayFile();
        try {
            this.latestReplay = this.replayFile.readRecords();
        }
        catch (IOException e) {
            LOG.warn("Failed to read the latest backup from {}. Continue with a new file. Error = {}", (Object)this.replayFile.getPath(), (Object)e.getMessage());
            try {
                this.replayFile = this.newReplayFile();
            }
            catch (IOException ee) {
                throw new RuntimeException(ee);
            }
            this.latestReplay = Collections.emptyList();
        }
        this.latestReplayIdx = 0;
        this.corruptionDetected = false;
        if (!this.kafkaTopicClient.isTopicExists(this.topicName) && this.latestReplay.size() > 0) {
            this.corruptionDetected = true;
        }
        LOG.info("Command topic will be backup on file: {}", (Object)this.replayFile.getPath());
    }

    @Override
    public void close() {
        this.replayFile.close();
    }

    @VisibleForTesting
    BackupReplayFile getReplayFile() {
        return this.replayFile;
    }

    private boolean isRestoring() {
        return this.latestReplayIdx < this.latestReplay.size();
    }

    private boolean isRecordInLatestReplay(ConsumerRecord<byte[], byte[]> record) {
        Pair<byte[], byte[]> latestReplayRecord = this.latestReplay.get(this.latestReplayIdx);
        if (Arrays.equals((byte[])record.key(), (byte[])latestReplayRecord.getLeft()) && Arrays.equals((byte[])record.value(), (byte[])latestReplayRecord.getRight())) {
            ++this.latestReplayIdx;
            return true;
        }
        return false;
    }

    @Override
    public void writeRecord(ConsumerRecord<byte[], byte[]> record) {
        if (this.corruptionDetected) {
            throw new CommandTopicCorruptionException("Failed to write record due to out of sync command topic and backup file. " + String.format("partition=%d, offset=%d", record.partition(), record.offset()));
        }
        if (record.key() == null || record.value() == null) {
            LOG.warn(String.format("Can't backup a command topic record with a null key/value: partition=%d, offset=%d", record.partition(), record.offset()));
        }
        if (Arrays.equals((byte[])record.key(), InternalTopicSerdes.serializer().serialize("", (Object)CommandTopicMigrationUtil.MIGRATION_COMMAND_ID))) {
            LOG.warn(String.format("Can't backup migration command topic record offset=%d", record.offset()));
            this.corruptionDetected = true;
            return;
        }
        if (this.isRestoring()) {
            if (this.isRecordInLatestReplay(record)) {
                return;
            }
            this.corruptionDetected = true;
            throw new CommandTopicCorruptionException("Failed to write record due to out of sync command topic and backup file. " + String.format("partition=%d, offset=%d", record.partition(), record.offset()));
        }
        if (this.latestReplay.size() > 0) {
            this.latestReplay.clear();
        }
        try {
            this.replayFile.write(record);
        }
        catch (Exception e) {
            LOG.warn("Failed to write to file {}. The command topic backup is not complete. Make sure the file exists and has permissions to write. KSQL must be restarted afterwards to complete the backup process. Error = {}", (Object)this.replayFile.getPath(), (Object)e.getMessage());
        }
    }

    @Override
    public boolean commandTopicCorruption() {
        return this.corruptionDetected;
    }

    @VisibleForTesting
    BackupReplayFile openOrCreateReplayFile() {
        try {
            Optional<BackupReplayFile> backupReplayFile = this.latestReplayFile();
            if (backupReplayFile.isPresent()) {
                return backupReplayFile.get();
            }
            return this.newReplayFile();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private BackupReplayFile newReplayFile() throws IOException {
        return BackupReplayFile.writable(Paths.get(this.backupLocation.getAbsolutePath(), String.format("%s%s_%s", PREFIX, this.topicName, this.ticker.get())).toFile());
    }

    private Optional<BackupReplayFile> latestReplayFile() throws IOException {
        String prefixFilename = String.format("%s%s_", PREFIX, this.topicName);
        File[] files = this.backupLocation.listFiles((f, name) -> name.toLowerCase().startsWith(prefixFilename));
        File latestBakFile = null;
        if (files != null) {
            long latestTs = 0L;
            for (File bakFile : files) {
                String bakTimestamp = bakFile.getName().substring(prefixFilename.length());
                try {
                    long ts = Long.parseLong(bakTimestamp);
                    if (ts <= latestTs) continue;
                    latestTs = ts;
                    latestBakFile = bakFile;
                }
                catch (NumberFormatException e) {
                    LOG.warn("Invalid timestamp '{}' found in backup replay file (file ignored): {}", (Object)bakTimestamp, (Object)bakFile.getName());
                }
            }
        }
        if (latestBakFile != null) {
            return Optional.of(BackupReplayFile.writable(latestBakFile));
        }
        return Optional.empty();
    }

    private static void ensureDirectoryExists(File backupsDir) {
        if (!backupsDir.exists()) {
            if (!backupsDir.mkdirs()) {
                throw new KsqlServerException("Couldn't create the backups directory: " + backupsDir.getPath() + "\n Make sure the directory exists and is readable/writable for KSQL server \n or its parent directory is readable/writable by KSQL server\n or change it to a readable/writable directory by setting 'ksql.metastore.backup.location' config in the properties file.");
            }
            try {
                Files.setPosixFilePermissions(backupsDir.toPath(), PosixFilePermissions.fromString("rwx------"));
            }
            catch (IOException e) {
                throw new KsqlServerException(String.format("Couldn't set POSIX permissions on the backups directory: %s. Error = %s", backupsDir.getPath(), e.getMessage()));
            }
        }
        if (!backupsDir.isDirectory()) {
            throw new KsqlServerException(backupsDir.getPath() + " is not a directory.\n Make sure the directory exists and is readable/writable for KSQL server \n or its parent directory is readable/writable by KSQL server\n or change it to a readable/writable directory by setting 'ksql.metastore.backup.location' config in the properties file.");
        }
        if (!(backupsDir.canWrite() && backupsDir.canRead() && backupsDir.canExecute())) {
            throw new KsqlServerException("The backups directory is not readable/writable for KSQL server: " + backupsDir.getPath() + "\n Make sure the directory exists and is readable/writable for KSQL server \n or change it to a readable/writable directory by setting 'ksql.metastore.backup.location' config in the properties file.");
        }
    }
}

