/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.rest.server;

import com.clearspring.analytics.util.Lists;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.AbstractScheduledService;
import com.google.common.util.concurrent.ServiceManager;
import io.confluent.ksql.engine.KsqlEngine;
import io.confluent.ksql.rest.server.ServerUtil;
import io.confluent.ksql.rest.util.DiscoverRemoteHostsUtil;
import io.confluent.ksql.services.ServiceContext;
import io.confluent.ksql.util.HostStatus;
import io.confluent.ksql.util.KsqlHostInfo;
import java.net.URI;
import java.net.URL;
import java.time.Clock;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.kafka.streams.state.HostInfo;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class HeartbeatAgent {
    private static final int SERVICE_TIMEOUT_SEC = 2;
    private static final int CHECK_HEARTBEAT_DELAY_MS = 1000;
    private static final int SEND_HEARTBEAT_DELAY_MS = 100;
    private static final int DISCOVER_CLUSTER_DELAY_MS = 50;
    private static final Logger LOG = LogManager.getLogger(HeartbeatAgent.class);
    private final KsqlEngine engine;
    private final ServiceContext serviceContext;
    private final HeartbeatConfig config;
    private final List<HostStatusListener> hostStatusListeners;
    private final ConcurrentHashMap<KsqlHostInfo, TreeMap<Long, HeartbeatInfo>> receivedHeartbeats;
    private final ConcurrentHashMap<KsqlHostInfo, HostStatus> hostsStatus;
    private final ScheduledExecutorService scheduledExecutorService;
    private final ServiceManager serviceManager;
    private final Clock clock;
    private KsqlHostInfo localHost;
    private URL localUrl;

    public static Builder builder() {
        return new Builder();
    }

    private HeartbeatAgent(KsqlEngine engine, ServiceContext serviceContext, HeartbeatConfig config, List<HostStatusListener> hostStatusListeners) {
        this.engine = Objects.requireNonNull(engine, "engine");
        this.serviceContext = Objects.requireNonNull(serviceContext, "serviceContext");
        this.config = Objects.requireNonNull(config, "configuration parameters");
        this.hostStatusListeners = Objects.requireNonNull(hostStatusListeners, "heartbeatListeners");
        this.scheduledExecutorService = Executors.newScheduledThreadPool(config.threadPoolSize);
        this.serviceManager = new ServiceManager(Arrays.asList(new AbstractScheduledService[]{new DiscoverClusterService(), new SendHeartbeatService(), new CheckHeartbeatService()}));
        this.receivedHeartbeats = new ConcurrentHashMap();
        this.hostsStatus = new ConcurrentHashMap();
        this.clock = Clock.systemUTC();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void receiveHeartbeat(KsqlHostInfo hostInfo, long timestamp) {
        TreeMap heartbeats;
        TreeMap treeMap = heartbeats = this.receivedHeartbeats.computeIfAbsent(hostInfo, key -> new TreeMap());
        synchronized (treeMap) {
            LOG.debug("Receive heartbeat at: {} from host: {} ", (Object)timestamp, (Object)hostInfo);
            heartbeats.put(timestamp, new HeartbeatInfo(timestamp));
        }
    }

    public Map<KsqlHostInfo, HostStatus> getHostsStatus() {
        return Collections.unmodifiableMap(this.hostsStatus);
    }

    @VisibleForTesting
    void setHostsStatus(Map<KsqlHostInfo, HostStatus> status) {
        this.hostsStatus.putAll(status);
    }

    void startAgent() {
        try {
            this.serviceManager.startAsync().awaitHealthy(2L, TimeUnit.SECONDS);
        }
        catch (IllegalStateException | TimeoutException e) {
            LOG.error("Failed to start heartbeat services with exception " + e.getMessage(), (Throwable)e);
        }
    }

    void stopAgent() {
        try {
            this.serviceManager.stopAsync().awaitStopped(2L, TimeUnit.SECONDS);
        }
        catch (IllegalStateException | TimeoutException e) {
            LOG.error("Failed to stop heartbeat services with exception " + e.getMessage(), (Throwable)e);
        }
        finally {
            this.scheduledExecutorService.shutdownNow();
        }
    }

    void setLocalAddress(String applicationServer) {
        HostInfo hostInfo = ServerUtil.parseHostInfo(applicationServer);
        this.localHost = new KsqlHostInfo(hostInfo.host(), hostInfo.port());
        try {
            this.localUrl = new URL(applicationServer);
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to convert remote host info to URL. remoteInfo: " + this.localHost.host() + ":" + this.localHost.host());
        }
        Preconditions.checkState((boolean)this.hostsStatus.isEmpty(), (Object)"expected empty host status map on startup");
        this.hostsStatus.putIfAbsent(this.localHost, new HostStatus(true, this.clock.millis()));
    }

    public static class Builder {
        private int nestedThreadPoolSize;
        private long nestedHeartbeatSendIntervalMs;
        private long nestedHeartbeatCheckIntervalMs;
        private long nestedDiscoverClusterIntervalMs;
        private long nestedHeartbeatWindowMs;
        private long nestedHeartbeatMissedThreshold;
        private List<HostStatusListener> nestedHostStatusListeners = Lists.newArrayList();

        Builder threadPoolSize(int size) {
            this.nestedThreadPoolSize = size;
            return this;
        }

        Builder heartbeatSendInterval(long interval) {
            this.nestedHeartbeatSendIntervalMs = interval;
            return this;
        }

        Builder heartbeatCheckInterval(long interval) {
            this.nestedHeartbeatCheckIntervalMs = interval;
            return this;
        }

        Builder heartbeatWindow(long window) {
            this.nestedHeartbeatWindowMs = window;
            return this;
        }

        Builder heartbeatMissedThreshold(long missed) {
            this.nestedHeartbeatMissedThreshold = missed;
            return this;
        }

        Builder discoverClusterInterval(long interval) {
            this.nestedDiscoverClusterIntervalMs = interval;
            return this;
        }

        Builder addHostStatusListener(HostStatusListener listener) {
            this.nestedHostStatusListeners.add(listener);
            return this;
        }

        public HeartbeatAgent build(KsqlEngine engine, ServiceContext serviceContext) {
            return new HeartbeatAgent(engine, serviceContext, new HeartbeatConfig(this.nestedThreadPoolSize, this.nestedHeartbeatSendIntervalMs, this.nestedHeartbeatCheckIntervalMs, this.nestedHeartbeatWindowMs, this.nestedHeartbeatMissedThreshold, this.nestedDiscoverClusterIntervalMs), this.nestedHostStatusListeners);
        }
    }

    static class HeartbeatConfig {
        private final int threadPoolSize;
        private final long heartbeatSendIntervalMs;
        private final long heartbeatCheckIntervalMs;
        private final long heartbeatWindowMs;
        private final long heartbeatMissedThreshold;
        private final long discoverClusterIntervalMs;

        HeartbeatConfig(int threadPoolSize, long heartbeatSendIntervalMs, long heartbeatCheckIntervalMs, long heartbeatWindowMs, long heartbeatMissedThreshold, long discoverClusterIntervalMs) {
            this.threadPoolSize = threadPoolSize;
            this.heartbeatSendIntervalMs = heartbeatSendIntervalMs;
            this.heartbeatCheckIntervalMs = heartbeatCheckIntervalMs;
            this.heartbeatWindowMs = heartbeatWindowMs;
            this.heartbeatMissedThreshold = heartbeatMissedThreshold;
            this.discoverClusterIntervalMs = discoverClusterIntervalMs;
        }
    }

    class DiscoverClusterService
    extends AbstractScheduledService {
        DiscoverClusterService() {
        }

        protected void runOneIteration() {
            try {
                List currentQueries = HeartbeatAgent.this.engine.getPersistentQueries();
                if (currentQueries.isEmpty()) {
                    return;
                }
                Set<HostInfo> uniqueHosts = DiscoverRemoteHostsUtil.getRemoteHosts(currentQueries, HeartbeatAgent.this.localHost);
                for (HostInfo hostInfo : uniqueHosts) {
                    KsqlHostInfo host = new KsqlHostInfo(hostInfo.host(), hostInfo.port());
                    HeartbeatAgent.this.hostsStatus.computeIfAbsent(host, key -> new HostStatus(true, HeartbeatAgent.this.clock.millis()));
                }
            }
            catch (Throwable t) {
                LOG.error("Failed to discover cluster with exception " + t.getMessage(), t);
            }
        }

        protected AbstractScheduledService.Scheduler scheduler() {
            return AbstractScheduledService.Scheduler.newFixedRateSchedule((long)50L, (long)HeartbeatAgent.this.config.discoverClusterIntervalMs, (TimeUnit)TimeUnit.MILLISECONDS);
        }

        protected ScheduledExecutorService executor() {
            return HeartbeatAgent.this.scheduledExecutorService;
        }
    }

    class SendHeartbeatService
    extends AbstractScheduledService {
        SendHeartbeatService() {
        }

        protected void runOneIteration() {
            for (Map.Entry<KsqlHostInfo, HostStatus> hostStatusEntry : HeartbeatAgent.this.hostsStatus.entrySet()) {
                KsqlHostInfo remoteHost = hostStatusEntry.getKey();
                try {
                    if (remoteHost.equals((Object)HeartbeatAgent.this.localHost)) continue;
                    URI remoteUri = ServerUtil.buildRemoteUri(HeartbeatAgent.this.localUrl, remoteHost.host(), remoteHost.port());
                    LOG.debug("Send heartbeat to host {} at {}", (Object)remoteHost, (Object)HeartbeatAgent.this.clock.millis());
                    HeartbeatAgent.this.serviceContext.getKsqlClient().makeAsyncHeartbeatRequest(remoteUri, HeartbeatAgent.this.localHost, HeartbeatAgent.this.clock.millis());
                }
                catch (Throwable t) {
                    LOG.error("Request to server: " + String.valueOf(remoteHost) + " failed with exception: " + t.getMessage(), t);
                }
            }
        }

        protected AbstractScheduledService.Scheduler scheduler() {
            return AbstractScheduledService.Scheduler.newFixedRateSchedule((long)100L, (long)HeartbeatAgent.this.config.heartbeatSendIntervalMs, (TimeUnit)TimeUnit.MILLISECONDS);
        }

        protected ScheduledExecutorService executor() {
            return HeartbeatAgent.this.scheduledExecutorService;
        }
    }

    class CheckHeartbeatService
    extends AbstractScheduledService {
        CheckHeartbeatService() {
        }

        protected void runOneIteration() {
            long now = HeartbeatAgent.this.clock.millis();
            long windowStart = now - HeartbeatAgent.this.config.heartbeatWindowMs;
            this.runWithWindow(windowStart, now);
        }

        @VisibleForTesting
        void runWithWindow(long windowStart, long windowEnd) {
            try {
                this.processHeartbeats(windowStart, windowEnd);
            }
            catch (Throwable t) {
                LOG.error("Failed to process heartbeats for window start = " + windowStart + " end = " + windowEnd + " with exception " + t.getMessage(), t);
            }
        }

        protected AbstractScheduledService.Scheduler scheduler() {
            return AbstractScheduledService.Scheduler.newFixedRateSchedule((long)1000L, (long)HeartbeatAgent.this.config.heartbeatCheckIntervalMs, (TimeUnit)TimeUnit.MILLISECONDS);
        }

        protected ScheduledExecutorService executor() {
            return HeartbeatAgent.this.scheduledExecutorService;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void processHeartbeats(long windowStart, long windowEnd) {
            if (HeartbeatAgent.this.receivedHeartbeats.isEmpty()) {
                HeartbeatAgent.this.hostsStatus.replaceAll((host, status) -> {
                    if (!host.equals((Object)HeartbeatAgent.this.localHost)) {
                        return status.withHostAlive(false);
                    }
                    return status;
                });
                this.notifyListeners();
                return;
            }
            for (Map.Entry<KsqlHostInfo, HostStatus> hostEntry : HeartbeatAgent.this.hostsStatus.entrySet()) {
                TreeMap<Long, HeartbeatInfo> copy;
                KsqlHostInfo ksqlHostInfo = hostEntry.getKey();
                HostStatus hostStatus = hostEntry.getValue();
                if (ksqlHostInfo.equals((Object)HeartbeatAgent.this.localHost)) continue;
                TreeMap<Long, HeartbeatInfo> heartbeats = HeartbeatAgent.this.receivedHeartbeats.get(ksqlHostInfo);
                if (heartbeats == null || heartbeats.isEmpty()) {
                    HeartbeatAgent.this.hostsStatus.computeIfPresent(ksqlHostInfo, (host, status) -> status.withHostAlive(false));
                    continue;
                }
                TreeMap<Long, HeartbeatInfo> treeMap = heartbeats;
                synchronized (treeMap) {
                    LOG.debug("Process heartbeats: {} of host: {}", heartbeats, (Object)ksqlHostInfo);
                    heartbeats.headMap(windowStart).clear();
                    copy = new TreeMap<Long, HeartbeatInfo>((SortedMap<Long, HeartbeatInfo>)heartbeats.subMap(windowStart, true, windowEnd, true));
                    LOG.debug("Process heartbeats: {} of host: {}, window start: {}, window end: {}", copy, (Object)ksqlHostInfo, (Object)windowStart, (Object)windowEnd);
                }
                boolean isAlive = this.decideStatus(ksqlHostInfo, windowStart, windowEnd, copy);
                if (!isAlive) {
                    LOG.info("Host: {} marked as dead.", (Object)ksqlHostInfo);
                }
                HeartbeatAgent.this.hostsStatus.computeIfPresent(ksqlHostInfo, (host, status) -> status.withHostAlive(isAlive).withLastStatusUpdateMs(windowEnd));
            }
            this.notifyListeners();
        }

        private void notifyListeners() {
            for (HostStatusListener listener : HeartbeatAgent.this.hostStatusListeners) {
                try {
                    listener.onHostStatusUpdated(HeartbeatAgent.this.getHostsStatus());
                }
                catch (Throwable t) {
                    LOG.error("Error while notifying listener", t);
                }
            }
        }

        private boolean decideStatus(KsqlHostInfo ksqlHostInfo, long windowStart, long windowEnd, TreeMap<Long, HeartbeatInfo> heartbeats) {
            long ts;
            long missedCount = 0L;
            long prev = windowStart;
            if (heartbeats.isEmpty()) {
                return false;
            }
            Iterator<Long> iterator = heartbeats.keySet().iterator();
            while (iterator.hasNext() && (ts = iterator.next().longValue()) < windowEnd) {
                if (ts - HeartbeatAgent.this.config.heartbeatSendIntervalMs > prev) {
                    missedCount = (ts - prev - 1L) / HeartbeatAgent.this.config.heartbeatSendIntervalMs;
                    LOG.debug("Host: {} missed: {} heartbeats, current heartbeat: {}, previous heartbeat: {}, send interval: {}.", (Object)ksqlHostInfo, (Object)missedCount, (Object)ts, (Object)prev, (Object)HeartbeatAgent.this.config.heartbeatSendIntervalMs);
                } else {
                    missedCount = 0L;
                }
                prev = ts;
            }
            if (windowEnd - prev - 1L > 0L) {
                missedCount = (windowEnd - prev - 1L) / HeartbeatAgent.this.config.heartbeatSendIntervalMs;
                LOG.debug("Host: {} missed: {} heartbeats, window end: {}, previous heartbeat: {}, send interval: {}.", (Object)ksqlHostInfo, (Object)missedCount, (Object)windowEnd, (Object)prev, (Object)HeartbeatAgent.this.config.heartbeatSendIntervalMs);
            }
            LOG.debug("Host: {} has {} missing heartbeats", (Object)ksqlHostInfo, (Object)missedCount);
            return missedCount < HeartbeatAgent.this.config.heartbeatMissedThreshold;
        }
    }

    public static class HeartbeatInfo {
        private final long timestamp;

        public HeartbeatInfo(long timestamp) {
            this.timestamp = timestamp;
        }

        public long getTimestamp() {
            return this.timestamp;
        }

        public String toString() {
            return String.valueOf(this.timestamp);
        }
    }

    public static interface HostStatusListener {
        public void onHostStatusUpdated(Map<KsqlHostInfo, HostStatus> var1);
    }
}

