/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.rest.server.computation;

import com.google.common.annotations.VisibleForTesting;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import io.confluent.ksql.rest.Errors;
import io.confluent.ksql.rest.server.computation.Command;
import io.confluent.ksql.rest.server.computation.CommandQueue;
import io.confluent.ksql.rest.server.computation.CommandRunnerMetrics;
import io.confluent.ksql.rest.server.computation.InteractiveStatementExecutor;
import io.confluent.ksql.rest.server.computation.QueuedCommand;
import io.confluent.ksql.rest.server.computation.RestoreCommandsCompactor;
import io.confluent.ksql.rest.server.resources.IncompatibleKsqlCommandVersionException;
import io.confluent.ksql.rest.server.state.ServerState;
import io.confluent.ksql.rest.util.ClusterTerminator;
import io.confluent.ksql.rest.util.PersistentQueryCleanupImpl;
import io.confluent.ksql.services.KafkaTopicClient;
import io.confluent.ksql.util.Pair;
import io.confluent.ksql.util.QueryMetadata;
import io.confluent.ksql.util.RetryUtil;
import java.io.Closeable;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.Temporal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.kafka.clients.consumer.OffsetOutOfRangeException;
import org.apache.kafka.common.errors.SerializationException;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class CommandRunner
implements Closeable {
    private static final Logger LOG = LogManager.getLogger(CommandRunner.class);
    private static final int STATEMENT_RETRY_MS = 100;
    private static final int MAX_STATEMENT_RETRY_MS = 5000;
    private static final Duration NEW_CMDS_TIMEOUT = Duration.ofMillis(5000L);
    private static final int SHUTDOWN_TIMEOUT_MS = 15000;
    private static final int COMMAND_TOPIC_THRESHOLD_LIMIT = 10000;
    private final InteractiveStatementExecutor statementExecutor;
    private final CommandQueue commandStore;
    private final ExecutorService executor;
    private final Function<List<QueuedCommand>, List<QueuedCommand>> compactor;
    private volatile boolean closed = false;
    private final int maxRetries;
    private final ClusterTerminator clusterTerminator;
    private final ServerState serverState;
    private final CommandRunnerMetrics commandRunnerMetric;
    private final AtomicReference<Pair<QueuedCommand, Instant>> currentCommandRef;
    private final AtomicReference<Instant> lastPollTime;
    private final Duration commandRunnerHealthTimeout;
    private final Clock clock;
    private final Deserializer<Command> commandDeserializer;
    private final Consumer<QueuedCommand> incompatibleCommandChecker;
    private final Errors errorHandler;
    private boolean incompatibleCommandDetected;
    private final Supplier<Boolean> commandTopicExists;
    private boolean commandTopicDeleted;
    private Status state = new Status(CommandRunnerStatus.RUNNING, CommandRunnerDegradedReason.NONE);

    public CommandRunner(InteractiveStatementExecutor statementExecutor, CommandQueue commandStore, int maxRetries, ClusterTerminator clusterTerminator, ServerState serverState, String ksqlServiceId, Duration commandRunnerHealthTimeout, String metricsGroupPrefix, Deserializer<Command> commandDeserializer, Errors errorHandler, KafkaTopicClient kafkaTopicClient, String commandTopicName, Metrics metrics) {
        this(statementExecutor, commandStore, maxRetries, clusterTerminator, Executors.newSingleThreadExecutor(r -> new Thread(r, "CommandRunner")), serverState, ksqlServiceId, commandRunnerHealthTimeout, metricsGroupPrefix, Clock.systemUTC(), RestoreCommandsCompactor::compact, queuedCommand -> {
            queuedCommand.getAndDeserializeCommandId();
            queuedCommand.getAndDeserializeCommand(commandDeserializer);
        }, commandDeserializer, errorHandler, () -> kafkaTopicClient.isTopicExists(commandTopicName), metrics);
    }

    @VisibleForTesting
    CommandRunner(InteractiveStatementExecutor statementExecutor, CommandQueue commandStore, int maxRetries, ClusterTerminator clusterTerminator, ExecutorService executor, ServerState serverState, String ksqlServiceId, Duration commandRunnerHealthTimeout, String metricsGroupPrefix, Clock clock, Function<List<QueuedCommand>, List<QueuedCommand>> compactor, Consumer<QueuedCommand> incompatibleCommandChecker, Deserializer<Command> commandDeserializer, Errors errorHandler, Supplier<Boolean> commandTopicExists, Metrics metrics) {
        this.statementExecutor = Objects.requireNonNull(statementExecutor, "statementExecutor");
        this.commandStore = Objects.requireNonNull(commandStore, "commandStore");
        this.maxRetries = maxRetries;
        this.clusterTerminator = Objects.requireNonNull(clusterTerminator, "clusterTerminator");
        this.executor = Objects.requireNonNull(executor, "executor");
        this.serverState = Objects.requireNonNull(serverState, "serverState");
        this.commandRunnerHealthTimeout = Objects.requireNonNull(commandRunnerHealthTimeout, "commandRunnerHealthTimeout");
        this.currentCommandRef = new AtomicReference<Object>(null);
        this.lastPollTime = new AtomicReference<Object>(null);
        this.commandRunnerMetric = new CommandRunnerMetrics(ksqlServiceId, this, metricsGroupPrefix, metrics);
        this.clock = Objects.requireNonNull(clock, "clock");
        this.compactor = Objects.requireNonNull(compactor, "compactor");
        this.incompatibleCommandChecker = Objects.requireNonNull(incompatibleCommandChecker, "incompatibleCommandChecker");
        this.commandDeserializer = Objects.requireNonNull(commandDeserializer, "commandDeserializer");
        this.errorHandler = Objects.requireNonNull(errorHandler, "errorHandler");
        this.commandTopicExists = Objects.requireNonNull(commandTopicExists, "commandTopicExists");
        this.incompatibleCommandDetected = false;
        this.commandTopicDeleted = false;
    }

    public void start() {
        this.executor.execute(new Runner());
        this.executor.shutdown();
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.closeEarly();
        }
        this.commandRunnerMetric.close();
    }

    private void closeEarly() {
        try {
            this.closed = true;
            this.commandStore.wakeup();
            this.executor.awaitTermination(15000L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public void processPriorCommands(PersistentQueryCleanupImpl queryCleanup) {
        try {
            Optional<QueuedCommand> terminateCmd;
            List<QueuedCommand> restoreCommands = this.commandStore.getRestoreCommands();
            List<QueuedCommand> compatibleCommands = this.checkForIncompatibleCommands(restoreCommands);
            LOG.info("Restoring previous state from {} commands.", (Object)compatibleCommands.size());
            if (compatibleCommands.size() > 10000) {
                LOG.warn("Command topic size exceeded. [commands={}, threshold={}]", (Object)compatibleCommands.size(), (Object)10000);
            }
            if ((terminateCmd = CommandRunner.findTerminateCommand(compatibleCommands, this.commandDeserializer)).isPresent()) {
                LOG.info("Cluster previously terminated: terminating.");
                this.terminateCluster(terminateCmd.get().getAndDeserializeCommand(this.commandDeserializer));
                return;
            }
            List<QueuedCommand> compacted = this.compactor.apply(compatibleCommands);
            compacted.forEach(command -> {
                this.currentCommandRef.set((Pair<QueuedCommand, Instant>)new Pair(command, (Object)this.clock.instant()));
                RetryUtil.retryWithBackoff((int)this.maxRetries, (int)100, (int)5000, () -> this.statementExecutor.handleRestore((QueuedCommand)command), (Class[])new Class[]{WakeupException.class});
                this.currentCommandRef.set(null);
            });
            List queries = this.statementExecutor.getKsqlEngine().getPersistentQueries();
            if (this.commandStore.corruptionDetected()) {
                LOG.info("Corruption detected, queries will not be started.");
                queries.forEach(QueryMetadata::setCorruptionQueryError);
            } else {
                LOG.info("Restarting {} queries.", (Object)queries.size());
                queries.forEach(QueryMetadata::start);
                queryCleanup.cleanupLeakedQueries(queries);
            }
            LOG.info("Restore complete");
        }
        catch (Exception e) {
            LOG.error("Error during restore", (Throwable)e);
            throw e;
        }
    }

    void fetchAndRunCommands() {
        this.lastPollTime.set(this.clock.instant());
        List<QueuedCommand> commands = this.commandStore.getNewCommands(NEW_CMDS_TIMEOUT);
        if (commands.isEmpty()) {
            if (!this.commandTopicExists.get().booleanValue()) {
                this.commandTopicDeleted = true;
            }
            return;
        }
        List<QueuedCommand> compatibleCommands = this.checkForIncompatibleCommands(commands);
        Optional<QueuedCommand> terminateCmd = CommandRunner.findTerminateCommand(compatibleCommands, this.commandDeserializer);
        if (terminateCmd.isPresent()) {
            this.terminateCluster(terminateCmd.get().getAndDeserializeCommand(this.commandDeserializer));
            return;
        }
        LOG.debug("Found {} new writes to command topic", (Object)compatibleCommands.size());
        for (QueuedCommand command : compatibleCommands) {
            if (this.closed) {
                return;
            }
            this.executeStatement(command);
        }
    }

    private void executeStatement(QueuedCommand queuedCommand) {
        String commandId = queuedCommand.getAndDeserializeCommandId().toString();
        LOG.info("Executing statement: " + commandId);
        Runnable task = () -> {
            if (this.closed) {
                LOG.info("Execution aborted as system is closing down");
            } else {
                this.statementExecutor.handleStatement(queuedCommand);
                LOG.info("Executed statement: " + commandId);
            }
        };
        this.currentCommandRef.set((Pair<QueuedCommand, Instant>)new Pair((Object)queuedCommand, (Object)this.clock.instant()));
        RetryUtil.retryWithBackoff((int)this.maxRetries, (int)100, (int)5000, (Runnable)task, (Class[])new Class[]{WakeupException.class});
        this.currentCommandRef.set(null);
    }

    private static Optional<QueuedCommand> findTerminateCommand(List<QueuedCommand> restoreCommands, Deserializer<Command> commandDeserializer) {
        return restoreCommands.stream().filter(command -> command.getAndDeserializeCommand(commandDeserializer).getStatement().equalsIgnoreCase("TERMINATE CLUSTER;")).findFirst();
    }

    private void terminateCluster(Command command) {
        this.serverState.setTerminating();
        LOG.info("Terminating the KSQL server.");
        this.close();
        List<String> deleteTopicList = command.getOverwriteProperties().getOrDefault("deleteTopicList", Collections.emptyList());
        this.clusterTerminator.terminateCluster(deleteTopicList);
        this.serverState.setTerminated();
        LOG.info("The KSQL server was terminated.");
        this.closeEarly();
        LOG.debug("The KSQL command runner was closed.");
    }

    private List<QueuedCommand> checkForIncompatibleCommands(List<QueuedCommand> commands) {
        ArrayList<QueuedCommand> compatibleCommands = new ArrayList<QueuedCommand>();
        try {
            for (QueuedCommand command : commands) {
                this.incompatibleCommandChecker.accept(command);
                compatibleCommands.add(command);
            }
        }
        catch (IncompatibleKsqlCommandVersionException | SerializationException e) {
            LOG.info("Incompatible command record detected when processing command topic", (Throwable)e);
            this.incompatibleCommandDetected = true;
        }
        return compatibleCommands;
    }

    @SuppressFBWarnings(value={"EI_EXPOSE_REP"}, justification="should be mutable")
    public CommandQueue getCommandQueue() {
        return this.commandStore;
    }

    public CommandRunnerStatus checkCommandRunnerStatus() {
        if (this.state.getStatus() == CommandRunnerStatus.DEGRADED) {
            return CommandRunnerStatus.DEGRADED;
        }
        Pair<QueuedCommand, Instant> currentCommand = this.currentCommandRef.get();
        this.state = currentCommand == null ? (this.lastPollTime.get() == null || Duration.between(this.lastPollTime.get(), this.clock.instant()).toMillis() < NEW_CMDS_TIMEOUT.toMillis() * 3L ? new Status(CommandRunnerStatus.RUNNING, CommandRunnerDegradedReason.NONE) : new Status(CommandRunnerStatus.ERROR, CommandRunnerDegradedReason.NONE)) : (Duration.between((Temporal)currentCommand.right, this.clock.instant()).toMillis() < this.commandRunnerHealthTimeout.toMillis() ? new Status(CommandRunnerStatus.RUNNING, CommandRunnerDegradedReason.NONE) : new Status(CommandRunnerStatus.ERROR, CommandRunnerDegradedReason.NONE));
        return this.state.getStatus();
    }

    public ServerState.State checkServerState() {
        return this.serverState.getState();
    }

    public CommandRunnerDegradedReason getCommandRunnerDegradedReason() {
        return this.state.getDegradedReason();
    }

    public String getCommandRunnerDegradedWarning() {
        return this.getCommandRunnerDegradedReason().getMsg(this.errorHandler);
    }

    public static class Status {
        private final CommandRunnerStatus status;
        private final CommandRunnerDegradedReason degradedReason;

        public Status(CommandRunnerStatus status, CommandRunnerDegradedReason degradedReason) {
            this.status = status;
            this.degradedReason = degradedReason;
        }

        public CommandRunnerStatus getStatus() {
            return this.status;
        }

        public CommandRunnerDegradedReason getDegradedReason() {
            return this.degradedReason;
        }
    }

    public static enum CommandRunnerStatus {
        RUNNING,
        ERROR,
        DEGRADED;

    }

    public static enum CommandRunnerDegradedReason {
        NONE(errors -> ""),
        CORRUPTED(Errors::commandRunnerDegradedCorruptedErrorMessage),
        INCOMPATIBLE_COMMAND(Errors::commandRunnerDegradedIncompatibleCommandsErrorMessage);

        private final Function<Errors, String> msgFactory;

        public String getMsg(Errors errors) {
            return this.msgFactory.apply(errors);
        }

        private CommandRunnerDegradedReason(Function<Errors, String> msgFactory) {
            this.msgFactory = msgFactory;
        }
    }

    private class Runner
    implements Runnable {
        private Runner() {
        }

        @Override
        public void run() {
            try {
                while (!CommandRunner.this.closed) {
                    if (CommandRunner.this.incompatibleCommandDetected) {
                        LOG.warn("CommandRunner entering degraded state due to encountering an incompatible command");
                        CommandRunner.this.state = new Status(CommandRunnerStatus.DEGRADED, CommandRunnerDegradedReason.INCOMPATIBLE_COMMAND);
                        CommandRunner.this.closeEarly();
                        continue;
                    }
                    if (CommandRunner.this.commandStore.corruptionDetected()) {
                        LOG.warn("CommandRunner entering degraded state due to encountering corruption between topic and backup");
                        CommandRunner.this.state = new Status(CommandRunnerStatus.DEGRADED, CommandRunnerDegradedReason.CORRUPTED);
                        CommandRunner.this.closeEarly();
                        continue;
                    }
                    if (CommandRunner.this.commandTopicDeleted) {
                        LOG.warn("CommandRunner entering degraded state due to command topic deletion.");
                        CommandRunner.this.state = new Status(CommandRunnerStatus.DEGRADED, CommandRunnerDegradedReason.CORRUPTED);
                        CommandRunner.this.closeEarly();
                        continue;
                    }
                    LOG.trace("Polling for new writes to command topic");
                    CommandRunner.this.fetchAndRunCommands();
                }
            }
            catch (WakeupException wue) {
                if (!CommandRunner.this.closed) {
                    throw wue;
                }
            }
            catch (OffsetOutOfRangeException e) {
                LOG.warn("The command topic offset was reset. CommandRunner thread exiting.");
                CommandRunner.this.state = new Status(CommandRunnerStatus.DEGRADED, CommandRunnerDegradedReason.CORRUPTED);
                CommandRunner.this.closeEarly();
            }
            finally {
                LOG.info("Closing command store");
                CommandRunner.this.commandStore.close();
            }
        }
    }
}

