/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.kafka.common.InvalidRecordException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.compress.Compression;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.CorruptRecordException;
import org.apache.kafka.common.errors.InvalidTimestampException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.errors.UnsupportedForMessageFormatException;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.AbstractRecords;
import org.apache.kafka.common.record.CompressionType;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.MemoryRecordsBuilder;
import org.apache.kafka.common.record.MutableRecordBatch;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.RecordValidationStats;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.requests.ProduceResponse;
import org.apache.kafka.common.utils.BufferSupplier;
import org.apache.kafka.common.utils.CloseableIterator;
import org.apache.kafka.common.utils.PrimitiveRef;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.interceptor.RecordInterceptor;
import org.apache.kafka.storage.internals.log.AppendOrigin;
import org.apache.kafka.storage.internals.log.KafkaLogValidator;
import org.apache.kafka.storage.internals.log.RecordValidationException;

public class LogValidator
implements KafkaLogValidator {
    private final MemoryRecords records;
    private final TopicPartition topicPartition;
    private final Time time;
    private final CompressionType sourceCompressionType;
    private final Compression targetCompression;
    private final boolean compactedTopic;
    private final byte toMagic;
    private final TimestampType timestampType;
    private final long timestampBeforeMaxMs;
    private final long timestampAfterMaxMs;
    private final List<RecordInterceptor> interceptors;
    private final int partitionLeaderEpoch;
    private final AppendOrigin origin;

    public LogValidator(MemoryRecords records, TopicPartition topicPartition, Time time, CompressionType sourceCompressionType, Compression targetCompression, boolean compactedTopic, byte toMagic, TimestampType timestampType, long timestampBeforeMaxMs, long timestampAfterMaxMs, List<RecordInterceptor> interceptors, int partitionLeaderEpoch, AppendOrigin origin) {
        this.records = records;
        this.topicPartition = topicPartition;
        this.time = time;
        this.sourceCompressionType = sourceCompressionType;
        this.targetCompression = targetCompression;
        this.compactedTopic = compactedTopic;
        this.toMagic = toMagic;
        this.timestampType = timestampType;
        this.timestampBeforeMaxMs = timestampBeforeMaxMs;
        this.timestampAfterMaxMs = timestampAfterMaxMs;
        this.interceptors = interceptors;
        this.partitionLeaderEpoch = partitionLeaderEpoch;
        this.origin = origin;
    }

    @Override
    public ValidationResult validateMessagesAndAssignOffsets(PrimitiveRef.LongRef offsetCounter, long currentTimeMs, MetricsRecorder metricsRecorder, BufferSupplier bufferSupplier) {
        if (this.sourceCompressionType == CompressionType.NONE && this.targetCompression.type() == CompressionType.NONE) {
            if (!this.records.hasMatchingMagic(this.toMagic)) {
                return this.convertAndAssignOffsetsNonCompressed(offsetCounter, currentTimeMs, metricsRecorder);
            }
            return this.assignOffsetsNonCompressed(offsetCounter, currentTimeMs, metricsRecorder);
        }
        return this.validateMessagesAndAssignOffsetsCompressed(offsetCounter, currentTimeMs, metricsRecorder, bufferSupplier);
    }

    private static MutableRecordBatch getFirstBatchAndMaybeValidateNoMoreBatches(MemoryRecords records, CompressionType sourceCompression) {
        Iterator batchIterator = records.batches().iterator();
        if (!batchIterator.hasNext()) {
            throw new InvalidRecordException("Record batch has no batches at all");
        }
        MutableRecordBatch batch = (MutableRecordBatch)batchIterator.next();
        if ((batch.magic() >= 2 || sourceCompression != CompressionType.NONE) && batchIterator.hasNext()) {
            throw new InvalidRecordException("Compressed outer record has more than one batch");
        }
        return batch;
    }

    private ValidationResult convertAndAssignOffsetsNonCompressed(PrimitiveRef.LongRef offsetCounter, long currentTimeMs, MetricsRecorder metricsRecorder) {
        long startNanos = this.time.nanoseconds();
        int sizeInBytesAfterConversion = AbstractRecords.estimateSizeInBytes((byte)this.toMagic, (long)offsetCounter.value, (CompressionType)CompressionType.NONE, (Iterable)this.records.records());
        MutableRecordBatch firstBatch = LogValidator.getFirstBatchAndMaybeValidateNoMoreBatches(this.records, CompressionType.NONE);
        long producerId = firstBatch.producerId();
        short producerEpoch = firstBatch.producerEpoch();
        int sequence = firstBatch.baseSequence();
        boolean isTransactional = firstBatch.isTransactional();
        ByteBuffer newBuffer = ByteBuffer.allocate(sizeInBytesAfterConversion);
        MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)newBuffer, (byte)this.toMagic, (Compression)Compression.NONE, (TimestampType)this.timestampType, (long)offsetCounter.value, (long)currentTimeMs, (long)producerId, (short)producerEpoch, (int)sequence, (boolean)isTransactional, (int)this.partitionLeaderEpoch);
        for (RecordBatch batch : this.records.batches()) {
            LogValidator.validateBatch(this.topicPartition, (RecordBatch)firstBatch, batch, this.origin, this.toMagic, metricsRecorder);
            ArrayList<ApiRecordError> recordErrors = new ArrayList<ApiRecordError>(0);
            int recordIndex = 0;
            for (Record record : batch) {
                Optional<ApiRecordError> recordError = LogValidator.validateRecord(batch, this.topicPartition, record, recordIndex, currentTimeMs, this.timestampType, this.timestampBeforeMaxMs, this.timestampAfterMaxMs, this.compactedTopic, this.interceptors, metricsRecorder);
                recordError.ifPresent(e -> recordErrors.add((ApiRecordError)e));
                if (recordErrors.isEmpty()) {
                    builder.appendWithOffset(offsetCounter.value++, record);
                }
                ++recordIndex;
            }
            LogValidator.processRecordErrors(recordErrors);
        }
        MemoryRecords convertedRecords = builder.build();
        MemoryRecordsBuilder.RecordsInfo info = builder.info();
        RecordValidationStats recordValidationStats = new RecordValidationStats((long)builder.uncompressedBytesWritten(), builder.numRecords(), this.time.nanoseconds() - startNanos);
        return new ValidationResult(convertedRecords, info.maxTimestamp, true, recordValidationStats);
    }

    public ValidationResult assignOffsetsNonCompressed(PrimitiveRef.LongRef offsetCounter, long currentTimeMs, MetricsRecorder metricsRecorder) {
        long maxTimestamp = -1L;
        MutableRecordBatch firstBatch = LogValidator.getFirstBatchAndMaybeValidateNoMoreBatches(this.records, CompressionType.NONE);
        for (MutableRecordBatch batch : this.records.batches()) {
            LogValidator.validateBatch(this.topicPartition, (RecordBatch)firstBatch, (RecordBatch)batch, this.origin, this.toMagic, metricsRecorder);
            long maxBatchTimestamp = -1L;
            ArrayList<ApiRecordError> recordErrors = new ArrayList<ApiRecordError>(0);
            int recordIndex = 0;
            for (Record record : batch) {
                Optional<ApiRecordError> recordError = LogValidator.validateRecord((RecordBatch)batch, this.topicPartition, record, recordIndex, currentTimeMs, this.timestampType, this.timestampBeforeMaxMs, this.timestampAfterMaxMs, this.compactedTopic, this.interceptors, metricsRecorder);
                recordError.ifPresent(e -> recordErrors.add((ApiRecordError)e));
                ++offsetCounter.value;
                if (batch.magic() > 0 && record.timestamp() > maxBatchTimestamp) {
                    maxBatchTimestamp = record.timestamp();
                }
                ++recordIndex;
            }
            LogValidator.processRecordErrors(recordErrors);
            if (batch.magic() > 0 && maxBatchTimestamp > maxTimestamp) {
                maxTimestamp = maxBatchTimestamp;
            }
            batch.setLastOffset(offsetCounter.value - 1L);
            if (batch.magic() >= 2) {
                batch.setPartitionLeaderEpoch(this.partitionLeaderEpoch);
            }
            if (batch.magic() <= 0) continue;
            if (this.timestampType == TimestampType.LOG_APPEND_TIME) {
                batch.setMaxTimestamp(TimestampType.LOG_APPEND_TIME, currentTimeMs);
                continue;
            }
            batch.setMaxTimestamp(this.timestampType, maxBatchTimestamp);
        }
        if (this.timestampType == TimestampType.LOG_APPEND_TIME) {
            maxTimestamp = this.toMagic == 0 ? -1L : currentTimeMs;
        }
        return new ValidationResult(this.records, maxTimestamp, false, RecordValidationStats.EMPTY);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ValidationResult validateMessagesAndAssignOffsetsCompressed(PrimitiveRef.LongRef offsetCounter, long currentTimeMs, MetricsRecorder metricsRecorder, BufferSupplier bufferSupplier) {
        boolean inPlaceAssignment = this.sourceCompressionType == this.targetCompression.type();
        long maxTimestamp = -1L;
        PrimitiveRef.LongRef expectedInnerOffset = PrimitiveRef.ofLong((long)0L);
        ArrayList<Record> validatedRecords = new ArrayList<Record>();
        int uncompressedSizeInBytes = 0;
        MutableRecordBatch firstBatch = LogValidator.getFirstBatchAndMaybeValidateNoMoreBatches(this.records, this.sourceCompressionType);
        if (firstBatch.magic() != this.toMagic || this.toMagic == 0) {
            inPlaceAssignment = false;
        }
        if (this.sourceCompressionType == CompressionType.NONE && firstBatch.isControlBatch()) {
            inPlaceAssignment = true;
        }
        for (MutableRecordBatch batch : this.records.batches()) {
            LogValidator.validateBatch(this.topicPartition, (RecordBatch)firstBatch, (RecordBatch)batch, this.origin, this.toMagic, metricsRecorder);
            uncompressedSizeInBytes += AbstractRecords.recordBatchHeaderSizeInBytes((byte)this.toMagic, (CompressionType)batch.compressionType());
            try (CloseableIterator recordsIterator = inPlaceAssignment && firstBatch.magic() >= 2 && this.interceptors.isEmpty() ? batch.skipKeyValueIterator(bufferSupplier) : batch.streamingIterator(bufferSupplier);){
                ArrayList<ApiRecordError> recordErrors = new ArrayList<ApiRecordError>(0);
                int recordIndex = 0;
                while (recordsIterator.hasNext()) {
                    Record record = (Record)recordsIterator.next();
                    ++expectedInnerOffset.value;
                    Optional<ApiRecordError> recordError = LogValidator.validateRecordCompression(this.sourceCompressionType, recordIndex, record);
                    if (recordError.isEmpty()) {
                        recordError = LogValidator.validateRecord((RecordBatch)batch, this.topicPartition, record, recordIndex, currentTimeMs, this.timestampType, this.timestampBeforeMaxMs, this.timestampAfterMaxMs, this.compactedTopic, this.interceptors, metricsRecorder);
                    }
                    if (recordError.isEmpty() && batch.magic() > 0 && this.toMagic > 0) {
                        long expectedOffset;
                        if (record.timestamp() > maxTimestamp) {
                            maxTimestamp = record.timestamp();
                        }
                        if (record.offset() != expectedOffset) {
                            inPlaceAssignment = false;
                        }
                    }
                    if (recordError.isPresent()) {
                        recordErrors.add(recordError.get());
                    } else {
                        uncompressedSizeInBytes += record.sizeInBytes();
                        validatedRecords.add(record);
                    }
                    ++recordIndex;
                }
                LogValidator.processRecordErrors(recordErrors);
            }
        }
        if (!inPlaceAssignment) {
            return this.buildRecordsAndAssignOffsets(offsetCounter, currentTimeMs, (RecordBatch)firstBatch, validatedRecords, uncompressedSizeInBytes);
        }
        offsetCounter.value += (long)validatedRecords.size();
        long lastOffset = offsetCounter.value - 1L;
        firstBatch.setLastOffset(lastOffset);
        if (this.timestampType == TimestampType.LOG_APPEND_TIME) {
            maxTimestamp = currentTimeMs;
        }
        if (this.toMagic >= 1) {
            firstBatch.setMaxTimestamp(this.timestampType, maxTimestamp);
        }
        if (this.toMagic >= 2) {
            firstBatch.setPartitionLeaderEpoch(this.partitionLeaderEpoch);
        }
        RecordValidationStats recordValidationStats = new RecordValidationStats((long)uncompressedSizeInBytes, 0, 0L);
        return new ValidationResult(this.records, maxTimestamp, false, recordValidationStats);
    }

    private ValidationResult buildRecordsAndAssignOffsets(PrimitiveRef.LongRef offsetCounter, long logAppendTime, RecordBatch firstBatch, List<Record> validatedRecords, int uncompressedSizeInBytes) {
        long startNanos = this.time.nanoseconds();
        int estimatedSize = AbstractRecords.estimateSizeInBytes((byte)this.toMagic, (long)offsetCounter.value, (CompressionType)this.targetCompression.type(), validatedRecords);
        ByteBuffer buffer = ByteBuffer.allocate(estimatedSize);
        MemoryRecordsBuilder builder = MemoryRecords.builder((ByteBuffer)buffer, (byte)this.toMagic, (Compression)this.targetCompression, (TimestampType)this.timestampType, (long)offsetCounter.value, (long)logAppendTime, (long)firstBatch.producerId(), (short)firstBatch.producerEpoch(), (int)firstBatch.baseSequence(), (boolean)firstBatch.isTransactional(), (int)this.partitionLeaderEpoch);
        for (Record record : validatedRecords) {
            builder.appendWithOffset(offsetCounter.value++, record);
        }
        MemoryRecords records = builder.build();
        MemoryRecordsBuilder.RecordsInfo info = builder.info();
        int conversionCount = builder.numRecords();
        RecordValidationStats recordValidationStats = new RecordValidationStats((long)(uncompressedSizeInBytes + builder.uncompressedBytesWritten()), conversionCount, this.time.nanoseconds() - startNanos);
        return new ValidationResult(records, info.maxTimestamp, true, recordValidationStats);
    }

    public static void validateBatch(TopicPartition topicPartition, RecordBatch firstBatch, RecordBatch batch, AppendOrigin origin, byte toMagic, MetricsRecorder metricsRecorder) {
        if (firstBatch.magic() != batch.magic()) {
            metricsRecorder.recordInvalidMagic();
            throw new InvalidRecordException("Batch magic " + batch.magic() + " is not the same as the first batch's magic byte " + firstBatch.magic() + " in topic partition " + String.valueOf(topicPartition));
        }
        if (batch.magic() >= 2) {
            LogValidator.validateBatchV2(topicPartition, firstBatch, origin, metricsRecorder);
        }
        if (batch.isTransactional() && toMagic < 2) {
            throw new UnsupportedForMessageFormatException("Transactional records cannot be used with magic version " + toMagic);
        }
        if (batch.hasProducerId() && toMagic < 2) {
            throw new UnsupportedForMessageFormatException("Idempotent records cannot be used with magic version " + toMagic);
        }
    }

    public static void validateBatchV2(TopicPartition topicPartition, RecordBatch batch, AppendOrigin origin, MetricsRecorder metricsRecorder) {
        if (origin == AppendOrigin.CLIENT) {
            long countFromOffsets = batch.lastOffset() - batch.baseOffset() + 1L;
            if (countFromOffsets <= 0L) {
                metricsRecorder.recordInvalidOffset();
                throw new InvalidRecordException("Batch has an invalid offset range: [" + batch.baseOffset() + ", " + batch.lastOffset() + "] in topic partition " + String.valueOf(topicPartition));
            }
            long count = batch.countOrNull().intValue();
            if (count <= 0L) {
                metricsRecorder.recordInvalidOffset();
                throw new InvalidRecordException("Invalid reported count for record batch: " + count + " in topic partition " + String.valueOf(topicPartition));
            }
            if (countFromOffsets != count) {
                metricsRecorder.recordInvalidOffset();
                throw new InvalidRecordException("Inconsistent batch offset range [" + batch.baseOffset() + ", " + batch.lastOffset() + "] and count of records " + count + " in topic partition " + String.valueOf(topicPartition));
            }
            if (batch.isControlBatch()) {
                metricsRecorder.recordInvalidOffset();
                throw new InvalidRecordException("Clients are not allowed to write control records in topic partition " + String.valueOf(topicPartition));
            }
            if (batch.hasProducerId() && batch.baseSequence() < 0) {
                metricsRecorder.recordInvalidSequence();
                throw new InvalidRecordException("Invalid sequence number " + batch.baseSequence() + " in record batch with producerId " + batch.producerId() + " in topic partition " + String.valueOf(topicPartition));
            }
        }
    }

    public static Optional<ApiRecordError> validateRecord(RecordBatch batch, TopicPartition topicPartition, Record record, int recordIndex, long now, TimestampType timestampType, long timestampBeforeMaxMs, long timestampAfterMaxMs, boolean compactedTopic, Iterable<RecordInterceptor> interceptors, MetricsRecorder metricsRecorder) {
        Optional<ApiRecordError> error;
        if (!record.hasMagic(batch.magic())) {
            metricsRecorder.recordInvalidMagic();
            return Optional.of(new ApiRecordError(Errors.INVALID_RECORD, new ProduceResponse.RecordError(recordIndex, "Record " + String.valueOf(record) + "'s magic does not match outer magic " + batch.magic() + " in topic partition " + String.valueOf(topicPartition))));
        }
        if (batch.magic() <= 1 && batch.isCompressed()) {
            try {
                record.ensureValid();
            }
            catch (CorruptRecordException e) {
                metricsRecorder.recordInvalidChecksums();
                throw e;
            }
            catch (InvalidRecordException e) {
                metricsRecorder.recordInvalidChecksums();
                throw new CorruptRecordException(e.getMessage() + " in topic partition " + String.valueOf(topicPartition));
            }
        }
        if ((error = LogValidator.validateKey(record, recordIndex, topicPartition, compactedTopic, metricsRecorder)).isPresent()) {
            return error;
        }
        error = LogValidator.validateTimestamp(batch, record, recordIndex, now, timestampType, timestampBeforeMaxMs, timestampAfterMaxMs, metricsRecorder);
        if (error.isPresent()) {
            return error;
        }
        return LogValidator.validateInterceptors(record, recordIndex, topicPartition, interceptors, metricsRecorder);
    }

    private static Optional<ApiRecordError> validateKey(Record record, int recordIndex, TopicPartition topicPartition, boolean compactedTopic, MetricsRecorder metricsRecorder) {
        if (compactedTopic && !record.hasKey()) {
            metricsRecorder.recordNoKeyCompactedTopic();
            return Optional.of(new ApiRecordError(Errors.INVALID_RECORD, new ProduceResponse.RecordError(recordIndex, "Compacted topic cannot accept message without key in topic partition " + String.valueOf(topicPartition))));
        }
        return Optional.empty();
    }

    public static Optional<ApiRecordError> validateTimestamp(RecordBatch batch, Record record, int recordIndex, long now, TimestampType timestampType, long timestampBeforeMaxMs, long timestampAfterMaxMs, MetricsRecorder metricsRecorder) {
        boolean isFutureTimeStamp;
        long timestampDiff = Math.abs(record.timestamp() - now);
        boolean bl = isFutureTimeStamp = record.timestamp() > now;
        if (timestampType == TimestampType.CREATE_TIME && record.timestamp() != -1L) {
            if (timestampDiff > TimeUnit.HOURS.toMillis(1L)) {
                if (isFutureTimeStamp) {
                    metricsRecorder.recordFutureTimeDiffMoreThanOneHour();
                } else {
                    metricsRecorder.recordPastTimeDiffMoreThanOneHour();
                }
            }
            if (timestampDiff > TimeUnit.DAYS.toMillis(1L)) {
                if (isFutureTimeStamp) {
                    metricsRecorder.recordFutureTimeDiffMoreThanOneDay();
                } else {
                    metricsRecorder.recordPastTimeDiffMoreThanOneDay();
                }
            }
            if (timestampDiff > TimeUnit.DAYS.toMillis(365L)) {
                if (isFutureTimeStamp) {
                    metricsRecorder.recordFutureTimeDiffMoreThanOneYear();
                } else {
                    metricsRecorder.recordPastTimeDiffMoreThanOneYear();
                }
            }
        }
        if (timestampType == TimestampType.CREATE_TIME && record.timestamp() != -1L) {
            if (LogValidator.recordHasInvalidTimestamp(record, now, timestampBeforeMaxMs, timestampAfterMaxMs)) {
                return Optional.of(new ApiRecordError(Errors.INVALID_TIMESTAMP, new ProduceResponse.RecordError(recordIndex, "Timestamp " + record.timestamp() + " of message with offset " + record.offset() + " is out of range. The timestamp should be within [" + (now - timestampBeforeMaxMs) + ", " + (now + timestampAfterMaxMs) + "]")));
            }
        } else if (batch.timestampType() == TimestampType.LOG_APPEND_TIME) {
            return Optional.of(new ApiRecordError(Errors.INVALID_TIMESTAMP, new ProduceResponse.RecordError(recordIndex, "Invalid timestamp type in message " + String.valueOf(record) + ". Producer should not set timestamp type to LogAppendTime.")));
        }
        return Optional.empty();
    }

    private static boolean recordHasInvalidTimestamp(Record record, long now, long timestampBeforeMaxMs, long timestampAfterMaxMs) {
        long timestampDiff = now - record.timestamp();
        return timestampDiff > timestampBeforeMaxMs || -1L * timestampDiff > timestampAfterMaxMs;
    }

    private static Optional<ApiRecordError> validateRecordCompression(CompressionType sourceCompression, int recordIndex, Record record) {
        if (sourceCompression != CompressionType.NONE && record.isCompressed()) {
            return Optional.of(new ApiRecordError(Errors.INVALID_RECORD, new ProduceResponse.RecordError(recordIndex, "Compressed outer record should not have an inner record with a compression attribute set: " + String.valueOf(record))));
        }
        return Optional.empty();
    }

    public static Optional<ApiRecordError> validateInterceptors(Record record, int batchIndex, TopicPartition topicPartition, Iterable<RecordInterceptor> interceptors, MetricsRecorder metricsRecorder) {
        for (RecordInterceptor interceptor : interceptors) {
            RecordInterceptor.RecordInterceptorResponse interceptorResponse = interceptor.onAppend(topicPartition, record);
            if (interceptorResponse == RecordInterceptor.RecordInterceptorResponse.REJECT) {
                metricsRecorder.recordRejectedRecords(topicPartition.topic(), interceptor.getClass().getName());
                return Optional.of(new ApiRecordError(Errors.INVALID_RECORD, new ProduceResponse.RecordError(batchIndex, "Record " + String.valueOf(record) + " is rejected by the record interceptor " + interceptor.getClass().getName())));
            }
            if (interceptorResponse != RecordInterceptor.RecordInterceptorResponse.RETRY) continue;
            metricsRecorder.recordRetriedRecords(topicPartition.topic(), interceptor.getClass().getName());
            return Optional.of(new ApiRecordError(Errors.UNKNOWN_TOPIC_OR_PARTITION, new ProduceResponse.RecordError(batchIndex, "Record " + String.valueOf(record) + " is validated fail with a retriable error by the record interceptor " + interceptor.getClass().getName())));
        }
        return Optional.empty();
    }

    public static void processRecordErrors(List<ApiRecordError> recordErrors) {
        if (!recordErrors.isEmpty()) {
            List<ProduceResponse.RecordError> errors = recordErrors.stream().map(e -> e.recordError).collect(Collectors.toList());
            if (recordErrors.stream().anyMatch(e -> e.apiError == Errors.INVALID_TIMESTAMP)) {
                throw new RecordValidationException((ApiException)new InvalidTimestampException("One or more records have been rejected due to invalid timestamp"), errors);
            }
            if (recordErrors.stream().anyMatch(e -> e.apiError == Errors.INVALID_RECORD)) {
                throw new RecordValidationException((ApiException)new InvalidRecordException("One or more records have been rejected due to " + errors.size() + " record errors in total, and only showing the first three errors at most: " + String.valueOf(errors.subList(0, Math.min(errors.size(), 3)))), errors);
            }
            if (recordErrors.stream().anyMatch(e -> e.apiError == Errors.UNKNOWN_TOPIC_OR_PARTITION)) {
                throw new RecordValidationException((ApiException)new UnknownTopicOrPartitionException("One or more records validation failed due to the schema is not cached, please retry the records"), errors);
            }
            throw new RecordValidationException((ApiException)new InvalidRecordException("One or more records has been rejected due to unknown errors"), errors);
        }
    }

    public static interface MetricsRecorder {
        public void recordInvalidMagic();

        public void recordInvalidOffset();

        public void recordInvalidSequence();

        public void recordInvalidChecksums();

        public void recordNoKeyCompactedTopic();

        public void recordRejectedRecords(String var1, String var2);

        public void recordRetriedRecords(String var1, String var2);

        public void recordPastTimeDiffMoreThanOneHour();

        public void recordPastTimeDiffMoreThanOneDay();

        public void recordPastTimeDiffMoreThanOneYear();

        public void recordFutureTimeDiffMoreThanOneHour();

        public void recordFutureTimeDiffMoreThanOneDay();

        public void recordFutureTimeDiffMoreThanOneYear();
    }

    public static class ValidationResult {
        public final MemoryRecords validatedRecords;
        public final long maxTimestampMs;
        public final boolean messageSizeMaybeChanged;
        public final RecordValidationStats recordValidationStats;

        public ValidationResult(MemoryRecords validatedRecords, long maxTimestampMs, boolean messageSizeMaybeChanged, RecordValidationStats recordValidationStats) {
            this.validatedRecords = validatedRecords;
            this.maxTimestampMs = maxTimestampMs;
            this.messageSizeMaybeChanged = messageSizeMaybeChanged;
            this.recordValidationStats = recordValidationStats;
        }
    }

    public static class ApiRecordError {
        final Errors apiError;
        final ProduceResponse.RecordError recordError;

        public ApiRecordError(Errors apiError, ProduceResponse.RecordError recordError) {
            this.apiError = apiError;
            this.recordError = recordError;
        }
    }
}

