/*
 * Decompiled with CFR 0.152.
 */
package io.confluent.ksql.serde.json;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.NullNode;
import com.fasterxml.jackson.databind.node.NumericNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.fasterxml.jackson.databind.ser.std.DateSerializer;
import com.google.common.collect.ImmutableMap;
import io.confluent.ksql.schema.connect.SqlSchemaFormatter;
import io.confluent.ksql.serde.SerdeUtils;
import io.confluent.ksql.serde.json.JsonSerdeUtils;
import io.confluent.ksql.util.DecimalUtil;
import io.confluent.ksql.util.KsqlException;
import java.io.IOException;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.sql.Time;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.common.errors.SerializationException;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.kafka.connect.data.Field;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.SchemaBuilder;
import org.apache.kafka.connect.data.Struct;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class KsqlJsonDeserializer<T>
implements Deserializer<T> {
    private static final Logger LOG = LogManager.getLogger(KsqlJsonDeserializer.class);
    private static final SqlSchemaFormatter FORMATTER = new SqlSchemaFormatter(word -> false, new SqlSchemaFormatter.Option[0]);
    private static final ObjectMapper MAPPER = new ObjectMapper().enable(DeserializationFeature.USE_BIG_DECIMAL_FOR_FLOATS).setNodeFactory(JsonNodeFactory.withExactBigDecimals((boolean)true)).registerModule((Module)new SimpleModule().addSerializer(Time.class, (JsonSerializer)new DateSerializer()));
    private static final Schema STRING_ARRAY = SchemaBuilder.array((Schema)Schema.OPTIONAL_STRING_SCHEMA).build();
    private static final Map<Schema.Type, Function<JsonValueContext, Object>> HANDLERS = ImmutableMap.builder().put((Object)Schema.Type.BOOLEAN, context -> JsonSerdeUtils.toBoolean(context.val)).put((Object)Schema.Type.INT32, KsqlJsonDeserializer::handleInt).put((Object)Schema.Type.INT64, KsqlJsonDeserializer::handleLong).put((Object)Schema.Type.FLOAT64, context -> JsonSerdeUtils.toDouble(context.val)).put((Object)Schema.Type.STRING, KsqlJsonDeserializer::processString).put((Object)Schema.Type.ARRAY, KsqlJsonDeserializer::enforceElementTypeForArray).put((Object)Schema.Type.MAP, KsqlJsonDeserializer::enforceKeyAndValueTypeForMap).put((Object)Schema.Type.STRUCT, KsqlJsonDeserializer::enforceFieldTypesForStruct).put((Object)Schema.Type.BYTES, KsqlJsonDeserializer::enforceValidBytes).build();
    private final Schema schema;
    private final boolean isJsonSchema;
    private final Class<T> targetType;
    private String target = "?";

    KsqlJsonDeserializer(Schema schema, boolean isJsonSchema, Class<T> targetType) {
        this.schema = Objects.requireNonNull(schema, "schema");
        this.isJsonSchema = isJsonSchema;
        this.targetType = Objects.requireNonNull(targetType, "targetType");
        SerdeUtils.throwOnSchemaJavaTypeMismatch(schema, targetType);
    }

    public void configure(Map<String, ?> map, boolean isKey) {
        this.target = isKey ? "key" : "value";
    }

    public T deserialize(String topic, byte[] bytes) {
        try {
            if (bytes == null) {
                return null;
            }
            JsonNode value = this.isJsonSchema ? JsonSerdeUtils.readJsonSR(bytes, MAPPER, JsonNode.class) : MAPPER.readTree(bytes);
            Object coerced = KsqlJsonDeserializer.enforceFieldType("$", new JsonValueContext(value, this.schema));
            if (LOG.isTraceEnabled()) {
                LOG.trace("Deserialized {}. topic:{}, row:{}", (Object)this.target, (Object)topic, coerced);
            }
            return SerdeUtils.castToTargetType(coerced, this.targetType);
        }
        catch (Exception e) {
            if (e instanceof JsonParseException) {
                ((JsonParseException)e).clearLocation();
            }
            throw new SerializationException("Failed to deserialize " + this.target + " from topic: " + topic + ". " + e.getMessage(), (Throwable)e);
        }
    }

    public static ObjectReader jsonReader() {
        return MAPPER.reader();
    }

    private static Object enforceFieldType(String pathPart, JsonValueContext context) {
        if (context.val == null || context.val instanceof NullNode) {
            return null;
        }
        try {
            Function<JsonValueContext, Object> handler = HANDLERS.getOrDefault(context.schema.type(), type -> {
                throw new KsqlException("Type is not supported: " + String.valueOf(type));
            });
            return handler.apply(context);
        }
        catch (CoercionException e) {
            throw new CoercionException(e.getRawMessage(), pathPart + e.getPath(), e);
        }
        catch (Exception e) {
            throw new CoercionException(e.getMessage(), pathPart, e);
        }
    }

    private static Object handleInt(JsonValueContext context) {
        if (context.schema.name() == "org.apache.kafka.connect.data.Time") {
            return JsonSerdeUtils.toTime(context.val);
        }
        if (context.schema.name() == "org.apache.kafka.connect.data.Date") {
            return JsonSerdeUtils.toDate(context.val);
        }
        return JsonSerdeUtils.toInteger(context.val);
    }

    private static Object handleLong(JsonValueContext context) {
        if (context.schema.name() == "org.apache.kafka.connect.data.Timestamp") {
            return JsonSerdeUtils.toTimestamp(context.val);
        }
        return JsonSerdeUtils.toLong(context.val);
    }

    private static String processString(JsonValueContext context) {
        if (context.val instanceof ObjectNode) {
            try {
                return MAPPER.writeValueAsString(MAPPER.treeToValue((TreeNode)context.val, Object.class));
            }
            catch (JsonProcessingException e) {
                throw new KsqlException("Unexpected inability to write value as string: " + String.valueOf(context.val));
            }
        }
        if (context.val instanceof ArrayNode) {
            return KsqlJsonDeserializer.enforceElementTypeForArray(new JsonValueContext(context.val, STRING_ARRAY)).stream().map(Objects::toString).collect(Collectors.joining(", ", "[", "]"));
        }
        return context.val.asText();
    }

    private static Object enforceValidBytes(JsonValueContext context) {
        if (DecimalUtil.isDecimal((Schema)context.schema)) {
            if (context.val instanceof NumericNode) {
                return DecimalUtil.ensureFit((BigDecimal)context.val.decimalValue(), (Schema)context.schema);
            }
            if (context.val instanceof TextNode) {
                return DecimalUtil.ensureFit((BigDecimal)new BigDecimal(context.val.textValue()), (Schema)context.schema);
            }
        } else if (context.val.isTextual()) {
            try {
                return ByteBuffer.wrap(context.val.binaryValue());
            }
            catch (IOException e) {
                throw new IllegalArgumentException("Value is not a valid Base64 encoded string: " + context.val.textValue());
            }
        }
        throw KsqlJsonDeserializer.invalidConversionException(context.val, context.schema);
    }

    private static List<?> enforceElementTypeForArray(JsonValueContext context) {
        if (!(context.val instanceof ArrayNode)) {
            throw KsqlJsonDeserializer.invalidConversionException(context.val, context.schema);
        }
        int idx = 0;
        ArrayNode list = (ArrayNode)context.val;
        ArrayList<Object> array = new ArrayList<Object>(list.size());
        for (JsonNode item : list) {
            Object element = KsqlJsonDeserializer.enforceFieldType("[" + idx++ + "]", new JsonValueContext(item, context.schema.valueSchema()));
            array.add(element);
        }
        return array;
    }

    private static Map<String, Object> enforceKeyAndValueTypeForMap(JsonValueContext context) {
        if (!(context.val instanceof ObjectNode)) {
            throw KsqlJsonDeserializer.invalidConversionException(context.val, context.schema);
        }
        ObjectNode map = (ObjectNode)context.val;
        HashMap<String, Object> ksqlMap = new HashMap<String, Object>(map.size());
        Iterator it = map.fields();
        while (it.hasNext()) {
            Map.Entry e = (Map.Entry)it.next();
            String key = (String)KsqlJsonDeserializer.enforceFieldType("." + (String)e.getKey() + ".key", new JsonValueContext((JsonNode)new TextNode((String)e.getKey()), Schema.OPTIONAL_STRING_SCHEMA));
            Object value = KsqlJsonDeserializer.enforceFieldType("." + (String)e.getKey() + ".value", new JsonValueContext((JsonNode)e.getValue(), context.schema.valueSchema()));
            ksqlMap.put(key, value);
        }
        return ksqlMap;
    }

    private static Struct enforceFieldTypesForStruct(JsonValueContext context) {
        if (!(context.val instanceof ObjectNode)) {
            throw KsqlJsonDeserializer.invalidConversionException(context.val, context.schema);
        }
        Struct columnStruct = new Struct(context.schema);
        ObjectNode jsonFields = (ObjectNode)context.val;
        Map<String, JsonNode> upperCasedFields = KsqlJsonDeserializer.upperCaseKeys(jsonFields);
        for (Field ksqlField : context.schema.fields()) {
            JsonNode fieldValue = jsonFields.get(ksqlField.name());
            if (fieldValue == null) {
                fieldValue = upperCasedFields.get(ksqlField.name());
            }
            Object coerced = KsqlJsonDeserializer.enforceFieldType("." + ksqlField.name(), new JsonValueContext(fieldValue, ksqlField.schema()));
            columnStruct.put(ksqlField.name(), coerced);
        }
        return columnStruct;
    }

    private static Map<String, JsonNode> upperCaseKeys(ObjectNode map) {
        HashMap<String, JsonNode> result = new HashMap<String, JsonNode>(map.size());
        Iterator it = map.fields();
        while (it.hasNext()) {
            Map.Entry entry = (Map.Entry)it.next();
            result.put(((String)entry.getKey()).toUpperCase(), (JsonNode)entry.getValue());
        }
        return result;
    }

    public void close() {
    }

    private static IllegalArgumentException invalidConversionException(Object value, Schema schema) {
        throw JsonSerdeUtils.invalidConversionException(value, FORMATTER.format(schema));
    }

    private static final class JsonValueContext {
        private final Schema schema;
        private final JsonNode val;

        JsonValueContext(JsonNode val, Schema schema) {
            this.schema = Objects.requireNonNull(schema, "schema");
            this.val = val;
        }
    }

    private static final class CoercionException
    extends RuntimeException {
        private final String path;
        private final String message;

        CoercionException(String message, String path, Throwable cause) {
            super(message + ", path: " + path, cause);
            this.message = Objects.requireNonNull(message, "message");
            this.path = Objects.requireNonNull(path, "path");
        }

        public String getRawMessage() {
            return this.message;
        }

        public String getPath() {
            return this.path;
        }
    }
}

