/*
 * Decompiled with CFR 0.152.
 */
package kafka.restore;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import kafka.restore.MessageEmitter;
import kafka.restore.OrchestratorStatus;
import kafka.restore.RestoreConfig;
import kafka.restore.RestoreMetricsManager;
import kafka.restore.RestorePartitionOperatorFactory;
import kafka.restore.RestoreUtil;
import kafka.restore.db.Job;
import kafka.restore.messages.AsyncTaskRequest;
import kafka.restore.messages.KafkaRequest;
import kafka.restore.messages.Message;
import kafka.restore.messages.MessageRequest;
import kafka.restore.messages.MessageResponse;
import kafka.restore.messages.MessageStatusCode;
import kafka.restore.messages.ObjectStoreRequest;
import kafka.restore.schedulers.AbstractAsyncServiceScheduler;
import kafka.restore.schedulers.AsyncServiceSchedulerResultsReceiver;
import kafka.restore.schedulers.AsyncTaskScheduler;
import kafka.restore.schedulers.KafkaConnectionPoolImpl;
import kafka.restore.schedulers.KafkaManager;
import kafka.restore.schedulers.ObjectStoreManager;
import kafka.restore.schedulers.ObjectStorePoolImpl;
import kafka.restore.snapshot.FtpsStateForRestore;
import kafka.restore.snapshot.PointInTimeTierPartitionStateBuilder;
import kafka.restore.snapshot.TierTopicConsumerForRestore;
import kafka.restore.statemachine.StateMachineController;
import kafka.restore.statemachine.api.FiniteStateMachine;
import kafka.restore.statemachine.api.State;
import kafka.restore.statemachine.events.KafkaRestoreEvent;
import kafka.server.KafkaConfig;
import kafka.tier.TopicIdPartition;
import kafka.tier.store.AzureBlockBlobTierObjectStore;
import kafka.tier.store.AzureBlockBlobTierObjectStoreConfig;
import kafka.tier.store.GcsTierObjectStore;
import kafka.tier.store.GcsTierObjectStoreConfig;
import kafka.tier.store.MockInMemoryTierObjectStore;
import kafka.tier.store.MockInMemoryTierObjectStoreConfig;
import kafka.tier.store.S3TierObjectStore;
import kafka.tier.store.S3TierObjectStoreConfig;
import kafka.tier.store.TierObjectStore;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RestoreOrchestrator
implements AsyncServiceSchedulerResultsReceiver,
MessageEmitter {
    private static final Logger LOGGER = LoggerFactory.getLogger(RestoreOrchestrator.class);
    private static final String DEFAULT_FTPS_FILE_DIRECTORY = "/mnt/ftps";
    private int restoreParallelism;
    private volatile OrchestratorStatus status;
    private final ArrayBlockingQueue<Message> responsesQueue;
    private StateMachineController stateMachineController;
    private AsyncTaskScheduler asyncTaskScheduler;
    private KafkaManager kafkaManager;
    private ObjectStoreManager objectStoreManager;
    private final PointInTimeTierPartitionStateBuilder ftpsBuilder;
    private Thread responseConsumerThread;
    private final RestoreMetricsManager restoreMetricsManager;
    private Time time;
    private TierTopicConsumerForRestore tierTopicConsumerForRestore;

    public RestoreOrchestrator(RestoreMetricsManager restoreMetricsManager, Time time) {
        this.restoreMetricsManager = restoreMetricsManager;
        this.time = time;
        this.restoreParallelism = Integer.parseInt(RestoreConfig.getProperty("RESTORE_PARALLELISM"));
        this.responsesQueue = new ArrayBlockingQueue(this.restoreParallelism);
        int kafkaConnectionPoolSize = Math.min(this.restoreParallelism, 20);
        KafkaConnectionPoolImpl kafkaConnectionPool = new KafkaConnectionPoolImpl(this, kafkaConnectionPoolSize, DEFAULT_FTPS_FILE_DIRECTORY, restoreMetricsManager, time);
        this.kafkaManager = new KafkaManager((AsyncServiceSchedulerResultsReceiver)this, kafkaConnectionPool);
        this.asyncTaskScheduler = new AsyncTaskScheduler((AsyncServiceSchedulerResultsReceiver)this, this.restoreParallelism, new RestorePartitionOperatorFactory());
        int objectStorePoolSize = this.restoreParallelism * 10;
        ThreadPoolExecutor objectThreadPool = RestoreUtil.createThreadPool(objectStorePoolSize, 50);
        TierObjectStore tierObjectStore = this.getTierObjectStore();
        ObjectStorePoolImpl objectStorePool = new ObjectStorePoolImpl(this, objectThreadPool, tierObjectStore, restoreMetricsManager, time);
        this.objectStoreManager = new ObjectStoreManager((AsyncServiceSchedulerResultsReceiver)this, objectStorePool);
        this.ftpsBuilder = new PointInTimeTierPartitionStateBuilder(tierObjectStore, objectThreadPool, restoreMetricsManager);
        this.status = OrchestratorStatus.NOT_STARTED;
    }

    @Override
    public void reportServiceSchedulerResponse(MessageResponse response) {
        try {
            this.responsesQueue.put(response);
        }
        catch (InterruptedException ie) {
            LOGGER.warn("interrupted while adding response int responseQueue: " + String.valueOf(response), (Throwable)ie);
        }
    }

    public synchronized boolean startUp() throws Exception {
        if (this.status == OrchestratorStatus.RUNNING) {
            LOGGER.debug("Orchestrator is already running, no action");
            return true;
        }
        if (this.status != OrchestratorStatus.NOT_STARTED && this.status != OrchestratorStatus.SHUTDOWN) {
            LOGGER.error("startUp() can only be called on a service scheduler that has not been started or is shutdown.");
            return false;
        }
        if (!(this.kafkaManager.startUp() && this.asyncTaskScheduler.startUp() && this.objectStoreManager.startUp())) {
            return false;
        }
        this.status = OrchestratorStatus.RUNNING;
        this.startResponseQueueConsumerThread();
        return true;
    }

    public synchronized boolean shutdown() {
        if (this.status == OrchestratorStatus.PAUSED || this.status == OrchestratorStatus.ERROR) {
            this.status = OrchestratorStatus.SHUTDOWN;
            return true;
        }
        if (this.status == OrchestratorStatus.SHUTDOWN) {
            LOGGER.info("RestoreOrchestrator is already shutdown");
            return true;
        }
        if (this.status != OrchestratorStatus.RUNNING) {
            LOGGER.error(String.format("shutdown() can only be called when status is running, paused, or in error state, current status: %s", new Object[]{this.status}));
            return false;
        }
        LOGGER.info("Shutting down RestoreOrchestrator");
        if (!(this.kafkaManager.shutdown() && this.asyncTaskScheduler.shutdown() && this.objectStoreManager.shutdown())) {
            LOGGER.error("Shutdown async service schedulers failed");
            this.status = OrchestratorStatus.ERROR;
            return false;
        }
        try {
            if (this.responseConsumerThread != null && this.responseConsumerThread.isAlive()) {
                this.responseConsumerThread.interrupt();
                this.responseConsumerThread.join();
            }
        }
        catch (InterruptedException e) {
            LOGGER.error("shutdown() was interrupted prior to background thread terminating", (Throwable)e);
            this.status = OrchestratorStatus.ERROR;
            return false;
        }
        if (this.tierTopicConsumerForRestore != null) {
            this.tierTopicConsumerForRestore.shutdown();
        }
        this.status = OrchestratorStatus.SHUTDOWN;
        LOGGER.info("RestoreOrchestrator is shutdown");
        return true;
    }

    public synchronized void forceShutdown() {
        this.kafkaManager.shutdown();
        this.asyncTaskScheduler.shutdown();
        this.objectStoreManager.shutdown();
        this.responseConsumerThread.interrupt();
        this.status = OrchestratorStatus.SHUTDOWN;
    }

    public synchronized boolean pause() throws InterruptedException {
        LOGGER.info("Pausing RestoreOrchestrator");
        if (this.status != OrchestratorStatus.RUNNING) {
            LOGGER.error("pause() can only be called on service scheduler that is running.");
            return false;
        }
        this.status = OrchestratorStatus.PAUSING;
        try {
            if (this.responseConsumerThread != null && this.responseConsumerThread.isAlive()) {
                this.responseConsumerThread.interrupt();
                this.responseConsumerThread.join();
            }
        }
        catch (InterruptedException e) {
            LOGGER.error("Pause was interrupted prior to completing", (Throwable)e);
            this.status = OrchestratorStatus.ERROR;
            return false;
        }
        if (!(this.kafkaManager.pause() && this.asyncTaskScheduler.pause() && this.objectStoreManager.pause())) {
            LOGGER.error("Pause async service schedulers failed");
            this.status = OrchestratorStatus.ERROR;
            return false;
        }
        this.status = OrchestratorStatus.PAUSED;
        LOGGER.info("RestoreOrchestrator is Paused");
        return true;
    }

    public synchronized boolean resume() {
        if (this.status != OrchestratorStatus.PAUSED) {
            LOGGER.error("resume() can only be called on service scheduler in PAUSED state.");
            return false;
        }
        LOGGER.info("Resuming RestoreOrchestrator");
        if (!(this.kafkaManager.resume() && this.asyncTaskScheduler.resume() && this.objectStoreManager.resume())) {
            LOGGER.error("resume async service schedulers failed");
            this.status = OrchestratorStatus.ERROR;
            return false;
        }
        this.status = OrchestratorStatus.RUNNING;
        this.startResponseQueueConsumerThread();
        LOGGER.info("RestoreOrchestrator is running");
        return true;
    }

    public synchronized OrchestratorStatus getStatus() {
        return this.status;
    }

    private void startResponseQueueConsumerThread() {
        LOGGER.info("start ResponseQueue Consumer Thread");
        this.responseConsumerThread = new Thread(this::consumeMessagesFromResponseQueue);
        this.responseConsumerThread.start();
    }

    private void consumeMessagesFromResponseQueue() {
        LOGGER.info("start consuming response queue");
        while (this.status == OrchestratorStatus.RUNNING) {
            try {
                Message message = this.responsesQueue.take();
                LOGGER.debug("take one message from response queue: " + String.valueOf(message));
                if (message instanceof MessageRequest) {
                    this.submitRequest((MessageRequest)message);
                    continue;
                }
                if (!(message instanceof MessageResponse)) continue;
                this.handleResponse((MessageResponse)message);
            }
            catch (InterruptedException ie) {
                LOGGER.info("ResponseConsumer loop interrupted, exit the loop.", (Throwable)ie);
                break;
            }
            catch (Exception e) {
                LOGGER.error("Unexpected exception caught in response consuming loop.", (Throwable)e);
            }
        }
        LOGGER.info("Stop consuming response queue");
    }

    private void handleResponse(MessageResponse response) {
        TopicPartition tp = new TopicPartition(response.getTopic(), response.getPartition());
        FiniteStateMachine fsm = this.stateMachineController.getFiniteStateMachineByTopicPartition(tp);
        if (fsm == null) {
            LOGGER.warn("FiniteStateMachine not found for " + String.valueOf(tp) + ", skip.");
            return;
        }
        State currentState = fsm.fire(new KafkaRestoreEvent(response));
        if (currentState == State.FAILED) {
            LOGGER.warn(String.format("[%s]: restore failed", tp));
            this.stateMachineController.moveToFailMap(tp);
            if (this.stateMachineController.inProgressCount() < this.restoreParallelism) {
                this.pickOnePartitionToStartRestore();
            }
        } else if (currentState == State.END_STATE) {
            this.stateMachineController.moveToCompleteSet(tp);
            LOGGER.info(String.format("[%s]: restore completed", tp));
            if (this.stateMachineController.inProgressCount() < this.restoreParallelism) {
                this.pickOnePartitionToStartRestore();
            }
        }
        if (currentState == State.FAILED || currentState == State.END_STATE) {
            fsm.cleanup();
        }
    }

    @Override
    public void submitRequest(MessageRequest request) {
        boolean bSuccess = false;
        AbstractAsyncServiceScheduler serviceScheduler = null;
        if (request instanceof KafkaRequest) {
            serviceScheduler = this.kafkaManager;
        } else if (request instanceof AsyncTaskRequest) {
            serviceScheduler = this.asyncTaskScheduler;
        } else if (request instanceof ObjectStoreRequest) {
            serviceScheduler = this.objectStoreManager;
        } else {
            LOGGER.warn(String.format("[%s]: Request type not support: %s, skip", request.getTopicPartition(), request.getClass().getSimpleName()));
            bSuccess = true;
        }
        if (serviceScheduler != null) {
            LOGGER.info(String.format("[%s]: submit request to scheduler: %s", request.getTopicPartition(), request));
            MessageStatusCode status = serviceScheduler.submitRequest(request);
            if (MessageStatusCode.SCHEDULED == status) {
                bSuccess = true;
            } else {
                LOGGER.warn(String.format("[%s]: submit request to scheduler failed, maybe because request queue full.", request.getTopicPartition()));
            }
        }
        if (!bSuccess) {
            try {
                LOGGER.info(String.format("[%s]: submit request to responseQueue: %s", request.getTopicPartition(), request));
                this.responsesQueue.put(request);
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    public StateMachineController stateMachineController() {
        return this.stateMachineController;
    }

    public void setKafkaManager(KafkaManager kafkaManager) {
        this.kafkaManager = kafkaManager;
    }

    public void setAsyncTaskScheduler(AsyncTaskScheduler asyncTaskScheduler) {
        this.asyncTaskScheduler = asyncTaskScheduler;
    }

    public void setObjectStoreManager(ObjectStoreManager objectStoreManager) {
        this.objectStoreManager = objectStoreManager;
    }

    public void runRestoreJob(Job job) {
        Thread buildFtpsThread = new Thread(() -> {
            HashMap<TopicIdPartition, FtpsStateForRestore> ftpsStateForRestoreMap = new HashMap();
            try {
                job.status = Job.JobStatus.IN_PROGRESS;
                ftpsStateForRestoreMap = this.buildFtpsFiles(job);
            }
            catch (Exception e) {
                LOGGER.error("build ftps files for restore failed", (Throwable)e);
            }
            this.mayStartTierTopicConsumer(ftpsStateForRestoreMap, job);
            this.stateMachineController = new StateMachineController(job, this, this.restoreMetricsManager, ftpsStateForRestoreMap);
            this.restoreMetricsManager.startRestoreRecord();
            int startPartitionCount = Math.min(this.restoreParallelism, this.stateMachineController.waitingCount());
            for (int i = 0; i < startPartitionCount; ++i) {
                this.pickOnePartitionToStartRestore();
            }
        });
        buildFtpsThread.start();
    }

    private void mayStartTierTopicConsumer(Map<TopicIdPartition, FtpsStateForRestore> ftpsStateForRestoreMap, Job job) {
        if (ftpsStateForRestoreMap.size() == 0) {
            return;
        }
        Map<Integer, Long> lastMaterializedOffsets = PointInTimeTierPartitionStateBuilder.calculateTierTopicLastMaterializedOffsets(ftpsStateForRestoreMap);
        if (job.getBrokerConnectionString() == null) {
            LOGGER.warn("no broker connection string found in job, can't initialize TierTopicConsumerForRestore");
            return;
        }
        this.tierTopicConsumerForRestore = new TierTopicConsumerForRestore(job.getBrokerConnectionString(), lastMaterializedOffsets, ftpsStateForRestoreMap.keySet());
        this.tierTopicConsumerForRestore.initialize();
        this.tierTopicConsumerForRestore.start();
        ftpsStateForRestoreMap.forEach((tierIdPartition, ftpsStateForRestore) -> {
            ftpsStateForRestore.liveConsumerRecords = this.tierTopicConsumerForRestore.getRecords((TopicIdPartition)tierIdPartition);
        });
    }

    private void pickOnePartitionToStartRestore() {
        this.restoreMetricsManager.record("RestorePartitionsWaitingCount", this.stateMachineController.waitingCount());
        this.restoreMetricsManager.record("RestorePartitionsInProgressCount", this.stateMachineController.inProgressCount());
        this.restoreMetricsManager.record("RestorePartitionsFailedCount", this.stateMachineController.failedCount());
        this.restoreMetricsManager.record("RestorePartitionsCompletedCount", this.stateMachineController.completedCount());
        FiniteStateMachine fsm = this.stateMachineController.pickOneNewPartitionToStartRestore();
        if (fsm != null) {
            MessageRequest request = StateMachineController.buildPreConditionCheckRequest(fsm);
            this.submitRequest(request);
            this.restoreMetricsManager.update("RestoreStarted", 1L);
        }
    }

    private Map<TopicIdPartition, FtpsStateForRestore> buildFtpsFiles(Job job) throws IOException, InterruptedException {
        if (job.partitionRestoreContextMap == null || job.partitionRestoreContextMap.size() == 0) {
            LOGGER.info("partition list is empty");
            return Collections.emptyMap();
        }
        long startMs = this.time.hiResClockMs();
        Map<TopicIdPartition, FtpsStateForRestore> ftpsStatesBuiltFromSnapshots = this.ftpsBuilder.buildFtpsFromSnapshot(job.partitionRestoreContextMap);
        long timeToBuildAllFTPSStatesFromSnapshots = this.time.hiResClockMs() - startMs;
        this.restoreMetricsManager.record("RestoreTimeToBuildFtpsStatesFromSnapshots", timeToBuildAllFTPSStatesFromSnapshots);
        return ftpsStatesBuiltFromSnapshots;
    }

    private TierObjectStore getTierObjectStore() {
        KafkaConfig config = RestoreConfig.kafkaConfig();
        TierObjectStore tierObjectStore = null;
        if (config != null && config.confluentConfig() != null && config.confluentConfig().tierFeature().booleanValue()) {
            switch (config.confluentConfig().tierBackend()) {
                case "S3": {
                    tierObjectStore = new S3TierObjectStore(new S3TierObjectStoreConfig(Optional.empty(), config), Optional.empty(), Time.SYSTEM, null);
                    break;
                }
                case "GCS": {
                    tierObjectStore = new GcsTierObjectStore(Time.SYSTEM, null, new GcsTierObjectStoreConfig(Optional.empty(), config), Optional.empty());
                    break;
                }
                case "AzureBlockBlob": {
                    tierObjectStore = new AzureBlockBlobTierObjectStore(new AzureBlockBlobTierObjectStoreConfig(Optional.empty(), config), Time.SYSTEM, null);
                    break;
                }
                case "mock": {
                    tierObjectStore = new MockInMemoryTierObjectStore(Time.SYSTEM, null, new MockInMemoryTierObjectStoreConfig(Optional.empty(), config));
                    break;
                }
                default: {
                    throw new IllegalStateException(String.format("Unknown TierObjectStore type: %s", config.confluentConfig().tierBackend()));
                }
            }
        }
        return tierObjectStore;
    }
}

