/*
 * Decompiled with CFR 0.152.
 */
package kafka.restore.schedulers;

import io.confluent.kafka.storage.checksum.Algorithm;
import io.confluent.kafka.storage.checksum.CheckedFileIO;
import java.io.File;
import java.nio.file.StandardOpenOption;
import java.time.Duration;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadPoolExecutor;
import kafka.restore.RestoreMetricsManager;
import kafka.restore.messages.CopyObjectInStoreRequest;
import kafka.restore.messages.CopyObjectInStoreResponse;
import kafka.restore.messages.ListObjectsInStoreRequest;
import kafka.restore.messages.ListObjectsInStoreResponse;
import kafka.restore.messages.MessageResult;
import kafka.restore.messages.MessageStatusCode;
import kafka.restore.messages.ObjectStoreRequest;
import kafka.restore.messages.RestoreObjectsInStoreRequest;
import kafka.restore.messages.UploadFtpsToStoreRequest;
import kafka.restore.messages.UploadFtpsToStoreResponse;
import kafka.restore.operators.OperatorUtil;
import kafka.restore.operators.SegmentStateAndPath;
import kafka.restore.schedulers.AsyncServiceSchedulerResultsReceiver;
import kafka.restore.schedulers.CompletableFutureRetryer;
import kafka.restore.schedulers.Constants;
import kafka.restore.schedulers.CopyObjectOperator;
import kafka.restore.schedulers.ListVersionsOperator;
import kafka.restore.schedulers.ObjectStorePool;
import kafka.restore.schedulers.RestoreObjectMultipleResponseHandler;
import kafka.restore.schedulers.RetryableException;
import kafka.tier.TopicIdPartition;
import kafka.tier.exceptions.TierSnapshotChecksumValidationFailedException;
import kafka.tier.state.ChecksumUtils;
import kafka.tier.state.FileTierPartitionState;
import kafka.tier.state.Header;
import kafka.tier.store.TierObjectStore;
import kafka.tier.store.objects.ObjectType;
import kafka.tier.store.objects.metadata.TierStateRestoreSnapshotMetadata;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ObjectStorePoolImpl
implements ObjectStorePool {
    private static final Logger LOGGER = LoggerFactory.getLogger(ObjectStorePoolImpl.class);
    private final AsyncServiceSchedulerResultsReceiver resultsReceiver;
    private volatile ObjectPoolState state;
    private ThreadPoolExecutor threadPool;
    private final TierObjectStore tierObjectStore;
    private Duration objectStoreRetryWaitInMs = Constants.DEFAULT_OBJECT_STORE_REQUEST_WAIT_BETWEEN_IN_MS;
    private CompletableFutureRetryer retries;
    private final RestoreMetricsManager metrics;
    private final Time time;

    public ObjectStorePoolImpl(AsyncServiceSchedulerResultsReceiver resultsReceiver, ThreadPoolExecutor threadPool, TierObjectStore tierObjectStore, RestoreMetricsManager restoreMetricsManager, Time time) {
        this.resultsReceiver = resultsReceiver;
        this.state = ObjectPoolState.OFF;
        this.threadPool = threadPool;
        this.tierObjectStore = tierObjectStore;
        this.metrics = restoreMetricsManager;
        this.time = time;
    }

    @Override
    public synchronized void startUp() {
        if (this.state == ObjectPoolState.RUNNING) {
            return;
        }
        this.state = ObjectPoolState.RUNNING;
        this.retries = new CompletableFutureRetryer(this.threadPool, this.objectStoreRetryWaitInMs, this.time);
    }

    @Override
    public synchronized void shutdown() {
        if (this.state == ObjectPoolState.OFF) {
            return;
        }
        if (this.threadPool != null) {
            this.threadPool.shutdown();
        }
        if (this.tierObjectStore != null) {
            this.tierObjectStore.close();
        }
        this.state = ObjectPoolState.OFF;
    }

    @Override
    public void submitObjectStoreRequest(ObjectStoreRequest objectStoreRequest) {
        if (objectStoreRequest instanceof RestoreObjectsInStoreRequest) {
            this.submitRestoreObjectsInStoreRequest((RestoreObjectsInStoreRequest)objectStoreRequest);
        } else if (objectStoreRequest instanceof UploadFtpsToStoreRequest) {
            this.submitUploadFtpsToStoreRequest((UploadFtpsToStoreRequest)objectStoreRequest);
        } else {
            throw new UnsupportedOperationException("objectStoreRequest of type " + String.valueOf(objectStoreRequest.getClass()) + " is not of a recognizable request type.");
        }
    }

    protected void submitRestoreObjectsInStoreRequest(RestoreObjectsInStoreRequest request) {
        LOGGER.info(String.format("[%s]: restoring %s segments", request.getTopicPartition(), request.getSegmentStateAndPathMap().size()));
        this.threadPool.execute(() -> {
            RestoreObjectMultipleResponseHandler multipleResponseHandler = new RestoreObjectMultipleResponseHandler(request, this.resultsReceiver, this.metrics);
            request.getSegmentStateAndPathMap().forEach((segmentId, segmentStateAndPath) -> {
                ListObjectsInStoreRequest listObjectsInStoreRequest = new ListObjectsInStoreRequest(0, request.getTopic(), request.getPartition(), (UUID)segmentId, (SegmentStateAndPath)segmentStateAndPath);
                ListVersionsOperator listObjectOperator = new ListVersionsOperator(listObjectsInStoreRequest, multipleResponseHandler, this.tierObjectStore, this.threadPool);
                CompletableFuture versionMapFuture = this.retries.withTimedRetries(listObjectOperator::listVersions, t -> t.getClass().equals(RetryableException.class), 10, this.metrics != null ? arg_0 -> ((Sensor)this.metrics.restoreListVersionsMs()).record(arg_0) : null, this.metrics != null ? this.metrics::recordRestoreListVersionsFailures : null);
                versionMapFuture.exceptionally(throwable -> {
                    LOGGER.error(String.format("[%s]: listVersions call completed with Exception", request.getTopicPartition()), throwable);
                    multipleResponseHandler.addReceivedResponse(new ListObjectsInStoreResponse(0, listObjectsInStoreRequest.getTopic(), listObjectsInStoreRequest.getPartition(), listObjectsInStoreRequest.getUuid(), MessageStatusCode.ILLEGAL_STATE_ERROR, MessageResult.FAILURE, (UUID)segmentId, null));
                    return null;
                });
                versionMapFuture.thenCompose(versionMap -> CompletableFuture.runAsync(() -> versionMap.forEach((object, version) -> {
                    LOGGER.debug(String.format("[%s]: handling: %s with version: %s", request.getTopicPartition(), object, version));
                    CopyObjectInStoreRequest copyObjectInStoreRequest = new CopyObjectInStoreRequest(0, request.getTopic(), request.getPartition(), (UUID)segmentId, (String)object, (String)version, (SegmentStateAndPath)segmentStateAndPath);
                    CopyObjectOperator copyObjectOperator = new CopyObjectOperator(copyObjectInStoreRequest, multipleResponseHandler, this.tierObjectStore, this.threadPool);
                    this.retries.withTimedRetries(copyObjectOperator::restoreObjectByCopy, t -> t.getClass().equals(RetryableException.class), 10, this.metrics != null ? arg_0 -> ((Sensor)this.metrics.restoreObjectCopyMs()).record(arg_0) : null, this.metrics != null ? this.metrics::recordRestoreObjectCopyFailures : null).exceptionally(throwable -> {
                        LOGGER.error(String.format("[%s]: restoreObjectByCopy call completed with Exception", request.getTopicPartition()), throwable);
                        multipleResponseHandler.addReceivedResponse(new CopyObjectInStoreResponse(0, copyObjectInStoreRequest.getTopic(), copyObjectInStoreRequest.getPartition(), copyObjectInStoreRequest.getUuid(), MessageStatusCode.INTERNAL_ERROR, MessageResult.FAILURE, (UUID)segmentId, copyObjectInStoreRequest.getObjectPath()));
                        return null;
                    });
                }), this.threadPool));
            });
        });
    }

    protected void submitUploadFtpsToStoreRequest(UploadFtpsToStoreRequest request) {
        this.threadPool.execute(() -> {
            LOGGER.debug(String.format("[%s]: upload ftps file at: %s", request.getTopicPartition(), request.getFtpsFilePath()));
            File ftpsFile = new File(request.getFtpsFilePath());
            TopicPartition topicPartition = request.getTopicPartition();
            try {
                ObjectStorePoolImpl.uploadFtps(topicPartition, this.tierObjectStore, ftpsFile);
                LOGGER.debug(String.format("[%s]: uploaded ftps file at: %s success, exit", request.getTopicPartition(), request.getFtpsFilePath()));
                this.reportUploadFtpsToStoreResponse(request, MessageStatusCode.OK, MessageResult.SUCCESS);
            }
            catch (Exception ex) {
                LOGGER.error(String.format("[%s]: Exception when uploading new ftps to object store", topicPartition), (Throwable)ex);
                this.reportUploadFtpsToStoreResponse(request, MessageStatusCode.INTERNAL_ERROR, MessageResult.FAILURE);
            }
        });
    }

    private static void uploadFtps(TopicPartition topicPartition, TierObjectStore objectStore, File file) throws Exception {
        try (CheckedFileIO fileChannel = CheckedFileIO.open(file.toPath(), StandardOpenOption.READ);){
            Algorithm ftpsAlgorithm = ChecksumUtils.tierStateFileAlgorithm(file.toPath());
            Optional<Header> headerOpt = FileTierPartitionState.readHeader(fileChannel);
            if (!headerOpt.isPresent()) {
                throw new Exception(String.format("[%s]: Header is not present for TierPartitionState being recovered", topicPartition));
            }
            Header header = headerOpt.get();
            String hash = OperatorUtil.computeMd5(fileChannel);
            TopicIdPartition topicIdPartition = new TopicIdPartition(topicPartition.topic(), header.topicId(), topicPartition.partition());
            TierStateRestoreSnapshotMetadata restoreMetadata = new TierStateRestoreSnapshotMetadata(topicIdPartition, header.startOffset(), header.endOffset(), hash, ftpsAlgorithm);
            LOGGER.debug(String.format("[%s]: restore metadata: %s", topicPartition, restoreMetadata));
            ObjectStorePoolImpl.ensureFileNotCorrupted(file.getAbsolutePath(), fileChannel);
            objectStore.putObject(restoreMetadata, file, ObjectType.TIER_STATE_SNAPSHOT);
        }
    }

    private static void ensureFileNotCorrupted(String path, CheckedFileIO fileChannel) {
        try {
            if (!fileChannel.validate()) {
                throw new TierSnapshotChecksumValidationFailedException(String.format("FTPS file: %s is corrupted. Corruption has been detected before upload to object store during restore.", path), null);
            }
        }
        catch (Throwable t) {
            throw new TierSnapshotChecksumValidationFailedException("Exception during FTPS checksum validation during restore", t);
        }
    }

    private void reportUploadFtpsToStoreResponse(UploadFtpsToStoreRequest request, MessageStatusCode statusCode, MessageResult result) {
        UploadFtpsToStoreResponse response = new UploadFtpsToStoreResponse(0, request.getTopic(), request.getPartition(), request.getUuid(), statusCode, result);
        this.resultsReceiver.reportServiceSchedulerResponse(response);
    }

    public ThreadPoolExecutor threadPool() {
        return this.threadPool;
    }

    private static enum ObjectPoolState {
        RUNNING,
        OFF;

    }
}

